!...|....1....|....2....|....3....|....4....|....5....|....6....|....7....|....8
!*******************************************************************************
!
! Routine:    linear_rsamp
! Purpose:    Performs linear interpolation/extrapolation.
!             Resamples data1(), sampled at locations in table1(), to
!             data2(), sampled at locations in table2(). table1() and table2()
!             may have arbirary spacing.
!
!                      i      i    i   i      o    i  o
! call linear_rsamp (table1,data1,n1,table2,data2,n2,ier)
!
! Arguments:
!     real     table1(n1)   Coordinates for data1() in ascending order
!     real     data1(n1)    Input data values
!     integer  n1           Length of vectors table1() and data1()
!     real     table2(n2)   Coordinates for data2() in ascending order
!     real     data2(n2)    Resampled output data
!     integer  n2           Length of vectors table2() and data2()
!     integer  ier          return 0 for no error, <> 0 if error found
!
! Error conditions:
!      If n1 < 2 or n2 < 1, the routine is aborted with ier=1
!
!-------------------------------------------------------------------------------
      subroutine linear_rsamp (table1,data1,n1, table2,data2,n2, ier)

      implicit none

! Argument declarations
      integer          :: n1, n2, ier
      real             :: table1(n1), data1(n1)
      real             :: table2(n2), data2(n2)

! Local variables
      integer          :: i, j, nwrk
      integer          :: j1, j2
      integer,allocatable :: indx(:)
      real,allocatable :: coef(:,:)
      real,allocatable :: wrk_tbl(:), wrk_dat(:)
      real             :: dd1, dt1

!-------------------------------------------------------------------------------
! Done with declarations
!-------------------------------------------------------------------------------
! Check if the problem is properly posed

      ier = 0
      if ( n1<2 .or. n2<1 ) then
        ier = 1
        return
      endif


! Allocate space for local variables

      allocate(wrk_tbl(n1))
      allocate(wrk_dat(n1))
      allocate(indx(n2))
      allocate(coef(n2,2))


! Initialization a list of indices and coefficients

      ! Squeeze out duplicate values of table1()
      j = 0
      do i = 1, n1-1
        if ( table1(i) == table1(i+1) ) cycle
        j = j+1
        wrk_tbl(j) = table1(i)
        wrk_dat(j) = data1(i)
      enddo
      nwrk = j+1
      wrk_tbl(nwrk) = table1(n1)
      wrk_dat(nwrk) = data1(n1)

      ! Check again if the problem is properly posed
      if ( nwrk < 2 ) then
        ier = 2
        return
      endif

      ! Fill indx()
      j = 2
      do i = 1, n2
        do 
          if ( table2(i) <= wrk_tbl(j) .or. j > (nwrk-1) ) exit
          j = j + 1
          cycle
        enddo
        indx(i) = j - 1
      enddo

      ! Calculate values of coef(), basically slope and intercept.
      do i = 1, n2

        j1 = indx(i)
        j2 = j1 + 1

        dd1 = wrk_dat(j2)-wrk_dat(j1)
        dt1 = wrk_tbl(j2)-wrk_tbl(j1)

        coef(i,1) = dd1/dt1
        coef(i,2) = wrk_dat(j1) - coef(i,1)*wrk_tbl(j1)

      enddo


! Interpolate data1() onto data2()

      data2(1:n2) = coef(1:n2,1)*table2(1:n2) + coef(1:n2,2)


! Clean up work space

      deallocate(wrk_tbl)
      deallocate(wrk_dat)
      deallocate(indx)
      deallocate(coef)

! All done

      return
      end subroutine linear_rsamp
