c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C Copyright 2002, Allied Geophysics, Inc. All Rights Reserved          *
C***********************************************************************
C Portions of this code and/or subroutines  used by this code are      *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  Main Routine -----------------------
c
c lmascii2usp - analize ascii .dat files barfed out by landmark or
c               use the information from the analysis in converting
c               horizons to a USP grid
c
c  Program Description:
c
c    lmascii2usp reads the ascii format lamdmark uses to dump maps as
c     .dat files and, based on user parameters, places the points on
c     a regular grid and writes the grid as a USP record. The grid
c     used by lmascii2usp is defined by starting in-line and cross-line
c     numbers, in-line and cross-line numbering increments, and the
c     total number of in-lines and cross-lines to grid.
c
c    Grid nodes that aren't represented in the .dat file are assigned
c     an embedded mask (emask) that can be anything the user wants.
c
c    To get the limits of in-lines and cross-lines the user can the
c     -I flag to inquire about the input file.
c
c  Program History:
c
c    Jan 22, 2002 - changes open statments to remove read,write stuff
c                    as would not compile with pgf77 --- PGAG
c    Jan 21, 2002 - added implicit none, found ixcnt error, should have 
c                   been xlcnt.  Made the change - PGAG
c    Jan  7, 2002 - call alloclun() to get unit for -P option
c    Dec 27, 2001 - minor cleanup on code and man page
c    Oct 30, 2001 - adopted emask as name for embedded mask
c    Sep 19, 2001 - original version
c
c get machine dependent parameters 

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c declare standard USP variables 
      integer     ilhd(3*SZLNHD)
     
      integer     luin , luout
      integer     obytes, lbytes, lbyout
      integer     argis, jerr

      character   ntap*255, otap*255, ptap*255, name*11

      logical     verbose

c Program Specific _ dynamic memory variables
      integer TrcSzOut,RecSzOut
      integer TotalMem
      integer errcd1, errcd2, abort

      integer TrcOut(2)
      real    RecOut(2)

      pointer (ptr_TrcOut, TrcOut)
      pointer (ptr_RecOut, RecOut)

c Program Specific - static memory variables
      integer origin_iline,delta_iline,n_iline
      integer origin_xline,delta_xline,n_xline
      integer n1, n2, n3, nbyt


      real    emask

      character cardin*80
      real    iline_in,xline_in, rjunk1,rjunk2, zdata

      integer hdr_index, tr_index
      integer ii, jj, kk, itrc, ipt

      integer lupicks
      logical lpicksout,lmultivalued

c variables used when reporting is requested
      logical lreport, lgrid, lfound
      integer ilcnt, xlcnt, ipt2

c     These are allocated (when needed) to SZLNHD and may
c     someday be too small
      real    ilines(2),xlines(2)
      pointer (ptr_ilines, ilines)
      pointer (ptr_xlines, xlines)

      real    iline_min, iline_max
      real    xline_min, xline_max
      real    zdata_min, zdata_max
      real    iline_delta, xline_delta
      real    iline_last, xline_last

      character card1*(80)

c Variables for access to RecNum and TrcNum trace headers
      integer ifmt_RecNum,l_RecNum,ln_RecNum
      integer ifmt_TrcNum,l_TrcNum,ln_TrcNum

c Initialize variables
      data abort/0/
      data name/"LMASCII2USP"/
      data iline_last/-1.0e+37/,xline_last/-1.0e+37/

c give command line help if requested
      if ( argis('-?')    .gt. 0 .or. 
     :     argis('-h')    .gt. 0 .or.
     :     argis('-help') .gt. 0 ) then
        call help(name)
        stop
      endif

c open printout file
#include <f77/open.h>

c get command line input parameters
      call cmdln
     :       (ntap, otap, ptap,
     :        origin_iline,delta_iline,n_iline,
     :        origin_xline,delta_xline,n_xline,
     :        emask, name, lreport, lgrid, verbose)

c open input
      if (ntap .ne. ' ') then
        luin = LUCARD
        open (unit=luin, access='SEQUENTIAL',
     :        file=ntap, status='OLD', iostat=errcd1)
        if (errcd1 .gt. 0 ) then
          write(LER,*)name,': Failed to open input file ',ntap
          write(LER,*)'FATAL'
          stop
        endif
        rewind luin
      else
        luin = LIN
      endif

c Set up for reporting
      IF (lreport) THEN

        TotalMem = 2*SZLNHD*SZSMPD
        call galloc(ptr_ilines, SZLNHD*SZSMPD, errcd1, abort)
        call galloc(ptr_xlines, SZLNHD*SZSMPD, errcd2, abort)

        if (errcd1.ne.0 .or. errcd2.ne.0) then
          write(LERR,*)' '
          write(LERR,*)
     :      'Unable to allocate workspace: ',TotalMem,' bytes'
          write(LERR,*)'FATAL'
          write(LERR,*)' '
          write(LER,*)' '
          write(LER,*)
     :      'Unable to allocate workspace: ',TotalMem,' bytes'
          write(LER,*)'FATAL'
          write(LER,*)' '
          stop
        else
          write(LERR,*)' '
          write(LERR,*)'Allocated ',TotalMem,' bytes of workspace'
          write(LERR,*)' '
        endif

        call vfill (emask,ilines,1,SZLNHD)
        call vfill (emask,xlines,1,SZLNHD)

        iline_min = +1e+37
        iline_max = -1e+37
        xline_min = +1e+37
        xline_max = -1e+37
        zdata_min = +1e+37
        zdata_max = -1e+37
        iline_delta = +1e+37
        xline_delta = +1e+37

      ENDIF

c Set up for gridding and trace output
      IF (lgrid) THEN

c Open the output
        call getln(luout,otap,'w',1)

c Clear the line header
        call vclr (ilhd, 1, 3*SZLNHD)

c get sizes for dynamic allocation and i/o traces
        n1=n_xline
        n2=n_iline
        n3=1

c       for traces
        TrcSzOut = SZTRHD + n1*SZSMPD

c       for arrays
        RecSzOut = n1*n2*SZSMPD

        TotalMem = TrcSzOut + RecSzOut

c       number bytes on input/output
        obytes = TrcSzOut

c dynamic memory allocation:  
        call galloc(ptr_TrcOut, TrcSzOut, errcd1, abort)
        call galloc(ptr_RecOut, RecSzOut, errcd2, abort)
    
        if (errcd1.ne.0 .or. errcd2.ne.0) then
          write(LERR,*)' '
          write(LERR,*)
     :      'Unable to allocate workspace: ',TotalMem,' bytes'
          write(LERR,*)'FATAL'
          write(LERR,*)' '
          write(LER,*)' '
          write(LER,*)
     :      'Unable to allocate workspace: ',TotalMem,' bytes'
          write(LER,*)'FATAL'
          write(LER,*)' '
          stop
        else
          write(LERR,*)' '
          write(LERR,*)'Allocated ',TotalMem,' bytes of workspace'
          write(LERR,*)' '
        endif

c initialize memory
        call vclr (TrcOut, 1, ITRWRD+n1)
        call vfill (emask, RecOut, 1, n1*n2)

c modify line header to reflect output sizes
        call savew(ilhd,'NumSmp', n1,LINHED)
        call savew(ilhd,'NumTrc', n2,LINHED)
        call savew(ilhd,'NumRec', n3,LINHED)
        call savew(ilhd,'SmpInt',  1,LINHED)
        call savew(ilhd,'Format',  3,LINHED)
        call savew(ilhd,'UnitSc',1.0,LINHED)

c set headers entries that control the initial wrtape
        lbytes = HSTOFF
        nbyt = 2 * SZHFWD
        call savew(ilhd,'HlhEnt', 0  ,LINHED)
        call savew(ilhd,'HlhByt',nbyt,LINHED)

c save out hlh and line header
        call savhlh (ilhd, lbytes, lbyout)
        call wrtape (luout, ilhd, lbyout)

c Get the trace header info for RecNum and TrcNum to adjust output
c numbering
        call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
        call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)

c only one record so fix that header here
        call savew2(TrcOut,ifmt_RecNum,l_RecNum,ln_RecNum,1,TRACEHEADER)

c set up output pick file if requested
        lpicksout = .false.
        if(ptap .ne. ' ') then
          lpicksout = .true.
          call alloclun(lupicks)
          open (unit=lupicks, access='SEQUENTIAL',
     :          file=ptap, status='UNKNOWN', iostat=errcd1)
          if (errcd1 .gt. 0 ) then
            write(LER,*)name,': Failed to open output pickfile ',ptap
            write(LER,*)'FATAL'
            stop
          endif
        endif

      ENDIF

c verbose output of all pertinent information before processing begins
      call verbal
     :       (ntap, otap, ptap,
     :        origin_iline,delta_iline,n_iline,
     :        origin_xline,delta_xline,n_xline,
     :        emask, name, lreport, lgrid, verbose)

c BEGIN PROCESSING 

c Read through the input file dumping z values onto the iline,xline grid
      lmultivalued = .false.
      ipt = 0
      errcd1 = 0
      do while (errcd1 .eq. 0)

        ipt = ipt+1
        read (unit=luin,fmt='(a80)',iostat=errcd1) cardin

        if (errcd1 .gt. 0) then
          write(LERR,*)name,': Read error on input at point ',ipt
          write(LERR,*)'FATAL'
          write(LER,*)name,': Read error on input at point ',ipt
          write(LER,*)'FATAL'
          stop
        endif

        if (errcd1 .lt. 0) then
          write(LERR,*)name,': End of input detected at point ',ipt
          write(LER,*)name,': End of input detected at point ',ipt
        endif

        if (errcd1 .eq. 0) then
          read (cardin,*,iostat=errcd2)
     :      iline_in,xline_in,rjunk1,rjunk2,zdata

          if (errcd2 .ne. 0) then
            write(LERR,*)name,': Format/Data error detected on line ',
     :                   ipt,' of input.  Card is'
            write(LERR,*)cardin
            write(LERR,*)name,': FATAL'
            write(LER,*)name,': Format/Data error detected on line ',
     :                   ipt,' of input.  Card is'
            write(LER,*)cardin
            write(LER,*)name,': FATAL'
            stop
          else
            if (lreport) then

              if (ipt .eq. 1) then
                card1 = cardin
                ilcnt = 1
                xlcnt = 1
                ilines(1) = iline_in
                xlines(1) = xline_in
                iline_min = min(iline_min,iline_in)
                iline_max = max(iline_max,iline_in)
                xline_min = min(xline_min,xline_in)
                xline_max = max(xline_max,xline_in)
              endif

              zdata_min = min(zdata_min,zdata)
              zdata_max = max(zdata_max,zdata)

              if (iline_in .ne. iline_last) then
                iline_delta = min(iline_delta,abs(iline_last-iline_in))
                ipt2 = 1
                lfound = .false.
                do while (.not.lfound    .and.
     :                    ipt2.le.SZLNHD .and.
     :                    ilines(ipt2).ne.emask)
                  if (iline_in.eq.ilines(ipt2)) lfound=.true.
                  ipt2 = ipt2+1
                enddo
                if(.not.lfound) then
                  ilcnt = ilcnt+1
                  ilines(ilcnt) = iline_in
                  iline_min = min(iline_min,iline_in)
                  iline_max = max(iline_max,iline_in)
                endif
                iline_last = iline_in
              endif

              if (xline_in .ne. xline_last) then
                xline_delta = min(xline_delta,abs(xline_last-xline_in))
                ipt2 = 1
                lfound = .false.
                do while (.not.lfound    .and.
     :                    ipt2.le.SZLNHD .and.
     :                    xlines(ipt2).ne.emask)
                  if (xline_in.eq.xlines(ipt2)) lfound=.true.
                  ipt2 = ipt2+1
                enddo
                if(.not.lfound) then
                  xlcnt = xlcnt+1
                  xlines(xlcnt) = xline_in
                  xline_min = min(xline_min,xline_in)
                  xline_max = max(xline_max,xline_in)
                endif
                xline_last = xline_in
              endif

            else

              ii = (xline_in-origin_xline)/delta_xline + 1
              jj = (iline_in-origin_iline)/delta_iline + 1
              if(ii.gt.0 .and. ii.le.n1 .and.
     :           jj.gt.0 .and. jj.le.n2      )then
c               must find the index pointer into 1D RecOut()
                kk = (jj-1)*n1 + ii
                if (RecOut(kk) .ne. emask) lmultivalued = .true.
                RecOut(kk) = zdata
                if(lpicksout) write(lupicks,'(a80)')cardin
              endif

            endif
          endif
        endif
      enddo

c report if a multivalued surface was found
      if (lmultivalued) then
        write(LERR,*)name,': A multivalued surface was detected.'
        write(LERR,*)
     :    name,': Could the in-line or cross-line increments be off?'
        write(LER,*)name,': A multivalued surface was detected.'
        write(LER,*)
     :    name,': Could the in-line or cross-line increments be off?'
      endif

c finish up by reporting or writing the map
      if (lreport) then

        ipt2 = 1
        do while (ilines(ipt2) .ne. emask)
          ipt2 = ipt2+1
        enddo
        n_iline = ipt2-1

        ipt2 = 1
        do while (xlines(ipt2) .ne. emask)
          ipt2 = ipt2+1
        enddo
        n_xline = ipt2-1

        write(LERR,*)' '
        write(LERR,*)name,':'
        write(LERR,*)'    Minimum in-line value        = ',iline_min
        write(LERR,*)'    Maximum in-line value        = ',iline_max
        write(LERR,*)'    Minimum in-line increment    = ',iline_delta
        write(LERR,*)'    Number of unique in-lines    = ',n_iline
        write(LERR,*)' '
        write(LERR,*)'    Minimum cross-line value     = ',xline_min
        write(LERR,*)'    Maximum cross-line value     = ',xline_max
        write(LERR,*)'    Minimum cross-line increment = ',xline_delta
        write(LERR,*)'    Number of unique cross-lines = ',n_xline
        write(LERR,*)' '
        write(LERR,*)'    Minimum value in data        = ',zdata_min
        write(LERR,*)'    Maximum value in data        = ',zdata_max
        write(LERR,*)' '
        write(LER,*)' '
        write(LER,*)name,':'
        write(LER,*)'    Minimum in-line value        = ',iline_min
        write(LER,*)'    Maximum in-line value        = ',iline_max
        write(LER,*)'    Minimum in-line increment    = ',iline_delta
        write(LER,*)'    Number of unique in-lines    = ',n_iline
        write(LER,*)' '
        write(LER,*)'    Minimum cross-line value     = ',xline_min
        write(LER,*)'    Maximum cross-line value     = ',xline_max
        write(LER,*)'    Minimum cross-line increment = ',xline_delta
        write(LER,*)'    Number of unique cross-lines = ',n_xline
        write(LER,*)' '
        write(LER,*)'    Minimum value in data        = ',zdata_min
        write(LER,*)'    Maximum value in data        = ',zdata_max
        write(LER,*)' '

c end of test for lreport
      endif

      if (lgrid) then

c       write the record
        tr_index = 1 - n1
        hdr_index = 1 - ITRWRD
        do itrc = 1,n2

c         First get the ptrs into the processed data and header arrays
          tr_index = tr_index + n1
          hdr_index = hdr_index + ITRWRD

c         Now move the output data to a header-attached trace
          call vmov (RecOut(tr_index), 1, TrcOut(ITHWP1), 1, n1)

c         Adjust the trace counter in the output header
c         (it might be interesting to stuff iline number too)
          call savew2 (TrcOut,ifmt_TrcNum,l_TrcNum,ln_TrcNum,itrc,
     :                 TRACEHEADER)

c         Now write the trace
          call wrtape (luout, TrcOut, obytes)

        enddo

c end of test for lgrid
      endif


c Close data files 
      close (luin)
      if (lgrid) call lbclos (luout)
      if (lpicksout) close (lupicks)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop

      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c provide terse online help [detailed help goes in man page]
      subroutine help(name)

      character name*(*)

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name
      write(LER,*)'   Convert a Landmark ascii horizon to a USP record.'
      write(LER,*)' '
      write(LER,*)'Input......................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]      -- input data set                 (stdin:)'
      write(LER,*)'-O[]      -- output data set               (stdout:)'
      write(LER,*)'-P[]      -- output picks within grid         (none)'
      write(LER,*)' '
      write(LER,*)'-I        -- Interrogate input for limits  (.false.)'
      write(LER,*)' '
      write(LER,*)'-ol[ol]   -- first in-line to grid               (1)'
      write(LER,*)'-dl[dl]   -- in-line numbering delta             (1)'
      write(LER,*)'-nl[nl]   -- number of in-lines in grid        (100)'
      write(LER,*)' '
      write(LER,*)'-ox[ox]   -- first cross-line to grid            (1)'
      write(LER,*)'-dx[dx]   -- cross-line numbering delta          (1)'
      write(LER,*)'-nx[nx]   -- number of cross-lines in grid     (100)'
      write(LER,*)' '
      write(LER,*)'-emask[emask] -- value for grid when no     (-1e+37)'
      write(LER,*)'                 input point is available'
      write(LER,*)' '
      write(LER,*)'-V        -- Verbose printout              (.false.)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'  lmascii2usp [ -N[] -O[] -P[] -ol[] -dl[] -nl[]'
      write(LER,*)'       -ox[] -dx[] -nx[] -emask[] -I -V ]'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c pick up command line arguments 
      subroutine cmdln
     :       (ntap, otap, ptap,
     :        origin_iline,delta_iline,n_iline,
     :        origin_xline,delta_xline,n_xline,
     :        emask, name, lreport, lgrid, verbose)

      implicit none

#include <f77/iounit.h>

      integer   argis
      integer   origin_iline,delta_iline,n_iline
      integer   origin_xline,delta_xline,n_xline
      real      emask
      character ntap*(*), otap*(*), ptap*(*), name*(*)
      logical   verbose, lreport, lgrid

      call argstr('-N', ntap, ' ', ' ') 
      call argstr('-O', otap, ' ', ' ') 
      call argstr('-P', ptap, ' ', ' ') 

      call argi4('-ol', origin_iline, 0, 0)
      call argi4('-dl',  delta_iline, 0, 0)
      call argi4('-nl',      n_iline, 0, 0)
      call argi4('-ox', origin_xline, 0, 0)
      call argi4('-dx',  delta_xline, 0, 0)
      call argi4('-nx',      n_xline, 0, 0)
      call argr4('-emask', emask, -1.0e+37, -1.0e+37 )

      lgrid = .true.
      lreport = .false.
      lreport= (argis('-I') .gt. 0)
      if (lreport) lgrid = .false.

      verbose = .false.
      verbose= (argis('-V') .gt. 0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's
      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c verbal printout of pertinent program particulars
      subroutine verbal
     :       (ntap, otap, ptap,
     :        origin_iline,delta_iline,n_iline,
     :        origin_xline,delta_xline,n_xline,
     :        emask, name, lreport, lgrid, verbose)

      implicit none
#include <f77/iounit.h>

      character  ntap*(*), otap*(*), ptap*(*), name*(*)
      integer    origin_iline,delta_iline,n_iline
      integer    origin_xline,delta_xline,n_xline
      real       emask
      logical    verbose, lreport, lgrid
      integer    nblen,len

      write(LERR,*)' '
      write(LERR,*)' File names for input and output'
      write(LERR,*)' '
      if (ntap(1:1) .ne. ' ') then
        len = nblen(ntap)
        write(LERR,*)'   input data set name   = ',ntap(1:len)
      else
        write(LERR,*)'   input data is on         stdin:'
      endif
      if (lgrid) then
        if (otap(1:1) .ne. ' ') then
          len = nblen(otap)
          write(LERR,*)'   output data set name  = ',otap(1:len)
        else
          write(LERR,*)'   output data is on        stdout:'
        endif
        if (ptap(1:1) .ne. ' ') then
          len = nblen(ptap)
          write(LERR,*)'   output pick filename  = ',ptap(1:len)
        endif
      endif
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Other command line parameters'
      write(LERR,*)' '
      if(lreport) then
        write(LERR,*)'   Interrogation of input requested'
      endif
      if (lgrid) then
        write(LERR,*)'   First in-line    in grid  = ',origin_iline
        write(LERR,*)'   First cross-line in grid  = ',origin_xline
        write(LERR,*)'   in-line    grid delta     = ',delta_iline
        write(LERR,*)'   cross-line grid delta     = ',delta_xline
        write(LERR,*)'   Number of in-lines        = ',n_iline
        write(LERR,*)'   Number of cross-lines     = ',n_xline
        write(LERR,*)' '
        write(LERR,*)'   Embedded mask value       = ',emask
        if (verbose) then
          write(LERR,*)' '
          write(LERR,*)'   Verbose printout requested'
        endif
        write(LERR,*)' '
        write(LERR,*)' '
        write(LERR,*)' Output data dimensions'
        write(LERR,*)' '
        write(LERR,*)'   samples per trace         = ',n_iline
        write(LERR,*)'   traces per record         = ',n_xline
        write(LERR,*)'   number of records         = ',1
      endif
      write(LERR,*)' '

      return
      end
