c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C Copyright 2001, Allied Geophysics, Inc. All Rights Reserved          *
C***********************************************************************
C Portions of this code and/or subroutines  used by this code are      *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  Main Routine -----------------------
c
c uspstats - Barf out a bunch of trivial statistics from a USP dataset
c
c  Program Description:
c
c     uspstats compiles and reports several statistical measures of
c       samples in a usp dataset. The user can limit the indices to
c       scan on each of first 3 axes, can indicate a value of an
c       embedded mask, and can optionally write a standard deviation
c       volume... a volume that indicates how many standard deviations
c       an input value is from the mean of the input samples.
c
c  Program History:
c
c    Jan 17, 2002 - implicit none, catch undefined variables and kill
c                   unused variables - PGAG
c    Dec 29, 2001 - minor cleanup of code and man page
c    Nov 23, 2001 - fix index limit bugs, add alternate arg style
c    Oct 30, 2001 - adopted emask name for embedded mask
c    Sep 26, 2001 - original version
c
c get machine dependent parameters 

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c declare standard USP variables 
      integer     ilhd(3*SZLNHD)
     
      integer     luin, luout
      integer     obytes, lbytes, lbyout, nbytes
      integer     n1,n2,n3,n1_out,n2_out,n3_out
      integer     argis, jerr

      character   ntap*255, otap*255, name*8

      logical     verbose

c Program Specific - dynamic memory variables
      integer TrcSize, TotalMem
      integer ier1,abort

      integer Trace(2)
      pointer (ptr_Trace,Trace)

c Program Specific variables
      real    emask, default_mask

      integer i2,i3
      integer   i1s,i1e, i2s,i2e, i3s,i3e

      logical lstdev

c statistical variables
      integer  nlive
      real*8   dlive
      real     data_min,data_max,data_mean,data_rms,data_std,gamma
      real*8   cum_val,cum_sqr,dmean_val,dstd_val,cum_gamma

c NOTE: Higher order moments could be computed if an array of the
c       accumulated values was used. e.g. data_cum(order) instead
c       of cum_val,cum_squ.

c Initialize variables
      data default_mask/-1e+37/
      data abort/0/
      data name/"USPSTATS"/
      data data_min/ 1.0e+37/
      data data_max/-1.0e+37/
      data nlive/0/
c
c ----------------------------------------------------------------------
c Done with declarations
c ----------------------------------------------------------------------

c Give command line help if requested
      if ( argis('-?')    .gt. 0 .or.
     :     argis('-h')    .gt. 0 .or.
     :     argis('-help') .gt. 0 ) then
        call help(name)
        stop
      endif

c Open printout file
#include <f77/open.h>

c Get command line input parameters
      call cmdln (ntap, otap, i1s,i1e,i2s,i2e,i3s,i3e,
     :            emask, default_mask, name, lstdev, verbose)

c Open the input and output files
      call getln(luin,ntap,'r',0)
      if (lstdev) call getln(luout,otap,'w',1)

c Read input line header and save certain parameters
      call rtape(luin,ilhd,lbytes)
      if(lbytes.eq.0)then
        write(LER,*)name,': no line header on input file',ntap
        write(LER,*)'FATAL'
        stop
      endif

c Print HLH to printout file
      call hlhprt (ilhd, lbytes, name, 4, LERR)

c Get size of input (and output) volume(s)
      call saver(ilhd, 'NumSmp', n1, LINHED)
      call saver(ilhd, 'NumTrc', n2, LINHED)
      call saver(ilhd, 'NumRec', n3, LINHED)
      if (i1s.lt.1  .or. i1s.gt.n1) i1s=1
      if (i2s.lt.1  .or. i2s.gt.n2) i2s=1
      if (i3s.lt.1  .or. i3s.gt.n3) i3s=1
      if (i1e.eq.-1 .or. i1e.gt.n1) i1e=n1
      if (i2e.eq.-1 .or. i2e.gt.n2) i2e=n2
      if (i3e.eq.-1 .or. i3e.gt.n3) i3e=n3
      n1_out = i1e - i1s + 1
      n2_out = i2e - i2s + 1
      n3_out = i3e - i3s + 1

c Verbose output of all pertinent information before processing begins
      call verbal
     :      (ntap, otap, i1s,i1e,i2s,i2e,i3s,i3e, emask,default_mask,
     :       lstdev,verbose, n1,n2,n3,n1_out,n2_out,n3_out)

c Get sizes for dynamic memory allocations
      TrcSize = SZTRHD + n1*SZSMPD
      TotalMem = TrcSize

c Dynamically allocate a trace with attached header
      call galloc(ptr_Trace, TrcSize, ier1, abort)
    
      if (ier1.ne.0) then
        write(LERR,*)' '
        write(LERR,*)name,
     :    ': Unable to allocate workspace: ',TotalMem,' bytes'
        write(LERR,*)'FATAL'
        write(LERR,*)' '
        write(LER,*)' '
        write(LER,*)name,
     :    ': Unable to allocate workspace: ',TotalMem,' bytes'
        write(LER,*)'FATAL'
        write(LER,*)' '
        stop
      else
        write(LERR,*)' '
        write(LERR,*)name,': Allocated ',TotalMem,' bytes of workspace'
        write(LERR,*)' '
      endif

c BEGIN PROCESSING 

c Skip to first axis 3 index to inspect
      call recskp(1,i3s-1,luin,n2,Trace)

c Loop over axis 3
      do i3 = i3s,i3e

c Skip to first axis 2 index to inspect
        call trcskp(i3,1,i2s-1,luin,n2,Trace)

c Loop over axis 2
        do i2 = i2s,i2e

          nbytes = 0
          call rtape(luin, Trace, nbytes)
          if(nbytes.eq.0)then
            write(LERR,*)name,
     :        ': Read error on input at record = ',i3,' trace= ',i2
            stop
          endif

c Accumulate values needed for statistics being determined
          call stats1 (Trace(ITRWRD+i1s), n1_out, nlive,
     :                 data_min, data_max, cum_val, cum_sqr,
     :                 emask, default_mask)

        enddo

c Skip past remaining indices of axis 2
        call trcskp(i3,i2e+1,n2,luin,n2,Trace)

      enddo

      dlive     = dble(nlive)
      data_mean = cum_val/dlive
      dmean_val = cum_val/dlive
      data_rms  = sqrt(cum_sqr/dlive)
      dstd_val  = cum_sqr - 2.0*cum_val*dmean_val + dlive*dmean_val**2.0
      data_std  = sqrt(dstd_val/dlive)

      IF (lstdev) THEN
c
c Second pass over the data
c   - compute and write sigma volume
c   - get skewness as a bonus
c
c Rewind input - actually close, reopen and pop off the line header
        call lbclos(luin)
        call getln(luin,ntap,'r',0)
        call rtape(luin,ilhd,lbytes)

c Fix sizes, adjust history, and write line header
        call savew(ilhd, 'NumSmp', n1_out, LINHED)
        call savew(ilhd, 'NumTrc', n2_out, LINHED)
        call savew(ilhd, 'NumRec', n3_out, LINHED)
        call savhlh(ilhd, lbytes, lbyout)
        call wrtape(luout, ilhd, lbyout)

c Number bytes for an output trace
        obytes = SZTRHD + n1_out*SZSMPD

c Loop over axis 3 and axis 2 processing and writing as we go.
        call recskp(1,i3s-1,luin,n2,Trace)
        do i3 = i3s,i3e
          call trcskp(i3,1,i2s-1,luin,n2,Trace)
          do i2 = i2s,i2e
            call rtape(luin,Trace,nbytes)
            call stats3
     :             (Trace(ITRWRD+i1s),n1_out, data_mean,data_std,
     :              cum_gamma, emask,default_mask)
            call data2stdev
     :             (Trace(ITRWRD+i1s),n1_out, data_mean,data_std,
     :              emask,default_mask)
            call vmov(Trace(ITRWRD+i1s),1,Trace(ITRWRD+1),1,n1_out)
            call wrtape(luout,Trace,obytes)
          enddo
          call trcskp(i3,i2e+1,n2,luin,n2,Trace)
        enddo
        gamma = cum_gamma/nlive

      ENDIF

c
c Report statistics to print file and screen
c
      write(LERR,*)' '
      write(LERR,*)name,': Data Statistics: '
      write(LERR,*)' '
      write(LERR,*)name,': Number of live values = ',nlive
      write(LERR,*)name,': Minimum value         = ',data_min
      write(LERR,*)name,': Maximum value         = ',data_max
      write(LERR,*)name,': Mean value            = ',data_mean
      write(LERR,*)name,': RMS value             = ',data_rms
      write(LERR,*)name,': Standard Deviation    = ',data_std
      write(LERR,*)name,': Variance              = ',data_std*data_std
      write(LERR,*)name,': Coeff. of variation   = ',data_std/data_mean
      if (lstdev) then
        write(LERR,*)name,': Skewness              = ',gamma
      endif
      write(LERR,*)' '

      write(LER,*)' '
      write(LER,*)name,': Data Statistics: '
      write(LER,*)' '
      write(LER,*)name,': Number of live values = ',nlive
      write(LER,*)name,': Minimum value         = ',data_min
      write(LER,*)name,': Maximum value         = ',data_max
      write(LER,*)name,': Mean value            = ',data_mean
      write(LER,*)name,': RMS value             = ',data_rms
      write(LER,*)name,': Standard Deviation    = ',data_std
      write(LER,*)name,': Variance              = ',data_std*data_std
      write(LER,*)name,': Coeff. of variation   = ',data_std/data_mean
      if (lstdev) then
        write(LER,*)name,': Skewness              = ',gamma
      endif
      write(LER,*)' '

c Close data files 
      call lbclos(luin)
      if(lstdev)call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop

      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c provide terse online help [detailed help goes in man page]
      subroutine help(name)

      character name*(*)

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name
      write(LER,*)'    Barf out statistics of samples in a USP dataset.'
      write(LER,*)' '
      write(LER,*)'Input......................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]     -- input data set                  (stdin:)'
      write(LER,*)'-O[]     -- output data set                   (none)'
      write(LER,*)' '
      write(LER,*)'-1s[i1s] -- first sample of axis 1 to use        (1)'
      write(LER,*)'-1e[i1e] -- last  sample of axis 1 to use   (NumSmp)'
      write(LER,*)' '
      write(LER,*)'-2s[i2s] -- first sample of axis 2 to use        (1)'
      write(LER,*)'-2e[i2e] -- last  sample of axis 2 to use   (NumTrc)'
      write(LER,*)' '
      write(LER,*)'-3s[i3s] -- first sample of axis 3 to use        (1)'
      write(LER,*)'-3e[i3e] -- last  sample of axis 3 to use   (NumRec)'
      write(LER,*)' '
      write(LER,*)'-stdev   -- Flag indicating request for    (.false.)'
      write(LER,*)'            standard deviation volume.'
      write(LER,*)'            (Use in absence of -O[] to'
      write(LER,*)'             output on stdout: )'
      write(LER,*)' '
      write(LER,*)'-emask[emask]                               (-1e+37)'
      write(LER,*)'         -- value for samples where grid is'
      write(LER,*)'            not defined'
      write(LER,*)' '
      write(LER,*)'-V       -- Verbose printout               (.false.)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'  uspstats -N[] [ -O[] -1s[] -1e[] -2s[] -2e[]'
      write(LER,*)'           -3s[] -3e[] -stdev -emask[] -V ]'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c pick up command line arguments 
      subroutine cmdln (ntap, otap, i1s,i1e,i2s,i2e,i3s,i3e,
     :                  emask, default_mask, name, lstdev, verbose)

#include <f77/iounit.h>

      integer   argis
      integer   i1s,i1e, i2s,i2e, i3s,i3e
      real      emask, default_mask
      character ntap*(*), otap*(*), name*(*)
      logical   lstdev,verbose

c Input/output names
      call argstr('-N', ntap, ' ', ' ') 
      call argstr('-O', otap, ' ', ' ') 

c Set flag for calculating standard deviation volume
      lstdev= .false.
      lstdev= (argis('-stdev') .gt. 0)
      if (otap(1:1).ne.' ') lstdev = .true.

c The embedded mask value
      call argr4('-emask', emask, default_mask, default_mask)

c Index ranges the way I prefer (default)
      call argi4('-1s', i1s,  1,  1 )
      call argi4('-1e', i1e, -1, -1 )
      call argi4('-2s', i2s,  1,  1 )
      call argi4('-2e', i2e, -1, -1 )
      call argi4('-3s', i3s,  1,  1 )
      call argi4('-3e', i3e, -1, -1 )

c Index ranges the old usp way (as a backup)
      call argi4('-s',  i1s, i1s, i1s )
      call argi4('-e',  i1e, i1e, i1e )
      call argi4('-ns', i2s, i2s, i2s )
      call argi4('-ne', i2e, i2e, i2e )
      call argi4('-rs', i3s, i3s, i3s )
      call argi4('-re', i3e, i3e, i3e )

      verbose= .false.
      verbose= (argis('-V') .gt. 0)

c Now using alternate argument style
      call argi4('1s=', i1s, i1s, i1s )
      call argi4('1e=', i1e, i1e, i1e )
      call argi4('2s=', i2s, i2s, i2s )
      call argi4('2e=', i2e, i2e, i2e )
      call argi4('3s=', i3s, i3s, i3s )
      call argi4('3e=', i3e, i3e, i3e )
      if(.not.verbose) verbose=(argis('verbose=').gt.0)

c Check for extraneous arguments and abort if found to
c catch all manner of user typo's
      call xtrarg (name, ler,.FALSE.,.FALSE.)
      call xtrarg (name,lerr,.FALSE.,.TRUE.)

c Make sure input is not piped when output is requested
c (the input will need to be rewound)
      if(lstdev .and. ntap(1:1).eq.' ') then

        write(LERR,*)' '
        write(LERR,*)name,
     :    ': ERROR - Writing an output is not compatible with'
        write(LERR,*)'        ',
     :    '  piped input. Computing a standard deviation volume'
        write(LERR,*)'        ',
     :    '  requires two passes over the data but I cannot rewind'
        write(LERR,*)'        ',
     :    '  piped inputs.'
        write(LERR,*)' '

        write(LER,*)' '
        write(LER,*)name,
     :    ': ERROR - Writing an output is not compatible with'
        write(LER,*)'        ',
     :    '  piped input. Computing a standard deviation volume'
        write(LER,*)'        ',
     :    '  requires two passes over the data but I cannot rewind'
        write(LER,*)'        ',
     :    '  piped inputs.'
        write(LER,*)' '

        stop
      endif

      return
      end


c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c verbal printout of pertinent program particulars
      subroutine verbal
     :      (ntap, otap, i1s,i1e,i2s,i2e,i3s,i3e, emask,default_mask,
     :       lstdev,verbose, n1,n2,n3, n1_out,n2_out,n3_out)

#include <f77/iounit.h>

      character  ntap*(*), otap*(*)
      real       emask,default_mask
      integer    i1s,i1e, i2s,i2e, i3s,i3e
      integer    n1,n2,n3, n1_out,n2_out,n3_out
      logical    verbose,lstdev
      integer    nblen

      write(LERR,*)' '
      write(LERR,*)' File names for input and output'
      if (ntap(1:1) .ne. ' ') then
        write(LERR,*)' '
        write(LERR,*)'   Input data set name   = ',ntap(1:nblen(ntap))
      else
        write(LERR,*)' '
        write(LERR,*)'   Input data is on         stdin:'
      endif
      write(LERR,*)' '
      write(LERR,*)'      Number of input samples   = ',n1
      write(LERR,*)'      Number of input traces    = ',n2
      write(LERR,*)'      Number of input records   = ',n3
      if(lstdev) then
        if (otap(1:1) .ne. ' ') then
          write(LERR,*)' '
          write(LERR,*)'   Output data set name  = ',otap(1:nblen(otap))
        else
          write(LERR,*)' '
          write(LERR,*)'   Output data is on        stdout:'
        endif
        write(LERR,*)' '
        write(LERR,*)'      Number of output samples   = ',n1_out
        write(LERR,*)'      Number of output traces    = ',n2_out
        write(LERR,*)'      Number of output records   = ',n3_out
      else
        write(LERR,*)' '
        write(LERR,*)'   No output will be written'
      endif
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Other command line parameters'
      write(LERR,*)' '
      write(LERR,*)'   First,Last sample to inspect  = ',i1s,i1e
      write(LERR,*)'              trace              = ',i2s,i2e
      write(LERR,*)'              record             = ',i3s,i3e
      if(emask .ne. default_mask) then
        write(LERR,*)' '
        write(LERR,*)'   Embedded mask value           = ',emask
      endif
      if (verbose) then
        write(LERR,*)' '
        write(LERR,*)'   Verbose printout requested'
      endif
      write(LERR,*)' '

      return
      end
