/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
**	list management
*/
#include "listdefs.h"
#include <cu.h>
#include <stdio.h>

#define LP(z)	    ((list_t *)(z))
#define	MEMGRAN	    8

char *malloc();
void free();
/*
**	private: check a list for a good magic number; abort on failure.
*/
static void checklist(pname, list)
char *pname;
LIST list;
{
	if(list == NULL)
		cuerror(1, "lists: %s passed a NULL list.\n", pname);
	if(LP(list)->lsmagic != LIST_MAGIC)
		cuerror(1, "lists: %s passed a bad list.\n", pname);
}
/*
**	private: make sure a list can accept one more entry.  Assumes
**		that list is valid and aborts on any failure.
*/
static void expandlist(list)
LIST list;
{
	list_t *l;

	l = LP(list);
	if( (l->listslots > l->listlength) ) return;
	l->listslots += MEMGRAN;
	l->listitems = (listitem_t *)curealloc( (VPTR)l->listitems,
			(int)(l->listslots * sizeof(listitem_t)) );
	return;
}
/*
**	private: return the index of entry with id or:
**		if die != 0, abort, else
**		return -1.
*/
static long lsindexof(list, id, die)
LIST list;
long id;
int die;
{
	list_t *l;
	long first, last, mid;
	long tid;

	l = LP(list);
	first = 0L;
	last = l->listlength - 1;
	while(last >= first) {
		mid = (first + last)/2;
		tid = l->listitems[mid].itemid;
		if(tid == id) return mid;
		if(tid > id)
			last = mid - 1;
		else
			first = mid + 1;
	}
	if(die != 0)
		cuerror(1, "lists: lsindexof: id (%ld) not in list.\n", id);
	return -1;
}
/*
**	private: insert the item at id.  it is an error if id is already
**		present in the list.  if id <= 0, use the largest id
**		in the list + 1 for the new item's id. in all cases,
**		return the item's id.
*/
static long lsinsertat(list, id, value)
LIST list;
long id;
itemvalue_t value;
{
	list_t *l;
	listitem_t i;
	long first, last, mid;
	long tid;

	l = LP(list);
	if(id <= 0) {
		if(l->listlength < 1)
			id = 1;
		else
			id = l->listitems[l->listlength - 1].itemid + 1;
	}
	i.itemid = id;
	i.itemvalue = value;
	expandlist(list);
/*
**	the special case of an initially empty list
*/
	if(l->listlength <= 0L) {
		l->listlength = 1L;
		l->listitems[0] = i;
		return id;
	}
/*
**	the special case of addition to the end of a list
*/
	if(l->listitems[l->listlength - 1].itemid < id) {
		++(l->listlength);
		l->listitems[(long)l->listlength - 1] = i;
		return id;
	}
/*
**	now deal with insertion: find the index before which to install
**		the new item.
*/
	first = 0L;
	last = l->listlength - 1;
	while(last > first) {
		mid = (last + first) / 2;
		tid = l->listitems[mid].itemid;
		if(tid == id)
			cuerror(1, "lsinsertat: id (%ld) already at %ld.\n",
					id, mid);
		if(tid < id)
			first = mid + 1;
		else
			last = mid;
	}
/*
**	check once more for duplication
*/
	if(tid == id)
		cuerror(1, "lsinsertat: id (%ld) already at %ld.\n", id, mid);
/*
**	bump first... up one and insert the new item.
*/
	for(mid = l->listlength; mid > first; mid--)
		l->listitems[mid] = l->listitems[mid - 1];
	l->listitems[first] = i;
	++(l->listlength);
	return id;
}
/*
**	remove the item at index and collapse the list.
**	trust that the index'd item exists.
*/
static void lsremoveindex(list, index)
LIST list;
long index;
{
	list_t *l;
	long i;

	l = LP(list);
	--(l->listlength);
	for(i = index; i < l->listlength; i++)
		l->listitems[i] = l->listitems[i + 1];
	return;
}


LIST ls_new(parent, itype, ltype, newunique)
LIST parent;
long itype, ltype;
VPTR newunique;
{
	list_t *l;

	if( (itype != ITEM_LONG) && (itype != ITEM_POINTER) ) {
		cuerror(1, "ls_new: bad itype = %ld\n", itype);
	}
	l = (list_t *)malloc(sizeof(list_t));
	l->lsmagic = LIST_MAGIC;
	l->itemtype = itype;
	l->listtype = ltype;
	l->parent = (list_t *) parent;
	l->unique = newunique;
	l->listlength = 0L;
	l->listslots = 0L;
	l->listitems = (listitem_t *) NULL;
	return (LIST)(l);
}

void ls_dispose(list)
LIST list;
{
	checklist("ls_dispose", list);
	if(LP(list)->listlength > 0L)
		free((char *)LP(list)->listitems);
	free((char *)list);
	return;
}

long ls_addl(list, id, val)
LIST list;
long id;
long val;
{
	itemvalue_t x;

	checklist("ls_addl", list);
	if( LP(list)->itemtype != ITEM_LONG )
		cuerror(1, "ls_addl: not a list of longs.\n" );
	x.longitem = val;
	return lsinsertat(list, id, x);
}

long ls_addp(list, id, val)
LIST list;
long id;
VPTR val;
{
	itemvalue_t x;

	checklist("ls_addp", list);
	if( LP(list)->itemtype != ITEM_POINTER )
		cuerror(1, "ls_addp: not a list of pointers.\n" );
	x.ptritem = val;
	return lsinsertat(list, id, x);
}

long ls_getlx(list, index)
LIST list;
long index;
{
	checklist("ls_getlx", list);
	if( LP(list)->itemtype != ITEM_LONG )
		cuerror(1, "ls_getlx: not a list of longs.\n" );
	if((index < 0) || (index >= LP(list)->listlength))
		cuerror(1, "ls_getlx: index out of range.\n");
	return LP(list)->listitems[index].itemvalue.longitem;
}

long ls_getl(list, id)
LIST list;
long id;
{
	return ls_getlx(list, lsindexof(list, id, 1));
}

VPTR ls_getpx(list, index)
LIST list;
long index;
{
	checklist("ls_getpx", list);
	if( LP(list)->itemtype != ITEM_POINTER )
		cuerror(1, "ls_getpx: not a list of pointers.\n" );
	if((index < 0) || (index >= LP(list)->listlength))
		cuerror(1, "ls_getpx: index out of range.\n");
	return LP(list)->listitems[index].itemvalue.ptritem;
}

VPTR ls_getp(list, id)
LIST list;
long id;
{
	return ls_getpx(list, lsindexof(list, id, 1));
}

void ls_delid(list, id)
LIST list;
long id;
{
	checklist("ls_delid", list);
	lsremoveindex(list, lsindexof(list, id, 1));
	return;
}

long ls_len(list)
LIST list;
{
	checklist("ls_len", list);
	return LP(list)->listlength;
}

long ls_type(list)
LIST list;
{
	checklist("ls_type", list);
	return LP(list)->listtype;
}

int ls_isl(list)
LIST list;
{
	checklist("ls_isl", list);
	return ( LP(list)->itemtype == ITEM_LONG );
}

int ls_isp(list)
LIST list;
{
	checklist("ls_isp", list);
	return ( LP(list)->itemtype == ITEM_POINTER );
}

long ls_idof(list, index)
LIST list;
long index;
{
	list_t *l;

	checklist("ls_idof", list);
	l = LP(list);
	if( (index < 0) || (index >= l->listlength) )
		cuerror(1, "ls_idof: index (%ld) out  of range.\n", index);
	return l->listitems[index].itemid;
}

LIST ls_parentof(list)
LIST list;
{
	checklist("ls_parentof", list);
	return (LIST)(LP(list)->parent);
}

VPTR ls_unique(list)
LIST list;
{
	checklist("ls_unique", list);
	return (VPTR)(LP(list)->unique);
}

long ls_myid(list)
LIST list;
{
	list_t *l;
	long i;
	VPTR z;

	checklist("ls_myid", list);
	l = LP(list)->parent;
	if(l == NULL)
		return 0;
	if(l->listlength <= 0)
		cuerror(1, "ls_myid: parent list had no items.\n");
	if(l->itemtype != ITEM_POINTER)
		cuerror(1, "ls_myid: parent list was not a pointer list.\n");
	z = (VPTR) list;
	for(i = 0; i < l->listlength; i++)
		if(l->listitems[i].itemvalue.ptritem == z)
			return l->listitems[i].itemid;
	cuerror(1, "ls_myid: target was not in parent.\n");
	return 0L;
}
	
int ls_hasid(list, id)
LIST list;
long id;
{
	checklist("ls_hasid", list);
	return (lsindexof(list, id, 0) >= 0) ? 1 : 0;
}

