C*****  AMX001  INITIALIZE TABLES FOR AMX002    REV 1.0     JAN 88
C
C  PURPOSE:
C       INITIALIZES CONSTANT, TWIDDLE FACTOR, FACTOR, AND
C       PERMUTATION TABLES NEEDED BY AMX002.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                DEC 86          D.R. BENUA, QTC
C       MATH ADVANTAGE VERSION  JAN 88          R.D. COLEMAN, QTC
C          CHANGED SUBROUTINE NAMES.
C
C  CALLING FORMAT:
C       CALL AMX001 (INC, N, IFLG, NFAC, M, CNST, STABLE, CTABLE,
C      &             ISRC, IDST, NPERM)
C
C  PARAMETERS:
C
C       INC     INTEGER INPUT SCALAR
C               STRIDE OF DATA ARRAYS USED BY AMX002.
C
C       N       INTEGER INPUT SCALAR
C               LENGTH OF DATA ARRAYS USED BY AMX002.
C
C       IFLG    INTEGER INPUT SCALAR
C               DIRECTION OF TRANSFORMS, POSITIVE FOR
C               FORWARD, NEGATIVE FOR INVERSE.
C
C       NFAC    INTEGER OUTPUT ARRAY OF LENGTH M
C               PRIME FACTORS OF N.
C
C       M       INTEGER OUTPUT SCALAR
C               NUMBER OF FACTORS IN ARRAY NFAC.
C
C       CNST    REAL OUTPUT ARRAY OF LENGTH 11
C               CONSTANTS USED BY AMX002, DEPENDENT
C               ON IFLG.
C
C       STABLE  REAL OUTPUT ARRAY OF LENGTH N
C               SINE TWIDDLE FACTORS, DEPENDENT
C               ON IFLG.
C
C       CTABLE  REAL OUTPUT ARRAY OF LENGTH N
C               COSINE TWIDDLE FACTORS.
C
C       ISRC    INTEGER OUTPUT ARRAY OF LENGTH NPERM
C               PERMUTATION INDICIES.
C
C       IDST    INTEGER OUTPUT ARRAY OF LENGTH NPERM
C               PERMUTATION INDICIES.
C
C       NPERM   INTEGER OUTPUT SCALAR
C               NUMBER OF INDICIES IN ISRC AND IDST,
C               ALWAYS LESS THAN N.
C
C  DESCRIPTION:
C       THIS ROUTINE FACTORS N, LOADS THE CONSTANT AND TWIDDLE
C       FACTOR TABLES, AND LOADS THE PERMUTATION ARRAYS.
C       IT MUST BE CALLED ONCE BEFORE ANY NUMBER OF CALLS TO
C       AMX002 THAT USE THE SAME N, INC, AND IFLG.
C
C  SUBPROGRAMS CALLED:
C       AMX003, AMX004
C
C  ERROR CONDITIONS:
C       IF M .EQ. 0, N IS NOT FACTORABLE BY 2, 3, 5, OR 7
C       AND THE TABLES ARE NOT INITIALIZED.
C
C-----------------------------------------------------------------------
C
        SUBROUTINE AMX001 (INC, N, IFLG, NFAC, M, CNST, STABLE,
     &                     CTABLE, ISRC, IDST, NPERM)
C
        REAL CNST(1), STABLE(1), CTABLE(1)
        INTEGER INC, N, IFLG, NFAC(1), M, ISRC(1), IDST(1), NPERM
        INTEGER I, ICYC, J
        REAL RAD, S1, S2, S3, S4, AK, C2
C
C-----------------------------------------------------------------------
C
C---------------------------------------------------------
C  Find the M prime factors of N, returning them in NFAC.
C  If N can't be factored with 2,3,5, & 7,  AMX003 will
C  return M = 0.
C---------------------------------------------------------
C
        CALL AMX003 (N, NFAC, M)
C
        IF (M .EQ. 0) RETURN
C
C------------------------------------------------------
C  Using the prime factor array, load the array STABLE
C  with digit reversed values of it's indicies.  Note
C  that STABLE is just used as a temporary arrray here.
C  It will be loaded with sine values before the END
C  of this routine.
C------------------------------------------------------
C
        CALL AMX004 (STABLE, NFAC, M)
C
C---------------------------------------------------
C  Convert the digit reversed values in STABLE by
C  subtracting them from N + 2.  This adjusts for
C  reversed coefficient ordering of Singleton's
C  algorithm.
C--------------------------------------------------
C
       DO 10 I=2,N
               STABLE(I) = FLOAT(N+2) - STABLE(I)
10     CONTINUE
C
C-------------------------------------------------------------
C  Load the arrays ISRC and IDST using the indicies of STABLE
C  as sources and the values in STABLE as destinations.  We
C  load them in order so that ISRC(J) = IDST(J-1) as frequently
C  as possible.  This technique allows us to only have one
C  (complex) temporary storage location to sort the A and B
C  arrays in place.  As we load ISRC and IDST, count the number
C  of source - destination pairs and store the total in NPERM.
C-------------------------------------------------------------
C
        NPERM = 0
        I = 1
20      I = I + 1
        IF (I .GE. N) GOTO 40
        IF (STABLE(I) .EQ. FLOAT(I)) GOTO 20
        ICYC = I
 
30      NPERM = NPERM + 1
        ISRC(NPERM) = I
        IDST(NPERM) = INT( STABLE(I) )
        J = I
        I = IDST(NPERM)
        STABLE(J) = FLOAT(J)
        IF (I .NE. ICYC) GOTO 30
        GOTO 20
C
C--------------------------------------------
C  If INC is greater than 1, we must correct
C  the indicies stored in ISRC and IDST.
C--------------------------------------------
C
40      IF (INC .GT. 1) THEN
C
                DO 50 I=1,NPERM
                        ISRC(I) = (ISRC(I) - 1) * INC + 1
                        IDST(I) = (IDST(I) - 1) * INC + 1
50              CONTINUE
        ENDIF
C
C---------------------------------
C  Initialize the constant array
C---------------------------------
C
        RAD = ATAN(1.0)
        S1 = RAD / 0.625
        CNST(1) = SQRT(0.75)
        CNST(4) = COS(S1)
        CNST(5) = SIN(S1)
        CNST(2) = CNST(4)**2 - CNST(5)**2
        CNST(3) = 2.0 * CNST(4) * CNST(5)
        S2 = RAD / 0.875
        S3 = S2 + S2
        S4 = S3 + S2
        CNST(6) = COS(S2)
        CNST(7) = SIN(S2)
        CNST(8) = COS(S3)
        CNST(9) = SIN(S3)
        CNST(10) = COS(S4)
        CNST(11) = SIN(S4)
C
C----------------------------
C  If an inverse transform,
C  correct constants.
C----------------------------
C
        IF (IFLG .LT. 0) THEN
                CNST(1) = -CNST(1)
                CNST(3) = -CNST(3)
                CNST(5) = -CNST(5)
                CNST(7) = -CNST(7)
                CNST(9) = -CNST(9)
                CNST(11) = -CNST(11)
                RAD = -RAD
        ENDIF
C
C-----------------------------------------
C  Initialize the sine and cosine tables.
C  Note that direction of transform affects
C  sign of STABLE values.
C-----------------------------------------
C
        AK = (RAD * 8.0) / FLOAT(N)
        DO 60 I=1,N
                C2 = AK * FLOAT(I)
                STABLE(I) = SIN( C2 )
                CTABLE(I) = COS( C2 )
60      CONTINUE
C
        RETURN
        END
