C****** ARDEC  AUTO REGRESIVE DECONVOLUTION            MTADV EXT REL 1.0
C
C      ** COPYRIGHT 1988  QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C        CALL ARDEC (X,NSAMP,Y,NFILT,LENWIN,INCWIN)
C
C        where,
C
C        X        Real input vector of length NSAMP, input trace.
C
C        NSAMP    Real input scaler number of elements in the trace.
C
C        Y        Real Output vector of length NSAMP, filtered trace.
C
C        NFILT    Real input scaler number of filter points.  NFILT should
C                 be less than 20.
C
C        LENWIN   Real input scaler length of window.   LENWIN must be less
C                 than or equal to 500.
C
C        INCWIN   Real input scaler window increment.
C
C  DESCRIPTION
C
C        This routine does a maximum entropy autoregressive deconvolution
C        on the input trace.   The performance of this routine is sensitive
C        to the values of the input parameters NFILT, LENWIN, and INCWIN.
C
C        NFILT corresponds to twice the number of frequency resonances
C        expected in the seismic trace.  This correspondance is rough since
C        the ARDEC.F source will certainly contain some resonances itself.
C        Hence, even if
C        only one resonance is expected, NFILT should be about 4.  Large values
C        of NFILT (eg. 20) should be avoided, since this many resonances would
C        be poorly defined and other deconvolution algorithms would be more
C        appropriate.
C
C        The window length, LENWIN, should be chosen to be  5 to 10 times the
C        value of NFILT.
C
C        INCWIN, the window increment, controls how often the maximum entropy
C        filter is calculated as the window is slid along the trace.  If the
C        trace changes rapidly with time the filter coefficents will need to be
C        updated frequenctly.  One should experiment with different values of
C        INCWIN as substantial savings in CPU time can be realized with large
C        values of INCWIN.
C
C        If LENWIN is greater than 500 or if INCWIN, NSAMP or NFILT are less
C        than or equal to zero, this routine exits immediately without
C        modifying the array Y.
C
C  REFERENCE
C
C        Andersen N., 1974, On The Calculation of Filter Coefficients For
C        Maximum Entropy Spectral Analysis, Geophysics vol. 39, No. 1, p. 69.
C
C  EXAMPLE
C
C        CALL ARDEC (X, 15, Y, 2, 10, 3)
C
C        Input Operands:
C
C            X  =    25.67532
C                    93.72925
C                    179.6309
C                    249.8744
C                    271.1777
C                    220.0597
C                    90.42228
C                   -102.8267
C                   -325.1960
C                   -529.0173
C                   -663.6033
C                   -687.0007
C                   -576.7116
C                   -336.8671
C                    1.5365263E-04
C
C        Output Operands:
C
C            Y  =    0.00000000E+00
C                    0.00000000E+00
C                    1.08194710E+02
C                    8.19863892E+01
C                    7.29766235E+01
C                   -3.76570129E+01
C                   -6.36061096E+01
C                   -6.96245728E+01
C                   -2.43738022E+02
C                   -1.30589645E+02
C                   -1.59115936E+02
C                   -6.78890381E+01
C                    4.22326050E+01
C                    5.66062012E+01
C                    3.04163269E+02
C
C  HISTORY
C        (1) JUNE 88           T.G. Mattson           Original
C
C-----------------------------------------------------------------------
C
      SUBROUTINE ARDEC(X,NSAMP,Y,NFILT,LENWIN,INCWIN)
      INTEGER NSAMP, NFILT, LENWIN, INCWIN
      REAL    X(*),  Y(*)
C
C  Local variables
C
      INTEGER  MXNLEN, NFILT1, I, IP, IST, IT, NM, NN, N, K
      INTEGER  MM, M, IP1, IP2, JJ
      PARAMETER (MXNLEN=500)
      REAL    AA(MXNLEN), B1(MXNLEN), B2(MXNLEN), A(MXNLEN)
      REAL    YI, B1T, B2T, AAM, ADEN, ANUM
C
C-----------------------------------------------------------------------
C
C  LENWIN must be less than or equal to MXNLEN.
C
      IF ( (LENWIN .GT. MXNLEN) .OR. (INCWIN .LE. 0) .OR.
     &     (NFILT  .LE. 0)      .OR. (NSAMP  .LE. 0) ) THEN
         RETURN
      ENDIF
C
C  Roll onto the trace.
C
      NFILT1 = NFILT + 1
      DO 100 I =1,NFILT1
           Y(I)=0.0
 100  CONTINUE
C
C  Slide the processing window over the trace.  The filter coefficents are
C  calculated for each window position.   IP indexes the end of the window
C  while IST is the begining.  N is the actual width of the window which will
C  usually be equal to LENWIN.
C
      DO 500 IP=NFILT1,NSAMP,INCWIN
        IF(IP .LT. LENWIN) THEN
           IST=1
        ELSE
           IST=IP-LENWIN+1
        ENDIF
        N  = IP - IST + 1
        NN = N  - 1
C
C  The filter coefficents are calculated by a recurrsive scheme similar to
C  that used to solve a Toeplitz matrix.  The first stage of the reccursion
C  is handled separately.  The array B1 is the backward and B2 is the forward
C  deconvolved output traces.  B2 is commonly called the prediction error
C  operator.  The A array contains the filter coefficents.
C
        B1(1)   = X(IST)
        B2(N-1) = X(IP)
        K       = IST
        DO 200 IT=2,NN
          K        = K + 1
          B1(IT)   = X(K)
          B2(IT-1) = X(K)
 200    CONTINUE
        ANUM = 0.0
        ADEN = 0.0
        DO 250 IT=1,NN
           ANUM = ANUM + B1(IT) * B2(IT)
           ADEN = ADEN + B1(IT) * B1(IT) + B2(IT) * B2(IT)
 250    CONTINUE
        A(1) = 0.0
        IF ( ADEN .NE. 0.) THEN
           A(1) = 2*ANUM/ADEN
        ENDIF
C
C  Do the other stages in the recursive calculation of the filter coefficents.
C
        DO 400 M = 2,NFILT
           MM = M-1
           NM = N-M
           DO 300 IT=1,MM
              AA(IT) = A(IT)
 300       CONTINUE
           AAM = AA(M-1)
           B1T = B1(1)
           B2T = B2(1)
           DO 350 IT=1,NM
              B1(IT) = B1T   - AAM * B2T
              B1T    = B1(IT+1)
              B2T    = B2(IT+1)
              B2(IT) = B2T - AAM * B1T
 350       CONTINUE
           ANUM = 0.0
           ADEN = 0.0
           NM   = N-M
           DO 375 IT=1,NM
              ANUM = ANUM + B1(IT)*B2(IT)
              ADEN = ADEN + B1(IT)*B1(IT) + B2(IT)*B2(IT)
 375       CONTINUE
           A(M) = 0.0
           IF ( ADEN .NE. 0. )  THEN
              A(M) = 2*ANUM/ADEN
           ENDIF
           AAM = A(M)
           DO 385 IT=1,MM
              A(IT) = AA(IT) - AAM * AA(M-IT)
 385       CONTINUE
 400    CONTINUE
C
C  Store the output trace value.
C
        Y(IP) = B2(N-NFILT)
C
C  For window increments other than one, apply the current filter coefficents
C  to the next INCWIN-1 values in the input trace.
C
        IF ( (INCWIN .GT. 1) .AND. (IP .LT. NSAMP) ) THEN
           IP1 = IP + 1
           IP2 = IP + INCWIN -1
           IF ( IP2 .GT. NSAMP) IP2=NSAMP
           DO 460 I=IP1,IP2
              YI = X(I)
              DO 420 JJ=1,NFILT
                 YI = YI - X(I-JJ) * A(JJ)
 420         CONTINUE
              Y(I) = YI
 460      CONTINUE
        ENDIF
 500  CONTINUE
C
      RETURN
      END
