C*****  CMFUFC  Complex Matrix Factor             MATH ADVANTAGE REL 3.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL CMFUFC (A,NRA,N,ZTOL,IPERM,IERR)
C
C       where,
C
C       A       Complex input/output matrix.  On input, A contains
C               the matrix to be factored.  On output, A contains
C               the factored form.
C
C       NRA     Integer input number of rows in full matrix A.
C
C       N       Integer input order of submatrix A.
C
C       ZTOL    Real input scalar, pivot zero tolerance.
C
C       IPERM   Integer output permutation vector of length N.
C
C       IERR    Integer output completion code:
C                   =0 if the routine terminated normally.
C                   >0 if the routine aborted because a pivot
C                      was less than or equal to ZTOL.
C                      This implies the matrix is singular.
C                      The value of IERR is the index of the column
C                      where it aborted.
C
C
C  DESCRIPTION
C
C       Using partial pivoting, this routine factors a complex
C       submatrix A into LU form.  L is lower triangular and
C       U is upper triangular. Both L and U are superimposed in
C       the same matrix and overlaid on A. The diagonal elements
C       of L are reciprocated.
C
C       The permutation vector, IPERM, is output by this routine
C       for use as input to CMFUSV.
C
C
C  REFERENCE
C
C       C. R. Rao.  1962.  A note on a generalized inverse of a
C       matrix with application to problems in mathematical
C       statistics.  J. R. Statis. Soc., Vol. B24,  pp. 152-
C       158.
C
C       J. H. Wilkinson.  1965.  The algebraic eigenvalue prob-
C       lem.  New York: Oxford University Press.
C
C       M. J. R. Healy.  1968.  Triangular decomposition of a
C       symmetric matrix.  Appl. Statis., Vol. 17,  pp. 195-
C       197.
C
C       G. W. Stewart.  1973.  Introduction to matrix computa-
C       tions.  New York: Academic Press.
C
C
C  EXAMPLE
C
C       CALL CMFUFC (A,3,3,ZTOL,IPERM,IERR)
C
C       Input Operands:
C
C       A = (1.000,1.000)  (0.000,0.000)  ( 1.000, 5.000)
C           (1.000,2.000)  (3.000,1.000)  (-1.000,18.000)
C           (0.000,0.000)  (2.000,3.000)  (-6.000, 8.000)
C
C       ZTOL = 1.0E-5
C
C       Output Operands:
C
C       A = (0.200,-0.400)  ( 1.000,-1.000)  ( 7.000,4.000)
C           (0.000, 0.000)  ( 0.154,-0.231)  ( 0.923,2.615)
C           (1.000, 1.000)  (-2.000, 0.000)  (-0.25, 1.250)
C
C       IPERM = 2
C               3
C               1
C
C       IERR = 0
C
C  HISTORY
C         1) Nov 84     D. Cooper       Original.
C                       R. Coleman
C         2) Jan 88     L. Shanbeck     Expanded IERR functionality
C
      SUBROUTINE CMFUFC(A,NRA,N,ZTOL,IPERM,IERR)
C
      INTEGER NRA,N,I,IERR,IMAX,IPERM(1),ITEMP,LA2,LASTPT,IA,IA2,IA3
      REAL    A(1),TEMP(2),VMAX,ZTOL,MAGN,ZTOLSQ
C
C   *******************************************************************
C
C     DO VALIDITY CHECKS AND INITIALIZATION
C
      IF (N.LE.0 .OR. NRA.LT.N) GO TO 700
C
      ZTOLSQ = ZTOL*ZTOL
      DO 110 I = 1, N
         IPERM(I) = I
110   CONTINUE
C
C     FACTOR FIRST COLUMN AND ROW
C
      I = 1
      LA2 = NRA + NRA
      LASTPT = N + N + (N-1)*LA2 - 1
      IF (N .EQ. 1) GO TO 420
C
      CALL QTC001(A,2,VMAX,IMAX,N)
C
      IF (VMAX.GT.ZTOLSQ) GOTO 130
         IERR = I
         GO TO 800
130   IF (IMAX.EQ.1) GOTO 150
         IPERM(1)    = IMAX
         IPERM(IMAX) = 1
         CALL QTC003(A,LA2,A(IMAX+IMAX-1),LA2,N)
150   CONTINUE
C
C---  A(1) = 1.0 / A(1)
      MAGN = A(1)*A(1) + A(2)*A(2)
      A(1) = A(1) / MAGN
      A(2) = -A(2) / MAGN
      CALL QTC002(A(LA2+1),LA2,A,A(LA2+1),LA2,N-1)
C
C     FACTOR THE REMAINING MATRIX
C
      IF (N .EQ. 2) GO TO 400
C
      IA = 3
      IA2 = 1 + LA2
      IA3 = IA2 + 2
      DO 390 I = 2, N-1
C
         CALL CNDOTP(A(IA),LA2,2,A(IA2),2,0,A(IA3),2,I-1,N-I+1,3)
C
         CALL QTC001(A(IA3),2,VMAX,IMAX,N-I+1)
         IMAX = IMAX + I - 1
C
         IF (VMAX.GT.ZTOLSQ) GOTO 200
            IERR = I
            GO TO 800
200      IF (IMAX.EQ.I) GOTO 300
            ITEMP       = IPERM(I)
            IPERM(I)    = IPERM(IMAX)
            IPERM(IMAX) = ITEMP
            CALL QTC003(A(IA),LA2,A(IMAX+IMAX-1),LA2,N)
300      CONTINUE
C
         MAGN = A(IA3)**2 + A(IA3+1)**2
         A(IA3)   = A(IA3) / MAGN
         A(IA3+1) = -A(IA3+1) / MAGN
         CALL CNDOTP(A(IA),LA2,0,A(IA2+LA2),2,LA2,A(IA3+LA2),
     +               LA2,I-1,N-I,3)
         CALL QTC002(A(IA3+LA2),LA2,A(IA3),A(IA3+LA2),LA2,N-I)
         IA = IA + 2
         IA2 = IA2 + LA2
         IA3 = IA3 + LA2 + 2
C
  390 CONTINUE
C
C     FACTOR A(N,N)
C
  400 CONTINUE
      CALL CDOTPR(A(N+N-1),LA2,A(1+(N-1)*LA2),2,TEMP,N-1)
      A(LASTPT)   = A(LASTPT) - TEMP(1)
      A(LASTPT+1) = A(LASTPT+1) - TEMP(2)
C
  420 CONTINUE
      MAGN = A(LASTPT)*A(LASTPT) + A(LASTPT+1)*A(LASTPT+1)
      IF (MAGN.GT.ZTOLSQ) GOTO 500
         IERR = I
         GO TO 800
500   CONTINUE
      A(LASTPT)   = A(LASTPT) / MAGN
      A(LASTPT+1) = -A(LASTPT+1) / MAGN
C
700   IERR = 0
C
  800 CONTINUE
      RETURN
      END
