C***** EUREKM   MULTI TRACE SYMMETRIC TOEPLITZ MATRIX SOVER  MTHADV EXT REL 1.0
C
C   ** COPYRIGHT 1988  QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL EUREKM (A, B, X, EP, Q, V, D, HOLD, ZTOL, NTRC, N, MXN, IERR)
C
C       where
C
C       A     Real input matrix the columns of which contain the  values
C             of the Toeplitz matrix diagonals for each trace.  Typically,
C             these are the autocorrelation coefficients.
C
C       B     Real input matrix the columns of which contain the right side
C             coeffiicents for each trace.
C
C       X     Real output matrix the colmuns of which contain the solution
C             to the Toeplitz matrix problem for each trace.  These are
C             typically the filter coefficients.
C
C       EP    Real output matrix the columns of which contain the prediction
C             error vector for each trace.
C
C       Q     Real work vector.
C
C       V     Real work vector.
C
C       D     Real work vector.
C
C       HOLD  Real work vector.
C
C       ZTOL  Real input scaler singularity tolerance.
C
C       NTRC  Integer input number of traces.  The number of matrix columns.
C
C       N     Integer input number of matrix rows.  The number of values in
C             each trace.
C
C       MXN   Integer input number of matrix rows in the dimensioning of the
C             matrices.
C
C       IERR  Integer output completion code with one of four values:
C               IERR = 0              Normal termination
C               IERR = -1             N is greater than MXN
C               IERR = -N+1           N is less than or equal zero
C               IERR = -NTRC+1        NTRC is less than or equal to zero
C               IERR = -MXN+1         MXN is less than or equal to zero
C               0  <  IERR < (N+1) ,  matrix has a singular principal minor.
C                                     The value indicates for which trace the
C                                     error was encountered.
C
C  DESCRIPTION:
C
C       A Toeplitz matrix is a special matrix form in which the elements of
C       each diagonal have a constant value.   The linear Toeplitz problem is
C       solved by a bordering method.   This recursive algorithm fails if any
C       of the diagonal, principal minors are zero.  Notice that this does not
C       indicate a singluar matrix and it may be possible to solve the
C       matrix with standard Gaussian Elimination techniques.  This Toeplitz
C       solver is setup to work with multiple traces simultaneously to produce
C       a more vectorizable solution.  Notice that the value of MXN should be
C       odd for computers with an interleaved memory (eg. the CRAY 2).
C
C  REFERENCES:
C
C       Robinson, E.A. and Treitel, S. 1980, Gelphysical Signal Analysis
C       (Englewood Cliffs, N.J.: Prentice-Hall), p. 163).
C
C  EXAMPLE:
C
C       CALL EUREKM (A, B, X, EP, Q, V, D, 1.0e-7, 3, 5, 10,
C    &                          IERR)
C
C       INPUT OPERANDS:
C
C       A  =   1.000000       3.000000       7.000000
C              2.000000       2.000000       5.000000
C              3.000000       1.000000       1.000000
C              4.000000       6.000000       2.000000
C              5.000000       2.000000       3.000000
C
C       B  =   3.000000       8.000000       1.000000
C              4.000000       9.000000       9.000000
C              1.000000       7.000000       4.000000
C              2.000000       2.000000       6.000000
C              5.000000       1.000000       7.000000
C
C       OUTPUT OPERANDS:
C
C       X  =   1.166667      -1.431035      -0.4905639
C             -2.000000     - 0.9698284     -0.1446533
C              2.000000       1.913793       1.918241
C              0.9999998      1.530173      -2.144654
C             -0.8333332      1.568966       2.509432
C
C       EP =   1.000000       1.000000       1.000000
C             -1.200000       0.6388891      0.1249998
C              0.000000       1.166667      -0.9375000
C              0.000000      -2.583333       1.229167
C             -0.200000      -0.6111112     -1.208333
C
C       IERR  = 0
C
C  SUBPROGRAMS:
C
C       INTRINSICS:     ABS
C       MATH ADVANTAGE: VMOV, VFIL, VMA, VDIV, VNEG, VSUB
C
C  HISTORY:
C       1) May 1988          T. G. Mattson               Original
C
C----------------------------------------------------------------------------
       SUBROUTINE EUREKM (A, B, X, EP, Q, V, D, ZTOL, NTRC, N, MXN,
     &                          IERR)
       INTEGER  NTRC,     N,        MXN,      IERR
       REAL     A(MXN,*), B(MXN,*), X(MXN,*), EP(MXN,*),
     &          Q(MXN),   V(MXN),   D(MXN),   HOLD, ZTOL
       INTEGER  I, L, L1, L2, J, K, L3
       REAL     QH, DH, AH
C
C----------------------------------------------------------------------------
C
       IF (N .LE. 0) THEN
          IERR = -N + 1
          RETURN
       ELSE IF( N .GT. MXN) THEN
          IERR = -1
          RETURN
       ELSE IF( NTRC .LE. 0) THEN
          IERR = -NTRC + 1
          RETURN
       ELSE IF( MXN .LE. 0) THEN
          IERR = -MXN + 1
          RETURN
       ENDIF
C
C  Initialize recursion
C
       IERR = 0
       DO 100 I=1,NTRC
          V(I)    = A(1,I)
          D(I)    = A(2,I)
          EP(1,I) = 1.0
          IF( ABS (V(I)) .LE. ZTOL) THEN
             IERR = I
             RETURN
          ENDIF
 100   CONTINUE
       CALL VDIV (B(1,1), MXN, V, 1, X(1,1), MXN, NTRC)
       CALL VMUL (X(1,1), MXN, D, 1, Q, 1, NTRC)
       IF( N .EQ. 1)THEN
           RETURN
       ENDIF
C
C  Main recursion loop
C
       DO 400 L=2,N
C
C  Find the "prediction error operator" MATRIX, EP
C
          DO 110 I=1,NTRC
             EP(L,I) = -D(I)/V(I)
 110      CONTINUE
          IF(L .GT. 2) THEN
             L1 = (L-2)/2
             L2 = (L1+1)
             DO 210 J=2,L2
                DO 200 I=1,NTRC
                   HOLD = EP(J,I)
                   K    = L-J+1
                   EP(J,I) = EP(J,I) + EP(L,I)*EP(K,I)
                   EP(K,I) = EP(K,I) + EP(L,I)*HOLD
 200            CONTINUE
 210         CONTINUE
             IF (2*L1 .NE. L-2) THEN
                CALL VMA (EP(L,1), MXN, EP(L2+1,1), MXN, EP(L2+1,1),
     &                    MXN, EP(L2+1,1), MXN, NTRC)
             ENDIF
          ENDIF
C
C  Find the solution vector, X
C
          DO 250 I=1,NTRC
             V(I) = V(I) + EP(L,I)*D(I)
             IF(ABS (V(I)) .LE. ZTOL)THEN
                IERR = I
                RETURN
             ENDIF
 250      CONTINUE
C
C  Use D as work space in the VSUB call to yield vectorizable code
C
          CALL VSUB (B(L,1), MXN, Q, 1, D, 1, NTRC)
          CALL VDIV (D, 1, V, 1, X(L,1), MXN, NTRC)
          L3    = L-1
          DO 300 J=1,L3
             K    = L-J+1
             CALL VMA (X(L,1), MXN, EP(K,1), MXN, X(J,1), MXN,
     &                              X(J,1), MXN, NTRC)
 300      CONTINUE
C
C  Exit test
C
          IF (L .EQ. N)THEN
              RETURN
          ENDIF
C
C  Setup for next iteration
C
          DO 385 J=1,NTRC
             QH = 0.0
             DH = 0.0
             DO 375 I=1,L
                AH = A(L-I+2,J)
                DH = DH + AH*EP(I,J)
                QH = QH + AH*X(I,J)
 375         CONTINUE
             Q(J) = QH
             D(J) = DH
 385      CONTINUE
 400   CONTINUE
C
        RETURN
        END
 
