C***** QTC052  COMPLEX FFT (MIXED-RADIX)       REV 1.0     JAN 88
C
C  PURPOSE:
C       COMPUTES A FORWARD OR INVERSE COMPLEX IN-PLACE FFT WHERE
C       THE LENGTH OF THE ARRAY IS FACTORABLE BY 2, 3, 5, AND/OR 7.
C
C  LANGUAGE:
C       FORTRAN 77
C
C  HISTORY:
C       ORIGINAL                DEC 86          D.R. BENUA, QTC
C       MATH ADVANTAGE VERSION  JAN 88          R.D. COLEMAN, QTC
C          CHANGED SUBROUTINE NAMES AND REMOVED SCALING.
C
C  CALLING FORMAT:
C       CALL QTC052 (A, B, INC, N, IFLG, NFAC, M, CNST, STABLE,
C      &             CTABLE, ISRC, IDST, NPERM)
C
C  PARAMETERS:
C
C       A       REAL INPUT/OUTPUT VECTOR OF LENGTH N
C               REAL ARRAY ELEMENTS.
C
C       B       REAL INPUT/OUTPUT VECTOR OF LENGTH N
C               IMAGINARY ARRAY ELEMENTS.
C
C       INC     INTEGER INPUT SCALAR
C               STRIDE OF VECTORS A AND B.
C
C       N       INTEGER INPUT SCALAR
C               LENGTH OF VECTORS A, B, STABLE, AND
C               CTABLE.
C
C       IFLG    INTEGER INPUT SCALAR
C               TRANSFORM DIRECTION FLAG, POSITIVE
C               FOR FORWARD, NEGATIVE FOR INVERSE.
C
C       NFAC    INTEGER INPUT ARRAY OF LENGTH M
C               PRIME FACTORS OF N.
C
C       M       INTEGER INPUT SCALAR
C               NUMBER OF PRIME FACTORS IN ARRAY NFAC.
C
C       CNST    REAL INPUT ARRAY OF LENGTH 11
C               CONSTANTS, DEPENDENT ON IFLG.
C
C       STABLE  REAL INPUT ARRAY OF LENGTH N
C               SINE TWIDDLE FACTORS.
C
C       CTABLE  REAL INPUT ARRAY OF LENGTH N
C               COSINE TWIDDLE FACTORS.
C
C       ISRC    INTEGER INPUT ARRAY OF LENGTH NPERM
C               PERMUTATION INDICIES.
C
C       IDST    INTEGER INPUT ARRAY OF LENGTH NPERM
C               PERMUTATION INDICIES.
C
C       NPERM   INTEGER INPUT SCALAR
C               NUMBER OF ELEMENTS IN ARRAYS ISRC
C               AND IDST, ALWAYS LESS THAN N.
C
C  DESCRIPTION:
C       THIS ROUTINE COMPUTES THE FORWARD OR INVERSE FAST FOURIER
C       TRANSFORM OF A COMPLEX ARRAY.  THE LENGTH OF THE ARRAY,
C       N, MUST BE FACTORABLE BY THE PRIME FACTORS 2, 3, 5, AND/OR
C       7.  THE TRANFORM IS PERFORMED IN PLACE USING PRE-COMPUTED
C       SINE, COSINE, FACTOR, CONSTANT, AND PERMUTATION TABLES.
C       THE TABLES ARE UNIQUE FOR EACH VALUE OF N, INC, AND IFLG.
C
C       THE ALGORITHM USED HERE IS ADAPTED FROM THE MIXED RADIX
C       FFT PROGRAM BY R. C. SINGLETON PUBLISHED IN "PROGRAMS FOR
C       DIGITAL SIGNAL PROCESSING", IEEE PRESS, 1979, SEC. 1.4.
C       THE PRIMARY DIFFERENCES ARE THAT THE PRIME FACTORS ARE
C       RESTRICTED TO 2, 3, 5, AND 7, SINE AND COSINE TABLES ARE
C       USED, A PERMUTATION TABLE IS USED, AND THE ORDER OF THE
C       RESULT ARRAY IS FLIPED ABOUT THE N/2 POINT. (EG. FOR A
C       FORWARD TRANFORM, THE POSITIVE FREQUENCY COEFFICIENTS
C       APPEAR IN THE FIRST HALF OF THE ARRAY RATHER THAN IN THE
C       LAST HALF AS THEY DO IN THE SINGLETON ALGORITHM)  THE
C       RESULTS PRODUCED BY THIS ROUTINE FOLLOW THE ORDER DEFINED
C       IN THE FORMAL DEFINITION OF THE DISCRETE FOURIER TRANSFORM
C       GIVEN BY MOST AUTHORS.
C
C       THE DESIGN STRATEGY FOR THIS ROUTINE ASSUMES THAT IT WILL
C       BE CALLED MANY TIMES WITH THE SAME VALUES OF N, INC, AND
C       IFLG, SO THAT THE OVERHEAD ASSOCIATED WITH CREATING THE
C       TABLES IS INSIGNIFICANT.  SEE THE ROUTINE QTC051.FOR FOR
C       THE TABLE INITIALIZATION ALGORITHMS.  WHEN SETTING UP FOR
C       AN INVERSE TRANSFORM AFTER DOING A NUMBER OF FORWARD
C       TRANSFORMS WITH THE SAME N AND INC, IT IS ONLY NECESSARY
C       TO CHANGE THE SIGNS OF THE VALUES IN STABLE AND SOME OF
C       THE CONSTANTS IN CNST.
C
C       THE CODE IN THESE SUBROUTINES WAS DESIGNED TO PROVIDE A CLEAR
C       MODEL FOR A TRANSLATION INTO MICROCODE OR ASSEMBLY LANGUAGE.
C       THE AUTHOR DOES NOT CLAIM THAT THIS IS THE BEST WAY TO
C       IMPLEMENT THE ALGORITHM FOR A GENERAL FORTRAN MACHINE.
C       THE DESIGN ALSO ASSUMES THAT A LARGE CACHE OR FAST MEMORY
C       IS AVAILABLE TO STORE FULL LENGTH SINE AND COSINE TABLES AND
C       PERMUTATION VECTORS. OBVIOUSLY, MEMORY REQUIREMENTS FOR THE
C       TWIDDLE FACTORS TABLES COULD BE REDUCED, BUT ONLY AT THE
C       EXPENSE OF INCREASED COMPUTATION TO DERIVE THEM AT RUN TIME.
C       REGARDING THE PERMUTATION VECTORS, NOTE THAT IN THE SORTING
C       LOOP AT THE END OF THIS ROUTINE THERE IS A CONDITIONAL TEST.
C       THE RESULT OF THIS TEST DEPENDS ONLY ON THE PRECOMPUTED
C       VALUES IN THE ARRAYS ISRC AND IDST.  IF IT PROVES MORE
C       EFFICIENT ON THE TARGET MACHINE, THE LOGICAL VALUE OF THE
C       TEST COULD BE ENCODED IN A THIRD PERMUTATION VECTOR, OR IN
C       THE HIGH BITS OF THE PRESENT PERMUTATION VECTORS.
C
C  SUBPROGRAMS CALLED:
C       NONE
C
C  ERROR CONDITIONS:
C       NONE
C
C-----------------------------------------------------------------------
C
        SUBROUTINE QTC052 (A, B, INC, N, IFLG, NFAC, M, CNST, STABLE,
     &                     CTABLE, ISRC, IDST, NPERM)
C
        REAL A(1), B(1), CNST(1), STABLE(1), CTABLE(1)
        REAL AA, AJ, AJM, AJP, AK, AKM, AKP
        REAL AT1, AT2, AT3, AT4, AT5, AT6, AT7
        REAL BB, BJ, BJM, BJP, BK, BKM, BKP
        REAL BT1, BT2, BT3, BT4, BT5, BT6, BT7
        REAL C1, C2, S1, S2
        INTEGER INC, N, IFLG, NFAC(1), M, ISRC(1), IDST(1), NPERM
C
        INTEGER I,NT,KSPAN,NN,KK,K,KSPNN,K1,K2,K3,K4,K5,K6
        INTEGER INCD,THETA1,THETA2
C
        IF (N .LT. 2 .OR. M .LE. 0) RETURN
C
C---------------------------------
C  Initialize indexing variables
C---------------------------------
C
        NT = INC*N
        KSPAN = NT
        NN = NT - INC
        I = 0
        INCD = 1
C
C-------------------------------------
C  Top of the main loop, one pass of
C  this loop for each factor in NFAC
C-------------------------------------
C
20      KK = 1
        IF (I .GT. 0) INCD = INCD * NFAC(I)
        I = I + 1
        K = NFAC(I)
        KSPNN = KSPAN
        KSPAN = KSPAN/K
C
C------------------------------------------------------------------
C  Radix 2 pass, this code includes twiddle factor multiplications.
C  Note that there are two loops here, the first one is used when
C  there are no twiddle factors.
C------------------------------------------------------------------
C
        IF (K .NE. 2) GOTO 80
        K1 = KSPAN + 2
 
  30    K2 = KK + KSPAN
        AK = A(K2)
        BK = B(K2)
        A(K2) = A(KK) - AK
        B(K2) = B(KK) - BK
        A(KK) = A(KK) + AK
        B(KK) = B(KK) + BK
        KK = K2 + KSPAN
        IF (KK .LE. NN) GO TO 30
C
        KK = KK - NN
        IF (KK .LE. INC) GO TO 30
C
        IF (KK .GT. KSPAN) GO TO 190
 
  40    THETA1 = INCD
        GO TO 60
 
  50    THETA1 = THETA1 + INCD
        IF (THETA1 .GT. N) THETA1 = THETA1 - N
C
  60    C1 = CTABLE(THETA1)
        S1 = STABLE(THETA1)
 
  70    K2 = KK + KSPAN
        AK = A(KK) - A(K2)
        BK = B(KK) - B(K2)
        A(KK) = A(KK) + A(K2)
        B(KK) = B(KK) + B(K2)
        A(K2) = C1*AK - S1*BK
        B(K2) = S1*AK + C1*BK
        KK = K2 + KSPAN
        IF (KK .LT. NT) GO TO 70
C
        K2 = KK - NT
        THETA1 = N/2 - THETA1
        IF (THETA1 .LE. 0) THETA1 = THETA1 + N
        KK = K1 - K2
        IF (KK .GT. K2) GO TO 60
        KK = KK + INC
        IF (KK .LE. K1/2) GO TO 50
C
        K1 = K1 + INC + INC
        KK = (K1-KSPAN)/2 + INC
        IF (KK .LE. INC+INC) GO TO 40
        GO TO 20
C
C----------------------------------------------------------------
C  Radix 3 Pass.  Note that this pass does not include twiddle
C  factor multiplication.  See code at label 130.
C----------------------------------------------------------------
C
80      IF (K .NE. 3) GOTO 100
C
90      K1 = KK + KSPAN
        K2 = K1 + KSPAN
        AK = A(KK)
        BK = B(KK)
        AJ = A(K1) + A(K2)
        BJ = B(K1) + B(K2)
        A(KK) = AK + AJ
        B(KK) = BK + BJ
        AK = -0.5 * AJ + AK
        BK = -0.5 * BJ + BK
        AJ = (A(K1) - A(K2)) * CNST(1)
        BJ = (B(K1) - B(K2)) * CNST(1)
        A(K1) = AK - BJ
        B(K1) = BK + AJ
        A(K2) = AK + BJ
        B(K2) = BK - AJ
        KK = K2 + KSPAN
        IF (KK.LT.NN) GO TO 90
C
        KK = KK - NN
        IF (KK.LE.KSPAN) GO TO 90
C
        GO TO 130
C
C-------------------------------------------------------------------
C  Radix 5 Pass. See code at 130 for twiddle factor multiplication
C-------------------------------------------------------------------
C
 100    IF (K .NE. 5) GOTO 120
C
 110    K1 = KK + KSPAN
        K2 = K1 + KSPAN
        K3 = K2 + KSPAN
        K4 = K3 + KSPAN
        AKP = A(K1) + A(K4)
        AKM = A(K1) - A(K4)
        BKP = B(K1) + B(K4)
        BKM = B(K1) - B(K4)
        AJP = A(K2) + A(K3)
        AJM = A(K2) - A(K3)
        BJP = B(K2) + B(K3)
        BJM = B(K2) - B(K3)
        AA = A(KK)
        BB = B(KK)
        A(KK) = AA + AKP + AJP
        B(KK) = BB + BKP + BJP
        AK = AKP * CNST(4) + AJP * CNST(2) + AA
        BK = BKP * CNST(4) + BJP * CNST(2) + BB
        AJ = AKM * CNST(5) + AJM * CNST(3)
        BJ = BKM * CNST(5) + BJM * CNST(3)
        A(K1) = AK - BJ
        A(K4) = AK + BJ
        B(K1) = BK + AJ
        B(K4) = BK - AJ
        AK = AKP * CNST(2) + AJP * CNST(4) + AA
        BK = BKP * CNST(2) + BJP * CNST(4) + BB
        AJ = AKM * CNST(3) - AJM * CNST(5)
        BJ = BKM * CNST(3) - BJM * CNST(5)
        A(K2) = AK - BJ
        A(K3) = AK + BJ
        B(K2) = BK + AJ
        B(K3) = BK - AJ
        KK = K4 + KSPAN
        IF (KK.LT.NN) GO TO 110
C
        KK = KK - NN
        IF (KK.LE.KSPAN) GO TO 110
C
        GO TO 130
C
C--------------------------------------------
C  Radix 7 Pass.  See code at label 130 for
C  twiddle factor multiplication.
C--------------------------------------------
C
120     K1 = KK + KSPAN
        K2 = K1 + KSPAN
        K3 = K2 + KSPAN
        K4 = K3 + KSPAN
        K5 = K4 + KSPAN
        K6 = K5 + KSPAN
        AT2 = A(K1) + A(K6)
        BT2 = B(K1) + B(K6)
        AT3 = A(K1) - A(K6)
        BT3 = B(K1) - B(K6)
        AT4 = A(K2) + A(K5)
        BT4 = B(K2) + B(K5)
        AT5 = A(K2) - A(K5)
        BT5 = B(K2) - B(K5)
        AT6 = A(K3) + A(K4)
        BT6 = B(K3) + B(K4)
        AT7 = A(K3) - A(K4)
        BT7 = B(K3) - B(K4)
C
        AA = A(KK)
        BB = B(KK)
        A(KK) = AA + AT2 + AT4 + AT6
        B(KK) = BB + BT2 + BT4 + BT6
C
        AK = AA + AT2*CNST(6) + AT4*CNST(8) + AT6*CNST(10)
        BK = BB + BT2*CNST(6) + BT4*CNST(8) + BT6*CNST(10)
        AJ = AT3*CNST(7) + AT5*CNST(9) + AT7*CNST(11)
        BJ = BT3*CNST(7) + BT5*CNST(9) + BT7*CNST(11)
        A(K1) = AK - BJ
        B(K1) = BK + AJ
        A(K6) = AK + BJ
        B(K6) = BK - AJ
C
        AK = AA + AT2*CNST(8) + AT4*CNST(10) + AT6*CNST(6)
        BK = BB + BT2*CNST(8) + BT4*CNST(10) + BT6*CNST(6)
        AJ = AT3*CNST(9) - AT5*CNST(11) - AT7*CNST(7)
        BJ = BT3*CNST(9) - BT5*CNST(11) - BT7*CNST(7)
        A(K2) = AK - BJ
        B(K2) = BK + AJ
        A(K5) = AK + BJ
        B(K5) = BK - AJ
C
        AK = AA + AT2*CNST(10) + AT4*CNST(6) + AT6*CNST(8)
        BK = BB + BT2*CNST(10) + BT4*CNST(6) + BT6*CNST(8)
        AJ = AT3*CNST(11) - AT5*CNST(7) + AT7*CNST(9)
        BJ = BT3*CNST(11) - BT5*CNST(7) + BT7*CNST(9)
        A(K3) = AK - BJ
        B(K3) = BK + AJ
        A(K4) = AK + BJ
        B(K4) = BK - AJ
C
        KK = KK + KSPNN
        IF (KK.LE.NN) GO TO 120
C
        KK = KK - NN
        IF (KK.LE.KSPAN) GO TO 120
C
C----------------------------------
C  Twiddle Factor Multiplications
C----------------------------------
C
130     IF (I.EQ.M) GO TO 190
        KK = INC + 1
C
140     THETA2 = INCD
        GO TO 160
C
150     THETA2 = THETA1 + INCD
        IF (THETA2 .GT. N) THETA2 = THETA2 - N
C
160     KK = KK + KSPAN
        THETA1 = THETA2
C
170     C2 = CTABLE(THETA2)
        S2 = STABLE(THETA2)
C
180     AK = A(KK)
        A(KK) = C2*AK - S2*B(KK)
        B(KK) = S2*AK + C2*B(KK)
        KK = KK + KSPNN
        IF (KK.LE.NT) GO TO 180
C
        THETA2 = THETA2 + THETA1
        IF (THETA2 .GT. N) THETA2 = THETA2 - N
        KK = KK - NT + KSPAN
        IF (KK.LE.KSPNN) GO TO 170
C
        KK = KK - KSPNN + INC
        IF (KK .LE. KSPAN) GO TO 150
C
        KK = KK - KSPAN
        IF (KK.GT.0) GO TO 20
C
        KK = KK + INC + INC
C       GOTO 140
C
C--------------------------------------------
C  Sort the arrays A and B according to the
C  index values stored in ISRC and IDST.
C--------------------------------------------
C
190     IF (NPERM .EQ. 0) RETURN
C
        AT1 = A(IDST(1))
        BT1 = B(IDST(1))
        A(IDST(1)) = A(ISRC(1))
        B(IDST(1)) = B(ISRC(1))
C
        DO 200 I=2,NPERM
                AT2 = A(IDST(I))
                BT2 = B(IDST(I))
                IF (IDST(I-1) .EQ. ISRC(I)) THEN
                        A(IDST(I)) = AT1
                        B(IDST(I)) = BT1
                ELSE
                        A(IDST(I)) = A(ISRC(I))
                        B(IDST(I)) = B(ISRC(I))
                ENDIF
                AT1 = AT2
                BT1 = BT2
200     CONTINUE
C
        RETURN
        END
