C*****  RMFUFC  Real Matrix Factor                MATH ADVANTAGE REL 2.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL RMFUFC (A,NRA,N,ZTOL,IPERM,IERR)
C
C       where,
C
C       A       Real input/output matrix.  On input, A contains the
C               matrix to be factored.  On output, A contains the
C               factored form.
C
C       NRA     Integer input number of rows in full matrix A.
C
C       N       Integer input order of submatrix A.
C
C       ZTOL    Real input scalar, pivot zero tolerance.
C
C       IPERM   Integer output permutation vector of length N.
C
C       IERR    Integer output completion code:
C                   =0 if the routine terminated normally.
C                   >0 if the routine aborted because a pivot
C                      was less than or equal to ZTOL.
C                      This implies the matrix is singular.
C                      The value of IERR is the index of the column
C                      where it aborted.
C
C
C  DESCRIPTION
C
C       Using partial pivoting, this routine factors a real
C       full matrix A into LU form.  L is lower triangular and
C       U is upper triangular. Both L and U are superimposed in
C       matrix A. The diagonal elements of L are reciprocated.
C
C       The permutation vector, IPERM, is output by this routine
C       for use as input to RMFUSV.
C
C
C  REFERENCE
C
C       C. R. Rao.  1962.  A note on a generalized inverse of a
C       matrix with application to problems in mathematical
C       statistics.  J. R. Statis. Soc., Vol. B24,  pp. 152-
C       158.
C
C       M. J. R. Healy.  1968.  Triangular decomposition of a
C       symmetric matrix.  Appl. Statis., Vol. 17,  pp. 195-
C       197.
C
C       G. W. Stewart.  1973.  Introduction to matrix computa-
C       tions.  New York: Academic Press.
C
C
C  EXAMPLE
C
C       CALL RMFUFC (A,5,5,ZTOL,IPERM,IERR)
C
C       Input Operands:
C
C       A =  0.000    -1.000     -5.000     11.000   8.000
C            8.000    16.000     -8.000     24.000   8.000
C           -4.000    -5.000      2.000     -4.000  -3.000
C           -3.000   -10.000      3.000    -13.000   5.000
C            2.000     6.000      6.000     -8.000  -2.000
C
C       ZTOL = 1.0E-5
C
C       Output Operands:
C
C       A =  0.125   2.000  -1.000   3.000   1.000
C           -3.000  -0.250   0.000   1.000  -2.000
C            2.000   2.000   0.125  -2.000   0.000
C            0.000  -1.000  -5.000   0.500   3.000
C           -4.000   3.000  -2.000   1.000   0.250
C
C       IPERM = 2
C               4
C               5
C               1
C               3
C
C       IERR = 0
C
C  HISTORY
C         1) Nov 84     D. Cooper       Original.
C                       R. Coleman
C         2) Jan 88     L. Shanbeck     Expanded IERR functionality
C
      SUBROUTINE RMFUFC(A,NRA,N,ZTOL,IPERM,IERR)
C
      INTEGER NRA,N,I,IERR,IMAX,IPERM(1),ITEMP,IA,IA2,LASTPT
      REAL    A(1),TEMP,VMAX,ZTOL
C
C   *******************************************************************
C
C     DO VALIDITY CHECKS AND INITIALIZATION
C
      IF (N.LE.0 .OR. NRA.LT.N) GO TO 700
C
      DO 110 I = 1, N
         IPERM(I) = I
110   CONTINUE
C
C     FACTOR FIRST COLUMN AND ROW
C
      LASTPT = N + (N-1) * NRA
      IF (N .EQ. 1) GO TO 420
C
      CALL MAXMGV(A,1,VMAX,IMAX,N)
C
      IF (VMAX.GT.ZTOL) GOTO 130
         IERR = 1
         GO TO 800
130   IF (IMAX.EQ.1) GOTO 150
         IPERM(1)    = IMAX
         IPERM(IMAX) = 1
         CALL VSWAP(A(1),NRA,A(IMAX),NRA,N)
150   CONTINUE
C
      A(1) = 1.0 / A(1)
      CALL VSMUL(A(1+NRA),NRA,A(1),A(1+NRA),NRA,N-1)
C
C     FACTOR THE REMAINING MATRIX
C
      IF (N .EQ. 2) GO TO 400
C
      IA = 1 + NRA
      IA2 = 2 + NRA
      DO 390 I = 2, N-1
C
         CALL RNDOTP(A(I),NRA,1,A(IA),1,0,A(IA2),1,I-1,N-I+1,3)
C
         CALL MAXMGV(A(IA2),1,VMAX,IMAX,N-I+1)
         IMAX = IMAX + I - 1
C
         IF (VMAX.GT.ZTOL) GOTO 200
            IERR = I
            GO TO 800
200      IF (IMAX.EQ.I) GOTO 300
            ITEMP       = IPERM(I)
            IPERM(I)    = IPERM(IMAX)
            IPERM(IMAX) = ITEMP
            CALL VSWAP(A(I),NRA,A(IMAX),NRA,N)
300      CONTINUE
C
         A(IA2) = 1.0 / A(IA2)
         CALL RNDOTP(A(I),NRA,0,A(IA+NRA),1,NRA,A(IA2+NRA),NRA,I-1,N-I,
     +    3)
         CALL VSMUL(A(IA2+NRA),NRA,A(IA2),A(IA2+NRA),NRA,N-I)
         IA = IA + NRA
         IA2 = IA2 + NRA + 1
C
  390 CONTINUE
C
C     FACTOR A(N,N)
C
  400 CONTINUE
      CALL DOTPR(A(N),NRA,A(1+(N-1)*NRA),1,TEMP,N-1)
      A(LASTPT) = A(LASTPT) - TEMP
C
  420 CONTINUE
      IF (ABS(A(LASTPT)).GT.ZTOL) GOTO 500
         IERR = N
         GO TO 800
500   CONTINUE
      A(LASTPT) = 1.0 / A(LASTPT)
C
700   IERR = 0
C
  800 CONTINUE
      RETURN
      END
