C*****  SCNRM2    Euclidean Length of Single Prec Complex    MATH ADV REL 3.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       SW = SCNRM2 (N,CX,INCX)
C
C       where,
C
C       N       Integer input element count.
C
C       CX      Complex input vector.
C
C       INCX    Integer input stride for vector CX.
C               May not be negative.
C
C
C  DESCRIPTION
C
C       This function returns the Euclidean length, also know as the
C       Euclidean norm, of the elements of complex vector CX.
C
C       If N <= 0 return with result = 0.0
C
C       Four phase method using two calculated constants that are
C         CUTLO = SQRT(U/EPS)
C         CUTHI = SQRT(V)
C       where
C         EPS = smallest number such that   EPS + 1.0 > 1.0
C         U   = smallest positive number   (underflow limit)
C         V   = largest number             (overflow  limit)
C
C       Brief outline of algorithm..
C
C       Phase 1    scans zero components.
C       Move to phase 2 when a component is nonzero and <= CUTLO
C       Move to phase 3 when a component is > CUTLO
C       move to phase 4 when a component is >= CUTHI/2*N
C
C
C  REFERENCE
C
C       J.J. Dongarra and C.B. Moler, J.R. Bunch, G.W. Stewart.  1979.
C       LINPACK Users' Guide.  Appendix A: Basic Linear Algebra Subprograms.
C       SIAM
C
C  EXAMPLE
C
C       SW = SCNRM2 (5,CX,1)
C
C       Input Operands:
C
C       CX = ( 1.000, 1.000)
C            ( 2.000, 2.000)
C            (-3.000, 3.000)
C            ( 4.000, 4.000)
C            ( 5.000,-5.000)
C
C       Output Result:
C
C       SW = 10.48809
C
C  HISTORY
C         1) Jan 88     L. Shanbeck     Original.
C
C
      REAL FUNCTION SCNRM2(N,CX,INCX)
      COMPLEX CX(1)
      REAL CUTLO,CUTHI,HITEST,SUM,XMAX,ZERO,ONE
      REAL ABSX,ABSR,ABSI
      REAL EPS,TINY,HUGE,S
      INTEGER N,INCX,I,J,IMAG
C
C   SMALL is used to prevent values below SMALL from being returned.
C   See the end of the program for usage.  Only for Microsoft Fortran.
C     REAL SMALL
C     DATA SMALL / 8.43E-37 /
C
      DATA ZERO, ONE /0.0, 1.0/
      DATA EPS, TINY, HUGE /0.0, 0.0, 0.0/
C
      SCNRM2  = ZERO
      IF(N.LE.0 .OR. INCX.LT.0) GO TO 999
C
C  SET MACHINE DEPENDENT VALUES, IF NECESSARY
      IF (HUGE .EQ. 0.0) THEN
C
         EPS = 1.0E0
   1     EPS = EPS/2.0E0
         S = 1.0E0 + EPS
         IF (S .GT. 1.0E0) GO TO 1
         EPS = 2.0E0*EPS
C
         S = 1.0E0
   2     TINY = S
         S = S/16.0E0
         IF (S*1.0 .NE. 0.0E0) GO TO 2
         TINY = (TINY/EPS)*100.0E0
         HUGE = 1.0E0/TINY
C
         CUTLO = SQRT(TINY/EPS)
         CUTHI = SQRT(HUGE)
      ENDIF
C
      SUM = ZERO
      NN = N * INCX
C
C  FOR COMPLEX SET HITEST = CUTHI/(2*N)
      HITEST = CUTHI/FLOAT( 2*N )
      XMAX = ZERO
C                        BEGIN MAIN LOOP
      I = 1
      IMAG = 1
   20 CONTINUE
      IF (IMAG .EQ. 1) THEN
         ABSX = ABS(REAL(CX(I)))
         IMAG = 0
      ELSE
         ABSX = ABS(AIMAG(CX(I)))
         IMAG = 1
      ENDIF
C
C                        PHASE 1.  SUM IS ZERO
C
      IF( ABSX .EQ. ZERO) THEN
         CONTINUE
C
C                        PHASE 2.  SUM IS SMALL
C                        SCALE TO AVOID DESTRUCTIVE UNDERFLOW.
C
      ELSE IF( ABSX .LE. CUTLO) THEN
         IF( ABSX .GT. XMAX) THEN
            SUM = ONE + SUM * (XMAX / ABSX)**2
            XMAX = ABSX
         ELSE
            SUM = SUM + (ABSX/XMAX)**2
         ENDIF
C
C                        PHASE 3.  SUM IS MID-RANGE.  NO SCALING.
C
      ELSE
         SUM = (SUM * XMAX) * XMAX
C  CHECK FOR LEFTOVER IMAGINARY TO BE PROCESSED SEPARATELY...
         IF (IMAG .EQ. 1) THEN
            I = I + INCX
            IF(ABSX .GE. HITEST) THEN
               J = I
               XMAX = ABSX
               SUM = (SUM / XMAX) / XMAX
               SUM = SUM + (ABSX/XMAX)**2
               GO TO 110
            ELSE
               SUM = SUM + ABSX**2
            ENDIF
         ENDIF
C  LOOP THRU REMAINING VECTOR ELEMENTS...
         DO 95 J =I,NN,INCX
            ABSR = ABS(REAL(CX(J)))
            ABSI = ABS(AIMAG(CX(J)))
            IF(ABSR .GE. HITEST .OR. ABSI .GE. HITEST) GO TO 100
            SUM = SUM + ABSR**2 + ABSI**2
   95    CONTINUE
         SCNRM2 = SQRT( SUM )
         GO TO 300
C
C                        PHASE 4.
C                        SUM IS LARGE.  SCALE TO AVOID OVERFLOW.
C
  100    IF (ABSR .GT. HITEST) THEN
            XMAX = ABSR
         ELSE
            XMAX = ABSI
         ENDIF
         SUM = (SUM / XMAX) / XMAX
C  LOOP THRU REMAINING VECTOR ELEMENTS...
  110    DO 120 I=J,NN,INCX
            ABSR = ABS(REAL(CX(I)))
            IF( ABSR .GT. XMAX) THEN
               SUM = ONE + SUM * (XMAX / ABSR)**2
               XMAX = ABSR
            ELSE
               SUM = SUM + (ABSR/XMAX)**2
            ENDIF
            ABSI = ABS(AIMAG(CX(I)))
            IF( ABSI .GT. XMAX) THEN
               SUM = ONE + SUM * (XMAX / ABSI)**2
               XMAX = ABSI
            ELSE
               SUM = SUM + (ABSI/XMAX)**2
            ENDIF
  120    CONTINUE
         IMAG = 1
      ENDIF
C
C                        END OF MAIN LOOP.
  200 CONTINUE
      IF (IMAG .EQ. 0) GO TO 20
      I = I + INCX
      IF (I .LE. NN) GO TO 20
C
C              COMPUTE SQUARE ROOT AND ADJUST FOR SCALING.
C
      SCNRM2 = XMAX * SQRT(SUM)
  300 CONTINUE
C
C *** 6 Jan 84, J. Fried :
C     This next line of FORTRAN was added to prevent values below
C     MS-FORTRAN's specified minimum from being returned.  Such
C     numbers have appeared during testing of the SQRDC routine
C     (test #9 in SQ).  The value of SMALL was taken from the
C     MS-FORTRAN manual.  Remember, this only applies to PC/MS-FORTRAN.
C
C     IF ( SCNRM2 .LE. SMALL ) SCNRM2 = SMALL
999   RETURN
      END
