C*****  SROTG   Construct Givens Plane Rotation             MATH ADV REL 3.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL SROTG (SA,SB,SC,SS)
C
C       where,
C
C       SA      Real input/output scalar.
C               On input, first rotational elimination parameter.
C               On output, value of R as described below.
C
C       SB      Real input/output scalar.
C               On input, second rotational elimination parameter.
C               On output, value of Z as described below.
C
C       SC      Real output scalar.  Cosine of angle of Givens rotation.
C
C       SS      Real output scalar.  Sine of angle of Givens rotation.
C
C
C  DESCRIPTION
C
C       This routine constructs Givens plane rotation variables as follows:
C
C                  R = (+/-)SQRT(SA**2 + SB**2)
C            where the sign is determined by the sign of SA or SB
C            whichever has the larger absolute value.
C
C                  SC = SA/R  and  SS = SB/R    if R <> 0.0
C                  SC = 1.0   and  SS = 0.0     if R  = 0.0
C
C                  Z = SS      if ABS(SA) > ABS(SB)
C                  Z = 1.0/SC  if ABS(SB) >= ABS(SA) and SC <> 0.0
C                  Z = 1.0     if SC = 0.0
C                  Z = 0.0     if SA = SB = 0.0
C
C       These values then satisfy the matrix equation
C
C                  | SC  SS|   |SA|     |R|
C                  |-SS  SC|   |SB|  =  |0|
C
C       R is returned in SA, and Z is returned in SB.
C
C  REFERENCE
C
C       J.J. Dongarra and C.B. Moler, J.R. Bunch, G.W. Stewart.  1979.
C       LINPACK Users' Guide.  Appendix A: Basic Linear Algebra Subprograms.
C       SIAM
C
C       Lawson, C.L. and R.J. Hanson, D.R. Kincaid, F.T. Krogh.
C       September, 1979.  Basic Linear Algebra Subprograms for Fortran Usage.
C       ACM Trans. Math. Software, Vol 5, Number 3
C
C  EXAMPLE
C
C       CALL SROTG (SA,SB,SC,SS)
C
C       Input Operands:
C
C       SA =   6.000
C
C       SB =  -8.000
C
C       Output Operands:
C
C       SA =  -10.00
C
C       SB =  -1.667
C
C       SC =  -0.600
C
C       SS =   0.800
C
C  HISTORY
C         1) Jan 88     L. Shanbeck     Original.
C
C
      SUBROUTINE SROTG(SA,SB,SC,SS)
      REAL SA,SB,SC,SS,RMAX,SCALE,R,Z
C
      IF (ABS(SA) .GT. ABS(SB)) THEN
         RMAX = SA
      ELSE
         RMAX = SB
      ENDIF
      SCALE = ABS(SA) + ABS(SB)
      IF (SCALE .EQ. 0.0) THEN
         SC = 1.0
         SS = 0.0
         R = 0.0
         Z = 0.0
      ELSE
         R = SCALE*SQRT((SA/SCALE)**2 + (SB/SCALE)**2)
         R = SIGN(1.0,RMAX)*R
         SC = SA/R
         SS = SB/R
         Z = 1.0
         IF (ABS(SA).GT.ABS(SB)) Z = SS
         IF (ABS(SB).GE.ABS(SA) .AND. SC.NE.0.0) Z = 1.0/SC
      ENDIF
      SA = R
      SB = Z
      RETURN
      END
