C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c     at3d : Arbitrary Traverse [extraction] from 3D Post Stack Data

c     Program Changes:

c      - original written: October/95

c     Program Description:

c      - this routine extracts from a post-stack 3d volume of seismic data
c        an arbitrary 2D traverse [fence diagram-like].  Input to the routine
c        includes a USP format seismic volume and vertex control either on the
c        command line or in an XSD pickfile.  Output is a USP format seismic
c        stack dataset of nrec records of 1 trace each.  For a more detailed
c        picture see ~usp/src/cmd/at3d.F

c Maintenance Notes: [Garossino]
c
c The Vetex control is contained in the following arrays:
c
c     Vertex_XY[] --> the (x,y) coordinates of the dataset
c     LI_DI[]     --> the nearest (li,di) coordinates corresponding to the above (x,y)
c     Vertex_XY_Transformed --> the (x,y) coordinates of the transformed rectangle
c
c Once the above is defined [command line or pickfile] then cdp locations can
c be derived.  CDP data is stored in the following arrays:
c
c     CDP_XY[]    -->  (x,y) coordinates of cdp locations in dataset coordinates
c     CDP_LIDI[]  -->  (li,di) coordinates of cdp locations in dataset coordinates
c     CDP_XY_Transformed[]  --> (x,y) cdp locations in coords of transformed rectangle
c     CDP_LIDI_XY_Transformed[] --> (x,y) of (li,di) location in coords of transformed 
c                                   rectangle
c
c
c Now go through the cdp locations one at a time, find the 4 nearest neighbor traces to
c a given location and sum then using a normalized cosine weighting based on offset from
c cdp location.  This logic uses the following arrays:
c
c     LI_Neighbors[]  --> sequential record of neighbor traces 
c     DI_Neighbors[]  --> sequential trace of neighbor traces
c     LIDI_Neighbors_X[] --> x coord [transformed rectangle] 
c     LIDI_Neighbors_Y[] --> y coord [transformed rectangle] 
c     CDPBCX[] --> bin center X coord in dataset coord
c     CDPBCY[] --> bin center Y coord in dataset coord
c     real_CDPBCX[] --> bin center X coord in dataset coord
c     real_CDPBCY[] --> bin center Y coord in dataset coord
c     CDPBCX_Transformed[] --> bin center X coord in transformed rectangle coord
c     CDPBCY_Transformed[] --> bin center Y coord in transformed rectangle coord
c     Weight[] --> weight to apply to neighbor traces prior to summation
c

c get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>

c dimension standard usp variables

      integer iform, nsamp, nrec, nreco, nsi, ntrc, ntrco
      integer luin, luout, argis, obytes, lbytes, lbyout
      integer itr(SZLNHD)

      real tri(SZLNHD)

      character ntap*255, otap*255, name*4

      logical verbos

c program dependant variables

      integer ifmt_TrcNum, l_TrcNum, ln_TrcNum
      integer ifmt_RecNum, l_RecNum, ln_RecNum, RecNum
      integer ifmt_DphInd, l_DphInd, ln_DphInd, DphInd
      integer ifmt_StaCor, l_StaCor, ln_StaCor
      integer ifmt_LinInd, l_LinInd, ln_LinInd, LinInd
      integer ifmt_CDPBCX, l_CDPBCX, ln_CDPBCX, CDPBCX(4)
      integer ifmt_CDPBCY, l_CDPBCY, ln_CDPBCY, CDPBCY(4)

      integer x1, y1, x2, y2, x3, y3, x4, y4
      integer incrLI, incrDI, minLI, maxLI, minDI, maxDI
      integer LI_DI(2,SZLNHD)
      integer CDP_LIDI(2,SZLNHD)
      integer LI_Neighbors(4), DI_Neighbors(4)
      integer lu_ptap
      integer NX, NY, NumVertices, CDP_counter

      integer XSD_NumSegs, XSD_PicksPerSeg(SZLNHD), XSD_SegNum
      integer XSD_SegColor, XSD_nsamp, XSD_ntrc
      integer tr_index, WalkOffDataCount, WalkOffMax

      real dx, dy
      real Vertex_XY(2,SZLNHD)
      real Vertex_XY_Transformed(2,SZLNHD)
      real LI_DI_XY(2,SZLNHD)
      real LI_DI_XY_Transformed(2,SZLNHD)
      real CDP_XY(2,SZLNHD)
      real CDP_XY_Transformed(2,SZLNHD)
      real CDP_LIDI_XY(2,SZLNHD)
      real CDP_LIDI_XY_Transformed(2,SZLNHD)
      real LIDI_Neighbors_X(4), LIDI_Neighbors_Y(4)
      real XSD_SmpUnits, XSD_SmpOffset, XSD_TrcUnits, XSD_TrcOffset
      real XSD_Record(SZLNHD), XSD_Trace(SZLNHD), XSD_Sample(SZLNHD)
      real CdpInt, CDP_X, CDP_Y
      real Neighbors(SZLNHD,4)
      real real_CDPBCX(4), real_CDPBCY(4) 
      real CDPBCX_Transformed(4), CDPBCY_Transformed(4)  
      real Weight(4)
      real junkx, junky, junkx_Transformed, junky_Transformed

      real*8 XX, XY, YX, YY, XXT, XYT, YXT, YYT, DE, DF, XYYXXY
      real*8 E, F

      character  ptap*255, Xsd_SegName*20

      logical  Abnormal, DI_ordered, Average

c initialize variables

      data name/'AT3D'/
      data ntrco/1/
      data nreco/0/
      data RecNum/1/
      data Abnormal/.false./
      data WalkOffDataCount/0/

C get help if necessary

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 )then
         call help()
         stop
      endif

c initialize variables

      do i = 1, 4
         CDPBCX(i) = 0
         LI_Neighbors(i) = 0
         DI_Neighbors(i) = 0
         LIDI_Neighbors_X(i) = 0.
         LIDI_Neighbors_Y(i) = 0.
         real_CDPBCX(i) = 0.
         real_CDPBCY(i) = 0.
         CDPBCX_Transformed(i) = 0.
         CDPBCY_Transformed(i) = 0.
         Weight(i) = 0.
         call vclr ( Neighbors(1,i), 1, SZLNHD )
      enddo

      do i = 1, SZLNHD
         LI_DI(1,i) = 0
         LI_DI(2,i) = 0
         CDP_LIDI(1,i) = 0         
         CDP_LIDI(2,i) = 0
         XSD_PicksPerSeg(i) = 0
         Vertex_XY(1,i) = 0.
         Vertex_XY(2,i) = 0.
         Vertex_XY_Transformed(1,i) = 0.
         Vertex_XY_Transformed(2,i) = 0.
         LI_DI_XY(1,i) = 0.
         LI_DI_XY(2,i) = 0.
         LI_DI_XY_Transformed(1,i) = 0.
         LI_DI_XY_Transformed(2,i) = 0.
         CDP_XY(1,i) = 0.
         CDP_XY(2,i) = 0.
         CDP_XY_Transformed(1,i) = 0.
         CDP_XY_Transformed(2,i) = 0.
         CDP_LIDI_XY(1,i) = 0.
         CDP_LIDI_XY(2,i) = 0.
         CDP_LIDI_XY_Transformed(1,i) = 0.
         CDP_LIDI_XY_Transformed(2,i) = 0.
      enddo

C open printout file

#include <f77/open.h>

C read command line parameters

      call cmdln( ntap, otap, ptap, x1, y1, x2, y2, x3, y3, x4, y4, dx, 
     :     dy, CdpInt, incrLI, incrDI, minLI, maxLI, minDI, maxDI, 
     :     NumVertices, Vertex_XY, LI_DI, DI_ordered, WalkOffMax, 
     :     Average, verbos )

c open input and output files

      call getln ( luin , ntap, 'r', 0 )
      call getln ( luout, otap, 'w', 1 )
 
c  read input line header 

      call rtape ( luin, itr, lbytes )
      if ( lbytes .eq. 0 ) then
         write(LERR,*)' '
         write(LERR,*)' no header read from unit ',luin
         write(LERR,*)' FATAL'
         write(LER,*)' '
         write(LER,*)' AT3D: no header read from unit ',luin
         write(LER,*)' FATAL'
         stop
      endif

c save global data parameters

      call saver (itr, 'NumSmp', nsamp, LINHED)
      call saver (itr, 'SmpInt', nsi  , LINHED)
      call saver (itr, 'NumTrc', ntrc , LINHED)
      call saver (itr, 'NumRec', nrec , LINHED)
      call saver (itr, 'Format', iform, LINHED)

c assign LI, DI defaults if not defined.  First check lineheader entries, then
c default to ntrc, nsamp

      if( minLI .eq. 9999999 ) call saver (itr, 'MnLnIn', minLI, LINHED)
      if( maxLI .eq. 9999999 ) call saver (itr, 'MxLnIn', maxLI, LINHED)
      if( minDI .eq. 9999999 ) call saver (itr, 'MnDpIn', minDI, LINHED)
      if( maxDI .eq. 9999999 ) call saver (itr, 'MxDpIn', maxDI, LINHED)

      if ( minLI .eq. maxLI .and. 
     :     minDI .eq. maxDI .and. 
     :     minLI .eq. minDI )then
         if ( .not. DI_ordered ) then
            minLI = 1
            maxLI = nrec
            minDI = 1
            maxDI = ntrc
         else
            minLI = 1
            maxLI = ntrc
            minDI = 1
            maxDI = nrec
         endif
      endif

c set up pointers to trace header entries required by this routine

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('LinInd',ifmt_LinInd,l_LinInd,ln_LinInd,TRACEHEADER)
      call savelu('CDPBCX',ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,TRACEHEADER)
      call savelu('CDPBCY',ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,TRACEHEADER)

c update the historical line header

      call savhlh (itr, lbytes, lbyout)

c set output bytes/trace size
 
      obytes = nsamp * SZSMPD + SZTRHD

c IF at this point the corner co-ordinates are still undefined 
c THEN assume the units are orthogonal to the x,y axes and base 
c the co-ordinate system numerics on the LI and DI information given 
c on the command line.  This will allow traverses to be extracted from
c volumes that do not contain (x,y) co-ordinate information as is often
c the case at various points during processing when slicing has occured
c and the headers have not been restored.  Also using .and. instead of
c .or. allows 0,0 to be one of the vertices.  Watch for DI or LI ordering
c of the records and build the coordinates appropriately.

      IF ( ( x1 .eq. 0 .or. y1 .eq. 0 ) .and.
     *     ( x2 .eq. 0 .or. y2 .eq. 0 ) .and.
     *     ( x3 .eq. 0 .or. y3 .eq. 0 ) .and.
     *     ( x4 .eq. 0 .or. y4 .eq. 0 ) ) then 

         x1 = 0
         y1 = 0

         x2 = x1

         if ( .not. DI_ordered ) then
c            y2 = dy * (((maxLI - minLI) / incrLI) + 1) + dy / 2.0
            y2 = dy * (((maxLI - minLI) / incrLI) + 1) + dy

c            x3 = dx * (((maxDI - minDI) / incrDI) + 1) + dx / 2.0
            x3 = dx * (((maxDI - minDI) / incrDI) + 1) + dx
         else
c            y2 = dy * (((maxDI - minDI) / incrDI) + 1) + dy / 2.0
            y2 = dy * (((maxDI - minDI) / incrDI) + 1) + dy

c            x3 = dx * (((maxLI - minLI) / incrLI) + 1) + dx / 2.0
            x3 = dx * (((maxLI - minLI) / incrLI) + 1) + dx
         endif

         y3 = y2

         x4 = x3
         y4 = y1

      ENDIF

c calculate mapping constants [parallelogram <--> rectangle] 

      CALL Transform_Init( X1, Y1, X2, Y2, X3, Y3, X4, Y4, DY, DX, 
     :     NX, NY, XX, XY, YX, YY, XXT, XYT, YXT, YYT)

      DE = DBLE(FLOAT(X1)) * XXT + DBLE(FLOAT(Y1)) * XYT
      DF = DBLE(FLOAT(X1)) * YXT + DBLE(FLOAT(Y1)) * YYT
      XYYXXY =  XYT * YXT - XXT * YYT

      if (XYYXXY .eq. dble(0.) )then
         write(LERR,*)' '
         write(LERR,*)'Attempt to use survey coords to fill in '
         write(LERR,*)'bin centers failed due to coord transform'
         write(LERR,*)'singularity (bad coord choice most likely)'
         write(LERR,*)'FATAL '
         write(LER,*)' '
         write(LER,*)'AT3D: Attempt to use survey coords to fill in '
         write(LER,*)'      bin centers failed due to coord transform'
         write(LER,*)'      singularity (bad coord choice most likely)'
         write(LER,*)'FATAL '
         goto 999
      endif

c get traverse control points and be sure both (x,y) and (li,di) arrays
c are complete.

      IF ( ptap .ne. ' ' ) then

c open the pickfile

         call alloclun(lu_ptap)
         length = lenth(ptap)
         open ( lu_ptap, file = ptap(1:length), status = 'old', 
     :        err = 991 )

c initialize XSD variables based on pick file attached.

         call XSDinitialize(lu_ptap, XSD_NumSegs, XSD_PicksPerSeg, 
     :        XSD_SmpUnits, XSD_SmpOffset, XSD_TrcUnits, XSD_TrcOffset, 
     :        XSD_nsamp, XSD_ntrc )

c read in the pick file data.  This logic assumes only one segment per pickfile.
c if more than one traverse is required then user must create a separate pickfile
c for each traverse.

         call XsdReadPicks(lu_ptap, XSD_Record, XSD_Trace, XSD_Sample, 
     :        XSD_PicksPerSeg(1), SZLNHD, XSD_SegNum, XSD_SegName, 
     :        XSD_SegColor)

c calculate LI_DI[], Vertex_XY[] and Vertex_XY_Transformed[] for input traverse
c vertices         
         
         do i = 1, XSD_PicksPerSeg(1)

c move sample origin to bottom left hand corner.  This is here since
c qdslice outputs a slice that must be plotted right to left to look like
c the plan view of the dataset it came from and samples in XSD start
c from 1 at the top not the bottom.  This logic converts the XSD_Trace[]
c and XSD_Sample[] numbers to sequential record and trace for use latter
c when recovering actual trace data.

            LI_DI(1,i) = nint ( XSD_Trace(i) * XSD_TrcUnits + 
     :           XSD_TrcOffset )
            LI_DI(2,i) = nint( XSD_Sample(i) * XSD_SmpUnits + 
     :           XSD_SmpOffset )

            Vertex_XY_Transformed(1,i) = 
     :           float(LI_DI(1,i)) * dx - dx / 2.0
            Vertex_XY_Transformed(2,i) = 
     :           float(LI_DI(2,i)) * dy - dy / 2.0

            E = (DBLE(LI_DI(1,i)) - 0.5) * dble(DX) + DE
            F = (DBLE(LI_DI(2,i)) - 0.5) * dble(DY) + DF

            Vertex_XY(1,i)  = sngl((F * XYT - E * YYT) / XYYXXY)

            if (YYT .ne. 0.) then
               Vertex_XY(2,i) = sngl((F - Vertex_XY(1,i) * YXT) / YYT)
            elseif (XYT .ne. 0.) then
               Vertex_XY(2,i) = sngl((E - Vertex_XY(1,i) * XXT) / XYT)
            endif

         enddo
 
         NumVertices = XSD_PicksPerSeg(1)

      ELSE

c vertex control came in on the command line as either LI_DI[] or Vertex_XY[]
c pairs.  Determine which and fill out the all coordinate arrays including
c Vertex_XY_Transformed

         if ( abs ( Vertex_XY(1,1) - 9999999. ) .gt. 1.e-32 ) then

            do i = 1, NumVertices

               call Coordinate_Transform ( 
     :              Vertex_XY(1,i), Vertex_XY(2,i), 
     :              LI_DI(1,i), LI_DI(2,i), 
     :              Vertex_XY_Transformed(1,i), 
     :              Vertex_XY_Transformed(2,i), 
     :              junkx, junky,
     :              IWARN, x1, y1, XX, XY, YX, YY, XXT, XYT, YXT, YYT, 
     :              dx, dy, NDI, NLI)

            enddo

         elseif ( LI_DI(1,1) .ne. 9999999 ) then

c here we assume that the li,di information is comming in on the command line
c in the coordinate system of the input dataset.  Will need to consider 
c minli, mindi and incrli, incrdi when building vertex_XY data so as to get
c the correct lengths associated with the lines.

            do i = 1, NumVertices

               Vertex_XY_Transformed(1,i) = 
     :              (float(LI_DI(1,i)-minLI+1)/ incrLI) * dx - dx / 2.0
               Vertex_XY_Transformed(2,i) = 
     :              (float(LI_DI(2,i)-minDI+1)/ incrDI) * dy - dy / 2.0

               E = (DBLE(LI_DI(1,i)) - 0.5) * dble(DX) + DE
               F = (DBLE(LI_DI(2,i)) - 0.5) * dble(DY) + DF

               Vertex_XY(1,i) = sngl((F * XYT - E * YYT) / XYYXXY)

               if (YYT .ne. 0.) then
                  Vertex_XY(2,i) = 
     :                 sngl((F - Vertex_XY(1,i) * YXT) / YYT)
               elseif (XYT .ne. 0.) then
                  Vertex_XY(2,i) = 
     :                 sngl((E - Vertex_XY(1,i) * XXT) / XYT)
               endif

            enddo

         endif

      ENDIF

c Generate CDP (x,y) data in Transformed space

      done = 0
      CDP_X = 0.
      CDP_Y = 0.
      CDP_counter = 0

      DO while ( done .eq. 0 )

         call cdpXY (  Vertex_XY_Transformed, NumVertices, CdpInt, 
     :        CDP_X, CDP_Y, nreco, done)
         if ( done .eq. 0 ) then
            CDP_counter = CDP_counter + 1
            CDP_XY_Transformed(1,CDP_counter) = CDP_X
            CDP_XY_Transformed(2,CDP_counter) = CDP_Y
         endif
         
      ENDDO

      if ( nreco .gt. SZLNHD ) then
         write(LERR,*)' '
         write(LERR,*)'You have asked to generate ',nreco
         write(LERR,*)'CDP traces.  This is more than this '
         write(LERR,*)'routine can handle.  Contact Garossino '
         write(LERR,*)'and ask him to put dynamic memory allocation '
         write(LERR,*)'into this routine or do your traverse in '
         write(LERR,*)'more segments '
         write(LER,*)' '
         write(LER,*)'AT3D: You have asked to generate ',nreco
         write(LER,*)'      CDP traces.  This is more than this '
         write(LER,*)'      routine can handle.  Contact Garossino '
         write(LER,*)'      and ask him to put dynamic memory '
         write(LER,*)'      allocation  into this routine or do your'
         write(LER,*)'       traverse in more segments '
         write(LER,*)'FATAL '
         goto 999
      endif

c echo all input parameters 

      call verbal (ntap, otap, ptap, x1, y1, x2, y2, x3, y3, x4, y4, dx, 
     :     dy, CdpInt, incrLI, incrDI, minLI, maxLI, minDI, maxDI, 
     :     NumVertices, Vertex_XY, LI_DI, Vertex_XY_Transformed, 
     :     nsamp, ntrc, nrec, iform, nsi, nreco, XSD_Record, XSD_Trace, 
     :     XSD_Sample, DI_ordered, WalkOffMax, Average, verbos)

c echo CDP coordinate information to printout file

      if ( verbos ) write (LERR, 100)
 100  FORMAT('   CDP         X           Y       X_Transformed   Y_Trans
     :formed   line index   depth index',/, 
     *     '  -----   ------------ ----------  -------------   ---------
     :----   ----------   -----------', /)

      DO i = 1, nreco
            
c compute (x,y) co-ords of the cdps

         E = Dble(CDP_XY_Transformed(1,i)) + DE 
         F = Dble(CDP_XY_Transformed(2,i)) + DF

         CDP_XY(1,i) = sngl((F * XYT - E * YYT) / XYYXXY)

         if ( YYT .ne. 0. ) then
            CDP_XY(2,i) = sngl((F - dble(CDP_XY(1,i)) * YXT) / YYT)
         elseif (XYT .ne. 0.) then
            CDP_XY(2,i) = sngl((E - dble(CDP_XY(1,i)) * XXT) / XYT)
         endif

c compute [li,di] locations for cdps

         call Coordinate_Transform ( CDP_XY(1,i), CDP_XY(2,i), 
     :        CDP_LIDI(1,i), CDP_LIDI(2,i),
     :        CDP_XY_Transformed(1,i), CDP_XY_Transformed(2,i), 
     :        CDP_LIDI_XY_Transformed(1,i), 
     :        CDP_LIDI_XY_Transformed(2,i), 
     :        IWARN, x1, y1, XX, XY, YX, YY, XXT, XYT, YXT, YYT, dx,  
     :        dy, NDI, NLI)

         if ( verbos ) write(LERR,101) i, CDP_XY(1,i), CDP_XY(2,i), 
     :        CDP_XY_Transformed(1,i), CDP_XY_Transformed(2,i), 
     :        CDP_LIDI(1,i), CDP_LIDI(2,i) 
 101     format( I5, 2X, 2f11.1, 1x, f11.1, 7x, f11.1, i12,1x, I12  )

      ENDDO

c build output lineheader

      call savew ( itr, 'NumRec', nreco, LINEHEADER )
      call savew ( itr, 'NumTrc', 1, LINEHEADER )

c output lineheader

      call wrtape ( luout, itr, lbyout )

c Processing Traverse - one trace at a time
      
      DO JJ = 1, nreco

c find four neighbor traces to the desired output cdp location
         
         call naybor (CDP_LIDI(1,JJ), CDP_LIDI(2,JJ), 
     :        CDP_XY_Transformed(1,JJ), CDP_XY_Transformed(2,JJ), 
     :        CDP_LIDI_XY_Transformed(1,JJ), 
     :        CDP_LIDI_XY_Transformed(2,JJ), 
     :        dx, dy,
     :        LI_Neighbors, DI_Neighbors, 
     :        LIDI_Neighbors_X, LIDI_Neighbors_Y,
     :        incrLI, incrDI)

c retrieve 4 neighbor traces

         do KK = 1, 4

c at this point the LI,DI information in these arrays corresponds to
c sequential record, sequential trace. LI's increase across records
c DI's increase across traces.

            if ( LI_Neighbors(KK) .gt. nrec .or. 
     :           DI_Neighbors(KK) .gt. ntrc .or.
     :           LI_Neighbors(KK) .lt. 1 .or. 
     :           DI_Neighbors(KK) .lt. 1 )then
               write(LERR,*)' '
               write(LERR,*)'A neighbor trace has been requested from '
               write(LERR,*)'outside the area of your data.  Either'
               write(LERR,*)'move your traverse vertices away from the'
               write(LERR,*)'edges or supply more data in the stack'
               write(LERR,*)'This happened at output trace ',JJ
               write(LERR,*)'requested LI = ', 
     :              LI_Neighbors(KK) * incrLI + minLI - 1
               write(LERR,*)'requested DI = ', 
     :              DI_Neighbors(KK) * incrDI + minDI - 1
               Abnormal = .true.
               if (LI_Neighbors(KK) .gt. nrec ) LI_Neighbors(KK) = nrec
               if (DI_Neighbors(KK) .gt. ntrc ) DI_Neighbors(KK) = ntrc
               if (LI_Neighbors(KK) .lt. 1 ) LI_Neighbors(KK) = 1
               if (DI_Neighbors(KK) .lt. 1 ) DI_Neighbors(KK) = 1
               write(LERR,*)'LI used = ',
     :              LI_Neighbors(KK) * incrLI + minLI - incrLI
               write(LERR,*)'DI used = ',
     :              DI_Neighbors(KK) * incrDI + minDI - incrDI
               write(LERR,*)' WARNING'

               WalkOffDataCount = WalkOffDataCount + 1
               if ( WalkOffDataCount .gt. WalkOffMax ) then
                  write(LERR,*)' '
                  write(LERR,*)' your traverse has requested so '
                  write(LERR,*)' many traces from outside your '
                  write(LERR,*)' volume that I think something'
                  write(LERR,*)' must be wrong with you basic'
                  write(LERR,*)' volume description on the command'
                  write(LERR,*)' command line.  Check that your '
                  write(LERR,*)' -dx, -dy parameters correspond to'
                  write(LERR,*)' the cell dimensions of your stacked'
                  write(LERR,*)' dataset.  If all looks well, better'
                  write(LERR,*)' call me in Tulsa, Garossino [3932]'
                  write(LERR,*)' FATAL'
                  write(LER,*)' '
                  write(LER,*)'AT3D: '
                  write(LER,*)'     your traverse has requested so '
                  write(LER,*)'     many traces from outside your '
                  write(LER,*)'     volume that I think something'
                  write(LER,*)'     must be wrong with you basic'
                  write(LER,*)'     volume description on the command'
                  write(LER,*)'     command line.  Check that your '
                  write(LER,*)'     -dx, -dy parameters correspond to'
                  write(LER,*)'     the cell dimensions of your stacked'
                  write(LER,*)'     dataset.  If all looks well, better'
                  write(LER,*)'     call me in Tulsa, Garossino [3932]'
                  write(LER,*)' FATAL'
                  goto 999
               endif

            endif
            
            tr_index = (LI_Neighbors(KK)-1) * ntrc + 
     :           ( DI_Neighbors(KK))
            call sisseek (luin, tr_index)
            lbytes = 0
            call rtape (luin, itr, lbytes )
            if ( lbytes .eq. 0 ) then
                  write(LERR,*)'Premature EOF on input:'
                  write(LERR,*)'Trying to read sequential record ',  
     :                 LI_Neighbors(KK)
                  write(LERR,*)'               sequential trace ',  
     :                 DI_Neighbors(KK)
                  go to 999
            endif

            call vmov ( itr(ITHWP1), 1, Neighbors(1,KK), 1, nsamp )

            call saver2 ( itr, ifmt_CDPBCX, l_CDPBCX, ln_CDPBCX, 
     :           CDPBCX(KK), TRACEHEADER )
            call saver2 ( itr, ifmt_CDPBCY, l_CDPBCY, ln_CDPBCY, 
     :           CDPBCY(KK), TRACEHEADER )

            real_CDPBCX(KK) = float(CDPBCX(KK))
            real_CDPBCY(KK) = float(CDPBCY(KK))

c only bother transforming the (x,y) data if it looks real,  this data is used
c in the CalcWeight subroutine to calculate weighting for the neighbor traces used
c in construction of the output traverse trace for this cdp.

            if ( ( ( CDPBCX(1) .ne. 0 ) .and. ( CDPBCY(1) .ne. 0 ) ) 
     :           .and.
     :           ( ( CDPBCX(1) .ne. 999999) .and. 
     :           ( CDPBCY(1) .ne. 999999 ) 
     :           ) .and. .not. Average ) then

               call Coordinate_Transform ( 
     :              real_CDPBCX(KK), real_CDPBCY(KK) 
     :              ,junkx, junky,
     :              CDPBCX_Transformed(KK), CDPBCY_Transformed(KK), 
     :              junkx_Transformed, junky_Transformed, 
     :              IWARN, x1, y1, XX, XY, YX, YY, XXT, XYT, YXT, YYT,   
     :              dx, dy, NDI, NLI)

c note: the junk parameters in the above call are being used since we are
c       only interested in the transformed CDPBCX and CPDBCY values. If 
c       LI,DI parameterization is being used there is no way to do weighting
c       based on offsets from bin centers so the weighting will default to
c       using bin centers.

            endif

         enddo

         if ( Average ) then

c perform simple average weighting

            do KK = 1, 4
               do i = 1, nsamp
                  Neighbors(i,KK) = Neighbors(i,KK) * 0.25
               enddo
            enddo
            
         elseif ( ( ( CDPBCX(1) .ne. 0 ) .and. ( CDPBCY(1) .ne. 0 ) ) 
     :           .and.
     :        ( ( CDPBCX(1) .ne. 999999) .and. ( CDPBCY(1) .ne. 999999 ) 
     :         ) ) then
            
c calculate weighting for trace stacking relative to offset from current output
c (x,y) location , if CDPBCX,Y information is not in the trace header then use 
c LIDI_Neighbors_X,Y data to do weighting.  If CDPBCX,Y data is there then weight 
c according to actual bin distance values

c do weight calculation  

            call CalcWeights( CDP_XY_Transformed(1,JJ), 
     :           CDP_XY_Transformed(2,JJ), 
     :           CDPBCX_Transformed, 
     :           CDPBCY_Transformed, 
     :           dx, dy, Weight )

c weight traces    

            do KK = 1, 4
               do i = 1, nsamp
                  Neighbors(i,KK) = Neighbors(i,KK) * Weight(KK) 
               enddo
            enddo
            
         else

c use calculated cdp bin center information for neighbors             
c do weight calculation  

            call CalcWeights ( CDP_XY_Transformed(1,JJ), 
     :           CDP_XY_Transformed(2,JJ), 
     :           LIDI_Neighbors_X, 
     :           LIDI_Neighbors_Y,
     :           dx, dy, Weight ) 

c weight traces
            
            do KK = 1, 4
               do i = 1, nsamp
                  Neighbors(i,KK) = Neighbors(i,KK) * Weight(KK)
               enddo
            enddo

         endif

c form output trace

         call vmov ( Neighbors(1,1), 1, tri, 1, nsamp ) 

         do KK = 2, 4 

            call vadd ( tri, 1, Neighbors(1,KK), 1, tri, 1, nsamp ) 
   
         enddo         

         call vmov ( tri, 1, itr(ITHWP1), 1, nsamp )

         call savew2 ( itr, ifmt_CDPBCX, l_CDPBCX, ln_CDPBCX, 
     :        nint(CDP_XY(1,JJ)), TRACEHEADER )
         call savew2 ( itr, ifmt_CDPBCY, l_CDPBCY, ln_CDPBCY, 
     :        nint(CDP_XY(2,JJ)), TRACEHEADER )
         LinInd = CDP_LIDI(1,JJ) * incrLI  + minLI - incrLI
         call savew2 ( itr, ifmt_LinInd, l_LinInd, ln_LinInd, 
     :        LinInd, TRACEHEADER )
         DphInd = CDP_LIDI(2,JJ) * incrDI + minDI - incrDI
         call savew2 ( itr, ifmt_DphInd, l_DphInd, ln_DphInd, 
     :        DphInd, TRACEHEADER )
         call savew2 ( itr, ifmt_RecNum, l_RecNum, ln_RecNum, 
     :        JJ, TRACEHEADER )
         call savew2 ( itr, ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :        JJ, TRACEHEADER )
         call savew2 ( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :        0, TRACEHEADER )

c update the trace header entries for this cdp output

         call wrtape ( luout, itr, obytes )

      ENDDO

      if ( Abnormal ) goto 999

      write(LER,*)'at3d: Normal Termination'
      write(LERR,*) 'Normal Termination'
      call lbclos (luin)
      call lbclos (luout)
      if ( ptap .ne. ' ' ) close(lu_ptap)
      stop

 991  continue

      write(LERR,*)' '
      write(LERR,*)' unable to open input pickfile ',ptap(1:length)
      write(LERR,*)' check existance/permissions and try again'
      write(LERR,*)' FATAL'
      write(LER,*)' '
      write(LER,*)' AT3D: unable to open input pickfile ',
     :     ptap(1:length)
      write(LER,*)'       check existance/permissions and try again'
      write(LER,*)' FATAL'
      goto 999

 999  continue
      
      call lbclos (luin)
      call lbclos (luout)
      if ( ptap .ne. ' ' ) close(lu_ptap)
      write(LERR,*)' Abnormal Termination'
      write(LER,*)' at3d: Abnormal Termination'

      stop
      END

