C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C
C     PROGRAM MODULE  dmoamp
C
C**********************************************************************C
C
*
* BORNDMO - Born theory transformation of offset data to zero offset.
*
*       SUMMARY --   The algorithm is 'input trace sequential', each
*                    input trace is 'spread' between source and receiver
*                    along its impulse response.  By repeated call of
*                    a DMO stack is built by superpostion.
*                    Memory requirements are low because only the
*                    zero offset data volume (nt,nx) is kept in memory.
*
*       VELOCITY --  The constant velocity theory is used ( abused? )
*                    while processing with variable RMS velocity field.
*
*       CAVEATS -- 1.Aliasing treatment is very strict; if the nyquist
*                    frequency is aliased, the operator is truncated.
*                    This acts to attenuate steep dips.  Caveat Emptor!
*                  2.Amplitude accuracy shows some 'drift' at far
*                    offsets: cause unknown.
*
*       TECHNICAL REFERENCE --  Liner, C. L., 1989, Mapping Reflection
*                               Seismic Data to Zero Offset, Ph.D. Thesis,
*                               Colorado School of Mines
*       ( also: Center for Wave Phenomena Research Report CWP-081 )
*
*
* Integer Variables:
*    ALIASFLAG  - flag to allow aliasing
*                       0 = no aliasing allowed
*                       1 = process all dips, even aliased ones
*    AMPFLAG    - flag to choose amplitude term
*                       0 = Born DMO common amp term
*                       1 = Born DMO common offset amp term
*                       2 = Kinematic DMO common offset amp term
*                           ( plus spreading corrections )
*    IPF        - input data file unit number
*    IREC       - receiver index
*    ISHOT      - shot index
*    IT0        - zero-offset time sample index
*    IT0        - zero-offset trace index
*    J          - general index
*    MT         - global maximum number of samples per trace
*    MX         - global maximum number of traces in output
*    NREC       - number of receivers per shot
*    NSHOT      - number of shots in the input data
*    NT         - number of time samples
*    NTRIN      - total number of input traces
*    NX         - number of traces ( midpoints ) in outdata
*    OPF        - output data file unit number
*    SPRDFLAG   - flag to remove zero-offset spreading
*                       0 = output includes zero-offset spreading
*    VERBOSE    - verbose flag
*                       0 = silent operation
*                       1 = chatty
*                       2 = very chatty
*
* Real Variables:
*    DMO(,)     - DMO'd zero offset data panel ( outdata )
*    DT         - time sample rate
*    DX         - midpoint spacing in outdata
*    DXG        - receiver interval
*    DXS        - source interval
*    ITRACE()   - space for holding imaginary part of complex trace
*                 (needed for complex fft called by 'iwtrace')
*    MAXDIP     - maximum anticipated reflector dip
*    NEAROFF    - near offset
*    OFFSET     - full offset
*    ONE        - 1.0
*    PI         - pi
*    ROW        - temporary vector needed for filling vmap
*    SINCS()    - sinc interpolation weights
*    TMP        - temporary variable
*    TRACE()    - input data trace
*    TWO        - 2.0
*    TWOPI      - 2.0 * pi
*    VMAP(,)    - RMS velocity map
*    W()        - frequency vector
*    XG         - receiver coordinate
*    XMIN       - minimum midpoint coordinate in outdata
*    XS         - source coordinate
*    XS1        - coordinate of first source
*    ZERO       - 0.0
*
* Complex Variables:
*    CTRACE()   - space for holding complex-valued trace
*                 (needed for complex fft called by 'iwtrace')
*
* Character Variables:
*    IPFILE     - input data file name
*    OPFILE     - output data file name
*

C
C SUBROUTINE CALLS: RTAPE, HLH, WRTAPE, SAVE
C
C**********************************************************************C
C
C     DECLARE VARIABLES
C

#include <localsys.h>
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      INTEGER     ITR  ( SZLNHD )
      INTEGER     ITR0 ( SZLNHD )
      INTEGER     NTR  ( SZLNHD )
      INTEGER     LHED ( SZLNHD )
      INTEGER     LHED0( SZLNHD )
      INTEGER     VHED ( SZLNHD )
      INTEGER     NSAMP, NSI, NTRC, NREC, IFORM
      INTEGER     LUIN , LBYTES, NBYTES,obytes
      integer     argis, ordfft, pipe
      integer     ist,iend,irs,ire,ns,ne

        integer   nshot, nrcvr
        integer   aliasflag,      ampflag,        sprdflag
        real      xs1,    dxs,    nearoff,        dxg
 
        integer   j
        integer   nt,     nx
 
        real      dt,     dx,     maxdip, offset
        real      sincs(41),      trace(2*SZLNHD)
        real      w(2*SZLNHD),  xmin,   xs,     xg
        real      spread( SZLNHD )
 
        real      itrace(2*SZLNHD),     tmp
        complex   ctrace(2*SZLNHD)
 
      real           zero,    one,    two,    pi,     twopi
      common /const/ zero,    one,    two,    pi,     twopi


      REAL        dmo, vmap
      REAL        vtr( SZLNHD )
      integer     itrh
      pointer     (wkaddr, dmo (1))
      pointer     (wkadr2, vmap(1))
      pointer     (wkadri, itrh(1))

      CHARACTER   NAME * 20, ntap * 256, otap * 256, vtap * 256
#include <f77/pid.h>
      logical     verbos,query,heap,split,restart,pre,first,source
      logical     debug
 
      EQUIVALENCE ( ITR (  1), LHED (1) )
      EQUIVALENCE ( ITR0(  1), LHED0(1) )
      EQUIVALENCE ( NTR (  1), VHED (1) )

      DATA NAME     /'DMOAMP_true_amp_dmo'/
      DATA LUIN / 1 /, LBYTES / 0 /, NBYTES / 0 /
      DATA  pipe/3/
      DATA itr0/SZLNHD*0/
      DATA first/.true./

c---------------------------------
c  get online help if necessary
c---------------------------------
      query = (argis('-?').gt.0 .or. argis('-h').gt.0)
      if ( query ) then
           call help ()
           stop
      endif

      debug = (argis('-debug') .gt. 0)

c       .. set constants
        zero  = 0.0
        one   = 1.0
        two   = 2.0
        pi    = 3.1415926
        twopi = two * pi

c------------------------------------
c  open printout file
c------------------------------------
#include <f77/open.h>

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM COMMAND LINE ARGUMENT STRING
C**********************************************************************C
c       .. readpar reads the parameter file bdmo.par,
c       .. and fills velocity map.

        call cmdln  ( ntap, iss, lss, xs1, dxs, nearoff, dxg,
     ;                otap, xmin, nx, dx, maxdip, vtap, idv,
     ;                aliasflag, ampflag, sprdflag, source,
     ;                verbos, ist,iend,irs,ire,ns,ne,ncdp,
     ;                dl1, dl2, dr1, dr2, far,split,restart,pre)

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C

      call getln ( luin, ntap, 'r', 0 )
      if (pre .and. .not.restart) then
          call getln (luout, otap, 'w', 1 )
      elseif (restart) then
          call getln (luout, otap, 'r+', 1 )
          call rwd (luout)
      else
          call getln (luout, otap, 'w', 1 )
      endif

      if (vtap(1:1) .ne. ' ') then
          call getln(luvel, vtap, 'r',-1)
      else
          write(LERR,*)'dmoamp assumed to be running inside IKP'
          call sisfdfit (luvel, pipe)
      endif
      if(luvel .lt. 0)   then
         write(lerr,*)'dmoamp error: velocity file -v not accessible'
      endif

      lbyte = 0
      call rtape4 ( luin, itr, lbyte, lbyoff )
      lbytes = lbyte
      if(lbytes .eq. 0) then
         write(LERR,*)'DMO: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif
      if (.not. restart)
     1CALL HLHprt    ( ITR , LBYTE, NAME, 20, LERR        )

c---------------------------------
c  save key header values
#include <f77/saveh.h>

c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('SoPtNm',ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

      if(nsamp .gt. SZLNHD) nsamp=SZLNHD

c
c ----- velocity data -----
c
 
      vflag=1
      lvbytes=0
 
      call rtape(luvel,ntr,lvbytes)
 
      if(lvbytes .eq. 0) then
 
         write(LERR,*)'FATAL: no lineheader read on',vtap
         write(LERR,*)' '
         stop
 
      endif
 
      call saver(ntr, 'NumSmp', nsampv, LINHED)
      call saver(ntr, 'SmpInt', nsiv  , LINHED)
      call saver(ntr, 'NumTrc', ntrcv , LINHED)
      call saver(ntr, 'NumRec', nrecv , LINHED)
      call saver(ntr, 'Format', iformv, LINHED)

      if (nrecv .gt. 1) then
         if (nx .eq. 0) then
            nx = nrecv
         endif
      else
         if (nx .eq. 0) then
            write(LERR,*)'For single velocity function must enter'
            write(LERR,*)'total number CDPs in output data set using'
            write(LERR,*)'-nm[] command line argument'
            go to 999
         endif
      endif

c----------------------------------
c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,iss,lss,ntrc,nrec)

      dt = float(nsi) * unitsc
      ist  = ist / nsi
      iend = iend / nsi

      if (ist .lt. 1)   ist = 1
      if (iend .lt. 1) iend = nsamp
      nsampo = iend - ist + 1

      nshot = nrec
      nrcvr = ntrc

      nu = ordfft( nsamp )
      nt = 2 ** nu

      nv = nsamp / idv + 1
      if (nv .gt. nsamp) nv = nsamp

      ixmax = abs (far) / dx + 1

c----------------------------------------------------------------
c  change output bytes to reflect change from time to # traces
      obytes = SZTRHD + SZSMPD * nsampo

c---------------------------------
c  verbos printout
        write(LERR,*)
        write(LERR,*)' Values read from input data set line header'
        write(LERR,*)
        write(LERR,*) ' # of Samples/Trace =  ', nsamp
        write(LERR,*) ' Sample Interval    =  ', nsi  
        write(LERR,*) ' Traces per Record  =  ', ntrc
        write(LERR,*) ' Records per Line   =  ', nrec
        write(LERR,*) ' Format of Data     =  ', iform
        write(LERR,*) ' Output # samples   =  ', nsampo
        write(LERR,*) ' Input # vel samples=  ', nsampv
        write(LERR,*) ' Output # traces    =  ', nx
        write(LERR,*) ' Output trace space =  ', dx
        write(LERR,*) ' length padded trcs =  ', nt
        write(LERR,*) ' '
        write(LERR,*) ' # shots                   =  ', nshot
        write(LERR,*) ' # receivers               =  ', nrcvr
        write(LERR,*) ' # midoints                =  ', nx
        write(LERR,*) ' First record to process   =  ',iss
        write(LERR,*) ' Last  record to process   =  ',lss
        write(LERR,*) ' First shot index on line  =  ',xs1
        write(LERR,*) ' Shot spacing (f,m)        =  ',dxs
        write(LERR,*) ' First midpoint coord      =  ',xmin
        write(LERR,*) ' CMP spacing (f,m)         =  ',dx
        write(LERR,*) ' Max # trcs in dmo ellipse =  ',ixmax
        write(LERR,*) ' Start time (samps)        =  ', ist
        write(LERR,*) ' End time (samps)          =  ', iend
        write(LERR,*) ' tme decimation of vel trc =  ', idv
        write(LERR,*) ' Number samps in vel trc   =  ', nv
        write(LER ,*) ' Number samps in vel trc   =  ', nv
        write(LERR,*) ' '
        write(LERR,*) ' Leftmost spread limit  = ',dl1
        write(LERR,*) ' NearLeft spread limit  = ',dl2
        write(LERR,*) ' NearRight spread limit = ',dr1
        write(LERR,*) ' Rightmost spread limit = ',dr2
        write(LERR,*) ' '
        if (pre) then
        write(LERR,*) ' pre-stack parameters:'
        write(LERR,*) ' Max abs far offset     =  ',far
        write(LERR,*) ' Trc spacing in CDP     =  ',dxg
        write(LERR,*) ' # trcs in CDP          =  ',ncdp
        write(LERR,*) ' Restart?               =  ',restart
        write(LERR,*) ' ... at shot            =  ', iss
        write(LERR,*) ' ... at trace           =  ', ns
        endif
        write(LERR,*) ' Flags:'
        write(LERR,*) ' aliasflag = ',aliasflag,':'
        write(LERR,*) '               1 = all dips; 0 = no aliasing'
        write(LERR,*) ' ampflag   = ',ampflag,':'
        write(LERR,*) '               0 = common shot; 1 = commn offset'
        write(LERR,*) '               2 = kinematic DMO'
        write(LERR,*) 'sprdflag   = ',sprdflag,'  1 = 0-offst spreading'
        write(LERR,*) '               1 = 0-offst spreading; 0 = no such
     : spreading'
        write(LERR,*) ' '

c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.

      if (pre) then
          itemd = nsampo * ixmax
      else
          itemd = nsampo * nx
      endif
      itemv = nx * nv
      itemi = nx * ITRWRD

      call galloc (wkaddr, itemd*SZSMPD, ierrcod, iabort)
      if (ierrcod .ne. 0) heap = .false.
      call galloc (wkadr2, itemv*SZSMPD, ierrcod2, iabort2)
      if (ierrcod2 .ne. 0) heap = .false.
      call galloc (wkadri, itemi*SZSMPD, ierrcodi, iaborti)
      if (ierrcodi .ne. 0) heap = .false.

      if (.not. heap) then
         write(LER ,*)'Unable to allocate workspace:'
         write(LER ,*)    nt*SZSMPD,'  bytes',wkaddr,ierrcod,iabort
         write(LER ,*) itemd*SZSMPD,'  bytes',wkadr2,ierrcod2,iabort2
         write(LER ,*) itemi*SZSMPD,'  bytes'
         write(LER ,*)'Machine memory too small? Try bigger machine'
         go to 999
      else
         write(LERR,*)'Allocated workspace:'
         write(LERR,*)    nt*SZSMPD,'  bytes',wkaddr,ierrcod,iabort
         write(LERR,*) itemd*SZSMPD,'  bytes',wkadr2,ierrcod2,iabort2
         write(LERR,*) itemi*SZSMPD,'  bytes'
      endif
c---------------------------------------------------

c-------
c compute spread model
c-------
         ngrp = ncdp
         ngrp2 = ngrp / 2
         call vfill (0.0, spread, 1, ngrp)

         if (split) then
 
            x  = -far -dxg
            do  j = 1, ngrp2
 
                x = x + dxg
                spread (j) = x
            enddo
            jj = 0
            do  j = 1, ngrp2
                jj = ngrp - j + 1
                spread (jj) = abs(spread (j))
            enddo
 
         else
 
            do  j = 1, ngrp
 
                if (far .gt. 0.) then
 
                       spread (j) = far - (ngrp-j) * dxg
 
                elseif (far .lt. 0.) then
 
                       spread (j) = far + (j-1) * dxg
                endif
            enddo
 
         endif
 
         call minmgv (spread, 1, rmin, lmin, ngrp)
         call maxmgv (spread, 1, rmax, lmax, ngrp)
 
         write(LERR,*)' '
         write(LERR,*)'Spread model:'
         write(LERR,*)'Number of groups in spread   = ',ngrp
         write(LERR,*)'Minimum distance located at grp= ',lmin
         write(LERR,*)(spread(i),i=1,ngrp)
         write(LERR,*)' '


c-----
c     skip unwanted shots
c-----
      call recskp(1,iss-1,luin,ntrc,itr)


C**********************************************************************C
C
C     READ VELOCITY TAPE IN TOTO
C
C**********************************************************************C

      DO  J = 1, nx

                 nbytes = 0
                 CALL RTAPE  ( LUVEL , NTR, NBYTES         )
                 if(nbytes .eq. 0) then
                    write(LERR,*)'WARNING: ',
     1              'End of file on velocity tape at',
     2              '  rec= ',J
                 else
c-------------------
c  put vels in array
                    call vmov (vhed(ITHWP1), 1, vtr, 1, nsamp)
                    if (nsampv .lt. nsamp) then
                       do  i = nsampv+1, nsamp
                           vtr (i) = vtr (nsampv)
                       enddo
                    endif
c-------------------
c  save tr headers
                    ishdr = (J-1)* ITRWRD
                    call vmov (vhed, 1, itrh(ishdr+1),1,ITRWRD)

                 endif
                 istrc = (J-1)* nv
                 do  iv = 1, nv

                     if (iv .eq. 1) then
                         ii = 1
                     else
                         ii = iv * idv
                         if (ii .gt. nsamp) ii = nsamp
                     endif
                     vmap (istrc+iv) = vtr (ii)
                 enddo

      ENDDO
      call maxmgv (vmap, 1, vmax, loc, nx*nv)
      call minmgv (vmap, 1, vmin, loc, nx*nv)
      write(LERR,*)' '
      write(LERR,*)' Maximum velocity = ',vmax
      write(LERR,*)' Minimum velocity = ',vmin
      write(LER ,*)' Maximum velocity = ',vmax
      write(LER ,*)' Minimum velocity = ',vmin
      write(LERR,*)' '
      call lbclos (luvel)

c-------------------
c       .. initializations:
c       .. zero arrays, get sinc weights and set frequency vector

        if (pre) then
           call init (dmo, nsampo, dt, nx, w, sincs, nt, ixmax)
           call savew( itr, 'NumTrc', ngrp  , LINHED)
        else
           call init (dmo, nsampo, dt, nx, w, sincs, nt, nx)
           call savew( itr, 'NumTrc',  1    , LINHED)
        endif

c------------------------------------------------------
c  save headers: exchange # traces/rec & # samples/rec
       call savew( itr, 'NumSmp', nsampo, LINHED)
       call savew( itr, 'NumRec', nx    , LINHED)
c-----------------------------------------------
c  write out output data volume with some reasonable
c  trace headers
c  then in the prestack case close the data set then re-open it
c  ready for business
      IF (pre) THEN

         if (.not.restart) then

            call savhlh ( itr, lbyte, lbyout )
            call wrtape(luout,itr,lbyout)
            do  j = 1, nx
                call savew2(lhed0,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                      j, TRACEHEADER)
                call savew2(lhed0,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                      j, TRACEHEADER)
                do  k = 1, ngrp

                    call savew2(lhed0,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                          k, TRACEHEADER)
                    call savew2(lhed0,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                          30000, TRACEHEADER)
                    call wrtape (luout, itr0, obytes)
                enddo
            enddo
            call lbclos (luout)
            call getln  (luout, otap, 'r+', -1)
            if (luout .lt. 0) then
               write(LERR,*)'Unable to re-open output disk file'
               stop
            endif
            call sislgbuf (luout, 'off')
            call rwd (luout)
            call rtape (luout, itr, lbytes)
         else
            call sislgbuf (luout, 'off')
         endif

      ELSE

         call savhlh ( itr, lbyte, lbyout )
         call wrtape(luout,itr,lbyout)

      ENDIF



C**********************************************************************C
C
C     READ RECORD, DO DMO, WRITE OUTPUT RECORD
C
C**********************************************************************C
 
      DO 100 JJ = iss, lss

           write(LER,*)'Working on shot ',jj,' of ',lss,' shots'
c--------------------------------------------------
c  read record & store
c----------------------
           nlive = 0
           ic = 0
           iflag = 1

           DO    KK = 1, nrcvr

                 nbytes = 0
                 CALL RTAPE  ( LUIN , ITR, NBYTES         )
                 if(nbytes .eq. 0) then
                    write(LERR,*)'WARNING'
                    write(LERR,*)'End of file on input:'
                    write(LERR,*)'  rec= ',jj,'  trace= ',kk
                    go to 999
                 endif
c                .. zero the input trace vector
                 call vclr (trace, 1, nt)
                 call vmov (lhed(ITHWP1), 1, trace, 1, nsamp)

                 call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                       istatic, TRACEHEADER)
                 call saver2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                       idis   , TRACEHEADER)
                 range = float (idis)
                 if (source) then
                    call saver2(lhed,ifmt_SoPtNm,l_SoPtNm, ln_SoPtNm,
     1                          isrc   , TRACEHEADER)
                 else
                    call saver2(lhed,ifmt_SrcLoc,l_SrcLoc, ln_SrcLoc,
     1                          isrc   , TRACEHEADER)
                    isrc = isrc / 10
                 endif
                 xsrc = isrc

                 if (istatic .eq. 30000) then
                    go to 2
                 else
                    nlive = nlive + 1
                 endif

                 if (xs1 .eq. -999999. .AND. first) then
                    if (JJ .ne. 1) then
                       write(LERR,*)'Default starting src point number'
                       write(LERR,*)'requires starting to process data'
                       write(LERR,*)'at beginning of line. Rerun with'
                       write(LERR,*)'-fr1 on cmd line'
                       write(LER ,*)'Default starting src point number'
                       write(LER ,*)'requires starting to process data'
                       write(LER ,*)'at beginning of line. Rerun with'
                       write(LER ,*)'-fr1 on cmd line'
                       go to 999
                    else
                       xs1 = xsrc
                    endif
                 endif

                 IF (range .ge. dl1 .AND. range .le. dl2
     1                              .OR.
     2               range .ge. dr1 .AND. range .le. dr2) THEN

c-------------------
c                   .. calc source position
c                   xs = ( xs1 + ( JJ - 1 ) ) * dxs
                    xs = ( xsrc - xs1 + 1.0 ) * dxs

c-------------------
c                   .. calc receiver position and offset
c                   .. (positive offsets to the right of source)
c                   xg = xs + nearoff + ( KK - 1 ) * dxg

                    xg = xs + range
                    offset = xg - xs

                    if (debug)
     1              write(0,*)'JJ/KK= ',jj,kk,' xs,xg,x= ',xsrc,xs1,xs,
     2                        xg,offset
c-------------------


c-------------------
c                   ... find rms sum of vector elements on this trace
c                   ... if the trace is dead, go get a new one

                    call rmsqv(trace(ist), 1, tmp, nsampo)
                    if ( tmp .eq. 0.0 ) goto 2
c-------------------
 
c-------------------
c                   .. do w**1/2 pre-processing of input trace
                    call iwtrace(trace, nt, dt, itrace, ctrace, w)
c-------------------
 
c-------------------
c   do dmo
c                   .. spray into impulse responses in 'born0 or born1'
                    IF (pre) THEN

                     call born1( trace,dmo,vmap,nsampo,dt,lbyoff,lhed,
     1                            xmin, nx, dx, xs, offset, xg, maxdip,
     2                            aliasflag, ampflag, sprdflag, sincs,
     3                            nsamp,nx,luout,spread,ngrp,obytes,dxg,
     4                            ifmt_RecNum,l_RecNum,ln_RecNum,
     5                            ifmt_DphInd,l_DphInd,ln_DphInd,vmax,
     6                            ifmt_TrcNum,l_TrcNum,ln_TrcNum,lmin,
     7                            ifmt_StaCor,l_StaCor, ln_StaCor,
     8                            ixmax,first,ist,iend,nv,idv)
                    ELSE

                     call born0( trace, dmo, vmap, nsamp, dt, nv,idv,
     1                            xmin, nx, dx, xs, offset, xg, maxdip,
     2                            aliasflag, ampflag, sprdflag, sincs,
     3                            nsampo, nx, ist, iend )
                    ENDIF
                    first = .false.

                 ENDIF

2                continue

           ENDDO


100   CONTINUE

c------------------------------------------------
c  post-stack output only

      IF (.not. pre) THEN

         DO  JJ = 1, nx
c-------------------
c  restore tr headers
             ishdr = (JJ-1)* ITRWRD
             call vmov (itrh(ishdr+1),1, lhed, 1, ITRWRD)
             icdp = nint (xmin) + (jj-1)
             call savew2(lhed,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                   icdp   , TRACEHEADER)
    
             istrc = (JJ-1) * nsampo
             call vmov (dmo (istrc+1), 1, lhed(ITHWP1), 1, nsampo)
 
             CALL WRTAPE  ( LUOUT , ITR, OBYTES         )
    
         ENDDO
      ENDIF
c------------------------------------------------



  999 continue
       call lbclos(luin)
       call lbclos(luout)

      END
