c***********************************************************************
c     
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c     NAME: COMPUSP
c     COMPARE TWO USP DATA FILES
c     REV 4.0  SEP 97 
c***********************************************************************
c     HISTORY:
c     ??? 9?          REV 1.0         R. D. Coleman,  CETech
c     
c     APR 95          REV 2.0         J. Cooperstein, CETech
c     --------------- Complete rewrite; keep only the name
c     
c     JUN 97          REV 3.0         J. Cooperstein, Axian Inc.
c     --------------- Change output
c     --------------- include line and trace header comparisons
c     --------------- general improvements
c     --------------- better portability
c     --------------- work with detached header files (nsmp=0)
c     
c     SEP 97          REV 4.0         J. Cooperstein, Axian Inc.
c     --------------- added some additional parameters
c     --------------- got rid of COMP* printout file and banner
c     
c***********************************************************************
c     compusp compares two usp data files, which need not be conformal;
c     i.e., they may have different numbers of samples, traces,
c     and records.  The second data file may be offset from the
c     second in all three variables (sample, trace, record), by
c     use of the offset parameters nsoff, ntoff, nroff.
c     
c     Its default use is:
c     
c     compusp -Afile1 -Bfile2 -Ooutfile
c     
c     which gives statistics for the entire conformal overlap
c     of the two files, summarizing for each record.
c     
c     If parameters EPSA and/or EPSR are specified (absolute and
c     relative difference thresholds), a side by side comparison of
c     the two files is given, with their relative and absolute
c     difference for each sample, for samples where the threshold is
c     exceeded.
c     
c     If -TRHEAD is specified on the command line, trace headers
c     are compared except when the offsets are non zero.
c     
c     If -LHEAD is specified on the command line, the first 1500
c     words of the line headers are compared
c     
c     The difference statistics reported are:
c     absolute difference = v1 - v2
c     relative difference = (v1 - v2 ) / ( (v1 + v2) / 2 )
c     
c     Averaged sums and the maximum absolute and relative differences
c     found on each record are reported
c     
c***********************************************************************

      program compusp

      implicit none

#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      character file1*100       ! name of file 1
      character file2*100       ! name of file 2
      character ofile*100       ! name of output file
      character ppname*7        ! program name
      character version*4       ! version number

      logical lhead             ! compare line headers
      logical trhead            ! compare trace headers
      logical verbos            ! verbose output flag

      integer ierr              ! error flag
      integer ier               ! error flag
      integer increc            ! record increment
      integer incsmp            ! sample increment
      integer inctrc            ! trace increment
      integer irec1             ! first input record to use
      integer irec2             ! last  input record to use
      integer ismp1             ! first input sample of trace to use
      integer ismp2             ! last  input sample of trace to use
      integer itrc1             ! beginning trace
      integer itrc2             ! ending trace
      integer jrec1             ! index of current input record, file 1
      integer jrec2             ! index of current input record, file 2
      integer jrec1p            ! index of previous input record, file 1
      integer jrec2p            ! index of previous input record, file 2
      integer linediffs         ! number of line header differences found
      integer linehdr1(SZLNHD)  ! length of file 1 line header (words)
      integer linehdr2(SZLNHD)  ! length of file 2 line header (words)
      integer luin1             ! logical unit of file 1
      integer luin2             ! logical unit of file 2
      integer luout             ! logical unit of output file
      integer nbytes            ! length of input file line header (bytes)
      integer nerr              ! number of diffs in trace header
      integer nerrtot           ! total number of diffs in trace headers
      integer nrec1             ! number of records in file 1
      integer nrec2             ! number of records in file 2
      integer nrec_used         ! number of records compared
      integer nroff             ! record offset, file 2
      integer nsmp_used         ! number of samples to check
      integer nsmp1             ! number of samples per trace in file 1
      integer nsmp2             ! number of samples per trace in file 2
      integer nsoff             ! sample offset, file 2
      integer ntoff             ! trace  offset, file 2
      integer ntrc_used         ! number of traces to check
      integer ntrc1             ! number of traces per record in file 1
      integer ntrc2             ! number of traces per record in file 2
      integer nvalues           ! total number of values compared

      real    absmax            ! max abs diff, record
      real    absmax_tot        ! max abs diff, all records
      real    epsa              ! absolute difference to print out
      real    epsr              ! relative difference to print out
      real    sumda             ! sum of abs diffs, record
      real    sumda_tot         ! sum of abs diffs, all records
      real    sumdr             ! sum of rel diffs, record
      real    sumdr_tot         ! sum of rel diffs, all records
      real    summag            ! sum of magnitudes, record
      real    summag_tot        ! sum of magnitudes, all records
      real    sumval            ! sum of values, record
      real    sumval_tot        ! sum of values, all records
      real    relmax            ! max rel diff, record
      real    relmax_tot        ! max rel diff, all records

c     dynamically allocated arrays

      integer headr1(1)         ! trace headers, file 1, current record
      integer headr2(1)         ! trace headers, file 2, current record
      real    rec1(1)           ! data record, file 1
      real    rec2(1)           ! data record, file 2
      real    trace(1)          ! scratch space, trace (including header)

c     variables for length (in words) of dynamically allocated arrays

      integer l_trace, l_rec1, l_rec2, l_headr1, l_headr2

c     pointers

      pointer ( p_headr1, headr1 )
      pointer ( p_headr2, headr2 )
      pointer ( p_trace, trace)
      pointer ( p_rec1,  rec1 )
      pointer ( p_rec2,  rec2  )

c     functions

      logical errchk            ! check if error, print warning message
      integer argis             ! is argument present function

c     data initialization

      data ier,ierr  / 0,0  /
      data luin1  / -1 /
      data luin2  / -1 /
      data ppname / 'COMPUSP' /
      data verbos  /.false./
      data version /'4.0 ' /

c***********************************************************************
 910  format(20x/' ******** Program ',a,' version ', a,' ********')
 902  format( /' ', 'Input file 1 name = ', A100/
     &     ' ', 'Input file 2 name = ', A100)
 904  format( /' ', 'Input file ', i1, ' line header values:')
 906  format(
     &     '   Number of samples/trace  =', i6/
     &     '   Number of traces/record  =', i6/
     &     '   Number of records        =', i6 )
 908  format(/' Number used:'/
     &     '             samples/trace  =', i6/
     &     '             traces/record  =', i6/
     &     '             records        =', i6 ) 
 9007 format(/' LINE HEADER DIFFERENCES'/
     &     '    FIELD      file1       file2')
 9008 format(/' ***** ', I6 
     &     , '   LINE HEADER DIFFERENCES DETECTED ****'/ )
 9100 format(/' nsoff = ', i5, ' ntoff = ', i5, ' nroff = ', i5/ )
 9110 format(' ismp1         = ', i5, ' ismp2         = ', i5,
     &     ' nsmp1 = ', i5
     &     / ' ismp1 + nsoff = ', i5, ' ismp2 + nsoff = ', i5
     &     , ' nsmp2 = ', i5/ )
 9120 format(' itrc1         = ', i5, ' itrc2         = ', i5,
     &     ' ntrc1 = ', i5
     &     / ' itrc1 + ntoff = ', i5, ' itrc2 + ntoff = ', i5
     &     , ' ntrc2 = ', i5/ )
 9130 format(' irec1         = ', i5, ' irec2         = ', i5
     &     ,      ' nrec1 = ', i5
     &     / ' irec1 + nroff = ', i5, ' irec2 + nroff = ', i5
     &     , ' nrec2 = ', i5/ )
 9200 format(/
     &     '  l_rec1   = ',i10, '      l_rec2   = ', i10/
     &     '  l_headr1 = ',i10, '      l_headr2 = ', i10/)
 9400 format(/'   SUM OVER ALL ',i6,' RECORDS,  (',i8,' VALUES )    '/
     &     , 'avg mag    avg-aerr    avg-rerr    max-aerr    max-rerr')
 9405 format( 1p5e12.3 )
 9500 format(/' ********** NO TRACE HEADER DIFFS DETECTED **********'/)
 9998 format( /' ', '***** NORMAL COMPLETION *****'/ )
 9999 format( /' ', '***** ABNORMAL COMPLETION CODE = ', i4, ' *****'/ )
c***********************************************************************
c***********************************************************************

c     check for help (quits if help desired)

      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) call help

c     read program parameters from command line

      call gcmdln( file1, file2, ofile, irec1, irec2, increc
     &     , itrc1, itrc2, inctrc, ismp1, ismp2, incsmp, epsa, epsr
     &     , verbos, trhead, lhead, nsoff, ntoff, nroff )

c     open files

      call lbopen( luin1, file1, 'r' )
      call lbopen( luin2, file2, 'r' )
c     
      if (ofile .eq. ' ') then
         luout = 6
      else
         luout = 27
         open (unit=luout, file=ofile, status='unknown')
      endif

c     read data line headers

      nbytes = 0
      call rtape( luin1, linehdr1, nbytes )
      if(errchk(nbytes.eq.0,'cannot read line header 1', ier)) go to 800
c     
      nbytes = 0
      call rtape( luin2, linehdr2, nbytes )
      if(errchk(nbytes.eq.0,'cannot read line header 2', ier)) go to 800

c     compare line headers

      if( lhead ) then
         write(luout,9007)
         call linehead(luout, linehdr1, linehdr2, linediffs )
         write(luout,9008) linediffs
      endif

c     get parameters from data line headers

      call saver( linehdr1, 'NumSmp', nsmp1, LINHED )
      call saver( linehdr1, 'NumRec', nrec1, LINHED )
      call saver( linehdr1, 'NumTrc', ntrc1, LINHED )
c     
      call saver( linehdr2, 'NumSmp', nsmp2, LINHED )
      call saver( linehdr2, 'NumRec', nrec2, LINHED )
      call saver( linehdr2, 'NumTrc', ntrc2, LINHED )

c     modify parameters

      if( irec2 .lt. irec1 ) irec2 = nrec1
      if( ismp2 .lt. ismp1 ) ismp2 = nsmp1

c     deal with detached header file? 

      if( nsmp1 .eq. 0 ) ismp1 = 0
      if( itrc2 .lt. itrc1 ) itrc2 = ntrc1
c     
      irec2 = min0( irec2, nrec1, nrec2 - nroff )
      ismp2 = min0( ismp2, nsmp1, nsmp2 - nsoff )
      itrc2 = min0( itrc2, ntrc1, ntrc2 - ntoff )
c     
      if( errchk( ismp2.lt.ismp1,' ismp2<ismp1 ', ier ) ) go to 800
      if( errchk( irec2.lt.irec1,' irec2<irec1 ', ier ) ) go to 800
      if( errchk( itrc2.lt.itrc1,' itrc2<itrc1 ', ier ) ) go to 800
      if( errchk( irec1+nroff.lt.1,'invalid record offset', ier ) )
     &     go to 800
      if( errchk( ismp1+nsoff.lt.0,'invalid sample offset', ier ) )
     &     go to 800
      if( errchk( itrc1+ntoff.lt.1,'invalid trace offset', ier ) )
     &     go to 800

c     allocate space for dynamic arrays

      l_headr1 = ITRWRD * ntrc1
      l_headr2 = ITRWRD * ntrc2
      l_trace  = ITRWRD + max0( nsmp1, nsmp2 )
      l_rec1 = nsmp1 * ntrc1
      l_rec2 = nsmp2 * ntrc2
      ntrc_used = ( itrc2 - itrc1 ) / inctrc + 1
      nsmp_used = ( ismp2 - ismp1 ) / incsmp + 1
      nrec_used = ( irec2 - irec1 ) / increc + 1
c     
      call galloc( p_trace,  ISZBYT*l_trace,    ierr, 'ABORT' )
      call galloc( p_rec1,   ISZBYT*l_rec1,     ierr, 'ABORT' )
      call galloc( p_rec2,   ISZBYT*l_rec2,     ierr, 'ABORT' )
      call galloc( p_headr1, ISZBYT * l_headr1, ierr, 'ABORT' )
      call galloc( p_headr2, ISZBYT * l_headr2, ierr, 'ABORT' )

c     
      if( verbos ) then
         write( luout, 910) ppname, version
         write( luout, 902 ) file1, file2
         write( luout, 904 ) 1
         write( luout, 906 ) nsmp1, ntrc1, nrec1
         write( luout, 904 ) 2
         write( luout, 906 ) nsmp2, ntrc2, nrec2
         write( luout, 908 ) nsmp_used, ntrc_used, nrec_used
         write( luout, 9100 ) nsoff, ntoff, nroff
         write( luout, 9110 )
     &        ismp1, ismp2, nsmp1, ismp1+nsoff, ismp2+nsoff, nsmp2
         write( luout, 9120 )
     &        itrc1, itrc2, ntrc1, itrc1+ntoff, itrc2+ntoff, ntrc2
         write( luout, 9120 )
     &        irec1, irec2, nrec1, irec1+nroff, irec2+nroff, nrec2
         write( luout, 9200 ) l_rec1, l_rec2, l_headr1, l_headr2
      endif
c     
c***********************************************************************
c***********************************************************************
c     
      call vclr( rec1, 1, l_rec1 )
      call vclr( rec2, 1, l_rec2 )

      sumval_tot = 0.0
      summag_tot = 0.0
      sumda_tot  = 0.0
      sumdr_tot  = 0.0
      absmax_tot = 0.0
      relmax_tot = 0.0
      jrec1p     = 0
      jrec2p     = 0
      nerrtot    = 0

c     begin loop over records
      do jrec1 = irec1, irec2, increc
         jrec2 = jrec1 + nroff

         if(jrec1.ne.jrec1p+1) call sisseek( luin1, 1+(jrec1-1)*ntrc1 )
         if(jrec2.ne.jrec2p+1) call sisseek( luin2, 1+(jrec2-1)*ntrc2 )

         jrec1p = jrec1
         jrec2p = jrec2

c     read records, files 1 and 2

         call rdrec( luin1, 0, jrec1, ntrc1, 1, ntrc1, nsmp1, 1, nsmp1
     &        , 0, nsmp1, 1, ntrc1, nsmp1, trace, headr1, rec1, ierr )
         if( errchk( ierr.ne.0,'reading record file 1',ier ) ) go to 800
c     
         call rdrec( luin2, 0, jrec2, ntrc2, 1, ntrc2, nsmp2, 1, nsmp2
     &        , 0, nsmp2, 1, ntrc2, nsmp2, trace, headr2, rec2, ierr )
         if( errchk( ierr.ne.0,'reading record file 2',ier ) ) go to 800

c     compare trace headers (if offsets are 0 )

         if( trhead ) then
            call tracehead( luout, headr1, headr2, jrec1, jrec2
     &           , itrc1, itrc2, inctrc, ntoff, nerr )
            nerrtot = nerrtot + nerr
         endif

c     compare records, files 1 and 2, unless detached header file(s)

         if( nsmp1. ne. 0 .and. nsmp2 .ne. 0 ) then

            call compare( luout, jrec1, jrec2, nsmp1, nsmp2, ntrc1
     &           , ntrc2, ismp1, ismp2, incsmp, itrc1, itrc2, inctrc
     &           , nsoff, ntoff, rec1, rec2, epsa, epsr
     &           , summag, sumval, sumda, sumdr, absmax, relmax )

            summag_tot  = summag_tot + summag
            sumval_tot  = sumval_tot + sumval
            sumda_tot   = sumda_tot  + sumda
            sumdr_tot   = sumdr_tot  + sumdr
            absmax_tot  = max(absmax, absmax_tot )
            relmax_tot  = max(relmax, relmax_tot )

         endif

      enddo
c     end loop over records

c***********************************************************************

c     global statistics

      nvalues = max0( nrec_used * nsmp_used * ntrc_used, 1)

      if( trhead .and. nerrtot .eq. 0 ) write (luout, 9500 )
      write( luout, 9400 ) nrec_used, nvalues
      write( luout, 9405 ) summag_tot/float(nvalues)
     &     , sumda_tot/float(nvalues)
     &     , sumdr_tot/float(nvalues)
     &     , absmax_tot, relmax_tot

c***********************************************************************

c     close files, clean-up, & exit

 800  continue

      if( luin1 .gt. 1 ) call lbclos (luin1)
      if( luin2 .gt. 1 ) call lbclos (luin2)

      if( ier .eq. 0 .and. ierr.eq.0 ) then
         write(luout,9998)
      else
         write(luout,9999) ier
         if( ierr.ne.0) write(luout,9999) ierr
      endif

c***********************************************************************
      end
c***********************************************************************
c***********************************************************************
      subroutine gcmdln( file1, file2, ofile, irec1, irec2, increc
     &     , itrc1, itrc2, inctrc, ismp1, ismp2, incsmp, epsa, epsr
     &     , verbos, trhead, lhead, nsoff, ntoff, nroff )
c***********************************************************************

c     GET COMMAND LINE ARGUMENTS

c***********************************************************************

      implicit none

c     parameters

      real      epsbig          ! large default value to turn off
      parameter( epsbig = 1.0e30 )

c     output parameters

      character file1*(*), file2*(*), ofile*(*)
      integer irec1, irec2, increc, itrc1, itrc2, inctrc, ismp1, ismp2
     &     , incsmp, nsoff, ntoff, nroff
      real    epsa, epsr
      logical lhead, trhead, verbos

c     functions

      integer argis             ! is argument present function
      logical errchk            ! check if error, print warning message
      integer ier

c***********************************************************************

      verbos = argis('-V')      .gt. 0
      trhead = argis('-TRHEAD') .gt. 0
      lhead  = argis('-LHEAD')  .gt. 0

      call argi4 ( '-BR', irec1 ,  1,  1 )
      call argi4 ( '-ER', irec2 , -1, -1 )
      call argi4 ( '-IR', increc,  1,  1 )
      call argi4 ( '-OR', nroff ,  0,  0 )

      call argi4 ( '-BT', itrc1 ,  1,  1 )
      call argi4 ( '-ET', itrc2 , -1, -1 )
      call argi4 ( '-IT', inctrc,  1,  1 )
      call argi4 ( '-OT', ntoff ,  0,  0 )

      call argi4 ( '-BS', ismp1 ,  1,  1 )
      call argi4 ( '-ES', ismp2 , -1, -1 )
      call argi4 ( '-IS', incsmp,  1,  1 )
      call argi4 ( '-OS', nsoff ,  0,  0 )

      call argr4 ( '-EPSA', epsa,  epsbig,  epsbig )
      call argr4 ( '-EPSR', epsr,  epsbig,  epsbig )

      call argstr( '-A', file1, ' ', ' ' )
      call argstr( '-B', file2, ' ', ' ' )
      call argstr( '-O', ofile, ' ', ' ' )

c     turn of TRHEAD if offsets are non-zero

      trhead = trhead .and.
     &     nroff .eq. 0 .and. nsoff .eq. 0 .and. ntoff .eq. 0

c     error checking

      if( errchk(file1.eq.' ', 'file 1 must be specified', ier ) ) stop
      if( errchk(file2.eq.' ', 'file 2 must be specified', ier ) ) stop
      if( errchk(irec1.le.0,   'irec1 must be > 0'       , ier ) ) stop
      if( errchk(itrc1.le.0,   'itrc1 must be > 0'       , ier ) ) stop
      if( errchk(ismp1.le.0,   'ismp1 must be > 0'       , ier ) ) stop

c***********************************************************************
      return
      end
c***********************************************************************
c***********************************************************************
      subroutine compare( luout, jrec1, jrec2, nsmp1, nsmp2, ntrc1
     &     , ntrc2, ismp1, ismp2, incsmp, itrc1, itrc2, inctrc
     &     , nsoff, ntoff, rec1, rec2, epsa, epsr
     &     , summag, sumval, sumda, sumdr, absmax, relmax )
c***********************************************************************

c     COMPARE 2 DATA FILES

c***********************************************************************
c     
      implicit none

c     parameters

      real   divchk             ! to avoid division by 0
      real   epsbig             ! large default value to turn off
      parameter( divchk = 1.e-38, epsbig = 1.0e30 )

c     input parameters

      real    epsa              ! absolute difference to print out
      real    epsr              ! relative difference to print out
      integer incsmp
      integer inctrc
      integer ismp1             ! first input sample of trace to use
      integer ismp2
      integer itrc1             ! beginning trace
      integer itrc2
      integer jrec1             ! index of current input record, file 1
      integer jrec2             ! index of current input record, file 2
      integer luout             ! logical unit of output file
      integer nsoff             ! sample offset, file 2
      integer nsmp1             ! number of samples per trace in file 1
      integer nsmp2             ! number of samples per trace in file 2
      integer ntoff             ! trace  offset, file 2
      integer ntrc1             ! number of traces per record in file 1
      integer ntrc2             ! number of traces per record in file 2
      real    rec1(nsmp1,ntrc1) ! data record, file 1
      real    rec2(nsmp2,ntrc2) ! data record, file 2

c     output parameters

      real    summag, sumval, sumda, sumdr, absmax, relmax

c     local parameters

      integer header_flag       ! whether or not to do printout heading
      integer jsmp1             ! sample index, record 1
      integer jsmp2             ! sample index, record 2
      integer jtrc1             ! trace index, record 1
      integer jtrc2             ! trace index, record 2
      integer ncnt              ! number of samples in record to check
      real    aerr              ! absolute difference rec1-rec2
      real    rerr              ! relative difference rec1-rec2
      real    v1                ! value from record 1
      real    v2                ! value from record 2

c     functions

      real fdiv, a, b
      fdiv( a, b ) = a / sign( max( abs(b), divchk ), b )
c     
      data header_flag / 1 /
c     
c***********************************************************************
 9000 format(/' rec1 trc1 smp1', '    value 1  '
     &     , ' rec2 trc2 smp2', '    value 2  '
     &     , ' abs-diff   rel-diff')
 9010 format( 3i5, 1pe13.5, 3i5, 1pe13.5, 1p2e11.3)
 9041 format(' SUM REC1 REC2      '
     &     ,' avg mag    avg-aerr    avg-rerr    max-aerr    max-rerr')
 9042 format(4x, 2i5, 3x, 1p5e12.3)
c***********************************************************************
c     
      if( min (epsa, epsr ) .lt. epsbig ) write( luout, 9000 )
      if(  header_flag .eq. 1 ) then
         write( luout, 9041 )
         header_flag = 0
      endif
c     
      sumval = 0.0
      summag = 0.0
      sumda  = 0.0
      sumdr  = 0.0
      absmax = 0.0
      relmax = 0.0
      ncnt   = 0
c     
      do jtrc1 = itrc1, itrc2, inctrc
         jtrc2 = jtrc1 + ntoff
c     
         do jsmp1 = ismp1, ismp2, incsmp
            ncnt = ncnt + 1
            jsmp2 = jsmp1 + nsoff
c     
            v1   = rec1(jsmp1,jtrc1)
            v2   = rec2(jsmp2,jtrc2)
c     
            aerr = ( v1 - v2 )
            rerr = fdiv ( aerr , 0.5 * (v1 + v2) )
c     
            sumval = sumval + v1
            summag = summag + abs(v1)
            sumda  = sumda  + abs(aerr)
            sumdr  = sumdr  + abs(rerr)
            absmax = max( abs(aerr), absmax )
            relmax = max( abs(rerr), relmax )
c     
            if( min (epsa, epsr ) .lt. epsbig ) then
               if( abs( aerr ) .ge. epsa .or. abs( rerr ) .ge. epsr )
     &              write( luout, 9010 ) jrec1, jtrc1, jsmp1, v1
     &              , jrec2, jtrc2, jsmp2, v2
     &              , aerr, rerr
            endif
         enddo
c     
      enddo
c     
      write( luout, 9042 ) jrec1, jrec2, summag/float(ncnt)
     &     , sumda/float(ncnt), sumdr/float(ncnt)
     &     , absmax, relmax
     
c***********************************************************************
      return
      end
c***********************************************************************
c***********************************************************************
      subroutine tracehead( luout, headr1, headr2, jrec1, jrec2
     &     , itrc1, itrc2, inctrc, ntoff, nerrtot )
c***********************************************************************

c     COMPARE 2 TRACE HEADERS

c***********************************************************************

      implicit none

c     input parameters

      integer*2  headr1(128,*)  ! trace headers, file 1, current record
      integer*2  headr2(128,*)  ! trace headers, file 2, current record
      integer itrc1             ! beginning trace
      integer itrc2             ! ending trace
      integer inctrc            ! trace increment
      integer jrec1             ! index of current input record, file 1
      integer jrec2             ! index of current input record, file 2
      integer luout             ! logical unit of output file
      integer ntoff             ! trace  offset, file 2
      integer nerrtot           ! total number of errors for this record

c     local parameters

      integer i                 ! word index, trace header
      integer jtrc1             ! trace index, record 1
      integer jtrc2             ! trace index, record 2
      integer nerr              ! number of diffs in trace header
c***********************************************************************
      character*6 trc_str(128)
      data (trc_str(i), i = 1,48) /
     & 	   'SGRNum', 'SGRAmp', 'srgdat', 'srgdat', 'FlReFN', 'FlStNm',
     &     'PREPIn', 'InStUn', 'InStAp', 'PREPRc', 'RcStUn', 'RcStAp', 
     &     'ToStAp', 'ToTmAA', 'ToStUn', 'ToTmAU', 'SrComp', 'RcComp',
     &     '      ', '      ', 'TrHdID', 'SrRcAz', 'SrPtXc', 'SrPtXc',
     &     'SrPtYc', 'SrPtYc', 'RcPtXc', 'RcPtXc', 'RcPtYc', 'RcPtYc',
     &     'SrRcMX', 'SrRcMX', 'SrRcMY', 'SrRcMY', 'CDPBCX', 'CDPBCX',
     &     'CDPBCY', 'CDPBCY', 'InTrCn', 'InTrCn', 'FlDteL', 'MulSkw',
     &     'PerSPO', 'InlSPO', 'SrXAzm', 'SrYRot', 'SrZRot', 'RcXAzm'/
      data (trc_str(i), i = 49, 96) /
     &     'RcYRot', 'RcZRot', 'TVPT01', 'TVPV01', 'TVPT02', 'TVPV02',
     &     'TVPT03', 'TVPV03', 'TVPT04', 'TVPV04', 'TVPT05', 'TVPV05', 
     &     'TVPT06', 'TVPV06', 'TVPT07', 'TVPV07', 'TVPT08', 'TVPV08', 
     &     'TVPT09', 'TVPV09', 'TVPT10', 'TVPV10', 'TVPT11', 'TVPV11',
     &     'TVPT12', 'TVPV12', 'TVPT13', 'TVPV13', 'TVPT14', 'TVPV14',
     &     'TVPT15', 'TVPV15', 'TVPT16', 'TVPV16', 'TVPT17', 'TVPV17',
     &     'TVPT18', 'TVPV18', 'TVPT19', 'TVPV19', 'TVPT20', 'TVPV20',
     &     'TVPT21', 'TVPV21', 'VPick1', 'VPick2', 'LRcCDP', 'CabDep'/
      data (trc_str(i), i = 97, 128) /
     &     'WDepDP', 'DPOPer', 'ShtDep', 'UphlTm', 'DpPtLt', 'DpPtLt',
     &     'DpPtLn', 'DpPtLn', 'FoldNm', 'RecNum', 'TrcNum', 'SrcPnt',
     &     'SrcLoc', 'PrRcNm', 'PrTrNm', 'SrPtEl', 'SrPrLt', 'SrPrLt ',
     &     'SrPrLn', 'SrPrLn', 'DstUsg', 'RecInd', 'DstSgn', 'GrpElv',
     &     'LinInd', 'DphInd', 'DePtEl', 'RfSrEl', 'StaCor', 'DatShf',
     &     'SoPtNm', 'SoPtAl' /
c***********************************************************************
 9000 format (' ', 'RECORD', 2I4, ', TRACE', 2I4
     &     , ' HEADER DIFFS =', I4)
c***********************************************************************

      nerrtot = 0

      do jtrc1 = itrc1, itrc2, inctrc
         jtrc2 = jtrc1 + ntoff
         nerr = 0
         do  i = 1, 128
            if (headr1(i,jtrc1) .ne. headr2(i,jtrc2) ) then
               nerr = nerr + 1
               write(luout,'(5x, a6,3i10)') trc_str(i), i
     &              , headr1(i,jtrc1), headr2(i,jtrc2)
            endif
         enddo
         if (nerr .ne. 0)
     &        write (luout, 9000) jrec1, jrec2, jtrc1, jtrc2, nerr
         nerrtot = nerrtot + nerr
      enddo
c***********************************************************************
      return
      end
c***********************************************************************
c***********************************************************************
      subroutine linehead( luout, headr1, headr2, linediffs )
c***********************************************************************

c     COMPARE 2 LINE HEADERS

c***********************************************************************

      implicit none
     
c     input parameters
    
      integer luout, headr1(*), headr2(*), linediffs

c     local parameters

      integer i, type, pos, lng
      character*6 str

#include <f77/save.h>
c***********************************************************************
      linediffs = 0
      do i = 1, lhdsiz
         str  = lhdstr(i)
         type = lhdind(i)
         pos  = lhdpos(i)
         lng  = lhdlng(i)

         if     ( type .eq. 0 ) then ! character
            if ( lng .gt. 1 .or. pos.eq.1 .or. pos.eq.8 ) then
               call charcomp( luout, headr1, headr2, pos, str, lng
     &              , linediffs )
            endif

         elseif ( type .eq. 1 ) then ! short int
            call int2comp( luout, headr1, headr2, pos, str, linediffs )

         elseif( type .eq. 2 ) then ! long int
            call int4comp( luout, headr1, headr2, pos, str, linediffs )

         elseif( type .eq. 3 ) then ! float
            call float4comp( luout, headr1, headr2, pos, str, linediffs)

         endif

      enddo

c***********************************************************************
      return
      end
c***********************************************************************
      subroutine int4comp( luout, headr1, headr2, pos, str, linediffs )
      implicit none
      integer luout, pos, linediffs
      integer headr1(*),headr2(*)
      character *6 str
      if (headr1(pos) .ne. headr2(pos) ) then
         write(luout,'(5x,a6,2i12)' ) str, headr1(pos), headr2(pos)
         linediffs = linediffs + 1
      endif
      return
      end
c***********************************************************************
      subroutine int2comp( luout, headr1, headr2, pos, str, linediffs )
      implicit none
      integer luout, pos, linediffs
      integer*2 headr1(*),headr2(*)
      character *6 str
      if (headr1(pos) .ne. headr2(pos) ) then
         write(luout,'(5x,a6,2i12)' ) str, headr1(pos), headr2(pos)
         linediffs = linediffs + 1
      endif
      return
      end
c***********************************************************************
      subroutine float4comp( luout, headr1, headr2, pos, str, linediffs)
      implicit none
      integer luout, pos, linediffs
      real *4 headr1(*),headr2(*)
      character *6 str
      if (headr1(pos) .ne. headr2(pos) ) then
         write(luout,'(5x,a6,1p2e12.3)') str, headr1(pos), headr2(pos)
         linediffs = linediffs + 1
      endif
      return
      end
c***********************************************************************
      subroutine charcomp( luout, headr1, headr2, pos, str, lng
     &     , linediffs )
      implicit none
      integer luout, pos, lng, linediffs
      character*(*) headr1
      character*(*) headr2
      character *6 str
      if (headr1(pos:pos+lng-1) .ne. headr2(pos:pos+lng-1) ) then
         write(luout,'(5x,a6,5x,a,5x,a)') str 
     &        ,headr1(pos:pos+lng-1), headr2(pos:pos+lng-1)
         linediffs = linediffs + 1
      endif
      return
      end
c***********************************************************************
c***********************************************************************
      logical function errchk( condition, string, ierr )
c***********************************************************************
      logical condition
      character *(*) string
      integer ierr
c***********************************************************************
      errchk = condition
      ierr = 0
      if( condition ) then
         write(0,'(''***** COMPUSP FATAL ERROR: '',a,'' *****'')')string
         ierr = -1 
      endif
c***********************************************************************
      return
      end
c***********************************************************************
