c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c***********************************************************************
c NAME: FXSMIG   F(X) SHOT PRE-STACK MIGRATION  REV 1.0  JAN 95 ********
c***********************************************************************
c
c  HISTORY:
c
c    JAN 95                       REV 1.0         J. Cooperstein, CETech
c
c    FEB 16, 1995                 REV 1.1              Mary Ann Thornton
c
c          - Added call to savhlh to save command line arguments in
c          -    the historical part of the line header
c
c          - Changed calculation of dx (dx=dx1000/1000, not 10000)
c
c          - Wrote a timer routine to replace 'second' which exists
c          -    only on the Crays - Removed the timing calculations
c          -    from fsmigr and left the calculations for timing in
c          -    the main program 'fsmain'
c
c          - Changed headr array to be integer*2 array for execution
c          -    on 32 bit machines.  (fsmain.F and fsfhdr.F)
c
c          - Changed the order of some of the variable type statments
c          -    because the Convex expects variables to be typed
c          -    before they appear as a subscript in another variable
c          -    type statment when "implicit none" is used.
c
c          - Changed name to fxsmig
c
c    JUL 5, 1995                  REV 1.2             Mary Ann Thornton
c
c          - Changed fsfhdr.F to indicate words 1:128 not 0:127
c
c    AUG 16, 1995                 REV 1.3        J. Cooperstein, CETech
c
c          - Corrected timing integral calculation so tmigr reported
c               accurately.
c
c          - Modified function timer to avoid #include <f77/localsys.h>
c               on the CRAY, which was unneccesary and caused problems.
c               Tacked timer.f onto the main routine.
c
c          - Changed the names of all routines except fsmain.F to be *.f
c
c          - Put in -CTAB option to permit optional reading/writing to
c               and from extrapolator coefficient table file. Arrays
c               ctab, cwrk? declared as real, not complex to assist this.
c
c          - Revise help listing, putting in new and undocumented options
c               and correcting some defaults. Put in subroutine fshelp.
c
c          - Reformat, indentation and comment style, cosmetic clean up.
c
c          - More sanity checking: if (nkx < 2*nap) stop, etc.
c
c          - Put in variable striding (Half aperture width a function of
c               frequency.) Two new command line parameters:
c                -MAXAP[nap_use] (half aperture width for each frequency)
c                -LAMBDA[xlam] (minimum number of wavelengths
c                               in half aperture)
c
c    SEP 13, 1995                 REV 2.0        J. Cooperstein, CETech
c
c          - Split functionality into two programs:
c	        1) fxprep -- reads GEOMETRY file from prepmig, cleans
c                            up data by padding dead traces and supplying
c                            headers for them, and centers output records
c                            about the shot.  Produces output geometry
c                            file for use by fxsmig.  Can be piped.
c               2) fxsmig -- performs the migration on a cleaner dataset.
c
c          - Renamed subroutines so all that are part of fxprep begin
c            with the prefix fxp-, and those for fxsmig begin with fxs-.
c            Most subroutines have been modified.
c
c          - Now works in piped mode
c
c
c***********************************************************************
c***********************************************************************
c
      program fxsmig
c
c
      implicit none
c
      integer   LPPNAM          ! number of characters in program name
      integer   MAXNKX          ! maximum number of kx's
      integer   MAXNW           ! maximum number of omegas
      parameter ( LPPNAM = 6, MAXNW = 2048, MAXNKX = 1024 )
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c
c variables:
c
      real      awov0           ! awov reference value
      real      awovmx          ! maximum awov (omega/v)
      real      base            ! controls sharpness of absorp bdry
      character ctape*128       ! name of coefficient table file
      real      dawov           ! delta awov
      integer   dheadr(SZLNHD)  ! length of input file line header (words)
      real      dt              ! delta t
      real      dx              ! delta x
      real      dzmul           ! delta z * izmul for multigriding
      real      dz              ! delta z in velocity model
      real      f1              ! low  cutoff frequency (Hz) for filter
      real      f2              ! low  corner frequency (Hz) for filter
      real      f3              ! high corner frequency (Hz) for filter
      real      f4              ! high cutoff frequency (Hz) for filter
      character gtape*128       ! name of geometry file
      integer   idt             ! Sampling interval used
      integer   idt0            ! Sampling interval from line header
      integer   idx1000         ! dx * 1000 in velocity header
      integer   idz1000         ! dz * 1000 in velocity header
      integer   ierr            ! error flag
      integer   ifac(19)        ! factors of NT
      integer   increc          ! increment between input records
      integer   irec1           ! first input record to use
      integer   irec2           ! last  input record to use
      integer   ismp1           ! first input sample of trace to use
      integer   ismp2           ! last  input sample of trace to use
      integer   itrc            ! trace index
      integer   itrc1           ! first trace to read in input dataset
      integer   itrc2           ! last  trace to read in input dataset
      integer   iw1             ! index of first omega to keep
      integer   izmul           ! multiplier for gridding in z dimension
      integer   jd              ! index of first element in column of DATA
      integer   jp              ! index of first element in row of PSI
      integer   jh              ! header index
      integer   jrec            ! index of current input record
      integer   jrecprev        ! index of last input record
      integer   junk            ! scratch integer
      integer   junk2           ! scratch integer
      integer   jw              ! frequency (omega) index
      integer   jzmax           ! maximum depth index for a shot
      integer   jzmin           ! minimum depth index for a shot
      integer   krec            ! index of current output record
      integer   luinp           ! logical unit of input dataset
      integer   luout           ! logical unit of output dataset
      integer   luvel           ! logical unit of velocity output dataset
      integer   mt              ! row dimension of input DATA (MT = NT+3)
      integer   nabs            ! number of absorbing zones
      integer   nap             ! number of apertures
      integer   nap_use         ! number of apertures to use
      integer   nawov           ! number of awov's
      integer   ndbytes         ! length of input file line header (bytes)
      integer   lbyout          ! length of input file line header (bytes)
      integer   nkx             ! number of kx's
      integer   nrec            ! number of records in input dataset
      integer   nrec2           ! number of records in output dataset
      integer   nshift          ! offset number between station numbers
                                ! and velocity model
      integer   nsmp            ! number of samples per trace in input
      integer   nspad           ! number pad sample
      integer   nsta            ! total number of stations in data
      integer   nsym            ! switch for symmetric window
      integer   nt              ! number of times ( samples-editing+pad )
      character ntape*128       ! name of input file
      integer   ntoff           ! number of zero samples at start of trace
      integer   ntrc            ! number of traces per record in input
      integer   nvbytes         ! length of velocity file line header (bytes)
      integer   nw              ! number frequencies (omega) kept
      integer   nx              ! number of x's (traces per record)
      integer   nxc             ! max number of x's on each side of shot
      integer   nxl             ! number of x's to left of shot
      integer   nxr             ! number of x's to right of shot
      integer   nxlout          ! # of traces to output on left of shot
      integer   nxrout          ! # of traces to output on right of shot
      integer   nxpad           ! # of 0-traces padded in x, each side
      integer   nxuse           ! number of input data traces used
      integer   nxluse          ! # of traces to use on left of shot
      integer   nxruse          ! # of traces to use on right of shot
      integer   nxwin           ! max number of x's to be used
      integer   nxout           ! number of x's to be used in output
      integer   nxv             ! number of x's in velocity file
      integer   nz              ! number of z's
      integer   nzout           ! number of z's (in output model)
      integer   nzmul           ! number of z's for multigriding
      character otape*128       ! name of output file
      character ppname*(LPPNAM) ! program name
      real      strfac          ! 2 pi * vmax / dx, used for migration
      real      switch          ! parameter for fft table building
      character title*66        ! title for printout banner
      real      t0              ! CPU time at program beginning
      real      tcoef           ! CPU time to generate coeffients
      real      ttot            ! total CPU time
      real      tmove0          ! used to calculate tmove
      real      tmove           ! CPU time spent doing geometry
      real      tmigr0          ! used to calculate tmigr
      real      tmigr           ! CPU time spent migrating
      real      tmute           ! muting time offset
      character tunits*5        ! name of time units
      logical   verbos          ! verbose flag
      character version*4       ! version number
      integer   vheadr(SZLNHD)  ! length of velocity file line header (words)
      real      vmin            ! minimum velocity
      real      vmax            ! maximum velocity
      real      vmute           ! muting velocity
      character vtape*128       ! name of velocity data file (input)
      real      xlam            ! number of wavelengths, half aperture
      real      zmax            ! maximum depth
c
c variables used to check coefficient table integrity
c
      real      awov0_chk, dawov_chk, dx_chk, dz_chk, switch_chk
      integer   nap_chk, nawov_chk, nkx_chk
c
c dynamically allocated arrays:
c
      real      ctab(1)         ! fft coefficient table
      real      cwrk1(1)        ! scratch space for fxsmigr
      real      cwrk2(1)        ! scratch space for fxsmigr
      real      cwrk3(1)        ! scratch space for fxsmigr
      real      cwrk4(1)        ! scratch space for fxsmigr
      real      data(1)         ! input/output array
      real      dataout(1)      ! input/output array for final image
      real      filtr(1)        ! frequency filter
      integer*2 headr(1)        ! trace headers, current record
      integer   jz(1)           ! centered depth index
      integer   nlive(1)        ! number of live stations in shot
      integer   nxldata(1)      ! # of traces to left of shot in data
      integer   nxrdata(1)      ! # of traces to right of shot in data
      real      omega(1)        ! angular frequency vector
      real      psi(1)          ! complex wave field (x,w) , shifted sites
      real      rwrk(1)         ! scratch vector for fsmigr
      integer   soptnm(1)       ! station number of shot
      real      tab(1)          ! table for FFT's
      real      trace(1)        ! scratch space, trace (including header)
      real      vshot(1)        ! centered velocity model
      real      vmodel(1)       ! velocity input data
      real      work(1)         ! scratch space
      real      zgrid(1)        ! array of z's for output
      real      zshot(1)        ! centered depths
      real      zsta(1)         ! depth at receiving stations
      real      zmul(1)         ! array of z's for multigrid
c
c variables for length (in words) of dynamically allocated arrays:
c
      integer   lctab, lcwrk1, lcwrk2, lcwrk3, lcwrk4, ldata, ldataout
     &         , lfiltr, lheadr, ljz, lnlive, lomega
     &         , lpsi, lrwrk, lsoptnm, ltab, ltrace, lvmodel
     &         , lvshot, lwork, lzgrid, lzshot, lzsta, lzmul, ltotal
c
c functions:
c
      integer   argis           ! is argument present function
      integer   nrfft5          ! computes integer for use in rfftmlt
      real      timer           ! returns elapsed CPU time
c
c pointers:
c
      pointer ( pctab,   ctab   )
      pointer ( pcwrk1,  cwrk1  )
      pointer ( pcwrk2,  cwrk2  )
      pointer ( pcwrk3,  cwrk3  )
      pointer ( pcwrk4,  cwrk4  )
      pointer ( pdata,   data   )
      pointer ( pdataout,dataout)
      pointer ( pfiltr,  filtr  )
      pointer ( pheadr,  headr  )
      pointer ( pjz,     jz     )
      pointer ( pnlive,  nlive  )
      pointer ( pnxldata, nxldata )
      pointer ( pnxrdata, nxrdata )
      pointer ( pomega,  omega  )
      pointer ( ppsi,    psi    )
      pointer ( prwrk,   rwrk   )
      pointer ( psoptnm, soptnm )
      pointer ( ptab,    tab    )
      pointer ( ptrace,  trace  )
      pointer ( pvmodel, vmodel )
      pointer ( pvshot,  vshot  )
      pointer ( pwork,   work   )
      pointer ( pzgrid,  zgrid  )
      pointer ( pzshot,  zshot  )
      pointer ( pzsta,   zsta   )
      pointer ( pzmul,   zmul   )
c
c data initialization:
c
      data ierr    /  0 /
      data luvel   / -1 /
      data ppname  /'FXSMIG'/
      data title   /                                                   '
     &                        F(X) Shot Migration
     &' /
      data verbos  /.false./
      data version /' 2.0'/
c***********************************************************************
c
c format statements:
c
 9100 format( /' ', ' INPUT DATASET    = '       / a128,
     &        /' ', ' OUTPUT DATASET   = '       / a128,
     &        /' ', ' VELOCITY DATASET = '       / a128,
     &        /' ', ' GEOMETRY DATASET = '       / a128)
 9110 format( /' ', 'INPUT LINE HEADER PARAMETERS:' )
 9120 format( /' ', 'OUTPUT LINE HEADER PARAMETERS:' )
 9130 format(
     &         ' ', '   NUMBER OF SEISMIC RECORDS IN THIS JOB =', i5,
     &        /' ', '   NUMBER OF TRACES PER SEISMIC RECORD   =', i5,
     &        /' ', '   NUMBER OF DATA SAMPLES PER TRACE      =', i5,
     &        /' ', '   SAMPLE INTERVAL (',a5,'SECONDS)        =', i5 )
 9140 format( /'    zmax     =', f13.6,
     &        /'    dz       =', f10.3,
     &        /'    dzmul    =', f10.3,
     &        /'    izmul    =', i6,
     &        /'    nz       =', i6,
     &        /'    nzmul    =', i6,
     &        /'    nzout    =', i6 )
 9150 format( /'    dx       =', f10.3,
     &        /'    nx       =', i6,
     &        /'    nxout    =', i6,
     &        /'    nxwin    =', i6,
     &        /'    nxc      =' ,i6,
     &        /'    nxv      =' ,i6,
     &        /'    ntrc     =' ,i6,
     &        /'    nxpad    =' ,i6,
     &        /'    nap      =' ,i6,
     &        /'    nap_use  =' ,i6,
     &        /'    xlam     =' ,f13.6,
     &        /'    switch   =' ,f13.6,
     &        /'    nkx      =' ,i6,
     &        /'    nabs     =' ,i6,
     &        /'    base     =' ,f13.6 )
 9160 format( /'    irec1    =', i6,
     &        /'    irec2    =', i6,
     &        /'    increc   =', i6,
     &        /'    nrec     =', i6 )
 9170 format( /'    dt       =', f13.6,
     &        /'    nsmp     =', i6,
     &        /'    ismp1    =', i6,
     &        /'    ismp2    =', i6,
     &        /'    nspad    =', i6,
     &        /'    ntoff    =', i6,
     &        /'    nt       =', i6 )
 9180 format(
     &        /'    f1       =', f8.1,
     &        /'    f2       =', f8.1,
     &        /'    f3       =', f8.1,
     &        /'    f4       =', f8.1,
     &        /'    nw       =', i6,
     &        /'    iw1      =', i6,
     &        /'    w1       =', f13.6,
     &        /'    dw       =', f13.6 )
 9200 format( /'    MEMORY ALLOCATION:',
     &        /'    lctab    =', i10,
     &        /'    lcwrk1   =', i10,
     &        /'    lcwrk2   =', i10,
     &        /'    lcwrk3   =', i10,
     &        /'    lcwrk4   =', i10,
     &        /'    ldata    =', i10,
     &        /'    ldataout =', i10,
     &        /'    lfiltr   =', i10,
     &        /'    lheadr   =', i10,
     &        /'    ljz      =', i10,
     &        /'    lnlive   =', i10,
     &        /'    lomega   =', i10,
     &        /'    lpsi     =', i10 )
 9210 format(
     &        /'    lrwrk    =', i10,
     &        /'    lsoptnm  =', i10,
     &        /'    ltab     =', i10,
     &        /'    ltrace   =', i10,
     &        /'    lvmodel  =', i10,
     &        /'    lvshot   =', i10,
     &        /'    lwork    =', i10,
     &        /'    lzgrid   =', i10,
     &        /'    lzshot   =', i10,
     &        /'    lzsta    =', i10,
     &        /'    lzmul    =', i10,
     &        /'    lnxldata =', i10,
     &        /'    lnxrdata =', i10,
     &        /'               ----------',
     &        /'    TOTAL    =', i10 )
 9240 format( /'    nawov    =', i8,
     &        /'    dawov    =', 1pe12.3,
     &        /'    awov0    =', 1pe12.3,
     &        /'    tmute    =', 1pe12.3,
     &        /'    vmute    =', 1pe12.3,
     &        /'    vmin     =', 1pe12.3,
     &        /'    vmax     =', 1pe12.3 /)
 9300 format(' nxl = ', i5, ' nxr = ', i5, ' jzmin = ', i5
     &                                   , ' jzmax = ', i5)
 9990 format(/
     &'***************************************************************'
     & /'       CPU time spent on coefficients = ',f12.3,
     & /'       CPU time spent on geometry     = ',f12.3,
     & /'       CPU time spent on migration    = ',f12.3,
     & /'       CPU time on everything else    = ',f12.3,
     & /'                                        --------'
     & /'                    Total CPU time    = ',f12.3,/
     &'***************************************************************')
 9998 format( /' ', '***** NORMAL COMPLETION *****'/ )
 9999 format( /' ', '***** ABNORMAL COMPLETION CODE = ', i4, ' *****'/ )
c***********************************************************************
c***********************************************************************
c
c=== begin program
c
c***********************************************************************
c
      t0 = timer(0.0)
c
c===  check for help
c
      if( argis( '-h' ) .gt. 0 .or. argis( '-?' ) .gt. 0 ) then
         call fxshelp( LER )
         stop
      endif
c
c===  create unique print file name from process id, open print file
c
      call openpr( LUPPRT, LUPRT, ppname, ierr)
      if( ierr .ne. 0 ) then
         write( LER,* ) '***** ERROR: ierr = ',ierr
     &       ,' reported by openpr *****'
         go to 800
      endif
c
c===  create print banner
c
#include <mbsdate.h>
c
      call gamoco( title, 1, LUPRT )
c
c===  read program parameters from command line
c
      call fxscmdln( LER, verbos, ntape, otape, vtape, gtape, ctape
     &              , base, dawov, dx, dz, f1, f2, f3, f4, switch
     &              , zmax, izmul, idt, increc, irec1, irec2, ismp1
     &              , ismp2, nabs, nap, nkx, nxpad, nspad, nxwin, nxout
     &              , nsym, tmute, vmute, nap_use, xlam )
c
      write( LUPRT,  9100 ) ntape, otape, vtape, gtape
c
c===  open files:     default for pipes: luinp = 0, luout = 1
c
      luinp = 0
      luout = 1
      if( ntape .ne. ' ' ) call lbopen( luinp, ntape, 'r' )
      if( otape .ne. ' ' ) call lbopen( luout, otape, 'w' )
      call lbopen (luvel, vtape, 'r')
c
c===  read data line header
c
      ndbytes = 0
      call rtape( luinp, dheadr, ndbytes )
      if( ndbytes .eq. 0 ) then
         write( LER,* ) ' ***** ERROR - DATA LINE HEADER READ ERROR ***'
         ierr = 1
         go to 800
      endif
c
c===  get parameters from data line header
c
      call saver( dheadr, 'SmpInt', idt0, LINHED )
      call saver( dheadr, 'NumRec', nrec, LINHED )
      call saver( dheadr, 'NumSmp', nsmp, LINHED )
      call saver( dheadr, 'NumTrc', ntrc, LINHED )
c
c===  use all the input records on a pipe
c
      if( ntape .eq. ' ' ) then
         irec1 = 1
         irec2 = nrec
         increc = 1
      endif
c
c===  set some parameters using these values
c
      if( idt .lt. 0 )   idt   = idt0
      if( irec2 .lt. 0 ) irec2 = nrec
      if( ismp2 .lt. 0 ) ismp2 = nsmp
c
c===  read velocity line header
c
      nvbytes = 0
      call rtape( luvel, vheadr, nvbytes )
      if( nvbytes .eq. 0 ) then
         write( LER, * )
     &         '***** ERROR: VELOCITY LINE HEADER READ ERROR ***** '
         ierr = 1
         go to 800
      endif
c
c===  get parameters from velocity line header
c
      call saver( vheadr, 'NumSmp', nz,      LINHED )
      call saver( vheadr, 'NumTrc', nxv,     LINHED )
c
      if( ismp1 .lt. 1 ) then
         ntoff = - ismp1 - 1
         ismp1 = 1
      else
         ntoff = 0
      endif
c
      nt = nrfft5( ismp2 - ismp1 + 1 + nspad + ntoff )
      mt = nt + 3
c
c===  set dx and dz from the velocity line header if not set by fsxcmdln
c
      if ( dx. le. 0.0 ) then
         call saver( vheadr, 'Dx1000', idx1000, LINHED )
         dx = float(idx1000)/1000.0
         if ( dx. le. 0.0 ) then
             write( LER, * ) ' ***** ERROR: dx not set in vel line'
     &           ,' header, must be set from command line *****'
     &           ,' idx1000 = ', idx1000
             ierr = 1
         endif
      endif
c
      if ( dz. le. 0.0 ) then
         call saver( vheadr, 'Dz1000', idz1000, LINHED )
         dz = float(idz1000)/1000.0
         if ( dz. le. 0.0 ) then
             write( LER, * ) ' ***** ERROR: dz not set in vel line'
     &           ,' header, must be set from command line *****'
     &           ,' idz1000 = ', idz1000
             ierr = 1
         endif
      endif
c
      if( ierr. ne. 0 ) go to 800
c
c***********************************************************************
c
c===  calculate x-window size parameters:
c          nxpad traces must be added to left and right
c          nxc is the maximum number of traces to the right or left of shot
c          nxwin is the user supplied number of pixels to use about shot
c                                  w/o padding
c          nxout is the user supplied number of pixels to output,
c                             no greater than nxwin
c          best to keep nxwin and nxout odd numbers;
c          need nx = 2*nxc+1 = nxwin + 2*nxpad
c          note ntrc must be odd, has been set in fxprep
c
      if( nxwin .le. 0 ) then
         nxwin = ntrc
      else
         if( mod( nxwin , 2 ) .eq. 0 ) nxwin = nxwin - 1
         nxwin = min0( nxwin, ntrc )
      endif
c
      if( nxout. le. 0 ) then
         nxout = nxwin
      else
         nxout = min0( nxout, nxwin )
      endif

 
      if( mod( nxout , 2 ) .eq. 0 ) nxout = nxout - 1
c
      nx = nxwin + 2*nxpad
      nxc = nx / 2
c
c***********************************************************************
c
c=== some sanity checking
c
      if( mod( ntrc, 2 ) .eq. 0 ) then
         write( LER, * ) ' ***** ERROR: ntrc must be odd, not ntrc = '
     &        , ntrc, '  ***** '
         ierr = 1
      endif
c
      if( (nabs .le. 0) .or. ( nabs .gt. nx/2 ) ) then
         write( LER, * ) ' ***** ERROR: nabs must be positive and less '
     &        ,'than nx/2 ***** '
         ierr = 1
      endif
c
      if( ( irec1 .le. 0 ) .or. ( irec1 .gt. irec2 ) ) then
          write( LER, * ) ' ***** ERROR: irec1 must be positive and '
     &        ,'no greater than irec2 *****'
         ierr = 1
      endif
c
      if( irec2 .gt. nrec ) then
          write( LER, * )
     &       ' ***** ERROR: irec2 must be no greater than nrec *****'
          ierr = 1
      endif
c
      if( increc .le. 0 ) then
          write( LER, * ) ' ***** ERROR: increc must be positive *****'
          ierr = 1
      endif
c
      if( ( ismp1 .le. 0 ) .or. ( ismp1 .gt. ismp2 ) ) then
         write( LER, * ) ' ***** ERROR: ismp1 must be positive '
     &         ,'and no more than ismp2 *****'
         ierr = 1
      endif
c
      if( ismp2 .gt. nsmp ) then
         write( LER, * )
     &        ' ***** ERROR: ismp2 must be no greater than nsmp *****'
         ierr = 1
      endif
c
      if( ( nkx .le. 0 ) .or. ( nkx .gt. MAXNKX ) ) then
         write( LER, * ) ' ***** ERROR: nkx must be positive '
     &          ,'and no more than MAXNKX *****'
         ierr = 1
      endif
c
      if( nxpad .ge. nxc ) then
         write( LER, * ) ' ***** ERROR: nxpad must be less than nxc '
         ierr = 1
      endif
c
      if ( ierr .ne. 0 ) go to 800
c
c***********************************************************************
c
c===  set dt
c
      if(idt.le.32) then
         dt = 0.001 * float( idt )
      else
         dt = 0.000001 * float( idt )
      endif
c
      if(dt.le.0.0) then
         write( LER, * ) ' ***** ERROR: dt must be positive *****'
         ierr = 1
         go to 800
      endif
c
c===  check and adjust frequency filter parameters
c
      if( f1 .le. 0.0 ) f1 = 0.0
      if( f4 .le. 0.0 ) f4 = float( nt/2 - 1 ) / ( float( nt ) * dt )
      if( f2 .le. 0.0 ) f2 = f1
      if( f3 .le. 0.0 ) f3 = f4
c
c===  allocate space for, compute filter, angular frequency, angle
c
      lfiltr = MAXNW
      lomega = MAXNW
c
      call galloc( pfiltr,  ISZBYT*lfiltr,    ierr, 'ABORT' )
      call galloc( pomega,  ISZBYT*lomega,    ierr, 'ABORT' )
c
      call gtfltr( f1, f2, f3, f4, nt, dt, MAXNW,
     &             nw, iw1, omega, filtr, ierr )
c
      if (ierr .ne. 0) then
         write( LER, * ) ' ***** ERROR in gtfltr, ierr = ',ierr,' *****'
         go to 800
      endif
c
c===  compute nzout
c
      if( zmax .eq. 0.0 ) then
         nzout = nz
      else
         nzout = min0( nint( zmax / dz ), nz )
      endif
c
c===  update and output line header
c
      call hlhprt( dheadr, ndbytes, PPNAME, LPPNAM, LUPRT )
c
      nrec2 = ( irec2 - irec1 ) / increc + 1
      call savew( dheadr, 'SmpInt', idt    , LINHED )
      call savew( dheadr, 'NumSmp', nzout+1, LINHED )
      call savew( dheadr, 'NumTrc', nxout  , LINHED )
      call savew( dheadr, 'NumRec', nrec2  , LINHED )
cmat
c     store command line arguments in historical part of line header
c
      call savhlh( dheadr, ndbytes, lbyout )
      call wrtape(  luout, dheadr,  lbyout )
cmat
c
c===  allocate space for and read velocity model
c              (note: model is transposed on the fly;
c              i.e.,model is vmodel(nxv,nz) )
c
      ltrace   = max0( ITRWRD + nsmp, ITRWRD + nzout + 1 )
      lvmodel  = nz * nxv
c
      call galloc( ptrace,   ISZBYT * ltrace,   ierr, 'ABORT' )
      call galloc( pvmodel,  ISZBYT * lvmodel,  ierr, 'ABORT' )
c
      call vclr( vmodel, 1, lvmodel )
      call rdrecnh( luvel, LUPRT, 1, nxv, 1, nxv, nz, 1, nzout, 0,
     &              nzout, nxv, nxv, 1, trace, vmodel, ierr )
c
      if( ierr .ne. 0 ) then
         write( LER, * ) ' ***** ERROR reading velocity model'
         go to 800
      endif
c
c===  find minimum and maximum velocities (look only for z.le.nzout )
c
      call maxv( vmodel, 1, vmax, junk, nzout*nxv )
      call minv( vmodel, 1, vmin, junk, nzout*nxv )
c
c===  compute dzmul and nzmul
c         should have (nzout-1) = (nzmul-1)*izmul ; end of grid the same
c         use input gridding in if izmul = 1
c         automatic multigridding to the maximum possible if izmul=0
c
      junk = nint( 0.25 * vmin / f4 /dz )
      if( izmul .eq. 0 ) izmul = junk
      write( LUPRT, * )
     &         ' ***** dzmax = ', junk*dz,' using ',izmul*dz,' ******'
c
      if( izmul .eq. 0 ) then
         write( LER, * ) ' ***** ERROR izmul cant be zero *****'
         ierr = 1
         go to 800
      endif
c
      dzmul = izmul * dz
      nzmul = 1 + (nzout-1)/izmul       !truncation important here
c
c***********************************************************************
c
c===  make sure: nap <= nxc
c                nap_use <= nap
c                nkx >= 2*nap
c
      if( nap .eq. 0 ) nap = nap_use
      nap = min0 ( nap, nxc )
      nap_use = min0( nap, nap_use )
      if( nkx .lt. 2 * nap ) then
         write( LER, * ) ' ***** ERROR nkx must be >= 2*nap '
         ierr = 1
         go to 800
      endif
c
c===  calculate factor for stride in migration
c
      strfac = 2.0 * 3.14159265358979323846 * vmax / dx
c
c===  set up for parameters for coefficient table
c
      awov0  = omega(1)  / vmax
      awovmx = omega(nw) / vmin
      nawov  = int( ( awovmx - awov0 ) / dawov ) + 2
c
c***********************************************************************
c
c===  allocate space for dynamic arrays
c
      lctab    = 2 * nap * nawov
      lcwrk1   = 2 * ( nx + 2 * nap - 1 )
      lcwrk2   = 2 * ( nx + 2 * nap - 1 )
      lcwrk3   = 2 * nx
      lcwrk4   = 2 * nx
      ldata    = max0( mt * nx, (nzmul + 1) * nx , mt * ntrc )
      ldataout = (nzout + 1) * nxout
      lheadr   = max0( ITRWRD * nx, ITRWRD * ntrc )
      ljz      = nx
      lnlive   = nrec
      lpsi     = 2 * nw * nx
      lrwrk    = nx
      lsoptnm  = nrec
      ltab     = 2 * nt
      lvshot   = nzmul * nx
      lwork    = max0( 2 * nt * nx, 2 * nap * nx ,nzmul+1 +6*(nzout+1))
      lzgrid   = nzout + 1
      lzmul    = nzmul + 1
      lzshot   = nx
      lzsta    = nxv
c
      call galloc( pctab,    ISZBYT * lctab,    ierr, 'ABORT' )
      call galloc( pcwrk1,   ISZBYT * lcwrk1,   ierr, 'ABORT' )
      call galloc( pcwrk2,   ISZBYT * lcwrk2,   ierr, 'ABORT' )
      call galloc( pcwrk3,   ISZBYT * lcwrk3,   ierr, 'ABORT' )
      call galloc( pcwrk4,   ISZBYT * lcwrk4,   ierr, 'ABORT' )
      call galloc( pdata,    ISZBYT * ldata,    ierr, 'ABORT' )
      call galloc( pdataout, ISZBYT * ldataout, ierr, 'ABORT' )
      call galloc( pheadr,   ISZBYT * lheadr,   ierr, 'ABORT' )
      call galloc( pjz,      ISZBYT * ljz,      ierr, 'ABORT' )
      call galloc( pnlive,   ISZBYT * lnlive,   ierr, 'ABORT' )
      call galloc( pnxldata, ISZBYT * nrec,     ierr, 'ABORT' )
      call galloc( pnxrdata, ISZBYT * nrec,     ierr, 'ABORT' )
      call galloc( ppsi,     ISZBYT * lpsi,     ierr, 'ABORT' )
      call galloc( prwrk,    ISZBYT * lrwrk,    ierr, 'ABORT' )
      call galloc( psoptnm,  ISZBYT * lsoptnm,  ierr, 'ABORT' )
      call galloc( ptab,     ISZBYT * ltab,     ierr, 'ABORT' )
      call galloc( pvshot,   ISZBYT * lvshot,   ierr, 'ABORT' )
      call galloc( pwork,    ISZBYT * lwork,    ierr, 'ABORT' )
      call galloc( pzgrid,   ISZBYT * lzgrid,   ierr, 'ABORT' )
      call galloc( pzmul,    ISZBYT * lzmul,    ierr, 'ABORT' )
      call galloc( pzshot,   ISZBYT * lzshot,   ierr, 'ABORT' )
      call galloc( pzsta,    ISZBYT * lzsta,    ierr, 'ABORT' )
c
c===  setup z-grids for resampling
c
      call vramp( 0.0, dzmul, zmul,  1, nzmul + 1 )
      call vramp( 0.0, dz,    zgrid, 1, nzout + 1 )
c
c***********************************************************************
c
c===  begin verbose output
c
      if( verbos ) then
c
         if( idt0 .le. 32 ) then
            tunits = 'MILLI'
         else
            tunits = 'MICRO'
         endif
         write( LUPRT, 9110 )
         write( LUPRT, 9130 ) nrec, ntrc, nsmp, tunits, idt0
c
         if( idt .le. 32 ) then
            tunits = 'MILLI'
         else
            tunits = 'MICRO'
         endif
         write( LUPRT, 9120 )
         write( LUPRT, 9130 ) nrec2, nx, nzout+1, tunits, idt
         ltotal =  lctab + lcwrk1 +lcwrk2 + lcwrk3 + lcwrk4 + ldata
     &           + ldataout + lfiltr + lheadr + ljz + lnlive + lomega
     &           + lpsi + lrwrk + lsoptnm + ltab + ltrace + lvmodel
     &           + lvshot + lwork + lzgrid + lzshot + lzsta + lzmul
     &           + nrec + nrec
         write( LUPRT, 9200 ) lctab, lcwrk1, lcwrk2, lcwrk3, lcwrk4
     &                      , ldata, ldataout, lfiltr, lheadr
     &                      , ljz, lnlive, lomega, lpsi
         write( LUPRT, 9210 ) lrwrk, lsoptnm, ltab, ltrace, lvmodel
     &              , lvshot, lwork, lzgrid, lzshot, lzsta, lzmul
     &              , nrec, nrec, ltotal
         write( LUPRT, 9140) zmax, dz, dzmul, izmul, nz, nzmul, nzout
         write( LUPRT, 9150) dx, nx, nxout, nxwin, nxc, nxv, ntrc, nxpad
     &                     , nap, nap_use, xlam, switch, nkx, nabs, base
         write( LUPRT, 9160) irec1, irec2, increc, nrec
         write( LUPRT, 9170) dt, nsmp, ismp1, ismp2, nspad, ntoff, nt
         write( LUPRT, 9180) f1, f2, f3, f4, nw, iw1, omega(1)
     &                     , omega(2) - omega(1)
         write( LUPRT, 9240 ) nawov, dawov, awov0, tmute, vmute
     &                     , vmin, vmax
      endif
c
c===  end verbose output
c
c***********************************************************************
c
c===  compute (or read and/or write) coefficient table
c     read from file ctape if name is defined and file exists
c     write to file ctape if name is defined but file doesnt exist
c
      tcoef = timer(0.0)
c
      if( ctape .eq. ' ' ) then
         call fxscoef(ctab, nap, nawov, awov0, dawov, dx, dzmul
     &                                            , switch, nkx )
      else
         open( unit=LUCARD, file=ctape, form='unformatted'
     &                    , status='old', iostat = ierr )
         if( ierr .eq. 0 ) then
            write(LER,*)' reading coefficient table'
            read( LUCARD )awov0_chk, dawov_chk, dx_chk, dz_chk
     &          , switch_chk, nap_chk, nawov_chk, nkx_chk
            if( ( awov0  .ne. awov0_chk  ) .or.
     &          ( dawov  .ne. dawov_chk  ) .or.
     &          (    dx  .ne. dx_chk     ) .or.
     &          ( dzmul  .ne. dz_chk     ) .or.
     &          ( switch .ne. switch_chk ) .or.
     &          (   nap  .ne. nap_chk    ) .or.
     &          ( nawov  .ne. nawov_chk  ) .or.
     &          (   nkx  .ne. nkx_chk    ) ) then
               write( LER, * )' *****ERROR: coefficient table '
     &      ,' written with different arguments; correct or recreate'
               ierr = 1
               go to 800
            endif
            read( LUCARD ) ( ctab(junk), junk = 1, lctab )
            close( LUCARD )
         else
            call fxscoef(ctab, nap, nawov, awov0, dawov, dx, dzmul
     &                                            , switch, nkx )
            open( unit=LUCARD, file=ctape, form='unformatted'
     &                       , status='new')
            write(LER,*)' writing coefficient table'
            write( LUCARD )awov0, dawov, dx, dzmul
     &          , switch, nap, nawov, nkx
            write( LUCARD ) ( ctab(junk), junk = 1, lctab )
            close( LUCARD )
            ierr = 0
         endif
      endif
c
      tcoef = timer(tcoef)
c
c***********************************************************************
c
c===  generate fft table
c
      call fftfax( nt, ifac, tab )
c
c===  read global shot geometry: get:
c                  some global informaton;
c                  shot locations and number of live stations;
c                  depth and integer depth index for stations.
c
      tmove0 = timer(0.0)
      call fxsgeom( LER, LUDISK, gtape, nxv, ntrc, nrec, dx, nshift
     &             , nlive, nsta, soptnm, nxldata, nxrdata, zsta
     &             , ierr )
      tmove = timer(tmove0)
c
      if( ierr .ne .0 ) then
         write( LER, * ) ' ***** ERROR reading geometry file ***** '
         go to 800
      endif
c
c***********************************************************************
c***************  BEGIN THE CALCULATION NOW ****************************
c***********************************************************************
c
      krec     = 0
      jrecprev = 0
      tmigr    = 0.0
c
c===  beginning of loop over records
      do jrec = irec1, irec2, increc
c
	 write( LER, * ) 'PROCESSING RECORD ', jrec
         krec = krec + 1
c
c======  get geometric information necessary for the shot and
c        set up center velocity and depth arrays
c
      tmove0 = timer( 0.0 )
         call fxsmove( LER, dzmul, izmul, nshift, nsym, nxc, ntrc
     &    , nxldata(jrec), nxrdata(jrec), soptnm(jrec)
     &    , nxout, nxpad, nxv, nzmul, nsta
     &    , vmodel, zsta, ierr, itrc1, itrc2, jz, jzmax, jzmin, nxl, nxr
     &    , nxlout, nxrout, nxuse, nxluse, nxruse, vshot, zshot )
      tmove = tmove + timer( tmove0 )
         if (verbos) write( LUPRT, 9300 ) nxl, nxr, jzmin, jzmax
         if( ierr. ne. 0 ) go to 800
c
c======  read record
c        NOTE: record is center justified about the shot
c
         if (verbos) write( LUPRT, * ) ' READING RECORD ', jrec
c
         if( jrec .ne. ( jrecprev + 1 ) ) then
            if( luinp .gt. 0 ) then
               call sisseek( luinp, 1 + ( jrec - 1 ) * ntrc )
            else
               ierr = 1
               write( LER, * ) ' ***** ERROR no siseek on pipe ***** '
               go to 800
            endif
         endif
c
         call vclr( data, 1, ldata )
         call rdrec( luinp, LUPRT, jrec, ntrc, 1, ntrc,
     &               nsmp, ismp1, ismp2, ntoff, nt, 1, ntrc, mt,
     &               trace, headr, data, ierr )
         if( ierr .ne. 0 ) then
            write( LER, * ) ' ***** ERROR reading record ***** '
            go to 800
         endif
c
c======  move to front of data ( done to avoid skipping traces, to
c                                 work with pipes )
         if( itrc1 .ne. 1 ) then
            call vmov( data((itrc1-1)*mt+1), 1, data(1), 1, nxuse*mt )
            do itrc = 1, nxuse
               do jh = 1, 128
                  headr(jh + 128 * (itrc-1) ) = 
     &            headr(jh + 128 * (itrc-1) + 128 * (itrc1-1) )
               enddo
            enddo
         endif
c
c======  mute ( if vmute > 0 )
c
         if( vmute .gt. 0.0 )
     &      call fxsmute( nxc, nxluse, nxruse, zshot, mt, dx, dt, vmute
     &                 , tmute, data )
c
c======  do forward fft
c
         call rfftmlt( data, work, tab, ifac, 1, mt, nt, nxuse, -1 )
c
c======  transpose & filter
c        (note nx for stride, nxuse for number of vectors to process)
c
         call vclr( psi, 1, lpsi )
         jd = 2 * iw1 - 1
         jp = 1 + 2*(nxc-nxluse)
         do jw = 1, nw
            call cvsmul( data(jd), mt, filtr(jw), psi(jp), 2, nxuse )
            jd = jd + 2
            jp = jp + 2 * nx
         enddo
c
c======  note: psi now contains the nx by nw complex wavefield
c              with nxuse values, the rest padded with zeros
c
c======  perform migration
c
      tmigr0 = timer( 0.0 )
         call fxsmigr(LUPRT, nw, nxc, nxl, nxr, nzmul, nap, nawov, awov0
     &                 , dawov, nabs, base, omega, vshot, jz, jzmin
     &                 , jzmax, ctab, psi, work, rwrk, cwrk1, cwrk2
     &                 , cwrk3, cwrk4, data, strfac, nap_use, xlam )
      tmigr = tmigr + timer( tmigr0 )
c
c======  note: data now contains the nx by nzmul+1 real output image
c
c======  resample image for output file
c        clear image first, discard padding on the fly
c
c======  junk  = starting index for output data in array "data"
c        junk2 = total number of real x's in output data
c
         junk = (nxc-nxlout) + 1
         junk2 = nxlout + nxrout + 1
c
         call vclr( dataout, 1, ldataout )
c
         call resamp(junk2, 1, 1, nzmul+1, nx, nzout+1, nxout, zmul
     &     , zgrid , work, data(junk), dataout)
c
c======  dataout now contains the resampled image
c
c======  write record - note: the image is transposed on the fly
c
         if (verbos) write( LUPRT, * ) ' WRITING RECORD ', krec
c
         call wrrec( luout, LUPRT, krec, nzout+1, nxout, nxout, 1,
     &               trace, headr, dataout, ierr )
c
         jrecprev = jrec
      enddo
c===  end of loop over records
c***********************************************************************
      ierr = 0
c
c===  close files, clean-up, & exit
c
800   continue
c
      if( luinp .ge. 0 ) call lbclos (luinp)
      if( luout .ge. 0 ) call lbclos (luout)
      if( luvel .ge. 0 ) call lbclos (luvel)
c
      ttot = timer(t0)
c
      write( LUPRT , 9990 ) tcoef, tmove, tmigr
     &      , ttot - tcoef - tmove - tmigr, ttot
c
      if( ierr .eq. 0 ) then
         write( LUPRT, 9998 )
         stop 0
      else
         write( LUPRT, 9999 ) ierr
         stop 1
      endif
c***********************************************************************
      end
c***********************************************************************
      function timer(sec)
c#include <f77/localsys.h>
#ifdef CRAYSYSTEM
      real sec
      timer = second() - sec
#else
#include <f77/localsys.h>
cmjo      real sec
cmjo      timer = secnds(sec)

      real sec, cpu_time,wall_time
      call timstr(cpu_time,wall_time)
      timer = cpu_time - sec
#endif
      end
c***********************************************************************
