C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
c gam2vel - March 10, 1995
c This is a new program that does a Dix-inversion of gamma scans relative
c to an existing velocity field.
c Gary Murphy         Version 1.0                        March 14, 1995*
c***********************************************************************
c
c     declare variables
c
c-----
c    get machine dependent parameters
c    these includes are picked up and expanded into the code by afp
c    and are found in ~usp/include/f77
 
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
c-----
c    SZLNHD is a value obtained from lhdrsz.h
c    The 3 vectors below are equivalenced and are
c    to access the trace header entries (whatever
c    they may be)
c-----
      integer     lhed ( SZLNHD )
      integer * 2 itr  ( SZLNHD )
      real        head ( SZLNHD )
 
      integer     nsamp, nsi, ntrc, nrec, nrecs, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne
 
c-----------------------------------------------------------------------
c     next lines needed for reading picks
 
      parameter (lpck=27, lpck2=28)
      parameter (maxseg = 3000, maxpnt = 3000, maxnum = maxseg*maxpnt)
      parameter (maxfil = 100, maxmed = maxfil*maxseg)
      integer icolor(1), npts(1)
      integer icolr2(1), npt2(1)
      integer npicko(1)
      real rec(1),trac(1),samp(1)
      real reco(1),traco(1),sampo(1)
      real rec2(1),trac2(1),samp2(1)
      real sego(1)
      integer imed(1),nmed(1)
      real jgamin(1),jgamot(1)
      pointer     (prec, rec)
      pointer     (ptrac, trac)
      pointer     (psamp, samp)
      pointer     (prec2, rec2)
      pointer     (ptrac2, trac2)
      pointer     (psamp2, samp2)
      pointer     (picolr2, icolr2)
      pointer     (pnpt2, npt2)
      pointer     (preco, reco)
      pointer     (ptraco, traco)
      pointer     (psampo, sampo)
      pointer     (psego, sego)
      pointer     (picolor, icolor)
      pointer     (pnpts, npts)
      pointer     (pnpicko, npicko)
      pointer     (pnmed, nmed)
      pointer     (pimed, imed)
      pointer     (pjgamin, jgamin)
      pointer     (pjgamot, jgamot)
      character*20 segnam(maxseg)
      character*20 segnm2(maxseg)
      real units(3),offset(3)
      real units2(3),offst2(3),pow,vmin,vmax
      character*128 picks, pick2
      integer nout,jwin
 
c     maxseg is maximum number of segments allowed
c     maxpnt is maximum number of points for any one segment
c-----------------------------------------------------------------------
 
c------
c  static memory allocation
c     real        bigar1(SZSPRD*SZSMPM)
c     real        bigar2(SZSPRD*SZSMPM)
c------
c  dynamic memory allocation for big arrays, eg whole records
      integer     itrhdr
      real        bigar1, bigar2, bigar3
      pointer     (wkadri, itrhdr(1))
      pointer     (wkadr1, bigar1(1))
      pointer     (wkadr2, bigar2(1))
      pointer     (wkadr3, bigar3(1))
c------
      integer     recnum, trcnum
      integer     srcloc, recind, dphind, dstsgn, stacor
c-----
c    SZSMPM is a value obtained from lhdrsz.h
c-----
      real        tri ( SZLNHD )
      character   ntap * 100, otap * 100, name*7, version*4
      logical     verbos, hlp, query
      logical     heap1, heap2, heap3, heapi
      logical     heap5, heapj, heapn, heapx, heapz

      logical     luinop, luouto
      integer     argis
 
c-----
c    we access the header values which can be shot or long integers
c    or real values.  The actual trace values start at position
c    ITRWRD1  (position 65 in the old SIS format).  This value is
c    set in lhdrsz.h but eventually could come in thru the line header
c    making the trace header format variable
c-----
      equivalence ( itr( 1), lhed (1), head(1) )
 
      data lbytes / 0 /, nbytes / 0 /, name/'GAM2VEL'/, version /' 1.2'/
      data luinop /.FALSE./, luouto /.FALSE./
      data no_seg / 0 /, no_pnt / 0 /
      data nout / 0 /
      data iabort / 0 /
 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
      if ( hlp ) then
           call help()
           stop
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/mbsopen.h>
 
      call gcmdln(ntap,otap,ns,ne,irs,ire,
     1            picks,pick2,pow,dgamma,ngamma,dz,
     1            jwin,vmin,vmax,agrad,ks,ke,dk,iord,verbos)
 
c-----
c     get logical unit numbers for input and output of seismic data
 
c     input values are strings ntap & otap (i.e. names of files), and the
c     read 'r' or write 'w' designations of these files, and the default
c     logical unit numbers if ntap & otap are blank strings, i.e. ntap = " "
c     in which case:
c     0 = default stdin
c     1 = default stdout
c     Note: default values other that 0 & 1 may be used
 
c     output values are the logical unit numbers accessing these disk files. if
c     these values are less than 0 it means there was a fatal error in trying to
c     open these files
c-----
      call getln(luin , ntap,'r', 0)
      luinop=.TRUE.
      call getln(luout, otap,'w', 1)
      luouto=.TRUE.
 
c-----
c     read line header of inputa DSN (rtape reads data into vector "itr"
c     lbytes is the number of bytes actually read
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'GAM2VEL: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
 
c------
c     save certain pace header rameters
 
c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
 
c-----------
c format values are:
 
c     integer    = 1
c     float      = 2
c     character  = 3
c     2-byte int = 4
c-----------
 
c------
c  here we mark out slots to be used for 4-byte floating point
c  storeage in the trace header.  we choose to use the time-velocity
c  area of the trace header but starting from the tail-end to minimize
c  clobbering those folks who do use this area for its intended purpose.
 
c  devlopers are wise to allow some freedom of the user to stake out
c  these slots so he can avoid trashing something he needs.
c  in the 2 cases below we grab the last 2 T-V pairs (for 2 reals)
 
      call savelu('TVPT20',ifmt_TVPT20,l_TVPT20,ln_TVPT20,TRACEHEADER)
      call savelu('TVPT21',ifmt_TVPT21,l_TVPT21,ln_TVPT21,TRACEHEADER)
 
      write(LERR,*)'TrcNum,ifmt,l_TrcNum,length= ',
     1             ifmt_TrcNum,l_TrcNum,ln_TrcNum
      write(LERR,*)'RecNum,ifmt,l_RecNum,length= ',
     1             ifmt_RecNum,l_RecNum,ln_RecNum
      write(LERR,*)'SrcLoc,ifmt,l_SrcLoc,length= ',
     1             ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc
      write(LERR,*)'RecInd,ifmt,l_RecInd,length= ',
     1             ifmt_RecInd,l_RecInd,ln_RecInd
      write(LERR,*)'DphInd,ifmt,l_DphInd,length= ',
     1             ifmt_DphInd,l_DphInd,ln_DphInd
      write(LERR,*)'DstSgn,ifmt,l_DstSgn,length= ',
     1             ifmt_DstSgn,l_DstSgn,ln_DstSgn
      write(LERR,*)'StaCor,ifmt,l_StaCor,length= ',
     1             ifmt_StaCor,l_StaCor,ln_StaCor
 
c     To get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c     LINHED is a value in the include file <f77/sisdef.h> that refers to the
c     lineheader
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrecs , LINHED)
      call saver(itr, 'Format', iform, LINHED)
 
c------
c     hlhprt prints out the historical line header of length lbytes AND
 
c     hlhprt takes "name", in this case 4 characters long and stuffs this
c     into the modified historical line header and returns the NEW length
c     of the line header in lbytes
c------
      call hlhprt (itr, lbytes, name, 7, LERR)
 
c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrecs)
 
c---------------------------------------------------
c  malloc only space we're going to use
      heapi = .true.
      heap1 = .true.
      heap2 = .true.
      heap3 = .true.
      heap5 = .true.
      heapj = .true.
      heapn = .true.
      heapx = .true.
      heapz = .true.
 
c--------------------------
c  note: these don't
c  have to be the same size
      nrecc = ire - irs + 1
      jtr   = ne - ns + 1
      if (ngamma.eq.0) ngamma = jtr
 
      itemi = ntrc * ITRWRD * SZSMPD
      item1 = ntrc * nsamp  * SZSMPD
      item2 = ntrc * nsamp  * SZSMPD
      item3 = ntrc * nsamp  * SZSMPD
 
c  note also SZSMPD is the native
c  size of a float or int in bytes
c--------------------------
 
c--------
c  galloc - general allocation (machine independent since it uses C
c  malloc internally
c  inputs to galloc are pointer, number of bytes to allocate
c  outputs are error codes:
c     errcod = 1  (allocation succeeded)
c     errcod = 0  (allocation failed)
c--------
 
      call galloc (wkadri, itemi, errcdi, aborti)
      call galloc (wkadr1, item1, errcd1, abort1)
      call galloc (wkadr2, item2, errcd2, abort2)
      call galloc (wkadr3, item3, errcd3, abort3)
 
      if (errcdi .ne. 0.) heapi = .false.
      if (errcd1 .ne. 0.) heap1 = .false.
      if (errcd2 .ne. 0.) heap2 = .false.
      if (errcd3 .ne. 0.) heap3 = .false.
 
      if (.not. heap1 .or. .not. heap1 .or. .not. heap2
     1                .or. .not. heap3) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*) item3,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*) item3,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------
 
 
c-----
c     modify line header to reflect actual number of traces output
c-----
      call savew(itr, 'NumRec', nrecc, LINHED)
      call savew(itr, 'NumTrc', jtr  , LINHED)
 
c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
 
c----------------------
c  inject command line into
c  historical LH:
c  inputs are current line header in vector "itr" and length lbytes (bytes)
c  outputs are line header modified by insertion of the command line into
c  the historical LH and the modified length of the new LH (lbyout)
 
      call savhlh(itr,lbytes,lbyout)
c----------------------
 
c------
c     write to unit number luout lbyout bytes contained in vector itr
c------
      call wrtape ( luout, itr, lbyout                 )
 
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrecs, iform,
     1                  picks,pick2,pow,dgamma,ngamma,dz,ntap,otap,
     1                  jwin,vmin,vmax,agrad,ks,ke,dk,iord)
      end if
 
c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary
 
      if (nsi .le. 32) then
         dt = real (nsi) /1000.
      else
         dt = real (nsi) /1000000.
      endif
 
C-----------------------------------------------------------------------
C Read the picks file with rdpic - parameter explanation:
c rec    = x point, trac = y point, samp = z point (returned from rdpic)
c icolor = color number for segment                (returned from rdpic)
c segnam  = name of segment                        (returned from rdpic)
c npts   = npts(1) = no. points in segment 1       (returned from rdpic)
c          npts(2) = no. points in segment 2, etc.
c units  = units for rec,trac,samp                 (returned from rdpic)
c offset = offset for rec,trac,samp                (returned from rdpic)
c maxseg = maximum no. of segments allowed
C no_seg = the no. of segments read                (returned from rdpic)
c maxpnt = maximum no. of pnts/segment allowed
C no_pnt = maximum no. of points in any 1 segment  (returned from rdpic)
C SZSMPD = size of a sample (defined in lhdrsz.h)
C nrec   = no. records in original 'picked' data   (returned from rdpic)
c ntrac  = no. traces in original 'picked' data    (returned from rdpic)
c nsamp  = no. samples in original 'picked' data   (returned from rdpic)
C LERR   = printfile
C lpck   = the input pick file
c jerr   = error flag                              (returned from rdpic)
C-----------------------------------------------------------------------
      if(picks.ne.' ')then
         open(unit=lpck,file=picks,status='old',iostat=jerr)
         if(jerr.ne.0)then
            write(LERR,*)'  Error opening picks file'
            write(LOT,*)' job completed abnormally'
            stop 50
         endif
      else
         write(LERR,*)' Picks filename must be supplied'
         write(LOT,*)' job completed abnormally'
         stop 100
      endif

      if(pick2.ne.' ')then
         open(unit=lpck2,file=pick2,status='old',iostat=jerr)
         if(jerr.ne.0)then
            write(LERR,*)'  Error opening pick2 file'
            write(LOT,*)' job completed abnormally'
            stop 50
         endif
      else
         write(LERR,*)' Pick2 filename must be supplied'
         write(LOT,*)' job completed abnormally'
         stop 100
      endif
 
      write(LERR,10)
   10 format (//, 27X, 'program parameters',//)
      write(LERR,37)picks
      write(LERR,38)pick2
   37 format(' input picks dataset = ',A128)
   38 format(' input horizon picks dataset = ',A128)
      call rdpici(lprt,lpck,jerr,no_seg,no_pnt)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error reading pick file header'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      if (no_seg .gt. maxseg) then
         write(LERR,*)'  Too many segments in picks file'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call rdpici(lprt,lpck,jerr,noseg2,nopnt2)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error reading pick2 file header'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      if (noseg2 .gt. maxseg) then
         write(LERR,*)'  Too many segments in pick2 file'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(prec,no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(ptrac,no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(psamp,no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(prec2,noseg2*nopnt2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(ptrac2,noseg2*nopnt2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(psamp2,noseg2*nopnt2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(preco,jtr*no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(ptraco,jtr*no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(psampo,jtr*no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(psego,jtr*no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(picolor,no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(picolr2,noseg2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pnpts,no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pnpt2,noseg2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pnpicko,no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pnmed,no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pimed,no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pjgamin,nrecs*no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      do i=1,nrecs*no_seg
         jgamin(i)=jtr/2+1
      enddo
      call galloc(pjgamot,nrecs*no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      mxseg=no_seg
      mxpnt=no_pnt
      do i=1,mxseg
         nmed(i)=0
         npicko(i)=0
      enddo
      call rdpic(rec,trac,samp,icolor,segnam,npts,
     &           units,offset,mxseg,no_seg,mxpnt,no_pnt,SZSMPD,
     &           nrec,ntrac,nsmp,LERR,lpck,jerr)
      print *, ' number of segments ', no_seg
      print *, ' number of points ', no_pnt
      if (jerr .ne. 0) goto 999
 
      mxseg2=noseg2
      mxpnt2=nopnt2
      call rdpic(rec2,trac2,samp2,icolr2,segnm2,npt2,
     &           units2,offst2,mxseg2,noseg2,mxpnt2,nopnt2,SZSMPD,
     &           nrec2,ntrac2,nsmp2,LERR,lpck2,jerr)
      print *, ' number of segments ', noseg2
      print *, ' number of points ', nopnt2
      if (jerr .ne. 0) goto 999

 
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
 
c-----
c     process desired trace records
c-----
      do 1000 jj = irs, ire
 
c----------------------
c  skip to start trace
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c----------------------
 
            ic = 0
            do 1001  kk = ns, ne
 
                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
 
c------
c     use previously derived pointers to trace header values
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_SrcLoc,l_SrcLoc, ln_SrcLoc,
     1                        srcloc , TRACEHEADER)
                  call saver2(lhed,ifmt_RecInd,l_RecInd, ln_RecInd,
     1                        recind , TRACEHEADER)
                  call saver2(lhed,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                        dphind , TRACEHEADER)
                  call saver2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        dstsgn , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        stacor , TRACEHEADER)
 
 
c------
 
 
                  if (stacor .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif
 
c----------------------
c  pack data into array
                  ic = ic + 1
                  istrc = (ic-1) * nsamp
                  ishdr = (ic-1) * ITRWRD
                  call vmov (tri,1, bigar1(istrc+1),1, nsamp)
                  call vmov (lhed,1, itrhdr(ishdr+1),1,ITRWRD)
 
1001        continue
 
c----------------------
c  skip to end of record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c----------------------
 
c-----------------------
c  here's the meat...
c  do something to data
 



                     call subs (jtr, nsamp, bigar1, bigar2, 
     1                          bigar3, nrecc, jj,
     1                          rec, trac, samp, npts, no_seg,
     1                          rec2, trac2, samp2, npt2, noseg2,
     1                          ngamma,dgamma,dz,
     1                          pow,jwin,vmin,vmax,agrad,ks,ke,dk,iord)
 
c  (USER: insert your subroutine above)
c-----------------------
 
c---------------------
c  extract traces from
c  output array and
c  write output data
            do 1002 kk = 1, jtr
 
                  istrc = (kk-1) * nsamp
                  ishdr = (kk-1) * ITRWRD
                  call vmov (bigar2(istrc+1),1,lhed(ITHWP1),1, nsamp)
                  call vmov (itrhdr(ishdr+1),1,lhed,1,ITRWRD)
                  call wrtape (luout, itr, obytes)
 
 
 1002             continue

 1000       continue
 
  999 continue
 
c-----
c     close data files
c     flush data left
c     in output buffer
c     NOTE: if the output buffer is not closed you can sometimes end up
c     with missing data
c-----
      if (luinop) call lbclos ( luin )
      if (luouto) call lbclos ( luout )
 
            write(LERR,*)'end of gam2vel, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       HELP                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  ???                                                       *
C  ENTRY POINTS:                                                       *
C      HELP                                                            *
C  ARGUMENTS:     NONE                                                 *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 93/11/10  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 93/11/10  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:                                                              *
C      LER  ( OUTPUT SEQUENTIAL ) -                                    *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'gam2vel updates a velocity using Dix:'
        write(LER,*)
     :'see manual pages for details ( online by typing xmbsman )'
        write(LER,*)' '
        write(LER,*)
     :'execute gam2vel by typing gam2vel and the program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]       (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]       (no default)      : output data file name'
        write(LER,*)
     :' -ns[ns]         (default = first) : start trace number'
        write(LER,*)
     :' -ne[ne]         (default = last)  : end trace number'
        write(LER,*)
     :' -rs[irs]        (default = first) : start record number'
        write(LER,*)
     :' -re[ire]        (default = last)  : end record number'
        write(LER,*)
     :' -P1[picks]      (no default)      : input xsd ',
     :'picks from stack'
        write(LER,*)
     :' -P2[pick2]      (no default)      : input xsd ',
     :'picks from horizon velocity scans'
        write(LER,*)
     :' -p[pow]         (default 2)       : 2=rms, 1=avg'
        write(LER,*)
     :' -dgamma[dgamma] (default .020)    : delta gamma'
        write(LER,*)
     :' -ngamma[ngamma] (default ntr)     : number of gammas'
        write(LER,*)
     :' -dz[dz]         (no default)      : delta z'
        write(LER,*)
     :' -jwin[jwin]     (default 11)      : smoothing window length'
        write(LER,*)
     :' -vmin[vmin]     (default 1480)    : minimum inverval velocity'
        write(LER,*)
     :' -vmax[vmax]     (default 6000)    : maximum inverval velocity'
        write(LER,*)
     :' -agrad[agrad]   (default 1)       : 1=use gradient;0=dix'
        write(LER,*)
     :' -ks[ks]         (default -.5)     : starting gradient'
        write(LER,*)
     :' -ke[ke]         (default 1.5)     : ending gradient'
        write(LER,*)
     :' -dk[dk]         (default .1)      : gradient increment'
        write(LER,*)
     :' -iord[iord]     (default no fit)  : order for polynomial fit'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   gam2vel -N[ntap] -O[otap] -ns[ns] -ne[ne] -rs[irs] '
        write(LER,*)
     :'                 -re[ire] -P1[picks] -P2[pick2] -p[pow]',
     :'                 -dgamma[dgamma] -ngamma[ngamma] -dz[dz]',
     :'                 -jwin[jwin] -vmin[vmin] -vmax[vmax]',
     :'                 -agrad[agrad] -ks[ks] -ke[ke] -dk[dk]',
     :'                 -iord[iord] -V'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       GCMDLN                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  ???                                                       *
C  ENTRY POINTS:                                                       *
C      GCMDLN  (NTAP,OTAP,NS,NE,IRS,IRE,PICKS,POUT,VERBOS,REV)         *
C  ARGUMENTS:                                                          *
C      NTAP    CHAR*(*)  ??IOU* -                                      *
C      OTAP    CHAR*(*)  ??IOU* -                                      *
C      NS      INTEGER   ??IOU* -                                      *
C      NE      INTEGER   ??IOU* -                                      *
C      IRS     INTEGER   ??IOU* -                                      *
C      IRE     INTEGER   ??IOU* -                                      *
C      PICKS   CHAR*(*)  ??IOU* -                                      *
C      POUT    CHAR*(*)  ??IOU* -                                      *
C      VERBOS  LOGICAL   ??IOU* -                                      *
C      REV     LOGICAL   ??IOU* -                                      *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 93/11/10  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 93/11/10  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      ARGSTR          -                                               *
C      ARGI4           -                                               *
C      ARGIS   INTEGER -                                               *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************

      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,
     1                  picks,pick2,pow,dgamma,ngamma,dz,
     1                  jwin,vmin,vmax,agrad,ks,ke,dk,iord,verbos)
c-----
c     get command arguments
c
c     ntap    - C*100    input file name
c     otap    - C*100    output file name
c     ns      - I*4      starting trace index
c     ne      - I*4      ending trace index
c     irs     - I*4      starting record index
c     ire     - I*4      ending record index
c     picks   - C*100    input file name
c     pick2   - C*100    input file name
c     pow    - R*4      2=rms, 1=avg
c     dgamma  - R*4      delta gamma
c     ngamma  - I*4      number of gammas
c     dz      - R*4      output file name
c     jwin    - I*4      window length for dix inversion
c     vmin    - R*4      vmin for inversion (interval)
c     vmax    - R*4      vmax for inversion (interval)
c     agrad   - R*4      =1 for all gradient; =0 for all dix;
c     ks      - R*4      starting gradient
c     ke      - R*4      ending gradient
c     dk      - R*4      gradient increment
c     iord    - I*4      order for polynomial fit
c     verbos  - L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), picks*(*), pick2*(*)
      real        pow,ks,ke,dk
      integer     ns, ne, irs, ire
      logical     verbos
      integer     argis
 
c-------
c     import values from the command line using keys, e.g. -N
c     to which are immediately attached the users values.
 
c     For example program gam2vel might be invoked in the following way:
 
c     gam2vel  -Nxyz -Oabc
 
c     in which case xyz is a string (the name of the input data set)
c     which will be imported into gam2vel and associated with the variable
c     "ntap"
 
c     see manual pages on the argument handler routines
c     for the meanings of these functions.  Briefly though
c     we can import from the command line strings, integers, reals,
c     double precision reals, and check the existence of command line
c     keys.  The last 2 fields are the value to be assigned the variable
c     (1) if ONLY the key is present (no value attached to it), or
c     (2) if NO key & no value are present
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argstr( '-P1', picks, ' ', ' ' )
            call argstr( '-P2', pick2, ' ', ' ' )
            call argr4 ( '-p', pow, 2.0, 2.0 )
            call argr4 ( '-dgamma', dgamma ,   0.020 ,  0.020    )
            call argi4 ( '-ngamma', ngamma ,  0,  0    )
            call argr4 ( '-dz', dz,   10.,  10.    )
            call argi4 ( '-jwin', jwin ,   11  ,  11    )
            call argr4 ( '-vmin', vmin,   1480.,  1480.    )
            call argr4 ( '-vmax', vmax,   6000.,  6000.    )
            call argr4 ( '-agrad', agrad,   1.,  1.    )
            call argr4 ( '-ks', ks,   -.5,  -.5    )
            call argr4 ( '-ke', ke,   1.5,  1.5    )
            call argr4 ( '-dk', dk,   .1,  .1    )
            call argi4 ( '-iord', iord,   0,  0    )
            verbos =   (argis('-V') .gt. 0)
 
c-------
c     Note:  argis() = 1 if the key is present
c            argis() = 0 if it isn't
 
      return
      end
 
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       VERBAL                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  ???                                                       *
C  ENTRY POINTS:                                                       *
C      VERBAL  (NSAMP,NSI,NTRC,NREC,IFORM,PICKS,POUT,NTAP,OTAP)        *
C  ARGUMENTS:                                                          *
C      NSAMP   INTEGER   ??IOU* -                                      *
C      NSI     INTEGER   ??IOU* -                                      *
C      NTRC    INTEGER   ??IOU* -                                      *
C      NREC    INTEGER   ??IOU* -                                      *
C      IFORM   INTEGER   ??IOU* -                                      *
C      PICKS   CHAR*(*)  ??IOU* -                                      *
C      POUT    CHAR*(*)  ??IOU* -                                      *
C      NTAP    CHAR*(*)  ??IOU* -                                      *
C      OTAP    CHAR*(*)  ??IOU* -                                      *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 93/11/10  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 93/11/10  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:                                                              *
C      LERR  ( OUTPUT SEQUENTIAL ) -                                   *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  picks,pick2,pow,dgamma,ngamma,dz,ntap,otap,
     1                  jwin,vmin,vmax,agrad,ks,ke,dk,iord)
c-----
c     verbose output of processing parameters
c
c     nsamp   - I*4     number of samples in trace
c     nsi     - I*4     sample interval in ms
c     ntrc    - I*4     traces per record
c     nrec    - I*4     number of records per line
c     iform   - I*4     format of data
c     ntap    - C*100   input file name
c     otap    - C*100   output file name
c     picks   - C*100   input pick
c     pow     - R*4     2=rms, 1=avg
c     dgamma  - R*4     delta gamma
c     ngamma  - I*4     search window size
c     dz      - R*4     delta z
c     jwin    - I*4     Smoothing window length for dix inversion
c     vmin    - R*4     Minimum velocity for dix inversion
c     vmax    - R*4     Maximum velocity for dix inversion
c     agrad   - R*4     Weight for gradient vrs dix inversion
c     ks      - R*4     starting gradient to test
c     ke      - R*4     ending gradient to test
c     dk      - R*4     increment gradient
c     iord    - I*4     order for polynomial fit
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec
      character   ntap*(*), otap*(*), picks*(*), pick2*(*)
      real        dgamma, dz, pow
      real        ks,ke,dk,vmin,vmax
      integer     ngamma,jwin
 
            write(LERR,*) ' '
            write(LERR,*) ' line header values after default check '
            write(LERR,*) ' # of samples/trace       = ', nsamp
            write(LERR,*) ' sample interval          = ', nsi
            write(LERR,*) ' traces per record        = ', ntrc
            write(LERR,*) ' records per line         = ', nrec
            write(LERR,*) ' format of data           = ', iform
            write(LERR,*) ' input data set name      = ', ntap
            write(LERR,*) ' output data set name     = ', otap
            write(LERR,*) ' input pick file          = ', picks
            write(LERR,*) ' input pick file          = ', pick2
            write(LERR,*) ' rms/avg vel flag         = ', pow
            write(LERR,*) ' delta gamma              = ', dgamma
            write(LERR,*) ' number of gammas         = ', ngamma
            write(LERR,*) ' delta z                  = ', dz
            write(LERR,*) ' rms/int smoothing        = ', jwin
            write(LERR,*) ' minimum int velocity     = ', vmin
            write(LERR,*) ' maximum int velocity     = ', vmax
            write(LERR,*) ' 1=gradient;0=dix         = ', agrad
            write(LERR,*) ' starting gradient        = ', ks
            write(LERR,*) ' ending gradient          = ', ke
            write(LERR,*) ' gradient increment       = ', dk
            write(LERR,*) ' order for polynomial fit = ', iord
            write(LERR,*) ' '
            write(LERR,*) ' '
 
      return
      end
