C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c***********************************************************************
c    hsp --- Despike and Scale Seismic tape based on amplitude histogram
c
c    A USP program to perform histogram scaling of amplitudes into a 
c    user specified range (default is -128 to +127).  A histogram of 
c    the data volume is made and then the central -S of the samples
c    are linearly scaled into the output range. values at the extrema
c    are clipped to -128 and +127.
c
c    Execute "hsp -h" for self documentation.
c
c    histscal reads in a USP dataset (or a subset of the dataset, 
c    depending on various command-line parameters), passes it gather by
c    gather through a subroutine, and writes the results to an output 
c    file.
c
c    NOTE:
c    afp, the "Amoco Fortran Preprocessor", is required to compile
c    this module.
c
c***********************************************************************

c***********************************************************************
c
c     Initialization:
c
c     Load Include Files
c     Declare variables
c
c***********************************************************************

c-----
c
c    Fun and games with the afp preprocessor.
c
c    Define various machine-dependent parameters.
c    These include files are inserted into the code by afp;
c    they can be found under "~usp/include".
 
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

c
c-----
c
c    Because this file is expanded by a pre-processor before the
c    Fortran compiler even sees it, we can also define
c    "magic parameters" such as "SLOP" using the preprocessor.
c    (The usp include files define many more of these in just this
c    way.) Keep in mind, though, that the 72-character line-length
c    limit applies AFTER all the names have been substituted, not
c    before!
c
c    SLOP is the extra space to allow at the end of a dynamically
c    allocated trace, so that buggy USP datasets that have slightly
c    longer traces than the line header promises will not cause
c    our program to core dump.
c
#define  SLOP	(4)
c
c-----

c-----
c    The 3 vectors below are all used to access the LINE header.
c    Which one to use depends on what sort of value you need;
c    header values can be short integers, long integers, or real values.
c    (Or characters, but that case isn't covered here.)
c
c    We might as well declare these statically because we have no way
c    of knowing how big the input line header is before we read it.

      integer     llhed1 ( SZLNHD )
      integer * 2 ilhed1 ( SZLNHD*2 )
      real        rlhed1 ( SZLNHD )
      equivalence ( ilhed1(1), llhed1 (1), rlhed1(1) )

c
c-----

c-----
c    The 3 vectors below are all used to access the TRACE header.
c    Which one to use depends on what sort of value you are using;
c    header values can be short integers, long integers, or real values.
c    (Or characters, but that case isn't covered here.) The actual trace
c    values start at position ITHWP1 = ITRWRD + 1, position 65 in the
c    old SIS format. This value is currently set in lhdrsz.h but
c    in the future may be set in the line header, allowing the trace
c    header length to become a variable.

c    The dynamic way of allocating memory.
c    Declare the three ways we're going to refer to the trace...

      integer     lthed1
      integer * 2 ithed1
      real        rthed1

c    then point all three to the same place.
c    (The Cray fortran manual says not to do this, but in our
c    case we really do want all three to overlay, and there is no
c    obvious other way to do it. Equivalencing the three arrays
c    is specifically disallowed.)
c    The "2" is just to let the compiler know this is an array; some
c    Fortran compilers won't accept 1-element dummy arrays here.

      pointer     (wkadt1, lthed1(2))
      pointer     (wkadt1, ithed1(2))
      pointer     (wkadt1, rthed1(2))

c    Also declare space for a data trace WITHOUT the header.

      real        dtrace
      pointer     (wkadrd, dtrace(2))

c    We'll use galloc later to allocate the memory, after we've
c    found out from the line header how long traces are in this
c    dataset.
c
c-----

c-----
c    Now let's declare two 2-D areas of memory, sufficient for storing
c    entire gathers in. Also declare space for the associated trace
c    headers, which will be kept separately in itrhdr.
c    Again, we can do this either statically or dynamically.

c    dynamic memory allocation for big 2-D arrays

      integer     itrhd1, irecar1
      real        recar1, aryout, hstogrm

      pointer     (wkadh1, itrhd1(2))
      pointer     (wkadi1, irecar1(2))
      pointer     (wkadr1, recar1(2))
      pointer     (wkadr2, aryout(2))
      pointer     (wkadr3, hstogrm(2))

c-----

c-----
c    Miscellaneous declarations
c
c-----

      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin1, luout, luhist, lh1byts, nbytes
      integer     irs,ire,ns,ne
      integer     RecordSize
 
      integer     recnum, trcnum, hstnm
      integer     srcloc, recind, dphind, dstsgn, stacor

      real        vmin, vmax, scal, amin, amax, scalar, bias

      character   ntap*100, otap*100, hfile*100, name*100
      logical     verbose, query, pltr, lbias, lrec, histout
      integer     argis
      character   getname * 100

      data lh1byts / 0 /, nbytes / 0 /, luhist /27/

      vmin = ( 1.0e+42)
      vmax = (-1.0e+42)

c-----


c*********************************************************************
c
c Executable Code begins here
c
c*********************************************************************

c-----
c
c    Find out the name of this program. getname() returns the actual
c    name from argument 0 (after suitable massaging to delete any
c    path and to convert it to upper case).
c
c    Unfortunately FORTRAN has no easy way of shortening a character
c    variable so it doesn't have gratuitous padding on the end!
c    So if we want to grab the name from getname we have to use
c    "name(1:nblen(name))" in error messages in main instead
c    of just "name".
c
      name = getname()
c
c-----

 
c*********************************************************************
c
c Prepare for IO
c
c*********************************************************************
 
c-----
c    First, see if this run is just for self-documentation.
c    (The self-doc routine "help" is defined later on in this source
c    file.)
c

      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0
     1          .or. argis ( '-help' ) .gt. 0 )

      if ( query ) then
        call help(name(1:nblen(name)))
        stop
      endif

c
c-----
 
c-----
c    Not just for self-doc, so open printout files in preparation
c    for processing. The output file consists of a 2-character
c    identifier, the process id number, and the child process id number:
c    XY.nnnnn.mmmmm
c    this should be unique even for multiple occurences of the same
c    process in a pipeline.
c

#include <f77/open.h>

c
c-----

c-----
c    Scan the command line for arguments (the routine gcmdln is not
c    a library routine; it is defined near the end of this source file).
c

      call gcmdln(name(1:nblen(name)),ntap,otap,ns,ne,irs,ire,scal,
     1        amin,amax,hstnm,verbose,pltr,lbias,lrec,histout,hfile)

c
c-----

c-----
c    Get logical unit numbers for input and output of seismic data
c    using the USP library routine "getln".
c
c    Input values are strings ntap and otap (names for the input and
c    output files, respectively), the read 'r' or write 'w'
c    designations of these files, and the default logical unit numbers
c    to use if ntap and/or otap are blank strings. Usually you will
c    want the default unit numbers to be:
c
c     0 = stdin for reading
c     1 = stdout for writing
c
c    Output values are the logical unit numbers to use for accessing
c    these disk files. (If these values are less than 0 it means there
c    was a fatal error opening the corresponding file. Usually
c    getln will not get the chance to return in case of error, however,
c    because the lower-level SIS I/O routines will have aborted the
c    entire program the instant the error occurred.)
c
c    Note if the user did not specify -N or -O on the command line,
c    ntap and otap will be blanks, and the call below will set up
c    stdin and stdout for input and output (respectively).

      call getln(luin1, ntap ,'r', 0)
      call getln(luout, otap ,'w', 1)
      if (histout) then
        open(unit=luhist,file=hfile,status='unknown',iostat=ierr)
      endif

c
c-----

c-----
c
c    Read the input dataset's line header, which comprises the
c    very first record in a USP dataset.
c    Among other useful things, the line header tells us the
c    dimensions of the dataset we're about to process.
c
c    One call to the standard input subroutine rtape reads one
c    record of data. For USP datasets, this is either a line header
c    or a trace header and trace. Here rtape reads data into the
c    array ilhed1; lh1byts is the number of bytes actually read.
c

      call rtape  (luin1, ilhed1, lh1byts)

c
c-----

c-----
c
c    Complain if we can't read the line header.

      if(lh1byts .eq. 0) then
        write(LER ,*) name(1:nblen(name)),
     :  ': no line header read from input1, unit ', luin1
        write(LERR,*) name(1:nblen(name)),
     :  ': no line header read from input1, unit ', luin1
        write(LER ,*) name(1:nblen(name)), ' FATAL'
        stop
      endif

c
c-----
c    Update the historical part of the line header.
c

      call hlhprt (ilhed1, lh1byts, name, nblen(name), LERR)

c    Now inject the current command line into the historical line
c    header as well. 
 
      call savhlh(ilhed1,lh1byts,lh1byts)
c
c-----

c-----
c
c    Make sure to specify keywords in single quotes. FORTRAN
c    converts strings in double quotes to upper case; saver
c    and related subroutines won't recognize those!

      call saver(ilhed1, 'NumSmp', nsamp, LINEHEADER)
      call saver(ilhed1, 'SmpInt', nsi  , LINEHEADER)
      call saver(ilhed1, 'NumTrc', ntrc , LINEHEADER)
      call saver(ilhed1, 'NumRec', nrec , LINEHEADER)
      call saver(ilhed1, 'Format', iform, LINEHEADER)

c
c-----

c-----
c
c    Do some basic sanity checking... this input dataset may just be
c    random junk for all we know!
c

      if (nsamp .le. 0 .or. ntrc .le. 0 .or. nrec .le. 0) then
        write(LER ,*) name(1:nblen(name)),
     1           ': Input line header indicates null input dataset.'
        write(LERR,*) name(1:nblen(name)),
     1           ': Input line header indicates null input dataset.'
        write(LERR,*) 'NumSmp=', nsamp
        write(LERR,*) 'NumTrc=', ntrc
        write(LERR,*) 'NumRec=', nrec
        write(LER ,*) name(1:nblen(name)), ' FATAL'
        stop
      endif

c
c    Ensure that the command-line values for starting and ending trace,
c    record, etc, are compatible with the dimensions of the data set.
c    Command-line parameters are modified to fit as necessary.
c    gcmdln by default returns ns, ne, irs, and ire all zero;
c    zero or negative values are changed to the defaults (to use the
c    entire dataset) by cmdchk.
c    Note cmdchk is a standard USP library routine.
c

      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c
c-----

c-----
c    Modify the line header to be consistent with the number of
c    traces that will be output. savew is the reverse of saver;
c    it writes values into the slot in a trace or line header buffer
c    corresponding to a given keyword.
c
      nrecout = ire - irs + 1
      ntrcout = ne - ns + 1

      if (.not. histout) then

        call savew(ilhed1, 'NumRec', nrecout, LINEHEADER)
        call savew(ilhed1, 'NumTrc', ntrcout  , LINEHEADER)

c    Recalculate the number of output bytes in a trace.
        obytes = SZTRHD + nsamp * SZSMPD
c
c-----

c-----
c    We've finished modifying the line header; write it out.
c    (Write to unit luout the lh1byts bytes in the array ilhed1.)

        call wrtape ( luout, ilhed1, lh1byts  )

      endif
c
c-----

c-----
c
c    Here is an example of how to use savelu to find out the
c    offsets of various trace header parameters.
c

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c
c-----
 
c--------------------------------------------------
c    Compute sample interval in seconds. For historical reasons,
c    if SmpInt is 32 or less, it is in milliseconds. If greater
c    than 32, it is in microseconds!
 
      if (nsi .le. 32) then
        dt = real (nsi) / 1000.
      else
        dt = real (nsi) / 1000000.
      endif

      if (dt .le. 0.) then
        write(LER ,*) name(1:nblen(name)),
     1        ': warning, sample interval ', dt, ' is not positive.'
        write(LERR,*) name(1:nblen(name)),
     1        ': warning, sample interval ', dt, ' is not positive.'
      endif
 
c
c--------------------------------------------------
c
c    Dynamically allocate space to store a single trace.
c
c    SZSMPD is the size of a native float or long integer in bytes.
c    ITRWRD * SZSMPD = SZTRHD is the number of bytes in a trace header;
c    nsamp * SZSMPD is the number of bytes in a data trace.
c
c    SLOP is defined near the top of this file; it allocates a bit
c    of extra space to account for screwed-up datasets that have traces
c    a few elements longer than the line header claims they
c    should be. (This is a result of "off by one" bugs in some
c    processing codes.)
c
c    galloc is a general memory allocation USP library routine.
c    The input to galloc is the number of bytes to allocate, and an
c    integer flag telling it whether to abort on error or not.
c    The output is a pointer to a chunk of memory of the requested
c    size, and an integer error code:
c     ierr =  0  (allocation succeeded)
c     ierr = -1  (allocation failed)
c-----

c
c    Tell galloc not to abort in case of error;
c    we'll check the return code and print our own message.

      iabort = 0

c    Allocate space for a trace with a header attached

c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
      call galloc (wkadt1, SZTRHD + (nsamp+SLOP)*SZSMPD, ierr1, iabort)

c    Allocate space for a data trace, no header

      call galloc (wkadrd, (nsamp + SLOP)*SZSMPD, ierr2, iabort)

      if (ierr1 .ne. 0 .or. ierr2 .ne. 0) then
        write(LER ,*) name(1:nblen(name)),
     1           ': Could not allocate enough space for a trace ',
     2           nsamp, ' samples long.'
        write(LERR,*) name(1:nblen(name)),
     1           ': Could not allocate enough space for a trace ',
     2           nsamp, ' samples long.'
        write(LER ,*) name(1:nblen(name)), ' FATAL'
        stop
      endif

c
c-----

c---------------------------------------------------
c
c    Allocate storage space for an entire record (gather), with
c    trace headers stored separately. Allow space for an extra
c    copy of the trace data, so we can copy from one to the other
c    while performing our "arcane geophysical process".
c
      iabort = 0
      RecordSize = nsamp*ntrcout
 
c    Allocate space for ntrcout trace headers.

      call galloc (wkadh1,  ntrcout   * SZTRHD, ierr1, iabort)
      call galloc (wkadi1, RecordSize * SZSMPD, ierr2, iabort)

c    Allocate space for ntrcout data traces, no headers.

      call galloc (wkadr1, RecordSize * SZSMPD, ierr3, iabort)
      call galloc (wkadr2, RecordSize * SZSMPD, ierr4, iabort)
      call galloc (wkadr3,   hstnm    * SZSMPD, ierr5, iabort)
 
 
      if (ierr1 .ne. 0 .or. ierr2 .ne. 0 .or. ierr3 .ne. 0 .or.
     1    ierr4 .ne. 0 .or. ierr5 .ne. 0) then
        write(LER ,*) name(1:nblen(name)),
     :  ': Unable to allocate workspace.'
        write(LERR,*) name(1:nblen(name)),
     1            ': Unable to allocate workspace for 3 copies of'
        write(LERR,*) ntrcout,' traces of'
        write(LERR,*) nsamp,' samples each.'
        write(LERR,*) ntrcout*SZTRHD, '  bytes for headers;'
        write(LERR,*) 2*RecordSize*SZSMPD, '  bytes for data.'
        write(LER ,*) name(1:nblen(name)), ' FATAL'
        stop
      else
        write(LERR,*) ' '
        write(LERR,*) name(1:nblen(name)),
     :  ': Allocated workspace for 3 copies of'
        write(LERR,*) ntrcout,' traces of'
        write(LERR,*) nsamp,' samples each.'
        write(LERR,*) ntrcout*SZTRHD, '  bytes for headers;'
        write(LERR,*) 2*RecordSize*SZSMPD, '  bytes for data.'
        write(LERR,*) ' '
      endif
 
c---------------------------------------------------

c-----
c     verbose output of all pertinent information before
c     processing begins
c
      if( verbose ) then
        call verbal(nsamp, nsi, ntrc, nrec, iform,
     1          ntap,otap,scal,amin,amax,hstnm,
     2          pltr,lrec)
      endif
c
c-----

c*********************************************************************
c    If running in record constant mode, skip ahead to reading 
c    trace records for scaling

      IF (lrec) GOTO 990

c*********************************************************************
c
c    BEGIN TRACE PROCESSING
c
c    Initially, scan input for min and max values
c
c*********************************************************************

      call scanvol (irecord,irs,ire,itrace,ns,ne,ntrc,ithed1,
     1        nbytes,nsamp,vmin,vmax,dtrace,luin1,lthed1)
c-----
c    If end-of-data encountered (nbytes=0) then bail out gracefully.
c    (In our case we know something's wrong with the input dataset
c    if we hit the end-of-file here.)
c
      if(nbytes .eq. 0) then
        write(LERR,*) name(1:nblen(name)),
     1             ': Unexpected End of file on input,'
        write(LERR,*)'  rec= ',irecord,'  trace= ',itrace
c
c    Bailing out (gracefully)!
c
        go to 999
      endif
c
c-----

      write(LER ,*) name(1:nblen(name)),
     :': Range of values on input : ',vmin,' to ',vmax
      write(LERR,*) name(1:nblen(name)),
     :': Range of values on input : ',vmin,' to ',vmax
      write(LERR,*) ' '

c-----
c
c    Rewind input and set the pointer at the beginning of trace data
c
      call rwd (luin1)
      call rtape (luin1, ilhed1, lh1byts)
c
c-----

c*********************************************************************
c
c    Build a histogram of the input volume and digest to determin 
c    linear amplitude ampping coefficients.
c

      if (scal .ne. 1.0) then

        call histo (irecord,irs,ire,itrace,ns,ne,ntrc,ithed1,
     1          nbytes,nsamp,vmin,vmax,dtrace,luin1,lthed1,
     2          hstogrm,hstnm,histout,luhist)
c-----
c
c    Rewind input and set the pointer at the beginning of trace data
c
        call rwd (luin1)
        call rtape (luin1, ilhed1, lh1byts)
c
c-----

c-----
c    If dumping histogram to output, quit here
        if (histout) then
          itottrcount = ntrcout
          ireccount   = nrecout
          goto 999
        endif

c-----
c
c    On the historgram, find the limits of the input data that will be
c    mapped into the output limits.
c
        call histlim (hstogrm, hstnm, scal, vmin, vmax,
     1                nsamp, ntrcout, nrecout)

      endif

c-----

      write(LER ,*) name(1:nblen(name)),
     :': Histogram clipping range : ',vmin,' to ',vmax
      write(LERR,*) name(1:nblen(name)),
     :': Histogram clipping range : ',vmin,' to ',vmax
      write(LERR,*) ' '

c-----
c-----
c
c    Calculate the linear scaling coefficients. When no bias is 
c    used, scalar is a function of the maximum magnitude.
c
      if (lbias) then

        call lineco (vmin,vmax,amin,amax,scalar,bias,lbias)

      else

        bias = 0.
        if (abs(vmin) .ge. abs(vmax)) then
          scalar = amin/vmin
        else
          scalar = amax/vmax
        endif

      endif

c-----

      write(LER ,*) name(1:nblen(name)),
     :': Linear mapping scale,bias: ',scalar,bias
      write(LERR,*) name(1:nblen(name)),
     :': Linear mapping scale,bias: ',scalar,bias
      write(LERR,*) ' '

c-----
c
c*********************************************************************

  990 CONTINUE

c-----
c
c    NOW PROCEED WITH READING THE VOLUME AND SCALING IT.
c
c*********************************************************************
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c*********************************************************************
c
c    Read traces, buffer them into 2-D arrays,
c    do strange and wonderful things to the 2-D data,
c    and then write the results to the output file until there
c    are no more gathers to process.
c
c*********************************************************************

c------------------------------
c    DIGEST A RECORD FROM INPUT
c------------------------------
c    Skip over unwanted initial records.
c

      call recskp(1,irs-1,luin1,ntrc,ithed1)
 
c
c-----

c-----
c    Process desired trace records;
c    Keep track of how many we've done in ireccount.
c    itottrcount is the total number of traces we've
c    processed in the entire input dataset so far.
c-----

      ireccount = 0
      itottrcount = 0
      do 1000 irecord = irs, ire

c-----
c    We're at the start of the correct record (gather)... now
c    skip to the desired starting trace within that record.
c

        call trcskp(irecord,1,ns-1,luin1,ntrc,ithed1)

c
c-----

c-----
c    Process desired traces within a record.
c    Keep count of how many traces we've done in itrcount.
c-----
        itrcount = 0
        do 1001  itrace = ns, ne
 
          nbytes = 0
          call rtape( luin1, ithed1, nbytes)

c-----
c    Copy the data part of the trace across to dtrace;
c    increment the trace counter.
c
c       CALL VMOV (A,IA,C,IC,N)
c       where,
c       A       Real input vector.
c       IA      Integer input stride for vector A.
c       C       Real output vector.
c       IC      Integer input stride for vector C.
c       N       Integer input element count.
c
          call vmov (lthed1(ITHWP1), 1, dtrace, 1, nsamp)
          itrcount = itrcount + 1
          itottrcount = itottrcount + 1
c
c-----
 
c-----
c    Use previously derived pointers to get trace header values
c    efficiently.
c
          call saver2(lthed1,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                recnum , TRACEHEADER)
          call saver2(lthed1,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                trcnum , TRACEHEADER)
          call saver2(lthed1,ifmt_SrcLoc,l_SrcLoc, ln_SrcLoc,
     1                srcloc , TRACEHEADER)
          call saver2(lthed1,ifmt_RecInd,l_RecInd, ln_RecInd,
     1                recind , TRACEHEADER)
          call saver2(lthed1,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                dphind , TRACEHEADER)
          call saver2(lthed1,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                dstsgn , TRACEHEADER)
          call saver2(lthed1,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                stacor , TRACEHEADER)

          if (verbose) then
            write(LERR,*) ' '
            write(LERR,*) ' Reading input'
            write(LERR,*) ' '
            write(LERR,*)
     1      'recnum,trcnum,srcloc,recind,dphind,dstsgn,stacor= '
     2      ,recnum,trcnum,srcloc,recind,dphind,dstsgn,stacor
          endif


c
c    By convention, a "Station Correction" of 30000 is used as
c    a dead trace flag! If the trace is dead, zero it out to make sure.
c
          if (stacor .eq. 30000) then
            call vclr (dtrace,1,nsamp)
          endif
c
c-----

c-----
c
c    Pack the trace data into a 2-D array, recar1.
c    itrcount-1 is the number of trace-lengths to offset into
c    the 2-D array. Each data trace is nsamp points long.
c    The trace headers are also saved in a separate 2-D array,
c    itrhdr.
c

          istrc = (itrcount-1) * nsamp
          ishdr = (itrcount-1) * ITRWRD
          call vmov (dtrace, 1, recar1(istrc+1), 1,  nsamp)
          call vmov (lthed1, 1, itrhd1(ishdr+1), 1, ITRWRD)
c
c-----

c-----
c    End of the loop over traces within a gather (record).

 1001   continue
        ireccount = ireccount + 1
c
c-----

c-----
c    Skip to the end of the record.
c

        call trcskp(irecord,ne+1,ntrc,luin1,ntrc,ithed1)

c
c-----
 
c*********************************************************************
c
c    Finally, here's the meat...
c    Do something to data.
c
c    USER, INSERT YOUR SUBROUTINE HERE!
c
c    irecord is the record (gather) number
c    ntrcout is the number of traces in a gather.
c    nsamp is the number of sample points in a trace.
c    recar1 is the input 2-D dataset.
c    aryout is the output 2-D dataset.
 
c-----
c
c    Record constant scaling coefficients are determined here.
c    They are done as in volume constant mode but with fewer
c    messages barfed out to the terminal.

        IF (lrec) then

          call vclr(hstogrm,1,hstnm)
          call minv(recar1,1,vmin,ismp,RecordSize)
          call maxv(recar1,1,vmax,ismp,RecordSize)
          call hist(recar1,1,hstogrm,RecordSize,hstnm,vmax,vmin)
          call histlim(hstogrm,hstnm,scal,vmin,vmax,nsamp,
     1                 ntrcout,1)

          if (lbias) then
            call lineco(vmin,vmax,amin,amax,scalar,bias,lbias)
          else
            bias = 0.
            if (abs(vmin) .ge. abs(vmax)) then
              scalar = amin/vmin
            else
              scalar = amax/vmax
            endif
          endif

        ENDIF
c
c-----
                     
c-----
c
c    Apply the scalar, offset and clipping of the input record, then 
c    float it back to a real vector.
c
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
c
c        call vsocfx (recar1,1,scalar,bias,amin,amax,irecar1,1,
c     1               RecordSize,0)
c
c        call vfloat (irecar1,1,aryout,1,RecordSize)
c
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..

        call vsmsa (recar1,1,scalar,bias,aryout,1,RecordSize)
        call vclip (aryout,1,amin,amax,aryout,1,RecordSize)
c        call clipit (aryout,amin,amax,aryout,RecordSize)

c
c-----
c***********************************************************************
c-------------------------------
c    READY TO WRITE OUT A RECORD
c-------------------------------
c
c    Now extract single traces from the 2-D output array aryout and
c    write out the filtered data.
c
c
        do 1003 itrace = 1, ntrcout
 
          istrc = (itrace-1) * nsamp
          ishdr = (itrace-1) * ITRWRD
          call vmov (itrhd1(ishdr+1),1,lthed1,1,ITRWRD)
          call vmov (aryout(istrc+1),1,lthed1(ITHWP1),1,nsamp)
          call wrtape (luout, ithed1, obytes)
 
 1003   continue
 
c
c-----
 
c-----
c
c    End of the loop over gathers (records).

 1000 continue

c
c-----
 
c-----
c
c    A place to jump when bailing out prematurely...

  999 continue

c
c-----
 
c-----
c    Close data files. This flushes any data left in the output
c    buffer. If the output buffer is NOT closed you can sometimes
c    end up with missing data, because data left in the buffer
c    at program exit may fail to be written.
c

      call lbclos ( luin1 )
      call lbclos ( luout )
      if (histout) close (luhist)
 
      write(LERR,*) 'End of "', name(1:nblen(name)),
     1   '"; processed', itottrcount, ' trace(s) in',
     2   ireccount, ' record(s).'
      write(LER ,*) 'End of "', name(1:nblen(name)),
     1   '"; processed', itottrcount, ' trace(s) in',
     2   ireccount, ' record(s).'

c
c-----

      stop      
      end

c***********************************************************************
c
c    Here's the routine that prints out the help message.
c
c
      subroutine help(name)
      character * (*)  name
#include <f77/iounit.h>


         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*) name,
     :': program to scale data based on a histogram of the'
        write(LER,*)
     :'data volume and a user specified output amplitude range.'
        write(LER,*)' '
        write(LER,*)
     :'Execute by typing "hsp',
     :'" followed by the program parameters.'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default; stdin if not specified)'
        write(LER,*)
     :'                                   : input seismic file name'
        write(LER,*)
     :' -O [otap]    (no default; stdout if not specified)'
        write(LER,*)
     :'                                   : output data file name'
        write(LER,*)
     :' -H [hfil]    (default = none)     : output histogram file'
        write(LER,*)
     :' -p [hstnm]   (default = amax-amin): number of histogram bins'
        write(LER,*)
     :' -ns[ns]      (default = first)    : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last )    : end trace number'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -re[ire]     (default = last )    : end record number'
        write(LER,*)' '
        write(LER,*)
c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
     :' -S [scal]    (default = .98  )    : fraction of samples to use'
        write(LER,*)
     :'                                     in determining scale factor'
        write(LER,*)' '
        write(LER,*)
     :' -amin[amin]  (default = -128.)    : user defined minimum and'
        write(LER,*)
     :' -amax[amax]  (default = +127.)      maximum output amplitudes'
        write(LER,*)' '
        write(LER,*)
     :' -PLTR                             : scale output -1024 to +1023'
        write(LER,*)
     :' -bias                             : use bias to fill the output'
        write(LER,*)
     :'                                     range(maximize dynamic rng)'
        write(LER,*)
     :' -rec                              : do record constant scaling'
        write(LER,*)
     :'                                     (enables piped inputs)'
        write(LER,*)
     :' -hist                             : dump off histogram only'
        write(LER,*)
     :' -V                                : verbose printout'
        write(LER,*) ' '
        write(LER,*)
     :'usage:',
     :'hsp -N[ntap] -O[otap] -ns[ns] -ne[ne] -rs[irs] -re[ire] -S[scal]'
        write(LER,*)
     :'          -amin[amin] -amax[amax] [-PLTR -bias -rec -hist -V]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
c***********************************************************************
c
c    Here's the routine that scans the command line looking for
c    relevant arguments.
c
c
      subroutine gcmdln(name,ntap,otap,ns,ne,irs,ire,scal,amin,amax,
     1                  hstnm,verbose,pltr,lbias,lrec,histout,hfile)
c-----
c    Set the following variables:
c
c     ntap       - C*100    input file name
c     otap       - C*100    output file name
c     hfile      - C*100    output histogram file name
c     ns         - I*4      starting trace index
c     ne         - I*4      ending trace index
c     irs        - I*4      starting record index
c     ire        - I*4      ending record index
c     scal       - R*4      Fraction of input amps to use
c     amin       - R*4      minimum value for output mapping
c     amax       - R*4      maximum value for output mapping
c     verbose    - L        verbose output or not
c     pltr       - L        scale output to 2047 or not
c     lbias      - L        set bias to zero before mapping
c     lrec       - L        record constant mode allowing pipes
c
c-----
      character   name*(*)
      character   ntap*(*), otap*(*), hfile*(*)
      integer     ns, ne, irs, ire, hstnm
      real        amin, amax, scal
      logical     verbose, pltr, lbias, lrec, histout
      integer     argis, argfre
      character   sargvv * 100
      character   args * 100
#include <f77/iounit.h>
 
c-----
c    Options in USP are specified on the command line.
c    For example prgm might be invoked in the following way:
c 
c    prgm  -Nxyz -Oabc
c
c    or
c
c    prgm  -N xyz -O abc
c 
c    In this case xyz is a string (the name of the input data set);
c    the subroutine call
c           call argstr( '-N', ntap, ' ', ' ' )
c    would then set the variable ntap to have the value "xyz".
c
c    In the calls below,
c    the first argument specifies the key to look for.
c    The second argument specifies the variable to return values in.
c    The last 2 fields are the values to assign a variable
c    (1) if ONLY the key is present (with no value attached to it),
c    or
c    (2) if the KEY is not even present.
c
c    We can import from the command line strings, integers, reals,
c    and double precision reals, and we can check for the presence
c    of a given command-line key.
c
c    After an argument has matched something once, it is removed
c    from consideration for future matches. (So you have to save
c    the value returned by a given call; you can't just call
c    a second time to see again what it was.)
c
c    See the manual pages on the argument handler routines
c    for more information about these functions.
c

      call argstr( '-N', ntap, ' ', ' ' )
      call argstr( '-O', otap, ' ', ' ' )
      call argstr( '-H',hfile, ' ', ' ' )
      call argi4 ( '-ns', ns , 0  ,  0  )
      call argi4 ( '-ne', ne , 0  ,  0  )
      call argi4 ( '-rs',irs , 0  ,  0  )
      call argr4 ( '-S',scal, .98, .98 )
      call argr4 ( '-amin',amin,-128.,-128.)
      call argr4 ( '-amax',amax, 127., 127.)
      call argi4 ( '-p ',hstnm , -1 ,  -1 )
c
c    Note:  argis() = 1 if the key is present
c           argis() = 0 if it isn't
c
      verbose =   (argis('-V') .gt. 0)
      pltr    =   (argis('-PLTR') .gt. 0)
      lbias   =   (argis('-bias') .gt. 0)
      lrec    =   (argis('-rec' ) .gt. 0)
      histout =   (argis('-hist' ) .gt. 0)
      call argi4 ( '-re',ire , 0  ,  0  )

c-----
c    Check for piped input. If yes, check for record constant
c    scaling. If the input is a pipe and record constant scaling
c    is NOT requested, bail out.

      if(ntap .eq. ' ' .and. .not. lrec) then
         write(LER,*) ' '
         write(LER,*) name(1:nblen(name)),
     :   ': Volume constant scaling of a piped input has been'
         write(LER,*) name(1:nblen(name)),
     :   ': requested but cannot be done.'
         write(LER,*) ' '
         write(LER,*) name(1:nblen(name)),
     :   ': hsp will accept piped inputs only when record constant'
         write(LER,*) name(1:nblen(name)),
     :   ': scaling is requested (use -rec on the command line).'
         write(LER,*) ' '
         stop
      endif
c
c-----

c-----
c    Look for scal greater than 1 and notify user
      if (scal .gt. 1.) then
        write(LER ,*) name(1:nblen(name)),
     :  ': parameter -scal(',scal,') is greater than 1. Resetting to 1.'
        scal = 1.
      endif
c
c-----

c-----
c
c    It's a good idea to warn the user that arguments they've
c    taken the trouble to list are actually being ignored.
c
c    Argfre, called with a  non-negative  argument,  returns  the
c    index  of  the next command-line argument which has not been
c    previously consumed, or 0 if no unconsumed arguments remain.
c
c    Argeat causes the argument associated with its non-negative
c    index  to  be marked as consumed so that subsequent calls to
c    argfre (or any other function  in  this  package)  will  not
c    detect it.
c
c    sargvv prints out the associated argument string.
c

 50   narg = argfre(0)
      if (narg .gt. 0) then
        args = sargvv(narg)
        write(LER ,*) name,
     :  ': unknown argument "', args(1:nblen(args)),
     :  '" ignored.'
        write(LERR,*) name,
     :  ': unknown argument "', args(1:nblen(args)),
     :  '" ignored.'
        call argeat (narg)
        goto 50
      endif
c
c-----
 
c-----
c    Limit ncolors to increments of at least 1 (only 256 colors 
c    available to openwin), turn boun into a fraction.
c
      if (pltr) then
        amin = -1024.
        amax =  1023.
      endif

c-----
c    The number of histogram buckets is somewhat arbitrary. Here,
c    it is tied to the resolution in the output data by using the 
c    number of integer steps between minimum and maximum output
c    values.

      if (hstnm .lt. 0 ) hstnm = int(amax-amin+1.5)

      return
      end
 
c***********************************************************************
c
c    Verbose output of processing parameters
c
c
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  ntap,otap,scal,amin,amax,hstnm,
     2                  pltr,lrec)
c
c-----
c
c     nsamp      - I*4     number of samples in trace
c     nsi        - I*4     sample interval in ms
c     ntrc       - I*4     traces per record
c     nrec       - I*4     number of records per line
c     iform      - I*4     format of data
c     ntap       - C*100   input file name
c     otap       - C*100   output file name
c     scal       - R*4     Fraction of input amps to use
c     amin       - R*4     minimum value for output mapping
c     amax       - R*4     maximum value for output mapping
c     hstnm      - I*4     number of histogram buckets
c
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec, hstnm
      real        scal, amin, amax
      logical     pltr, lrec
      character   ntap*(*), otap*(*)
 
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*) ' # of samples/trace = ', nsamp
      write(LERR,*) ' sample interval    = ', nsi
      write(LERR,*) ' traces per record  = ', ntrc
      write(LERR,*) ' records per line   = ', nrec
      write(LERR,*) ' format of data     = ', iform
      write(LERR,*)' '
      write(LERR,*) ' input  dataset name = ',ntap
      write(LERR,*) ' output dataset name = ',otap
      write(LERR,*)' '
      write(LERR,*) ' fraction of amps used       = ', scal
      write(LERR,*) ' output mapping minimum      = ', amin
      write(LERR,*) ' output mapping maximum      = ', amax
      write(LERR,*) ' number of histogram buckets = ', hstnm
      write(LERR,*)' '
      write(LERR,*) ' PLTR scaling                = ', pltr
      write(LERR,*) ' Record constant mode        = ', lrec
      write(LERR,*)' '
      write(LERR,*)' '

      return
      end
