C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       IVAD                                                 *
C  ROUTINE TYPE:  MAIN                                                 *
C  PURPOSE:       PROGRAM IVAD AUTOMATICALLY COMPUTES DIP USING        *
C                 THE SPIN ALGORITHM.  IT INTERPOLATES VELOCITIES      *
C                 ALONG DIP AND OUTPUTS A VELOCITY TAPE.               *
C  ENTRY POINTS:  MAINLINE ENTRY                                       *
C  ARGUMENTS:     NONE                                                 *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:    GARY MURPHY                     ORIGIN DATE:  90/03/01   *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 90/03/01  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      rdcard          -                                               *
C      LBOPEN          -                                               *
C      OPENPR          -                                               *
C      GAMOCO          -                                               *
C      ICOPEN  INTEGER -                                               *
C      RTAPE           -                                               *
C      HLHPRT          -                                               *
C      WRTAPE          -                                               *
C      VCLR            -                                               *
C      VMOV            -                                               *
C      VMUL            -                                               *
C      LBCLOS          -                                               *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      FLOAT   REAL -                                                  *
C  FILES:                                                              *
C      LCRD  ( INPUT  SEQUENTIAL ) -                                   *
C      LPRT  ( OUTPUT SEQUENTIAL ) -                                   *
C  COMMON:           NONE                                              *
C  STOP CODES:                                                         *
C      200      ( 1) -  OPEN PRINT FILE ERROR                          *
C      =BLANK=  ( 1) -  NO ERRORS                                      *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  GARY MURPHY                   REVISION DATE: 90/03/01  *
C            - INITIAL RELEASE                                         *
C  REVISED BY: MARY ANN THORNTON              DATE: 91/09/05           *
C            - Moved code to sun to be maintained, distributed there   *
C  REVISED BY: DAVID W.NELSON    V2.2         DATE: 92/01/07           *
C            - Changed NVMAX from 375 to 1000 to match pwmvzn          *
C            - Use NINT for LTR and NSO computations                   *
C  REVISED BY: Mary Ann Thornton V2.3         DATE: 92/03/25           *
C            - Call openpr with full program name for OS 6.1           *
c  revised by: gary murphy v2.4               data: 92/06/29           *
c            - allow output of dips and other calculations             *
c            - fully implement verbos option                           *
c            - check for duplicate velocity functions                  *
c            - remove include for mbs datestamp                        *
c  revised by: gary murphy v2.5               date: 92/08/20           *
c            - use interval velocity from top of layer instead of      *
c              bottom.                                                 *
c            - do flat-dip interpolation before the first velocity     *
c              function and after the last                             *
c  revised by: gary murphy v2.6               date: 92/09/08           *
c            - change maximum number of samples per trace from 2048 to *
c              8000                                                    *
c  revised by: gary murphy v2.7               date: 92/09/20           *
c            - allow total width to be off by one trace width          *
c            - I had to remove the include of ~usp/include/f77/
c            - HeaderSize.h because prg defined these things in
c            - ~usp/include/f77/lhdrsz.h   /Mary Ann Thornton
c  revised by: gary murphy v2.8               date: 92/10/01
c            - do flat-dip interpolation before the first velocity
c              function and after the last only if you have more than
c              three functions.
c  revised by: gary murphy v2.9               date: 92/11/04
c            - refined velocities so that flat layer models tie
c              with the new vlmx.
c  revised by: gary murphy v3.0               date: 92/11/09
c            - made code match man-page.
c            - made sure actual number of traces matches line header.
c  revised by: gary murphy v3.1               date: 93/02/24
c            - added option to interpolate under functions.
c  revised by: gary murphy v3.2               date: 93/03/93
c            - added option to interpolated up to a fault.
c            - added parameters to override flat-layer interplation
c              at the beginning and the end of the lines when there are
c              more than two velocity functions.
c         Also - added:
c                logical unit for HP (ler=7 on HP and =0 on others)
c                and changed ltrm = ler.  Line header size left at
c                8000 words                  -      Mary Ann Thornton
c  revised by: Mary Ann Thornton V3.3          date: 06/07/93
c              removed the #ifdef's and included the hp.h include file
c  revised by: Mary Ann Thornton V3.4          date: 06/10/93
c              Corrected command line argument                         
c  revised by: gary murphy v3.5               date: 09/20/93
c            - arranged command line arguments to match manual page
C***********************************************************************
#include <localsys.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <f77/hp.h>

      PARAMETER (LPRT=26, LCRD=25, LLIST=27)
      PARAMETER (MAXSMP=8000, MAXTRC=4000)
      PARAMETER (ITRC=1, ISMB=2, IDIP=3, IMDP=4,
     &           ifwt=5, ibwt=6, IFWD=7, ichp=8,
     &           IBCK=9)
      PARAMETER (NFMAX=75, NVMAX=1000)

      PARAMETER (MXWORD=MAXSMP+ITRWRD)
      PARAMETER (MAXBIN=MAXTRC*MAXSMP*IBCK)

      POINTER (PTR, BIN(1))
      POINTER (JPTR, JBIN(1))

      DIMENSION TRACE(MAXSMP)
      dimension ihead(mxword)
      integer*2 jhead(LNTRHD)
c     DIMENSION IZ(MAXSMP), ZZ(MAXSMP*4)
      DIMENSION X1(MAXSMP), X2(MAXSMP)
      INTEGER IHEADS(ITRWRD, MAXTRC)
      DIMENSION NV(NFMAX), IX(NFMAX)
      DIMENSION VIN(NVMAX,NFMAX), DEP(NVMAX,NFMAX)
      DIMENSION TIM(NVMAX)
      DIMENSION Z(MAXSMP)

      CHARACTER*128 NTAP,OTAP,INPUT,VCARD,ich
      CHARACTER*66 CTITLE
      CHARACTER*35 CHLH
      CHARACTER*80 CARD
      CHARACTER*4 VERSION
      CHARACTER*4 PPNAME

      LOGICAL VERBOS

      EQUIVALENCE (IHEAD(ITHWP1), TRACE(1))
      equivalence (jhead(1), ihead(1))
      DATA VERSION/' 3.5'/
      DATA PPNAME/'IVAD'/

      DATA CTITLE/'                   VELOCITY INTERPOLATION ALONG DIP  
     &             '/
      DATA ICC,  N /2*0/, LVCRD /28/
      ltrm = LER
C
C *** GET COMMAND LINE ARGUMENTS
C
      CALL rdcard(NTAP, OTAP, INPUT, IPIPI, IPIPO, LTRM, VERBOS, VCARD,
     &   MINDIP, MAXDIP, INCDIP, IBOX, IMED, THRESH, NSEMB, ITIME,
     &   ipnt, DZ, TDEPTH, DX, TWIDTH, NFUNC, iunder, ich, ixf, ixb)
C
C *** LUIN IS AN INPUT DATASET
C
      IF(IPIPI.EQ.0)THEN
         CALL LBOPEN(LUIN,NTAP,'r')
      ELSE
C
C *** WE KNOW LUIN IS A PIPE
C
         LUIN = 0
         LTRM = 2
      END IF
C
C *** LUOUT IS A OUTPUT DATASET
C
      IF(IPIPO.EQ.0)THEN
         CALL LBOPEN(LUOUT,OTAP,'w')
      ELSE
C
C *** LUOUT IS A PIPE
C
         LUOUT = 1
      ENDIF
c
c *** open ich dataset
c
      if (ich .ne. ' ') then
         call lbopen(luich,ich,'r')
      endif
C
C *** OPEN PRINT FILE
C
      CALL OPENPR(LLIST,LPRT,PPNAME,JERR)
      IF (JERR .NE. 0) THEN
         WRITE (LER, 10)
   10    FORMAT(/, 1X, '** M0010 ** ERROR DETECTED BY PROGRAM IVAD',/,
     &             2X, 'UNABLE TO OPEN PRINT FILE'               ,/,
     &            45X, 'REMOVE ANY FILES NAMED IVAD IN YOUR '    ,/,
     &            45X, 'WORKING DIRECTORY, OR, MAKE MORE ROOM '  ,/,
     &            45X, 'ON YOUR DISK.'                           ,/)
         ICC = 100
      END IF
C #include <mbsdate.h>
C
C *** MAKE TORCH AND OVAL
C
      NLIN = 1
      CALL GAMOCO (CTITLE, NLIN, LPRT)
C
C *** PRINT INPUT/OUTPUT DATASET INFO AFTER TORCH AND OVAL
C
      if (verbos)
     &WRITE (LPRT, 20) NTAP, OTAP, ich
   20 FORMAT(' INPUT DATASET  = ',/,A128,/,
     &       ' OUTPUT DATASET = ',/,A128,/,
     &       ' ICH DATASET    = ',/,A128,/)
C
C *** READ INPUT SIS LINE HEADER
C
      IEOF = 0
      CALL RTAPE (LUIN, IHEAD, IEOF)
      IF (IEOF .EQ. 0) THEN
         WRITE (LPRT, 30)
   30    FORMAT(/, 1X, '** M0030 ** ERROR DETECTED BY PROGRAM IVAD',/,
     &             2X, 'UNABLE TO OPEN SIS DATA SET.'            ,/,
     &            45X, 'MAKE SURE THAT YOU INPUT THE DATA '      ,/,
     &            45X, 'SET NAME CORRECTLY AND CHECK CASE.'      ,/)
         ICC = 100
      END IF
      CALL SAVER(IHEAD, 'NumTrc', NTRCS, LINHED)
      CALL SAVER(IHEAD, 'NumRec', NREC, LINHED)
      CALL SAVER(IHEAD, 'SmpInt', ISI, LINHED)
      CALL SAVER(IHEAD, 'NumSmp', NSAMPS, LINHED)
      CALL SAVER(IHEAD, 'Format', IFMT, LINHED)
C
C *** READ CARD DECK
C
      IF(INPUT.NE.' ')THEN
         OPEN(UNIT=LCRD,FILE=INPUT,STATUS='OLD',IOSTAT=JERR)
         IF(JERR.NE.0)THEN
            WRITE (LPRT, 33) 
   33       FORMAT(/,1X,'** M0033 ** ERROR DETECTED BY PROGRAM IVAD',/,
     &                2X, 'UNABLE TO OPEN PARAMETER CARD'           ,/)
            ICC = 100
         ELSE
            N = 1
         ENDIF
      ELSE
         N = ICOPEN('-ivad.crd',LCRD)
      ENDIF
      IF (N .NE. 0) THEN

         READ (LCRD, 40, END=8000) CARD
         if (verbos)
     &   WRITE (LPRT, 45) CARD
         IF (CARD(1:4) .NE. 'SPIN') THEN
            WRITE (LPRT, 35) CARD(1:4)
   35       FORMAT(/,1X,'** M0035 ** ERROR DETECTED BY PROGRAM IVAD',/,
     &                2X, 'EXPECTED TO READ A SPIN CARD'            ,/,
     &               45X, 'FOUND A ', A4, ' INSTEAD.'               ,/)
            ICC = 100
         END IF
         READ (LCRD, 40, END=8000) CARD
         if (verbos)
     &   WRITE (LPRT, 45) CARD
         READ (CARD, 50) MNDP, MXDP, INCDP, IBX, IMD, 
     &                   THRSH, NSMB, ITM, ipt 
   40    FORMAT (A80)
   45    FORMAT (5X, A80)
   50    FORMAT (4X, 3I8, 2I6, F8.0, I8, 2I5)
C
C *** NON-ZERO COMMAND LINE PARAMETERS OVERRIDE CARD PARAMETERS
C
         IF (MINDIP .EQ. 0) MINDIP = MNDP
         IF (MAXDIP .EQ. 0) MAXDIP = MXDP
         IF (INCDIP .EQ. 0) INCDIP = INCDP
         IF (IBOX   .EQ. 0) IBOX   = IBX
         IF (IMED   .EQ. 0) IMED   = IMD
         IF (THRESH .EQ.0.) THRESH = THRSH
         IF (NSEMB  .EQ. 0) NSEMB  = NSMB
         IF (ITIME  .EQ. 0) ITIME  = ITM
         if (ipnt   .eq. 0) ipnt   = ipt

         READ (LCRD, 40, END=8000) CARD
         if (verbos)
     &   WRITE (LPRT, 45) CARD
         IF (CARD(1:4) .NE. 'SIZE') THEN
            WRITE (LPRT, 55) CARD(1:4)
   55       FORMAT(/, 1X, '** M0055 ** ERROR DETECTED BY PROGRAM IVAD',
     &                2X, 'EXPECTED TO READ A SIZE CARD'            ,/,
     &               45X, 'FOUND A ', A4, ' INSTEAD.'               ,/)
            ICC = 100
         END IF
         READ (LCRD, 40, END=8000) CARD
         if (verbos)
     &   WRITE (LPRT, 45) CARD
         READ (CARD, 60) TDZ, TDPTH, TDX, TWDTH, NFNC
   60    FORMAT (4F10.0, I7)
C
C *** NON-ZERO COMMAND LINE PARAMETERS OVERRIDE CARD PARAMETERS
C
         IF (DZ     .EQ.0.) DZ     = TDZ
         IF (TDEPTH .EQ.0.) TDEPTH = TDPTH
         IF (DX     .EQ.0.) DX     = TDX
         IF (TWIDTH .EQ.0.) TWIDTH = TWDTH
         IF (NFUNC  .EQ. 0) NFUNC  = NFNC

      END IF
C
C *** CHECK MINIMUM DIP RATE
C
      IF ((MINDIP.LT.-200) .OR. (MINDIP.GT.100)) THEN
         WRITE (LPRT, 70)
   70    FORMAT(/, 1X, '** M0070 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'MINIMUM DIP RATE '                         ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC  5-12 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF
C
C *** CHECK MAXIMUM DIP RATE
C
      IF ((MAXDIP.LT.-100) .OR. (MAXDIP.GT.200)) THEN
         WRITE (LPRT, 80)
   80    FORMAT(/, 1X, '** M0080 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'MAXIMUM DIP RATE '                         ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC 13-20 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF
C
C *** CHECK STEP SIZE FOR COMPUTING SEMBLANCE
C
      IF  (INCDIP .EQ. 0)  INCDIP = 1
      IF ((INCDIP .LT. 1) .OR. (INCDIP .GT. 9)) THEN
         WRITE (LPRT, 90)
   90    FORMAT(/, 1X, '** M0090 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'STEP SIZE FOR COMPUTING SEMBLANCE '        ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC 21-28 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF
C
C *** CHECK LENGTH OF RUNNING WINDOW
C
      IF ((IBOX .LT. 0) .OR. (IBOX .GT. 500)) THEN
         WRITE (LPRT, 95)
   95    FORMAT(/, 1X, '** M0095 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'LENGTH OF RUNNING SUM WINDOW.'             ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC 29-34 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF
      if (ibox .eq. 0) then
         ibox = 1
         write (lprt, 96)
   96    format(/, 1x, '** m0096 ** warning from program ivad',
     &             2x, 'length of running sum window',
     &            45x, 'reset from 0 to 1.'
     &            45x, 'check cc 29-34 of the spin card.'        ,/)
      endif
C
C *** CHECK LENGTH OF SMOOTHING OPERATOR
C
      IF ((IMED .LT. 0) .OR. (IMED .GT. 500)) THEN
         WRITE (LPRT, 100)
  100    FORMAT(/, 1X, '** M0100 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'LENGTH OF SMOOTHING OPERATOR '             ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC 35-40 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF
      if (imed .eq. 0) then
         imed = 1
         write (lprt, 101)
  101    format(/, 1x, '** m0101 ** warning from program ivad',
     &             2x, 'length of smoothing operator ',
     &            45x, 'reset from 0 to 1.'
     &            45x, 'check cc 35-40 of the spin card.'        ,/)
      endif
C
C *** CHECK THRESHOLD
C
      IF ((THRESH .LT. .0001) .OR. (THRESH .GT. .99)
     &   .AND. (THRESH .NE. 0)) THEN
         WRITE (LPRT, 200)
  200    FORMAT(/, 1X, '** M0200 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'THRESHOLD FOR EDITING SEMBLANCE PICKS '    ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC 41-48 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF
C
C *** CHECK NUMBER OF TRACES FOR SEMBLANCE COMPUTATION
C
      IF ((NSEMB .LT. 4) .OR. (NSEMB .GT. 12)) THEN
         WRITE (LPRT, 210)
  210    FORMAT(/, 1X, '** M0210 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'NUMBER OF TRACES IN DIP SCAN '             ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC 49-55 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF
C
C *** CHECK TIME INPUT FLAG
C
      IF ((ITIME .LT. 0) .OR. (ITIME .GT. 1)) THEN
         WRITE (LPRT, 215)
  215    FORMAT(/, 1X, '** M0215 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'INPUT DATASET IN TIME FLAG '             ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC 56-60 OF THE SPIN CARD.'        ,/)
         ICC = 100
      END IF

C
C *** Check output data type
C

      if (ipnt  .le. itrc) ipnt   = itrc
      IF (ipnt .gt. ibck) then
         write (lprt, 216)
  216    format(/, 1x, '** m0216 ** error detected by program ivad',
     &             2x, 'output data type flag ',
     &                 'invalid',
     &            45x, 'check cc 61-65 of the spin card.',/)
         icc = 100
      endif

C
C *** CHECK DEPTH SPACING
C
      IF (DZ .LE. 0.) THEN
         WRITE (LPRT, 220)  
  220    FORMAT(/, 1X, '** M0220 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'DELTA-Z PARAMETER '                        ,
     &                 'INVALID'                                 ,/,
     &            45X, 'CHECK CC  5-10 OF THE SIZE CARD.'        ,/)
         ICC = 100
      END IF
C 
C *** CHECK TOTAL DEPTH
C 
      IF (TDEPTH .LE. DZ) THEN
         WRITE (LPRT, 230)   
  230    FORMAT(/, 1X, '** M0230 ** ERROR DETECTED BY PROGRAM IVAD', 
     &             2X, 'TOTAL DEPTH MUST BE GREATER THAN '         , 
     &                 'DEPTH INTERVAL'                          ,/, 
     &            45X, 'CHECK CC 11-20 OF THE SIZE CARD.'        ,/) 
         ICC = 100 
      END IF 
C 
C *** CHECK TRACE SPACING 
C 
      IF (DX .LE. 0.) THEN
         WRITE (LPRT, 240)   
  240    FORMAT(/, 1X, '** M0240 ** ERROR DETECTED BY PROGRAM IVAD', 
     &             2X, 'DELTA-X PARAMETER '                        , 
     &                 'INVALID'                                 ,/, 
     &            45X, 'CHECK CC 21-30 OF THE SIZE CARD.'        ,/) 
         ICC = 100 
      END IF 
C  
C *** CHECK TOTAL WIDTH
C  
      IF (TWIDTH .LE. DX) THEN 
         WRITE (LPRT, 250)    
  250    FORMAT(/, 1X, '** M0250 ** ERROR DETECTED BY PROGRAM IVAD',  
     &             2X, 'TOTAL WIDTH MUST BE GREATER THAN '         ,  
     &                 'TRACE INTERVAL'                          ,/, 
     &            45X, 'CHECK CC 31-40 OF THE SIZE CARD.'        ,/) 
         ICC = 100  
      END IF
C   
C *** CHECK TOTAL NUMBER OF SAMPLES AND TRACES
C
c
c *** to be consistent with other mbs programs, do not add one to the
c *** number of traces
c
      LTR = NINT(TWIDTH / DX)
      IF (LTR .GT. NTRCS) LTR = NTRCS
      IF (ITIME .EQ. 0) THEN
         NS = NINT(TDEPTH / DZ + 1.)
         NSO = NS
      ELSE
c
c *** to be consistent with other mbs programs, do not add one to the
c *** number of samples
c
         NSO = NINT(TDEPTH /DZ)
         NS = NSAMPS
      END IF
      IF (NS  .GT. NSAMPS) NS = NSAMPS
      IF (LTR*NS .GT. MAXTRC*MAXSMP) THEN
         WRITE (LPRT, 255)
  255    FORMAT(/, 1X, '** M0255 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'TOTAL WIDTH AND DEPTH EXCEEDS '            ,
     &                 'PROGRAM LIMITS.'                         ,/)
         ICC = 100
      END IF
      IF (LTR .LT. 3) THEN
         WRITE (LPRT, 257) 
  257    FORMAT(/, 1X, '** M0257 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'TOTAL NUMBER OF TRACES MUST EXCEED 2'   ,/)
         ICC = 100
      END IF
      IF (NS .LT. 3 .OR. NS .GT. MAXSMP) THEN
         WRITE (LPRT, 259) MAXSMP
  259    FORMAT(/, 1X, '** M0259 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'TOTAL NUMBER OF SAMPLES MUST EXCEED 2'   ,/,
     &            45X, 'AND BE LESS THAN', I9                    ,/)
         ICC = 100
      END IF
c
c *** Check iunder flag
c
      if ((iunder .lt. 0) .or. (iunder .gt. 1)) then
         write (lprt, 261) iunder
  261    format (/, 1x, '** m0261 ** error detected by program ivad',
     &              2x, 'the interpolation flag (iunder) ',
     &                  'is invalid '
     &             45x, 'check the command line argument (-u).',/)
         icc = 100
      endif

C   
C *** CHECK NUMBER OF FUNCTIONS 
C
      IF ((NFUNC .LE. 0) .OR. (NFUNC .GT. NFMAX)) THEN  
         WRITE (LPRT, 260)     
  260    FORMAT(/, 1X, '** M0260 ** ERROR DETECTED BY PROGRAM IVAD',   
     &             2X, 'THE NUMBER OF VELOCITY FUNCTIONS '         ,   
     &                 'IS INVALID '                             ,/,  
     &            45X, 'CHECK CC 41-47 OF THE SIZE CARD.'        ,/)  
         ICC = 100   
      END IF

c
c Check interpolation flags
c
      if (nfunc .ge. 3) then
         if (ixf .eq. -1) ixf = 1
         if (ixb .eq. -1) ixb = 1
      else
         if (ixf .eq. -1) ixf = 0
         if (ixb .eq. -1) ixb = 0
      endif
      if(ixf.ne.0)ixf=1
      if(ixb.ne.0)ixb=1
C     
C *** WRITE PARAMETERS TO USER REPORT
C     
      if (verbos)
     &WRITE (LPRT, 270)   MINDIP, MAXDIP, INCDIP, IBOX, IMED, THRESH,
     &                    NSEMB, ITIME, DZ, TDEPTH, DX, TWIDTH, iunder,
     &                    ixf, ixb, NFUNC
  270 FORMAT (//, 30X, 'INPUT PARAMETERS AFTER DEFAULTS:'         ,
     &        //, 23X, '  MINIMUM DIP RATE. . . . . . . .',4X,  I5,
     &        //, 23X, '  MAXIMUM DIP RATE. . . . . . . .',4X,  I5,
     &        //, 23X, '  STEP SIZE FOR SEMBLANCE . . . .',4X,  I5,
     &        //, 23X, '  LENGTH OF RUNNING SUM WINDOW. .',4X,  I5,
     &        //, 23X, '  LENGTH OF SMOOTHING WINDOW. . .',4X,  I5,
     &        //, 23X, '  THRESHOLD FOR EDITTING PICKS. .',2X,F7.2,
     &        //, 23X, '  NUMBER OF TRACES IN DIP SCAN. .',4X,  I5,
     &        //, 23X, '  DATASET TYPE (0=DEPTH, 1=TIME).',4X,  I5,
     &        //, 23X, '  DELTA-Z . . . . . . . . . . . .',2X,F7.2,
     &        //, 23X, '  TOTAL DEPTH . . . . . . . . . .',   F9.2,
     &        //, 23X, '  DELTA-X . . . . . . . . . . . .',2X,F7.2,
     &        //, 23X, '  TOTAL WIDTH . . . . . . . . . .',   F9.2,
     &        //, 23X, '  INTERPOLATE UNDER SHORT FUNCS .',4X   I5,
     &        //, 23X, '  FLAT AFTER LAST FUNCTION. . . .',4X   I5,
     &        //, 23X, '  FLAT BEFORE FIRST FUNCTION. . .',4X   I5,
     &        //, 23X, '  NUMBER OF VELOCITY FUNCTIONS. .',4X,  I5)
C
C *** OPEN VELOCITY DECK
C
      IF (VCARD .NE. ' ') THEN
         OPEN (UNIT=LVCRD, FILE=VCARD, STATUS='OLD', IOSTAT=JERR)
         IF (JERR .NE. 0) THEN
            WRITE (LPRT, 280) vcard
  280       FORMAT(/, 1X, '** M0280 ** ERROR DETECTED BY PROGRAM IVAD',
     &                2X, 'UNABLE TO OPEN VELOCITY CARD DECK.'      ,/,
     &               45X,  a,/,
     &               45X, 'MAKE SURE THAT YOU INPUT THE DATA '      ,/,
     &               45X, 'SET NAME CORRECTLY AND CHECK CASE.'      ,/)
            ICC = 100
         END IF
      ELSE
         LVCRD=LCRD
      END IF
C
C *** READ ALL VELOCITIES, EXPAND, AND PUT IN BIN
C
      VMIN = 100000.
      VMAX = 0.0
      ifunco = 1
      DO 370 IFUNC = 1, NFUNC
C
C *** MODEL CARD AND NUMBER OF VELOCITIES AND TRACE POSITION
C
         READ (LVCRD, 40, END=8000) CARD
         if (verbos)
     &   WRITE (LPRT, 45) CARD
         IF (CARD(1:5) .NE. 'MODEL') THEN
            WRITE (LPRT, 300) CARD(1:5)
  300       FORMAT(/, 1X, '** M0300 ** ERROR DETECTED BY PROGRAM IVAD',
     &                2X, 'EXPECTED TO READ A MODEL CARD'           ,/,
     &               45X, 'FOUND A ', A5, ' INSTEAD.'               ,/)
            ICC = 100
         ENDIF
         READ (CARD, 310) nvs, ixs
  310    FORMAT (10X, I10, 20X, I10)
c
c *** eliminate redundant entries
c
         if (ifunc .gt. 1) then
            if (ixs .eq. ix(ifunco)) then
               if (verbos)
     &            write (lprt, *)
     &                  'warning - duplicate entries at function ',
     &                   ix(ifunco)
            else
               ifunco = ifunco + 1
            endif
         endif
         ix(ifunco) = ixs
         nv(ifunco) = nvs
c
c *** accomodate the amoco way of miscounting samples
c
         if (IX(ifunco) .eq. ltr+1) then
            ltr = IX(ifunco)
            twidth = dx * ltr
            write (lprt, 315) twidth, ltr
  315       format (/, 1x, '** m0315 ** warning from program swath',
     &                 2x, 'total width parameter reset to ',f9.2,
     &                45x, 'number of output traces reset to ', i9,/)
         endif
         IF (IX(ifunco) .GT. LTR) THEN
            WRITE (LPRT, 320) 
  320       FORMAT(/, 1X, '** M0320 ** ERROR DETECTED BY PROGRAM IVAD',
     &                2X, 'TRACE NUMBER INPUT ON MODEL CARD'        ,/,
     &               45X, 'EXCEEDS TOTAL WIDTH.'                    ,/)
            ICC = 100
         END IF
C
C *** READ IN VELOCITY FUNCTION
C
         READ (LVCRD, 40, END=8000) CARD
         if (verbos)
     &   WRITE (LPRT, 45) CARD
         DO 360 IVEL = 1, NV(ifunco)
            READ (LVCRD, 40, END=8000) CARD
            if (verbos)
     &      WRITE (LPRT, 45) CARD
            READ (CARD, 330) INVEL, VIN(IVEL,ifunco), DEP(IVEL,ifunco)
  330       FORMAT (I10, F10.0, 10X, F10.0)
            IF (INVEL .NE. IVEL) THEN
               WRITE (LPRT, 340)
  340          FORMAT
     &            (/,1X,'** M0340 ** ERROR DETECTED BY PROGRAM IVAD',
     &               2X,'THE VELOCITY NUMBER MUST START AT '      ,/,
     &              45X,'1 AND INCREMENT BY 1.'                   ,/)
               ICC = 100
            ENDIF
            IF (IVEL .GT. 1) THEN
               IF (DEP(IVEL,ifunco) .LE. DEP(IVEL-1,ifunco))THEN
                  WRITE (LPRT, 350) 
  350             FORMAT
     &            (/,1X, '** M0350 ** ERROR DETECTED BY PROGRAM IVAD',
     &               2X, 'DEPTHS MUST INCREASE.'                   ,/)
                  ICC = 100
               ENDIF
            END IF
            IF(VIN(IVEL,ifunco).GT.VMAX)VMAX=VIN(IVEL,ifunco)
            IF(VIN(IVEL,ifunco).LT.VMIN)VMIN=VIN(IVEL,ifunco)
  360    CONTINUE
  370 CONTINUE
      if (ifunco .ne. nfunc) then
         nfunc = ifunco
         if (verbos)
     &      write (lprt, *) 'number of functions reset to ', ifunco
      endif
C
C *** CONVERT DEPTH-INTERVAL VELOCITIES TO TIME-INTERVAL VELOCITIES
C
      IF (ITIME .EQ. 1) THEN
         SR_SEC = FLOAT(ISI) * .001
         DO 410 IFUNC = 1, NFUNC
            T = DEP (1,IFUNC) / VIN(1,IFUNC)
            TIM (1) = T * 2. * DZ / SR_SEC
            DO 400 IVEL = 2, NV(IFUNC)
               T = T + ((DEP(IVEL,IFUNC) - DEP(IVEL-1,IFUNC))
     &           / VIN (IVEL,IFUNC))
               TIM (IVEL) = T * 2. * DZ / SR_SEC
  400       CONTINUE
            DO 405 IVEL = 1, NV(IFUNC)
               DEP (IVEL,IFUNC) = TIM (IVEL)
  405       CONTINUE
  410    CONTINUE
      END IF
C
C *** COME HERE ON CARD READ ERROR                        
C  
      GOTO 1000
 8000 CONTINUE
         WRITE (LPRT, 8010)
 8010    FORMAT(/, 1X, '** M8010 ** ERROR DETECTED BY PROGRAM IVAD',
     &             2X, 'END OF CARD DECK DETECTED.'              ,/)
         ICC = 100
 1000 CONTINUE
C
C *** UPDATE PROCESSING HISTORY AND WRITE OUT LINE HEADER
C  
      IF (ICC .NE. 0) GOTO 9000
      WRITE (CHLH, 500) 'IVAD (INTERPOLATE V(X,Z) ALONG DIP)'
  500 FORMAT (A35)
      IF (ITIME .EQ. 1) THEN
         CALL SAVEW(IHEAD, 'NumSmp', NSO, LINHED)
      END IF
      IDX = DX*1000.
      IDZ = DZ*1000.
      IVMIN = VMIN
      IVMAX = VMAX
      CALL SAVEW(IHEAD, 'Dx1000', IDX, LINHED)
      CALL SAVEW(IHEAD, 'Dz1000', IDZ, LINHED)
      CALL SAVEW(IHEAD, 'MinVel', IVMIN, LINHED)
      CALL SAVEW(IHEAD, 'MaxVel', IVMAX, LINHED)
      CALL SAVER(IHEAD, 'NumTrc', LTR, LINHED)
      CALL HLHPRT (IHEAD, IEOF, CHLH, LEN(CHLH), LPRT)
      CALL WRTAPE (LUOUT, IHEAD, IEOF )
C
C *** CLEAR BIG ARRAY
C
      MAXDYN = MAX (NS, NSO) * LTR * IBCK
      call galloc (ptr, maxdyn*ISZBYT, ierr, 'abort')
      JPTR = PTR
      CALL VCLR (BIN, 1, MAXDYN)

C
C *** PUT VELOCITIES IN BIG ARRAY
C
      DO 600 IFUNC = 1, NFUNC
          IBIN1 = (NS*LTR)*(IFWD-1) + NS*(IX(IFUNC)-1)
          IBIN2 = (NS*LTR)*(IBCK-1) + NS*(IX(IFUNC)-1)
          ibin3 = (ns*ltr)*(ifwt-1) + ns*(ix(ifunc)-1)
          ibin4 = (ns*ltr)*(ibwt-1) + ns*(ix(ifunc)-1)
          DO 575 IS = 1, int (DEP(1,IFUNC)/DZ+1.5)
             BIN(IBIN1+IS) = VIN (1,IFUNC)
             BIN(IBIN2+IS) = VIN (1,IFUNC)
             BIN(IBIN3+IS) = ix(ifunc)
             BIN(IBIN4+IS) = ix(ifunc)
  575     CONTINUE
          DO 590 IVEL = 2, NV(IFUNC)
             DO 580 IS = int(DEP(IVEL-1,IFUNC)/DZ+2.5),
     &                   int(DEP(IVEL,IFUNC)/DZ+1.5)
                BIN(IBIN1+IS) = VIN (IVEL,IFUNC)
                BIN(IBIN2+IS) = VIN (IVEL,IFUNC)
                BIN(IBIN3+IS) = ix(ifunc)
                BIN(IBIN4+IS) = ix(ifunc)
  580        CONTINUE
  590     CONTINUE
          DO 595 IS = int(DEP(NV(IFUNC),IFUNC)/DZ+2.5), NS
             BIN(IBIN1+IS) = VIN (NV(IFUNC),IFUNC)
             BIN(IBIN2+IS) = VIN (NV(IFUNC),IFUNC)
             if (iunder .eq. 0) then
                BIN(IBIN3+IS) = ix(ifunc)
                BIN(IBIN4+IS) = ix(ifunc)
             else
                BIN(IBIN3+IS) = 0.
                BIN(IBIN4+IS) = 0.
             endif
  595     CONTINUE
  600 CONTINUE
C
C *** READ DATA INTO BIG ARRAY
C
      if (ich .ne. ' ') then
         call rtape (luich, ihead, jeof)
         if (jeof .eq. 0) then
            write (lprt, 603)
  603       format(/, 1x, '** m0707 ** error from program ivad'
     &                2x, 'end of ich tape encountered' ,/,
     &               45x, 'job aborted',/)
            goto 9000
         endif
      endif
      JSAVE = 0
      DO 700 IT = 1, LTR
         IBIN1 = (NS*LTR)*(ITRC-1) + NS*(IT-1)
         IBIN2 = (NS*LTR)*(ichp-1) + NS*(IT-1)
         JEOF = 0
         CALL RTAPE (LUIN, IHEAD, JEOF)
         IF (jEOF .EQ. 0) THEN
            WRITE (LPRT, 605) IT
  605       FORMAT(/, 1X, '** M0605 ** WARNING FROM PROGRAM IVAD',
     &                2X, 'END OF TAPE ENCOUNTERED PREMATURELY'  ,/,
     &               45X, 'MAXIMUM TRACE NUMBER RESET TO', I9    ,/)
            GOTO 710
         END IF
         CALL SAVER(IHEAD, 'StaCor', ISTAT, TRCHED)
         IF (ISTAT .EQ. 30000) GO TO 700
         CALL VMOV (IHEAD(1), 1, IHEADS(1, IT), 1, ITRWRD)
         CALL VMOV (TRACE(1), 1, BIN(IBIN1+1),  1, NS)
         if (ich .ne. ' ') then
            call rtape (luich, ihead, jeof)
            if (jeof .eq. 0) then
               write (lprt, 607) it
  607          format(/, 1x, '** m0707 ** warning from program ivad'
     &                   2x, 'end of ich tape encountered' ,/,
     &                  45x, 'maximum trace number reset to ', i9 ,/)
               goto 710
            endif
            call vmov (trace(1), 1, bin(ibin2+1), 1, ns)
         endif
         if (verbos)
     &   WRITE (LPRT, 610) IT
 610     FORMAT(5X,'INPUT TRACE ',I9,' PROCESSED ')
  700 CONTINUE
      GOTO 720
  710 CONTINUE
      LTR = IT
  720 CONTINUE
C
C *** SET SOME COMMON VARIABLES
C
      CALL OPSPIN (MINDIP, MAXDIP, THRESH, IBOX, IMED, INCDIP, LPRT)
C
C *** COMPUTE SEMBLANCES, DIPS, AND SMOOTH DIPS
C
      LX = 3
      DO 790 IMIDT = 2, NSEMB/2
         IT = IMIDT - LX/2
         IBIN1 = (NS*LTR)*(ITRC-1) + NS*(IT-1) + 1
         IBIN2 = (NS*LTR)*(ISMB-1) + NS*(IT-1) + 1
         IBIN3 = (NS*LTR)*(IDIP-1) + NS*(IT-1) + 1
         IBIN4 = (NS*LTR)*(IMDP-1) + NS*(IT-1) + 1
         CALL SEMBXT (NS, LX, BIN(IBIN1), BIN(IBIN2), JBIN(IBIN3))
         CALL MEDFIL (NS, JBIN(IBIN3), JBIN(IBIN4))
         LX = LX + 2
  790 CONTINUE
      IBIN2 = (NS*LTR)*(ISMB-1) + 1
      IBIN3 = (NS*LTR)*(IDIP-1) + 1
      IBIN4 = (NS*LTR)*(IMDP-1) + 1
      CALL VMOV (BIN(IBIN2+NS), 1, BIN(IBIN2), 1, NS)
      CALL VMOV (JBIN(IBIN3+NS), 1, JBIN(IBIN3), 1, NS)
      CALL VMOV (JBIN(IBIN4+NS), 1, JBIN(IBIN4), 1, NS)
      DO 800 IMIDT = NSEMB/2+1, LTR-NSEMB/2
         IT = IMIDT - NSEMB/2
         IBIN1 = (NS*LTR)*(ITRC-1) + NS*(IT-1) + 1
         IBIN2 = (NS*LTR)*(ISMB-1) + NS*(IT-1) + 1
         IBIN3 = (NS*LTR)*(IDIP-1) + NS*(IT-1) + 1
         IBIN4 = (NS*LTR)*(IMDP-1) + NS*(IT-1) + 1
         CALL SEMBXT (NS, NSEMB, BIN(IBIN1), BIN(IBIN2), JBIN(IBIN3))
         CALL MEDFIL (NS, JBIN(IBIN3), JBIN(IBIN4))
  800 CONTINUE
      LX = NSEMB-2
      DO 810 IMIDT = LTR-NSEMB/2+1, LTR-1
         IT = IMIDT - LX/2
         IBIN1 = (NS*LTR)*(ITRC-1) + NS*(IT-1) + 1
         IBIN2 = (NS*LTR)*(ISMB-1) + NS*(IT-1) + 1
         IBIN3 = (NS*LTR)*(IDIP-1) + NS*(IT-1) + 1
         IBIN4 = (NS*LTR)*(IMDP-1) + NS*(IT-1) + 1
         CALL SEMBXT (NS, LX, BIN(IBIN1), BIN(IBIN2), JBIN(IBIN3))
         CALL MEDFIL (NS, JBIN(IBIN3), JBIN(IBIN4))
         LX = LX - 2
  810 CONTINUE
      IBIN2 = (NS*LTR)*(ISMB-1) + NS*(LTR-2) + 1
      IBIN3 = (NS*LTR)*(IDIP-1) + NS*(LTR-2) + 1
      IBIN4 = (NS*LTR)*(IMDP-1) + NS*(LTR-2) + 1
      CALL VMOV (BIN(IBIN2), 1, BIN(IBIN2+NS), 1, NS)
      CALL VMOV (JBIN(IBIN3), 1, JBIN(IBIN3+NS), 1, NS)
      CALL VMOV (JBIN(IBIN4), 1, JBIN(IBIN4+NS), 1, NS)
 
c do forward velocities
      RNS = NS - 2
      DO IFUNC = 1, NFUNC - 1
         DO 917 IT = IX(IFUNC)+1, IX(IFUNC+1)-1
            IBIN1 = (NS*LTR)*(IFWD-1) + NS*(IT-2) + 1
            IBIN2 = (NS*LTR)*(IMDP-1) + NS*(IT-1)
            IBIN3 = (NS*LTR)*(IFWD-1) + NS*(IT-1) + 1
            ibin4 = (ns*ltr)*(ifwt-1) + ns*(it-2) + 1
            ibin5 = (ns*ltr)*(ifwt-1) + ns*(it-1) + 1
            IBIN6 = (NS*LTR)*(Itrc-1) + NS*(IT-1) + 1
            ibin7 = (NS*LTR)*(ichp-1) + NS*(IT-2) + 1
            DO 916 IS = 1, NS
               RS = IS - 1
               Z(IS) = MAX(MIN(RS-FLOAT(JBIN(IBIN2+IS)),RNS),0.)
  916       CONTINUE
            DO 9166 IS = 1, NS-1
               IF (Z(IS) .GT. Z(IS+1)) THEN
                  Z(IS+1) = Z(IS)
               END IF
 9166       CONTINUE
            CALL VLINT (BIN(IBIN1),NS,Z(1),1,BIN(IBIN3),1,NS)
            CALL VLINT (BIN(IBIN4),NS,Z(1),1,BIN(IBIN5),1,NS)
            do is = 1, ns
               if (bin(ibin7+is-1) .ne. 0.0) then
                  bin(ibin5+is-1) = 0.0
               endif
            enddo
  917    CONTINUE
         it=ix(ifunc+1)
         wt=it
         ibin1 = (ns*ltr)*(ifwd-1) + ns*(it-2)
         ibin2 = (ns*ltr)*(ifwd-1) + ns*(it-1)
         ibin3 = (ns*ltr)*(ifwt-1) + ns*(it-2)
         ibin4 = (ns*ltr)*(ifwt-1) + ns*(it-1)
         ibin5 = (NS*LTR)*(ichp-1) + NS*(IT-2)
         ibin6 = (NS*LTR)*(ibck-1) + NS*(IT-1)
         ibin7 = (NS*LTR)*(ibwt-1) + NS*(IT-1)
         do is = 1, ns
            if(bin(ibin4+is).eq.0.)then
               bin(ibin2+is)=bin(ibin1+is)
               bin(ibin4+is)=bin(ibin3+is)
            endif
            if (bin(ibin5+is) .ne. 0.0) then
               bin(ibin4+is) = 0.0
            endif
            if ((ifunc+1) .eq. nfunc) then
               if (bin(ibin7+is) .eq. 0.0) then
                  bin(ibin6+is) = bin(ibin2+is)
                  bin(ibin7+is) = wt
               endif
            endif
         enddo
      enddo

c do backward velocities
      do ifunc = nfunc-1, 1, -1
         DO 919 IT = IX(IFUNC+1)-1, IX(IFUNC)+1, -1
            IBIN1 = (NS*LTR)*(IBCK-1) + NS*(IT)   + 1
            IBIN2 = (NS*LTR)*(IMDP-1) + NS*(IT-1)
            IBIN3 = (NS*LTR)*(IBCK-1) + NS*(IT-1) + 1
            ibin4 = (ns*ltr)*(ibwt-1) + ns*(it)   + 1
            ibin5 = (ns*ltr)*(ibwt-1) + ns*(it-1) + 1
            ibin6 = (ns*ltr)*(itrc-1) + ns*(it-1) + 1
            ibin7 = (NS*LTR)*(ichp-1) + NS*(IT)   + 1
            DO 918 IS = 1, NS
               RS = IS - 1  
               Z(IS) = MAX(MIN(RS+FLOAT(JBIN(IBIN2+IS)),RNS),0.)
  918       CONTINUE
            DO 9188 IS = 1, NS-1
               IF (Z(IS) .GT. Z(IS+1)) THEN
                  Z(IS+1) = Z(IS)
               END IF
 9188       CONTINUE
            CALL VLINT (BIN(IBIN1), NS, Z(1), 1, BIN(IBIN3), 1, NS)
            CALL VLINT (BIN(IBIN4), NS, Z(1), 1, BIN(IBIN5), 1, NS)
            do is = 1, ns
               if (bin(ibin7+is-1) .ne. 0.0) then
                  bin(ibin5+is-1) = 0.0
               endif
            enddo
  919    CONTINUE
         it=ix(ifunc)
         wt=it
         ibin1 = (ns*ltr)*(ibck-1) + ns*(it)
         ibin2 = (ns*ltr)*(ibck-1) + ns*(it-1)
         ibin3 = (ns*ltr)*(ibwt-1) + ns*(it)
         ibin4 = (ns*ltr)*(ibwt-1) + ns*(it-1)
         ibin5 = (NS*LTR)*(ichp-1) + NS*(IT)  
         ibin6 = (ns*ltr)*(ifwd-1) + ns*(it)
         ibin7 = (ns*ltr)*(ifwt-1) + ns*(it)
         do is = 1, ns
            if(bin(ibin4+is).eq.0.)then
               bin(ibin2+is)=bin(ibin1+is)
               bin(ibin4+is)=bin(ibin3+is)
            endif
            if (bin(ibin5+is) .ne. 0.0) then
               bin(ibin4+is) = 0.0
            endif
            if (ifunc .eq. 1) then
               if (bin(ibin7+is) .eq. 0.0) then
                  bin(ibin6+is) = bin(ibin2+is)
                  bin(ibin7+is) = wt
               endif
            endif
         enddo
      enddo

C
C *** USE BACKWARD VELOCITIES FOR BEGINNING OF LINE
C
      DO 915 IT = IX(1)-1, 1, -1
         IBIN1 = (NS*LTR)*(IBCK-1) + NS*(IT)   + 1
         IBIN2 = (NS*LTR)*(IMDP-1) + NS*(IT-1)
         IBIN3 = (NS*LTR)*(IBCK-1) + NS*(IT-1) + 1
         IBIN4 = (NS*LTR)*(ITRC-1) + NS*(IT-1) + 1
         IBIN5 = (ns*ltr)*(ibwt-1) + ns*(it)   + 1
         IBIN6 = (ns*ltr)*(ibwt-1) + ns*(it-1) + 1
         DO 905 IS = 1, NS
            RS = IS - 1
            if (ixb .eq. 0) then
               Z(IS) = MAX(MIN(RS+FLOAT(JBIN(IBIN2+IS)),RNS),0.)
            else
               Z(IS) = MAX(MIN(RS,RNS),0.)
            endif
  905    CONTINUE
         DO 907 IS = 1, NS-1
            IF (Z(IS) .GT. Z(IS+1)) THEN
              Z(IS+1) = Z(IS)
            END IF
  907    CONTINUE
         CALL VLINT (BIN(IBIN1), NS, Z(1), 1, BIN(IBIN3), 1, NS)
         CALL VMOV (BIN(IBIN3),1,BIN(IBIN4),1,NS)
         call VLINT (bin(ibin5), ns, z(1), 1, bin(ibin6), 1, NS)
  915 CONTINUE
      IBIN1 = (NS*LTR)*(IBCK-1) + NS*(IX(1)-1) + 1
      IBIN3 = (NS*LTR)*(ITRC-1) + NS*(IX(1)-1) + 1
      CALL VMOV (BIN(IBIN1),1,BIN(IBIN3),1,NS)

C
C *** USE FORWARD VELOCITIES FOR END OF LINE
C
      DO 935 IT = IX(NFUNC)+1, LTR
         IBIN1 = (NS*LTR)*(IFWD-1) + NS*(IT-2) + 1
         IBIN2 = (NS*LTR)*(IMDP-1) + NS*(IT-1)
         IBIN4 = (NS*LTR)*(IFWD-1) + NS*(IT-1) + 1
         IBIN3 = (NS*LTR)*(ITRC-1) + NS*(IT-1) + 1
         ibin5 = (ns*ltr)*(ifwt-1) + ns*(it-2) + 1
         ibin6 = (ns*ltr)*(ifwt-1) + ns*(it-1) + 1
         DO 833 IS = 1, NS
            RS = IS - 1
            if (ixf .eq. 0) then
               Z(IS) = MAX(MIN(RS-FLOAT(JBIN(IBIN2+IS)),RNS),0.)
            else
               Z(IS) = MAX(MIN(RS,RNS),0.)
            endif
  833    CONTINUE
         DO 834 IS = 1, NS-1
            IF (Z(IS) .GT. Z(IS+1)) THEN
              Z(IS+1) = Z(IS)
            END IF
  834    CONTINUE
         CALL VLINT (BIN(IBIN1), NS, Z(1), 1, BIN(IBIN3), 1, NS)
         CALL VMOV (BIN(IBIN3),1,BIN(IBIN4),1,NS)
         CALL VLINT (BIN(IBIN5), NS, Z(1), 1, BIN(IBIN6), 1, NS)
  935 CONTINUE

c merge forward and backward velocities
      do ifunc = 1, nfunc - 1
       CMAX = IX(IFUNC+1) - IX(IFUNC)
       cmax1 = cmax
       DO 920 IT = IX(IFUNC), IX(IFUNC+1) - 1
        IBIN1 = (NS*LTR)*(IFWD-1) + NS*(IT-1) + 1
        IBIN2 = (NS*LTR)*(IBCK-1) + NS*(IT-1) + 1
        IBIN3 = (NS*LTR)*(ITRC-1) + NS*(IT-1) + 1
        ibin4 = (ns*ltr)*(ifwt-1) + ns*(it-1) + 1
        ibin5 = (ns*ltr)*(ibwt-1) + ns*(it-1) + 1
        ibin6 = (ns*ltr)*(ifwd-1) + ns*(it-2) + 1
        C = (IT - IX(IFUNC)) / CMAX
c       CALL VINTB (BIN(IBIN1),1,BIN(IBIN2),1,C,BIN(IBIN3),1,NS)
c       vintb does d(i) = a(i) + c*(b(i)-a(i)) for i=1,ns
c       we are now weighting the velocities
        do is = 1, ns
         cmax = bin(ibin5+is-1)-bin(ibin4+is-1)
         if (cmax .eq. 0) cmax = cmax1
         c = (it - bin(ibin4+is-1)) / cmax
         if((bin(ibin4+is-1).eq.0.).and.(bin(ibin5+is-1).eq.0.))then
          bin(ibin3+is-1)=bin(ibin6+is-1)
          if(bin(ibin3+is-1).eq.0.)bin(ibin3+is-1)=vmin
         elseif((bin(ibin4+is-1).eq.0.).and.(bin(ibin5+is-1).ne.0.))then
          bin(ibin3+is-1)=bin(ibin2+is-1)
         elseif((bin(ibin4+is-1).ne.0.).and.(bin(ibin5+is-1).eq.0.))then
          bin(ibin3+is-1)=bin(ibin1+is-1)
         else
          bin(ibin3+is-1)=bin(ibin1+is-1)+c*(bin(ibin2+is-1)
     :                  -bin(ibin1+is-1))
         endif
        enddo
  920  CONTINUE
      enddo
      IF (NFUNC .GT. 1) THEN
         IBIN2 = (NS*LTR)*(IFWD-1) + NS*(IX(NFUNC)-1) + 1
         IBIN3 = (NS*LTR)*(ITRC-1) + NS*(IX(NFUNC)-1) + 1
         CALL VMOV (BIN(IBIN2),1,BIN(IBIN3),1,NS)
      END IF
C
C *** OUTPUT VELOCITY TAPE
C 
      NBYTES = (NSO+ ITRWRD) * SZSMPD
      INIT = 1
      X2(1) = 0.
      DO 936 I = 2, NSO
         X2(I) = (I-1) * DZ
  936 CONTINUE
C
C *** If the user would like to see the dips or the semblances...
C *** ipnt will indicate the appropriate data type.
C
      DO 950 IT = 1, LTR
         IBIN1 = (NS*LTR)*(ipnt-1) + NS*(IT-1)
         CALL VMOV (IHEADS(1,IT),  1, IHEAD(1), 1, ITRWRD)
         if ((ipnt .eq. 3) .or. (ipnt .eq. 4)) then
            do 9377 i = 1, ns
               bin(ibin1+i) = jbin(ibin1+i)
 9377       continue
         endif
         IF (ITIME .EQ. 1) THEN
            X1(1) = 0
            DO 937 I = 2, NS
               X1(I) = X1(I-1) + .5*SR_SEC*BIN(IBIN1+I)
  937       CONTINUE
c           CALL FLIINT (X1(1), BIN(IBIN1+1), NS, X2(1),
c    &                   TRACE(1), NSO, IZ(1), ZZ(1), INIT)
c
c *** the following do loop replaces the linear interpolation of
c *** the velocity tape from time to depth by using the nearest 
c *** sample.  this prevents numbers from appearing that were 
c *** never there
c
            maxj = 0
            jold = 0
            do i = 1, ns
               j=nint(x1(i)/dz)+1
               if (j .gt. maxj) maxj = j
               if (j .gt. nso) j = nso
               if (j .lt. 1)   j = 1
               do k = jold+1, j
                  trace(k) = bin(ibin1+i)
               enddo
               jold = j
            enddo
            do i = maxj+1, nso
               trace(i) = bin(ibin1+ns)
            enddo
         ELSE
            CALL VMOV (BIN(IBIN1+1), 1, TRACE(1), 1, NS)
         END IF
         IEOF = NBYTES
         jhead (106) = 1
         jhead (107) = it
         CALL WRTAPE(LUOUT, IHEAD, IEOF)
         if (verbos)
     &   WRITE (LPRT, 940) IT
  940    FORMAT(5X,'OUTPUT TRACE ',I9,' PROCESSED ')
  950 CONTINUE
 9000 CONTINUE
      WRITE (LPRT, 9045)
 9045 FORMAT (1X, '  END OF JOB ')
      CALL LBCLOS (LUIN)
      CALL LBCLOS (LUOUT)
      STOP
      END
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       rdcard                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  ???                                                       *
C  ENTRY POINTS:                                                       *
c     rdcard(NTAP, OTAP, INPUT, IPIPI, IPIPO, LTRM, VERBOS, VCARD,
c    &   MINDIP, MAXDIP, INCDIP, IBOX, IMED, THRESH, NSEMB, ITIME,
c    &   ipnt, DZ, TDEPTH, DX, TWIDTH, NFUNC, iunder, ich, ixf, ixb)
C  ARGUMENTS:                                                          * 
C                                                                      *
C      OTAP    CHAR*(NIHEAD)  ??IOU* -                                 *
C      INPUT   CHAR*(NIHEAD)  ??IOU* -                                 *
C      IPIPI   INTEGER   ??IOU* -                                      *
C      IPIPO   INTEGER   ??IOU* -                                      *
C      LTRM    INTEGER   ??IOU* -                                      *
C      VERBOS  LOGICAL   ??IOU* -                                      *
C      VCARD   CHAR*128  ??IOU* -                                      *
C      MINDIP   INTEGER   ??IOU* -                                      *
C      maxdip   INTEGER   ??IOU* -                                      *
C      INCDIP  INTEGER   ??IOU* -                                      *
C      IBOX    INTEGER   ??IOU* -                                      *
C      IMED    INTEGER   ??IOU* -                                      *
C      THRESH  REAL      ??IOU* -                                      *
C      NSEMB   INTEGER   ??IOU* -                                      *
C      DZ      REAL      ??IOU* -                                      *
C      TDEPTH  REAL      ??IOU* -                                      *
C      DX      REAL      ??IOU* -                                      *
C      TWIDTH  REAL      ??IOU* -                                      *
C      NFUNC   INTEGER   ??IOU* -                                      *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 88/05/05  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 88/12/12  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      ARGIS   INTEGER -                                               *
C      ARGSTR          -                                               *
C      ARGR4           -                                               *
C      ARGI4           -                                               *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:                                                              *
C      LTRM  ( OUTPUT SEQUENTIAL ) -                                   *
C  COMMON:           NONE                                              *
C  STOP CODES:                                                         *
C      =BLANK=  ( 1) -                                                 *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      subroutine rdcard
     &  (NTAP, OTAP, INPUT, IPIPI, IPIPO, LTRM, VERBOS, VCARD,
     &   MINDIP, MAXDIP, INCDIP, IBOX, IMED, THRESH, NSEMB, ITIME,
     &   ipnt, DZ, TDEPTH, DX, TWIDTH, NFUNC, iunder, ich, ixf, ixb)
      INTEGER ARGIS
      LOGICAL HELP,VERBOS
      CHARACTER*(*) NTAP,OTAP,INPUT,VCARD,ich
C
C *** SET DEFAULTS TO NO PIPES
C
      IPIPI=0
      IPIPO=0
      HELP  = (ARGIS( '-h' ).GT.0) .OR. (ARGIS( '-?').GT.0)
      IF(HELP)THEN
         WRITE(LTRM,*)'COMMAND LINE ARGUMENTS--INTERPOLATION ALONG DIP'
         WRITE(LTRM,*)' '
         WRITE(LTRM,*)' INPUT '
         WRITE(LTRM,*)'-N[input]      . INPUT DATASET NAME'
         WRITE(LTRM,*)'-O[output]     . OUTPUT DATASET NAME'
         WRITE(LTRM,*)'-C[ivad.card]  . INPUT CARDS'
         WRITE(LTRM,*)'-I[vfunctions] . INPUT VELOCITY CARDS'
         WRITE(LTRM,*)'-P[ich]        . INPUT ICH DATASET'
         WRITE(LTRM,*)'-V             . VERBOSE PRINTOUT'
         WRITE(LTRM,*)'-ND[mindip]    . MINIMUM DIP'
         WRITE(LTRM,*)'-XD[maxdip]    . MAXIMUM DIP'
         WRITE(LTRM,*)'-ID[incdip]    . DIP CALCULATION INCREMENT'
         WRITE(LTRM,*)'-B[ibox]       . BOX CAR LENGTH'
         WRITE(LTRM,*)'-M[imed]       . MEDIAN FILTER LENGHT'
         WRITE(LTRM,*)'-T[thresh]     . SEMBLANCE THRESHOLD'
         WRITE(LTRM,*)'-S[nsemb]      . NUMBER OF TRACES IN SCAN'
         WRITE(LTRM,*)'-TM[itime]     . INPUT DATASET IS IN TIME'
         WRITE(LTRM,*)'-D[ipnt]       . 1=vel, 2=semb, 3=dip, 4=med' 
         WRITE(LTRM,*)'-DZ[dz]        . DELTA-Z'
         WRITE(LTRM,*)'-Z[depth]      . TOTAL DEPTH'
         WRITE(LTRM,*)'-DX[dx]        . DELTA-X'
         WRITE(LTRM,*)'-X[width]      . TOTAL WIDTH'
         WRITE(LTRM,*)'               . 7=forward, 9=backward, 8=picks'
         WRITE(LTRM,*)'-NF[functions] . NO. VELOCITY FUNCTIONS TO READ'
         WRITE(LTRM,*)'-u[iunder]     . 1=interplate under, 0=between'
         WRITE(LTRM,*)'-ixf[ixf]      . 1=flat after last function'
         WRITE(LTRM,*)'-ixb[ixb]      . 1=flat before first function'
         WRITE(LTRM,*)'USAGE:'
         WRITE(LTRM,*)'ivad -N[] -O[] -C[] -I[]'
         WRITE(LTRM,*)'          OR'
         WRITE(LTRM,*)'ivad -N[] -O[] -I[] -P[] -V '
         WRITE(LTRM,*)' -ND[] -XD[] -ID[] -B[] -M[] -T[] -S[] '
         WRITE(LTRM,*)' -TM[] -D[] -DZ[] -Z[] -DX[] -X[] -NF[] '
         WRITE(LTRM,*)' -u[] -ixf[] -ixb[] -V'
         STOP
      ENDIF

      CALL ARGSTR('-N',NTAP,' ',' ')
      CALL ARGSTR('-O',OTAP,' ',' ')
      CALL ARGSTR('-C',INPUT,' ',' ')
      CALL ARGSTR('-I',VCARD,' ',' ')
      CALL ARGSTR('-P',ich,' ',' ')
      VERBOS =   (ARGIS( '-V' ).GT.0)
      CALL ARGI4 ('-ND',MINDIP,0,0)
      CALL ARGI4 ('-XD',MAXDIP,0,0)
      CALL ARGI4 ('-ID',INCDIP,0,0)
      CALL ARGI4 ('-B',IBOX,0,0)
      CALL ARGI4 ('-M',IMED,0,0)
      CALL ARGR4 ('-T',THRESH,0.0,0.0)
      CALL ARGI4 ('-S',NSEMB,0,0)
      CALL ARGI4 ('-TM',ITIME,0,0)
      CALL ARGI4 ( '-D',ipnt ,0  ,0  )
      CALL ARGR4 ('-DZ',DZ,0.0,0.0)
      CALL ARGR4 ('-Z',TDEPTH,0.0,0.0)
      CALL ARGR4 ('-DX',DX,0.0,0.0)
      CALL ARGR4 ('-X',TWIDTH,0.0,0.0)
      CALL ARGI4 ( '-NF',NFUNC ,0  ,0  )
      CALL ARGI4 ( '-u',iunder ,0  ,0  )
      CALL ARGI4 ( '-ixf',ixf ,-1  ,-1  )
      CALL ARGI4 ( '-ixb',ixb ,-1  ,-1  )
C
C *** MAKE THE NTAP A PIPE
C
      IF(NTAP.EQ.' ' ) IPIPI=1
C
C *** MAKE THE OTAP A PIPE
C
      IF(OTAP.EQ.' ' ) IPIPO=1
      RETURN
      END
      SUBROUTINE OPSPIN (MNDIP,MXDIP,THRSH,IBOX,IMED,INCDIP,LPRT)
C***********************************************************************
C                 
C     PROGRAM NAME: OPSPIN (SET COMMON BLOCK FOR SPIN SUBROUTINES)
C                 
C     LANGUAGE: FORTRAN
C                 
C     AUTHOR: G.MURPHY
C                 
C     DATE WRITTEN: 03/23/88
C                 
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                 
C                 
C     ABSTRACT: SET COMMON BLOCK FOR SPIN SUBROUTINES.
C                 
C     USAGE:    CALL  OPSPIN (MNDIP, MXDIP, THRSH, IBOX, IMED, INCDIP)
C                 
C     MODIFICATION HISTORY: 03/23/88  -  INITIAL RELEASE
C                 
C***********************************************************************
C                 
      COMMON/FXLU/   LUC, LUD, LUI, LUO, LUO2, LUP                      
      COMMON/FXPARM/ LDSIGN, LOPER, LSOVLP, IRS, LMED,ISS,
     &               PW, IMUTE, IISI, IFMT, ALTAPE,
     &               NTJB,MODE,LDIP,MDIP,THRESH,PERCNT,IPTYPE
      LDIP=MNDIP  
      MDIP=MXDIP  
      THRESH=THRSH
      IRS=IBOX    
      LMED=IMED   
      ISS=INCDIP  
      LUP=LPRT
      RETURN
      END
      SUBROUTINE SEMBXT (LX,ITUS,X,SMAX,DIP)                            
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       SEMBFX                                                
C  ROUTINE TYPE:  SUBROUTINE  SINGLE_ENTRY                              
C  PURPOSE:                                                             
C      COMPUTE SEMBLANCE COEFFICIENTS FOR EACH RATE OF DIP IN THE       
C      SPECIFIED RANGE (LDIP TO MDIP), THEN DETERMINE THE BEST          
C      ALIGNMENT BY PICKING THE MAXIMUM SEMBLANCE AT EACH SAMPLE POINT  
C      (SMAX).                                                          
C  ARGUMENTS:                                                           
C      LX      I*4  I        -  LENGTH IN SAMPLES OF INPUT TRACES       
C                               (STORED IN ARRAY X)                     
C      ITUS    I*4  I        -  NUMBER OF TRACES TO SCAN TO DETERMINE   
C                               SEMBLANCE COEFFICIENTS                  
C      X       R*4  I  ( LX,*)- THE BUFFER HOLDING THE SET OF NX        
C                               INPUT TRACES                            
C      LDIP    I*4  I        -  MINIMUM DIP AT WHICH TO COMPUTE SEMBLANC
C      MDIP    I*4  I        -  MAXIMUM DIP AT WHICH TO COMPUTE SEMBLANC
C      SMAX    R*4  O  ( * ) -  ARRAY TO HOLD THE MAXIMUM SEMBLANCE     
C                               COEFFICIENT AT EACH SAMPLE POINT        
C      DIP     I*4  O  ( 1 ) -  ARRAY TO HOLD THE DIP RATE ASSOCIATED   
C                               WITH THE MAXIMUM SEMBLANCE COEFFICIENT  
C                               AT EACH SAMPLE POINT                    
C      Y       R*4  *  ( 1 ) -  ARRAY TO HOLD THE SEMBLANCE COEFFICIENTS
C                               RETURNED FROM THE ARRAY PROCESSOR FOR   
C                               EACH RATE OF DIP SCANNED                
C      THRESH  R*4  C        -  THRESHOLD VALUE TO USE IN EDITING THE   
C                               SEMBLANCE VALUES.  ANY SEMBLANCE COEFF. 
C                               LT THRESH HAS ITS ASSOCIATED DIP VALUE  
C                               REPLACED WITH ZERO.                     
C       ISS    I*4  I        -  INTERVAL FOR COMPUTING SEMBLANCE.       
C       IRS    I*4  I        -  LENGTH OF SMOOTHING OPERATOR.           
C  CATEGORY:  RELATIVE SEMBLANCE COMPUTE MAXIMUM                        
C  KEYWORDS:                                                            
C       +------------------------------------------------------+        
C       |               DEVELOPMENT INFORMATION                |        
C       +------------------------------------------------------+        
C  AUTHOR:    MARILYN A. MILLER               ORIGIN DATE:  83/10/03    
C  LANGUAGE:  FORTRAN VII                                               
C       +------------------------------------------------------+        
C       |                 EXTERNAL ENVIRONMENT                 |        
C       +------------------------------------------------------+        
C  ROUTINES CALLED:                                                     
C      VCLR     -  CLEARS TO ZERO A VECTOR IN THE A.P.                  
C      VADD     -  ADDS 2 VECTORS IN THE ARRAY PROCESSOR                
C      VMA      -  VECTOR MULTIPLY AND ADD IN THE A.P.                  
C      VSMUL    -  VECTOR SCALAR MULTIPLY                               
C      VSQ      -  VECTOR SQUARE                                        
C      VSMSA    -  VECTOR SCALAR MULTIPLY AND SCALAR ADD                
C      VDIV     -  VECTOR DIVIDE                                        
C      VMOV     -  MOVE DATA WITHIN THE A.P.                            
C      VFILL    -  FILL A VECTOR WITH A SCALAR                          
C      CONV     -  CONVOLVE OPERATOR W/ OPERAND IN A.P.                 
C       +------------------------------------------------------+        
C       |             OTHER DOCUMENTATION DETAILS              |        
C       +------------------------------------------------------+        
C  ERROR HANDLING:                                                      
C  GENERAL DESCRIPTION:                                                 
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C                                                                       
C        THIS ROUTINE COMPUTES SEMBLANCE OVER A SET OF TRACES FOR A     
C        GIVEN RANGE OF DIPS.  THE TRACES INPUT ARE ORIGINAL INPUT (X). 
C        SEMBLANCE IS COMPUTED AT EVERY ISS SAMPLE POINT (1,3,5,7 OR 9) 
C        AND THEN CAN OPTIONALLY BE SMOOTHED WITH A RUNNING SUM         
C        TECHNIQUE.                                                     
C                                                                       
C        MODIFIED FROM SEMB9V (FROM SEMB2 BY K.L.PEACOCK)               
C        FORTRAN BY M.A.MILLER   8-19-83                                
C                                                                       
      REAL X(LX,*),SMAX(*),Y(8000)                                    
      REAL XX(8400),SAV1(8400),SAV2(8400),WORK(8400),MOP(1001)        
      INTEGER DIP(*),POSN                                             
      COMMON/FXLU/   LUC, LUD, LUI, LUO, LUO2, LUP                      
      COMMON/FXPARM/ LDSIGN, LOPER, LSOVLP, IRS, LMED,ISS,              
     &               PW, IMUTE, IISI, IFMT, ALTAPE,
     &               NTJB,MODE,LDIP,MDIP,THRESH,PERCNT,IPTYPE
C ----------------------------------------------------------------------
C |           COMPUTE CONSTANTS                                        |
C ----------------------------------------------------------------------
      NL=LX/ISS                                                         
      DIV = 1./ITUS                                                     
      BIAS = 1.0 E -10                                                  
      ONE = 1.0                                                         
      DIV2 = 1.0/IRS                                                    
      LX1 = LX - 1                                                      
      ISS1 = ISS-1                                                      
C     MIDL=1                                                            
      MIDL=ITUS/2+1
      LCLR=LX+ITUS*(IABS(LDIP)+IABS(MDIP))                              
      IF(LCLR.GT.8400)LCLR=8400                                         
C     WRITE (LUP,1)ISS,ITUS,IRS                                          
C   1 FORMAT (1X,'SEMBXT ISS ITUS IRS', 3I10)                       
C ----------------------------------------------------------------------
C |           LOOP ON DIP RATE FROM LDIP TO MDIP                       |
C ----------------------------------------------------------------------
      K=0                                                               
      DO 7000 IDIP=LDIP,MDIP                                            
      K=K+1                                                             
C ----------------------------------------------------------------------
C |           LOOP ON TRACE INDEX WITHIN SET OF X                      |
C ----------------------------------------------------------------------
      CALL VCLR(SAV1,1,LCLR)                                            
      CALL VCLR(SAV2,1,LCLR)                                            
      DO 6000 IC = 1,ITUS                                               
C     WRITE(LUP,2)X(1,IC)                                                 
C   2 FORMAT(1X,'SAMPLE NUMBER ONE FROM SEMBTX',F10.0)                  
C     WRITE (LUP,4) (X(I,IC), I=1, 50)                                    
C   4 FORMAT (//,1X, 'UNSHIFTED TRACE',/,10(1X,5F10.3,/),/)             
      POSN=(IC-MIDL)*IDIP+1                                             
      CALL VCLR(XX,1,LCLR)                                              
      ISTRT=POSN                                                        
      IEND=LX-ISTRT+1                                                   
      LMOVE=IEND-ISTRT                                                  
      L=1                                                               
      IF(ISTRT.LE.0)THEN                                                
         L=IABS(ISTRT)+1                                                
         LMOVE=LMOVE-L                                                  
         ISTRT=1                                                        
      ENDIF                                                             
C ----------------------------------------------------------------------
C | MOVE DATA TO WORK AREA                                             |
C ----------------------------------------------------------------------
      CALL VMOV(X(ISTRT,IC),1,XX(L),1,LMOVE)                            
C     WRITE (LUP,5) (XX(I), I=1,50)                                       
C   5 FORMAT (//,1X, 'SHIFTED TRACE',/,10(1X,5F10.3,/),/)               
C ----------------------------------------------------------------------
C | GET SUM X                                                          |
C ----------------------------------------------------------------------
C     WRITE (LUP,6) ISTRT,L                                        
C   6 FORMAT (1X,'SHIFTED FROM,TO', 2I10)                        
      CALL VADD(SAV1,ISS,XX,ISS,SAV1,ISS,LX)                     
C ----------------------------------------------------------------------
C | GET X*X                                                            |
C ----------------------------------------------------------------------
      CALL VMUL(XX,ISS,XX,ISS,XX,ISS,LX)                                
C ----------------------------------------------------------------------
C | GET SUM X*X                                                        |
C ----------------------------------------------------------------------
      CALL VADD(SAV2,ISS,XX,ISS,SAV2,ISS,LX)                            
 6000 CONTINUE                                                          
C ----------------------------------------------------------------------
C | GET (SUM X)/M                                                      |
C ----------------------------------------------------------------------
      CALL VSMUL(SAV1,ISS,DIV,SAV1,ISS,LX)                              
C ----------------------------------------------------------------------
C | GET ((SUM X)/M)**2                                                 |
C ----------------------------------------------------------------------
      CALL VSQ(SAV1,ISS,SAV1,ISS,LX)                                    
C ----------------------------------------------------------------------
C | GET  (SUM X*X)/M+1.0E-10                                           |
C ----------------------------------------------------------------------
      CALL VSMSA(SAV2,ISS,DIV,BIAS,SAV2,ISS,LX)                         
C ----------------------------------------------------------------------
C | GET  ((SUM X)/M)**2/(SUM X*X)/M+1.0E-10)                           |
C ----------------------------------------------------------------------
      CALL VDIV(SAV1,ISS,SAV2,ISS,Y,ISS,LX)                             
      IF(ISS.NE.1)THEN                                                  
         do jj = 1, lx, iss
            jjss = min (lx, jj+iss-1)
            DO II=jj+1,jjss
               y(ii) = y(jj)
            enddo
         enddo
      ENDIF                                                             
      IF(IRS.GE.3)THEN                                                  
         LCON=IRS+LX-1                                                  
         LZRO=IRS/2                                                     
         CALL VCLR(WORK,1,8400)                                         
         CALL VFILL(1.,MOP,1,IRS)                                       
         CALL VMOV(Y,1,WORK(LZRO),1,LX)                                 
C ----------------------------------------------------------------------
C | NOTE THAT SINCE MOP IS SYMMETRIC, CAN DO CORRELATION INSTEAD OF    |
C | CONVOLUTION.                                                       |
C ----------------------------------------------------------------------
            CALL CONV(WORK,1,MOP,1,Y,1,LX,IRS,1)                        
         CALL VSMUL(Y,1,DIV2,Y,1,LX)                                    
      ENDIF                                                             
C ----------------------------------------------------------------------
C |           SET SMAX TO MAXIMUM SEMBLANCE COMPUTED                   |
C ----------------------------------------------------------------------
      IF(IDIP.GT.LDIP) GO TO 6200                                       
      DO 6100 L = 1,LX                                                  
      DIP(L) = IDIP                                                     
 6100 SMAX(L) = Y(L)                                                    
      GO TO 7000                                                        
 6200 DO 6300 L = 1,LX                                                  
      IF(Y(L).LE.SMAX(L)) GO TO 6300                                    
      SMAX(L) = Y(L)                                                    
      DIP(L) = IDIP                                                     
 6300 CONTINUE                                                          
 7000 CONTINUE                                                          
C ----------------------------------------------------------------------
C |           EDIT THE SEMBLANCE - IF MAX SEMB LT THRESH, SET DIP=0    |
C ----------------------------------------------------------------------
      IF(THRESH.NE.0.)THEN
      DO 7100 L = 1,LX                                                  
      IF(SMAX(L).LT.THRESH) DIP(L) = 0                                  
 7100 CONTINUE                                                          
      ENDIF                                                             
C +--------------------------------------------------------------------+
C |           RETURN TO MAIN                                           |
C +--------------------------------------------------------------------+
      RETURN                                                            
      END                                                               
      SUBROUTINE MEDFIL(LS,S,O)                                         
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C                                                                      *
C***********************************************************************
C  ROUTINE:       MEDFIL                                                
C  ROUTINE TYPE:  SUBROUTINE  SINGLE_ENTRY                              
C  PURPOSE:                                                             
C      THIS ROUTINE SELECTS THE VALUE WHICH OCCURS THE MOST NUMBER      
C      OF TIMES WITHIN AN INTERVAL OF SPECIFIED LENGTH SURROUNDING THE  
C      SAMPLE BEING EXAMINED (KNOWN AS A MEDIAN FILTER TECHNIQUE).      
C  ARGUMENTS/PARAMETERS:                                                
C      LDIP   I*4  C         MINIMUM DIP (IN SAMPLES)                   
C      MDIP   I*4  C         MAXIMUM DIP (IN SAMPLES)                   
C      LMED   I*4  C         LENGTH IN SAMPLES OF INTERVAL TO SEARCH    
C      LS     I*4  I         LENGTH OF ARRAY S (IN SAMPLES)             
C      S      I*4  I  ( 1 )  INPUT ARRAY (CONTAINS DIP VALUES)          
C      O      I*4  O  ( 1 )  OUTPUT ARRAY                               
C      IBUF   I*4  U  ( 1 )  WORKING SPACE FOR DETERMINING WHICH DIP    
C                            OCCURS MOST IN THE INTERVAL LENPP. MUST    
C                            BE DIMENSIONED TO AT LEAST                 
C                            (MDIP-LDIP+1) SAMPLES                      
C  CATEGORY:  RELATIVE MEDIAN FILTER DIP                                
C  KEYWORDS:                                                            
C       +------------------------------------------------------+        
C       |               DEVELOPMENT INFORMATION                |        
C       +------------------------------------------------------+        
C  AUTHOR:    MARILYN A. MILLER               ORIGIN DATE:  83/10/01    
C  LANGUAGE:  FORTRAN IV                                                
C       +------------------------------------------------------+        
C       |                 EXTERNAL ENVIRONMENT                 |        
C       +------------------------------------------------------+        
C  ROUTINES CALLED:                                                     
C      MOVE     -  MOVES, ZEROES OR BLANKS DATA ARRAYS                  
C       +------------------------------------------------------+        
C       |             OTHER DOCUMENTATION DETAILS              |        
C       +------------------------------------------------------+        
C  ERROR HANDLING:                                                      
C  GENERAL DESCRIPTION:                                                 
C        THE ROUTINE PERFORMS A MEDIAN FILTER TECHNIQUE ON AN           
C        INPUT ARRAY.  THE TECHNIQUE SELECTS THE VALUE WHICH            
C        OCCURS THE MOST NUMBER OF TIMES WITHIN A GIVEN                 
C        INTERVAL SURROUNDING THE INDEX BEING EXAMINED, AND             
C        OUTPUTS THE SELECTED VALUE AT THAT INDEX.  THIS                
C        FILTER TECHNIQUE IS PERFORMED ON EACH INDEX OF THE             
C        INPUT ARRAY S, AND THE FILTERED ARRAY O IS OUTPUT.             
C        WHEN THE COUNT THAT OCCURS MOST WITHIN AN INTERVAL IS          
C        DETERMINED, AND MORE THAN ONE VALUE HAS THAT COUNT,            
C        THE MINIMUM OF THE VALUES IS SELECTED AS THE OUTPUT            
C        VALUE.                                                         
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
CC                                                                      
CC       THIS SUBROUTINE SETS OUTPUT ARRAY O TO THE NUMDIP              
CC            VALUE(DIP)FOUND IN THE INTERVAL LENPP OF                  
CC            INPUT ARRAY S                                             
CC                                                                      
CC       LDIP    =  LEAST DIP                                           
CC       MDIP    =  MAXIMUM DIP                                         
CC       LS      =  LENGTH OF ARRAY S IN SAMPLES                        
CC       S       =  INPUT ARRAY (CONTAINS DIP VALUES)                   
CC       O       =  OUTPUT ARRAY (CONTAINS DIP VALUES)                  
CC       IBUF    =  WORKING SPACE FOR DETERMINING WHICH DIP             
CC                  O
CC                  (MUST BE DIMENSIONED TO AT LEAST                    
CC                     (MDIP-LDIP+1)  )                                 
CC       LENPP   =  LENGTH (IN SAMPLES) OF INTERVAL TO SEARCH           
CC                                                                      
      INTEGER S(1),O(1),IBUF(401)                                     
      COMMON/FXPARM/ LDSIGN, LOPER, LSOVLP, IRS, LMED,ISS,              
     &               PW, IMUTE, IISI, IFMT, ALTAPE,                     
     &               NTJB,MODE,LDIP,MDIP,THRESH,PERCNT,IPTYPE 
            LENPP=LMED                                                  
C...                                                                    
C        IF LENPP = -1, NO MEDIAN FILTER IS REQUESTED                   
C...     IF LENPP IS NEGATIVE, DO NOT EDIT BUT MOVE S INTO O            
      IF(LENPP.GT.0) GO TO 1                                            
C*****CALL MOVE (1,O,S,LS*4)                                            
      CALL VMOV (S,1,O,1,LS)
      GO TO 2000                                                        
C...                                                                    
C...     EDIT THE ARRAY S                                               
    1 NDIP = (MDIP-LDIP+1)                                              
      NBM = NDIP*4                                                      
      NUMDIP = 0                                                        
      LENPP2 = LENPP/2                                                  
      LL = LENPP2                                                       
C*****CALL MOVE (0,IBUF,0,NBM)
      CALL VCLR(IBUF,1,NDIP)                                          
      DO 10 I=1,LL                                                      
      J = (S(I)-LDIP+1)                                                 
      IBUF(J) = IBUF(J)+1                                               
      NUMDIP = NUMDIP + 1                                               
   10 CONTINUE                                                          
      LLS = LL-LENPP+2                                                  
      DO 1000 I=1,LS                                                    
      LL = LL+1                                                         
      IF(LL.GT.LS) GO TO 100                                            
      J = (S(LL)-LDIP+1)                                                
      IBUF(J) = IBUF(J)+1                                               
      NUMDIP = NUMDIP + 1                                               
  100 IND = 1                                                           
      IFIND = 0                                                         
      MEDIAN = NUMDIP/2+1                                               
      DO 200 J=1,NDIP                                                   
      IND = J                                                           
      IFIND = IFIND + IBUF(J)                                           
      IF(IFIND.GE.MEDIAN)GO TO 210                                      
  200 CONTINUE                                                          
  210 O(I) = LDIP+(IND-1)                                               
      IF(LLS.LT.1) GO TO 300                                            
      J = (S(LLS)-LDIP+1)                                               
      IBUF(J) = IBUF(J)-1                                               
      NUMDIP = NUMDIP - 1                                               
  300 LLS = LLS+1                                                       
 1000 CONTINUE                                                          
 2000 RETURN                                                            
      END                                                               
