C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       morggam                                              *
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters
c    these includes are picked up and expanded into the code by afp
c    and are found in ~usp/include/f77
 
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
c-----
c    SZLNHD is a value obtained from lhdrsz.h
c    The 3 vectors below are equivalenced and are
c    to access the trace header entries (whatever
c    they may be)
c-----
      integer     lhed ( SZLNHD )
      integer * 2 itr  ( SZLNHD )
      real        head ( SZLNHD )
 
      integer     nsamp, nsi, ntrc, nrec, nrecs, iform, obytes
      integer     luin1, luin2, luin3 
      integer     luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne
 
c-----------------------------------------------------------------------
c     next lines needed for reading picks
 
      parameter (lpck=27, lpck2=28, lpout=29)
      parameter (maxseg = 3000, maxpnt = 3000, maxnum = maxseg*maxpnt)
      integer icolor(1), npts(1)
      integer icolr2(1), npt2(1)
      integer npicko(1)
      real rec(1),trac(1),samp(1)
      real rec2(1),trac2(1),samp2(1)
      real sego(1)
      pointer     (prec, rec)
      pointer     (ptrac, trac)
      pointer     (psamp, samp)
      pointer     (prec2, rec2)
      pointer     (ptrac2, trac2)
      pointer     (psamp2, samp2)
      pointer     (picolr2, icolr2)
      pointer     (pnpt2, npt2)
      pointer     (psego, sego)
      pointer     (picolor, icolor)
      pointer     (pnpts, npts)
      character*20 segnam(maxseg)
      character*20 segnm2(maxseg)
      real units(3),offset(3)
      real units2(3),offst2(3)
      character*128 picks, pick2, pout
      integer nout
 
c     maxseg is maximum number of segments allowed
c     maxpnt is maximum number of points for any one segment
c-----------------------------------------------------------------------
 
c------
c  static memory allocation
c     real        vel1(SZSPRD*SZSMPM)
c     real        vel2(SZSPRD*SZSMPM)
c------
c  dynamic memory allocation for big arrays, eg whole records
      integer     itrhdr
      real        vel1, vel2, resid, semb
      pointer     (pitrhdr, itrhdr(1))
      pointer     (pvel1, vel1(1))
      pointer     (pvel2, vel2(1))
      pointer     (presid, resid(1))
      pointer     (psemb, semb(1))
c------
      integer     recnum, trcnum
      integer     srcloc, recind, dphind, dstsgn, stacor
c-----
c    SZSMPM is a value obtained from lhdrsz.h
c-----
      real        tri ( SZLNHD )
      character   ntap1 * 100, ntap2 * 100, ntap3 * 100
      character   otap * 100, name*7, version*4
      logical     verbos, hlp, query
      logical     luinp1, luinp2, luinp3, luouto
      integer     argis
 
c-----
c    we access the header values which can be shot or long integers
c    or real values.  The actual trace values start at position
c    ITRWRD1  (position 65 in the old SIS format).  This value is
c    set in lhdrsz.h but eventually could come in thru the line header
c    making the trace header format variable
c-----
      equivalence ( itr( 1), lhed (1), head(1) )
 
      data lbytes / 0 /, nbytes / 0 /, name/'MORFGAM'/, version /' 1.2'/
      data luinp1/.FALSE./,luinp2/.FALSE./,luinp3/.FALSE./
      data luouto/.FALSE./
      data no_seg / 0 /, no_pnt / 0 /
      data nout / 0 /
      data iabort / 0 /
 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query )then
            call help()
            stop 5
      endif
      if ( hlp ) then
           call help()
           stop 10
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/mbsopen.h>
 
      call gcmdln(ntap1,ntap2,ntap3,otap,ns,ne,irs,ire,
     1             picks,pick2,pout,dgamma,ngamma,dz,pow,
     1             iopt,vmin,vmax,verbos)
 
c-----
c     get logical unit numbers for input and output of seismic data
 
c     input values are strings ntap & otap (i.e. names of files), and the
c     read 'r' or write 'w' designations of these files, and the default
c     logical unit numbers if ntap & otap are blank strings, i.e. ntap = " "
c     in which case:
c     0 = default stdin
c     1 = default stdout
c     Note: default values other that 0 & 1 may be used
 
c     output values are the logical unit numbers accessing these disk files. if
c     these values are less than 0 it means there was a fatal error in trying to
c     open these files
c-----
      call getln(luin1,ntap1,'r',0)
      luinp1=.TRUE.
      if(iopt.eq.1)then
         call getln(luin2,ntap2,'r',0)
         luinp2=.TRUE.
      endif
      if(ntap3.eq.' ')then
         luin3=0
      else
         call getln(luin3,ntap3,'r',0)
      endif
      luinp3=.TRUE.
      call getln(luout,otap,'w',1)
      luouto=.TRUE.
 
c-----
c     read line header of input DSN (rtape reads data into vector "itr"
c     lbytes is the number of bytes actually read
c-----
      call rtape  ( luin3, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'MORFGAM: no header read from ',ntap3
         write(LOT,*)'FATAL'
         stop 15
      endif
 
c------
c     save certain pace header rameters
 
c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
 
c-----------
c format values are:
 
c     integer    = 1
c     float      = 2
c     character  = 3
c     2-byte int = 4
c-----------
 
c------
c  here we mark out slots to be used for 4-byte floating point
c  storeage in the trace header.  we choose to use the time-velocity
c  area of the trace header but starting from the tail-end to minimize
c  clobbering those folks who do use this area for its intended purpose.
 
c  devlopers are wise to allow some freedom of the user to stake out
c  these slots so he can avoid trashing something he needs.
c  in the 2 cases below we grab the last 2 T-V pairs (for 2 reals)
 
      call savelu('TVPT20',ifmt_TVPT20,l_TVPT20,ln_TVPT20,TRACEHEADER)
      call savelu('TVPT21',ifmt_TVPT21,l_TVPT21,ln_TVPT21,TRACEHEADER)
 
      write(LERR,*)'TrcNum,ifmt,l_TrcNum,length= ',
     1             ifmt_TrcNum,l_TrcNum,ln_TrcNum
      write(LERR,*)'RecNum,ifmt,l_RecNum,length= ',
     1             ifmt_RecNum,l_RecNum,ln_RecNum
      write(LERR,*)'SrcLoc,ifmt,l_SrcLoc,length= ',
     1             ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc
      write(LERR,*)'RecInd,ifmt,l_RecInd,length= ',
     1             ifmt_RecInd,l_RecInd,ln_RecInd
      write(LERR,*)'DphInd,ifmt,l_DphInd,length= ',
     1             ifmt_DphInd,l_DphInd,ln_DphInd
      write(LERR,*)'DstSgn,ifmt,l_DstSgn,length= ',
     1             ifmt_DstSgn,l_DstSgn,ln_DstSgn
      write(LERR,*)'StaCor,ifmt,l_StaCor,length= ',
     1             ifmt_StaCor,l_StaCor,ln_StaCor
 
c     To get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c     LINHED is a value in the include file <f77/sisdef.h> that refers to the
c     lineheader
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrecs , LINHED)
      call saver(itr, 'Format', iform, LINHED)
 
c------
c     hlhprt prints out the historical line header of length lbytes AND
 
c     hlhprt takes "name", in this case 4 characters long and stuffs this
c     into the modified historical line header and returns the NEW length
c     of the line header in lbytes
c------
      call hlhprt (itr, lbytes, name, 7, LERR)
 
c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrecs)
 
c--------------------------
c  note: these don't
c  have to be the same size
      nrecc = ire - irs + 1
      jtr   = ne - ns + 1
      if(ngamma.eq.0)ngamma=jtr
      
      call galloc (pitrhdr,nrecc*ITRWRD*SZSMPD,jerr,abort)
      if(jerr.eq.1)then
            write(LERR,*)'  Error allocating pitrhdr'
            write(LOT,*)' job completed abnormally'
            stop 20
      endif
      
      call galloc (pvel1,nrecc*nsamp*SZSMPD,jerr,iabort)
      if(jerr.eq.1)then
            write(LERR,*)'  Error allocating pvel1'
            write(LOT,*)' job completed abnormally'
            stop 25
      endif

      call galloc (pvel2,nrecc*nsamp*SZSMPD,jerr,iabort)
      if(jerr.eq.1)then
            write(LERR,*)'  Error allocating pvel2'
            write(LOT,*)' job completed abnormally'
            stop 30
      endif

      call galloc (presid,nrecc*nsamp*SZSMPD,jerr,iabort)
      if(jerr.eq.1)then
            write(LERR,*)'  Error allocating presid'
            write(LOT,*)' job completed abnormally'
            stop 35
      endif
      
      call galloc (psemb,ngamma*nsamp*SZSMPD,jerr,iabort)
      if(jerr.eq.1)then
            write(LERR,*)'  Error allocating psemb'
            write(LOT,*)' job completed abnormally'
            stop 40
      endif

c-----
c     modify line header to reflect actual number of traces output
c-----
      call savew(itr, 'NumRec', nrecc, LINHED)
      call savew(itr, 'NumTrc', ngamma,LINHED)
 
c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
 
c----------------------
c  inject command line into
c  historical LH:
c  inputs are current line header in vector "itr" and length lbytes (bytes)
c  outputs are line header modified by insertion of the command line into
c  the historical LH and the modified length of the new LH (lbyout)
      call savhlh(itr,lbytes,lbyout)
c----------------------
 
c------
c     write to unit number luout lbyout bytes contained in vector itr
c------
      call wrtape ( luout, itr, lbyout                 )
 
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrecs, iform,
     1                  picks,pick2,pout,dgamma,ngamma,dz,pow,
     1                  iopt,vmin,vmax,
     1                  ntap1,ntap2,ntap3,otap)
      end if
 
c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary
 
      if (nsi .le. 32) then
         dt = real (nsi) /1000.
      else
         dt = real (nsi) /1000000.
      endif
 
C-----------------------------------------------------------------------
C Read the picks file with rdpic - parameter explanation:
c rec    = x point, trac = y point, samp = z point (returned from rdpic)
c icolor = color number for segment                (returned from rdpic)
c segnam  = name of segment                        (returned from rdpic)
c npts   = npts(1) = no. points in segment 1       (returned from rdpic)
c          npts(2) = no. points in segment 2, etc.
c units  = units for rec,trac,samp                 (returned from rdpic)
c offset = offset for rec,trac,samp                (returned from rdpic)
c maxseg = maximum no. of segments allowed
C no_seg = the no. of segments read                (returned from rdpic)
c maxpnt = maximum no. of pnts/segment allowed
C no_pnt = maximum no. of points in any 1 segment  (returned from rdpic)
C SZSMPD = size of a sample (defined in lhdrsz.h)
C nrec   = no. records in original 'picked' data   (returned from rdpic)
c ntrac  = no. traces in original 'picked' data    (returned from rdpic)
c nsamp  = no. samples in original 'picked' data   (returned from rdpic)
C LERR   = printfile
C lpck   = the input pick file
c jerr   = error flag                              (returned from rdpic)
C-----------------------------------------------------------------------

c we don't use picks if we morf from gammas to slowness scans.
      if(iopt.eq.2)goto 100

      if(picks.ne.' ')then
         open(unit=lpck,file=picks,status='old',iostat=jerr)
         if(jerr.ne.0)then
            write(LERR,*)'  Error opening picks file'
            write(LOT,*)' job completed abnormally'
            stop 45
         endif
      else
         write(LERR,*)' Picks filename must be supplied'
         write(LOT,*)' job completed abnormally'
         stop 50
      endif

      if(pick2.ne.' ')then
         open(unit=lpck2,file=pick2,status='old',iostat=jerr)
         if(jerr.ne.0)then
            write(LERR,*)'  Error opening pick2 file'
            write(LOT,*)' job completed abnormally'
            stop 55
         endif
      else
         write(LERR,*)' Pick2 filename must be supplied'
         write(LOT,*)' job completed abnormally'
         stop 60
      endif
 
      if(pout.ne.' ')then
         open(unit=lpout,file=pout,iostat=jerr)
         if(jerr.ne.0)then
            write(LERR,*)'  Error opening output picks file'
            write(LOT,*)' job completed abnormally'
            stop 65
         endif
      else
         write(LERR,*)' Output picks filename must be supplied'
         write(LOT,*)' job completed abnormally'
         stop 70
      endif
      write(LERR,10)
   10 format (//, 27X, 'program parameters',//)
      write(LERR,37)picks
      write(LERR,38)pick2
      write(LERR,39)pout
   37 format(' input picks dataset = ',A128)
   38 format(' input horizon picks dataset = ',A128)
   39 format(' output picks dataset = ',A128)
      call rdpici(lprt,lpck,jerr,no_seg,no_pnt)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error reading pick file header'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      if (no_seg .gt. maxseg) then
         write(LERR,*)'  Too many segments in picks file'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call rdpici(lprt,lpck,jerr,noseg2,nopnt2)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error reading pick2 file header'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      if (noseg2 .gt. maxseg) then
         write(LERR,*)'  Too many segments in pick2 file'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(prec,no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(ptrac,no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(psamp,no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(prec2,noseg2*nopnt2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(ptrac2,noseg2*nopnt2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(psamp2,noseg2*nopnt2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(psego,jtr*no_seg*no_pnt*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(picolor,no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(picolr2,noseg2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pnpts,no_seg*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      call galloc(pnpt2,noseg2*SZSMPD,jerr,iabort)
      if (jerr .ne. 0) then
         write(LERR,*)'  Error allocating memory'
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      mxseg=no_seg
      mxpnt=no_pnt
      do i=1,mxseg
         npicko(i)=0
      enddo
      call rdpic(rec,trac,samp,icolor,segnam,npts,
     &           units,offset,mxseg,no_seg,mxpnt,no_pnt,SZSMPD,
     &           nrec,ntrac,nsmp,LERR,lpck,jerr)
      if (jerr .ne. 0) goto 999
 
      mxseg2=noseg2
      mxpnt2=nopnt2
      call rdpic(rec2,trac2,samp2,icolr2,segnm2,npt2,
     &           units2,offst2,mxseg2,noseg2,mxpnt2,nopnt2,SZSMPD,
     &           nrec2,ntrac2,nsmp2,LERR,lpck2,jerr)
      if (jerr .ne. 0) goto 999

 100  continue

c read two velocity fields 
      call rvtape(luin1,ntap1,vel1,nsamp,nrecc,minvel,maxvel,
     &            jerr,LOT,LERR)   
      if(vmin.eq.0.)vmin=minvel
      if(vmax.eq.0.)vmax=maxvel
      slowmn=1./vmax
      dslow=((1./vmin)-slowmn)/(ngamma-1)
      if(jerr.eq.1)then
         write(LERR,*)'  Error reading velocity field ',ntap1
         write(LOT,*)' job completed abnormally'
         goto 999
      endif
      if(iopt.eq.1)then
         call rvtape(luin2,ntap2,vel2,nsamp,nrecc,minvel,maxvel,
     &               jerr,LOT,LERR)  
         if(jerr.eq.1)then
            write(LERR,*)'  Error reading velocity field ',ntap2
            write(LOT,*)' job completed abnormally'
            goto 999
         endif
      endif


c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin3,ntrc,itr)
 
c-----
c     process desired trace records
c-----
      do 1000 jj = irs, ire
 
c----------------------
c  skip to start trace
            call trcskp(jj,1,ns-1,luin3,ntrc,itr)
c----------------------
 
            ic = 0
            do 1001  kk = ns, ne
 
                  nbytes = 0
                  call rtape( luin3, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
 
c------
c     use previously derived pointers to trace header values
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_SrcLoc,l_SrcLoc, ln_SrcLoc,
     1                        srcloc , TRACEHEADER)
                  call saver2(lhed,ifmt_RecInd,l_RecInd, ln_RecInd,
     1                        recind , TRACEHEADER)
                  call saver2(lhed,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                        dphind , TRACEHEADER)
                  call saver2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        dstsgn , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        stacor , TRACEHEADER)
 
 
c------
 
 
                  if (stacor .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif
 
c----------------------
c  pack data into array
                  ic = ic + 1
                  istrc = (ic-1) * nsamp
                  ishdr = (ic-1) * ITRWRD
                  call vmov (tri,1, semb(istrc+1),1, nsamp)
                  call vmov (lhed,1, itrhdr(ishdr+1),1,ITRWRD)
 
1001        continue
 
c----------------------
c  skip to end of record
            call trcskp(jj,ne+1,ntrc,luin3,ntrc,itr)
c----------------------
 
c-----------------------
c  here's the meat...
c  do something to data
            call subs (jtr, nsamp, vel1, vel2,
     &                 resid, semb, nrecc, jj,
     &                 rec, trac, samp, npts, no_seg, 
     &                 rec2, trac2, samp2, npt2, noseg2, 
     &                 ngamma, dgamma, dz, pow, 
     &                 iopt, slowmn, dslow)

c---------------------
c  extract traces from
c  output array and
c  write output data
            do 1002 kk = 1, jtr
 
                  istrc = (kk-1) * nsamp
                  ishdr = (kk-1) * ITRWRD
                  call vmov (semb(istrc+1),1,lhed(ITHWP1),1, nsamp)
                  call vmov (itrhdr(ishdr+1),1,lhed,1,ITRWRD)
                  call wrtape (luout, itr, obytes)
 
 
 1002             continue

 1000       continue
 
c generate the output picks (just add the sort indexes to the 
c modified input picks)

        if(iopt.eq.2)goto 999

        nout=0
        inpick=0
        do i=1,no_seg
 
           do j=1,npts(i)
              inpick=inpick+1
              inpck2=0
              do i2=1,i-1
                 inpck2=inpck2+npt2(i2)
              enddo
              do j2=1,npt2(i)
                 inpck2=inpck2+1
                 if(nint(trac(inpick)).eq.nint(trac2(inpck2)))then
                     nout=nout+1
                     sego(nout)=i
                  endif
               enddo
            enddo
        enddo

      offset(3)=0
      units(3)=1
      nrec=nrec2
      ntrac=ntrac2
      write(LERR,*)' offsets=',offset(1),offset(2),offset(3)
      write(LERR,*)' units=',units(1),units(2),units(3)
      write(LERR,*)' nrec,ntrac,nsamp=',nrec2,ntrac2,nsmp2
C-----------------------------------------------------------------------
C     This section will write out a new picks file
c     parameters are the same as in the rdpic call
C     To add or delete segments, change value of no_seg
c     To add or delete picks in segment 1, change value of npts(1), etc.
C-----------------------------------------------------------------------
      jerr = 0
      call swrpck(rec2,trac2,samp2,icolr2,segnm2,npt2,units2,offst2,
     &            nrec2,ntrac2,nsmp2,noseg2,nopnt2,LERR,lpout,
     &            sego,nout,jerr)
 
 
  999 continue
 
c-----
c     close data files
c     flush data left
c     in output buffer
c     NOTE: if the output buffer is not closed you can sometimes end up
c     with missing data
c-----
      if (luinp1) call lbclos ( luin1 )
      if (luinp2) call lbclos ( luin2 )
      if (luinp3) call lbclos ( luin3 )
      if (luouto) call lbclos ( luout )
 
            write(LERR,*)'end of MORFGAM, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'MORFGAM computes point-wise semblance traces along gamma scans:'
        write(LER,*)
     :'see manual pages for details ( online by typing xmbsman )'
        write(LER,*)' '
        write(LER,*)
     :'execute MORFGAM by typing MORFGAM and the program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N1 [ntap1]     (no default)      : input velocity field name'
        write(LER,*)
     :' -N2 [ntap2]     (no default)      : input modified velocity '
        write(LER,*)
     :' -N3 [ntap3]     (no default)      : input spectra file name'
        write(LER,*)
     :' -O [otap]       (no default)      : output data file name'
        write(LER,*)
     :' -ns[ns]         (default = first) : start trace number'
        write(LER,*)
     :' -ne[ne]         (default = last)  : end trace number'
        write(LER,*)
     :' -rs[irs]        (default = first) : start record number'
        write(LER,*)
     :' -ne[ire]        (default = last)  : end record number'
        write(LER,*)
     :' -P1[picks]      (no default)      : input xsd ',
     :'picks from stack'
        write(LER,*)
     :' -P2[pick2]      (no default)      : input xsd ',
     :'picks from horizon velocity scans'
        write(LER,*)
     :' -P3[pout]       (no default)      : output pick file'
        write(LER,*)
     :' -dgamma[dgamma] (default .020)    : delta gamma'
        write(LER,*)
     :' -ngamma[ngamma] (default ntr)     : number of gammas'
        write(LER,*)
     :' -dz[dz]         (no default)      : delta z'
        write(LER,*)
     :' -pow[pow]       (default 2.0)     : 1=avg; 2=rms'
        write(LER,*)
     :' -iopt[pow]      (default 1)       : 1=update gamma;'
        write(LER,*)
     :'                                     2=slowness scan'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   MORFGAM -N1[ntap1] -N2[ntap2] -N3[ntap3] -O[otap] '
        write(LER,*)
     :'                 -ns[ns] -ne[ne] -rs[irs] '
        write(LER,*)
     :'                 -re[ire] -P1[picks] -P2[pick2] -P3[pout]'
        write(LER,*)
     :'                 -dgamma[dgamma] -ngamma[ngamma] -dz[dz] '
        write(LER,*)
     :'                 -pow[pow] -iopt[iopt] -vmin[vmin] '
        write(LER,*)
     :'                 -vmax[vmax] [-V]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
      subroutine gcmdln(ntap1,ntap2,ntap3,otap,ns,ne,irs,ire,
     1                  picks,pick2,pout,dgamma,ngamma,dz,pow,
     1                  iopt,vmin,vmax,verbos)
c-----
c     get command arguments
c
c     ntap    - C*100    input file name
c     otap    - C*100    output file name
c     ns      - I*4      starting trace index
c     ne      - I*4      ending trace index
c     irs     - I*4      starting record index
c     ire     - I*4      ending record index
c     picks   - C*100    input file name
c     pick2   - C*100    input file name
c     pout    - C*100    output file name
c     dgamma  - R*4      delta gamma
c     ngamma  - I*4      number of gammas
c     dz      - R*4      output file name
c     pow     - R*4      1=avg; 2=rms
c     iopt    - I*4      1=update gammas;2=gamma to slowness
c     vmin  - R*4      minimum slowness
c     vmax   - R*4      delta slowness
c     verbos  - L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap1*(*), otap*(*), picks*(*), pick2*(*), pout*(*)
      character   ntap2*(*), ntap3*(*)
      integer     ns, ne, irs, ire
      logical     verbos
      integer     argis
 
c-------
c     import values from the command line using keys, e.g. -N
c     to which are immediately attached the users values.
 
c     For example program MORFGAM might be invoked in the following way:
 
c     MORFGAM  -Nxyz -Oabc
 
c     in which case xyz is a string (the name of the input data set)
c     which will be imported into MORFGAM and associated with the variable
c     "ntap"
 
c     see manual pages on the argument handler routines
c     for the meanings of these functions.  Briefly though
c     we can import from the command line strings, integers, reals,
c     double precision reals, and check the existence of command line
c     keys.  The last 2 fields are the value to be assigned the variable
c     (1) if ONLY the key is present (no value attached to it), or
c     (2) if NO key & no value are present
c-------
            call argstr( '-N1', ntap1, ' ', ' ' )
            call argstr( '-N2', ntap2, ' ', ' ' )
            call argstr( '-N3', ntap3, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argstr( '-P1', picks, ' ', ' ' )
            call argstr( '-P2', pick2, ' ', ' ' )
            call argstr( '-P3', pout, ' ', ' ' )
            call argr4 ( '-dgamma', dgamma ,   0.020 ,  0.020    )
            call argi4 ( '-ngamma', ngamma ,  0,  0    )
            call argr4 ( '-dz', dz,   10.,  10.    )
            call argr4 ( '-pow', pow, 2., 2. )
            call argi4 ( '-iopt', iopt ,  1,  1    )
            call argr4 ( '-vmin', vmin, 0., 0. )
            call argr4 ( '-vmax', vmax, 0., 0. )
            verbos =   (argis('-V') .gt. 0)
 
c-------
c     Note:  argis() = 1 if the key is present
c            argis() = 0 if it isn't
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrecs, iform,
     1                  picks,pick2,pout,dgamma,ngamma,dz,pow,
     1                  iopt,vmin,vmax,
     1                  ntap1,ntap2,ntap3,otap)

c-----
c     verbose output of processing parameters
c
c     nsamp   - I*4     number of samples in trace
c     nsi     - I*4     sample interval in ms
c     ntrc    - I*4     traces per record
c     nrecs   - I*4     number of records per line
c     iform   - I*4     format of data
c     ntap    - C*100   input file name
c     otap    - C*100   output file name
c     picks   - C*100   input pick
c     pout    - C*100   output pick
c     dgamma  - R*4     delta gamma
c     ngamma  - I*4     search window size
c     pow     - R*4     1=avg; 2=pow
c     dz      - R*4     delta z
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrecs
      character   ntap1*(*), otap*(*), picks*(*), pick2*(*), pout*(*)
      character   ntap2*(*), ntap3*(*)
      real        dgamma, dz, pow, vmin, vmax
      integer     ngamma
 
            write(LERR,*) ' '
            write(LERR,*) ' line header values after default check '
            write(LERR,*) ' # of samples/trace     =  ', nsamp
            write(LERR,*) ' sample interval        =  ', nsi
            write(LERR,*) ' traces per record      =  ', ntrc
            write(LERR,*) ' records per line       =  ', nrecs
            write(LERR,*) ' format of data         =  ', iform
            write(LERR,*) ' input velocity field   =  ', ntap1
            write(LERR,*) ' input modified velocity=  ', ntap2
            write(LERR,*) ' input semblance field  =  ', ntap3
            write(LERR,*) ' output data set name   =  ', otap
            write(LERR,*) ' input pick file        =  ', picks
            write(LERR,*) ' input pick file        =  ', pick2
            write(LERR,*) ' output pick file       =  ', pout
            write(LERR,*) ' delta gamma            =  ', dgamma
            write(LERR,*) ' number of gammas       =  ', ngamma
            write(LERR,*) ' delta z                =  ', dz
            write(LERR,*) ' pow                    =  ', pow
            write(LERR,*) ' iopt                   =  ', iopt
            write(LERR,*) ' vmin                   =  ', vmin
            write(LERR,*) ' vmax                  =  ', vmax
            write(LERR,*) ' '
            write(LERR,*) ' '
 
      return
      end

      subroutine rvtape(luin,name,array,nsamp,ntrcs,minvel,maxvel,
     &                  jerr,LOT,LERR)
      integer luin,nsamp,ntrcs
      character name*(*)
      real array(nsamp,ntrcs)

#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
      
      integer      lhed(SZLNHD)

      lbytes=SZLNHD*SZSMPD
      call rtape(luin,lhed,lbytes)
      if(lbytes.eq.0)then
         write(LOT,*)'no header read from ',name
         write(LOT,*)'FATAL'
         jerr=1
         return
      endif
      call saver(itr, 'MinVel', minvel, LINHED)
      call saver(itr, 'MaxVel', maxvel, LINHED)

      jeof=0
      lbytes=(nsamp+ITRWRD)*SZSMPD
      do jtr=1,ntrcs
         if(jeof.eq.0)then
            nbytes=lbytes
            call rtape(luin,lhed,nbytes)
            if(nbytes.eq.0) then
               write(LERR,*)'End of file on input:',name
               write(LERR,*)'  rec = 1 trace= ',jtr
               jeof=1
            endif
         endif
         call vmov(lhed(ITHWP1),1,array(1,jtr),1,nsamp)
      enddo

      jerr=0
      return
      end
