/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
**   lpinv.c - velocity manipulation functions
**
**   AUTHOR:  Curtis Kruse
**   REVISED  David Nelson              28 SEP 92
**            Call vrecip before and after smoothing in velextract so that
**            smoothing is performed in average slowness.
**   REVISED: David Nelson              17 NOV 92
**            Added functions DAVG2TAVG and NL2L to support VMOD
**   REVISED: David Nelson              20 NOV 92
**            Added function NLTLP2NLTINT to support VMOD
**   REVISED: David Nelson              24 NOV 92
**            1. Change SMOOTH to PSMOOTH and TIV2TLP to TINT2TLP so that
**            VMOD and P2IV can use this common file
**            2. Change linterp to nl2l in velextract when converting to depth
**            3. Moved all system dependent function defines to top of file
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <localsys.h>

#if defined( CRAYSYSTEM )
#include <fortran.h>
#define VRECIP VRECIP
#define DAVG2TAVG DAVG2TAVG
#define PSMOOTH PSMOOTH
#define LINTERP LINTERP
#define NL2L NL2L
#define TLP2TINT TLP2TINT
#define NLTLP2NLTINT NLTLP2NLTINT
#define TINT2TLP TINT2TLP
#define D2NLT D2NLT
#define NLD2NLT NLD2NLT
#define T2NLD T2NLD
#define TSMOOTH TSMOOTH
#define XSMOOTH XSMOOTH
#define VELEXTR VELEXTR
#define INFILL INFILL
#define VELBLOCK VELBLOCK
#endif

#if defined( SUNSYSTEM ) || defined( LINUXSYSTEM ) || defined( CONVEXSYSTEM ) || defined( SGISYSTEM )
#define VRECIP vrecip_
#define DAVG2TAVG davg2tavg_
#define PSMOOTH psmooth_
#define LINTERP linterp_
#define NL2L nl2l_
#define TLP2TINT tlp2tint_
#define NLTLP2NLTINT nltlp2nltint_
#define TINT2TLP tint2tlp_
#define D2NLT d2nlt_
#define NLD2NLT nld2nlt_
#define T2NLD t2nld_
#define TSMOOTH tsmooth_
#define XSMOOTH xsmooth_
#define VELEXTR velextr_
#define INFILL infill_
#define VELBLOCK velblock_
#endif

void davg2tavg();
void rms2int_t2d();
void psmooth();
void int2rms();
void linterp();
void nl2l();
void int2avg();
void avg2int();
void tlp2tint();
void nltlp2nltint();
void tint2tlp();
void d2nlt();
void nld2nlt();
void t2nld();
void indx2lp();
void pturbvint();
void velextract();
void velpert();
void extractvintz();
void xsmooth();
void xsmoothold();
void tsmooth();
void infill();
void velblock();

/*##################################################################*/
void DAVG2TAVG(vavg,depth,nd,time)
float *vavg,*depth,*time;
int *nd;
    {
    davg2tavg(vavg,depth,*nd,time);
    }
/*###################################################################*/
void davg2tavg(vavg,depth,nd,time)
float *vavg,*depth,*time;
int nd;
{
  int i;

  for (i=0; i<nd; i++)
    time[i]=(vavg[i]!=0.0)?2.0*depth[i]/vavg[i]:0.0;
}

/*##################################################################*/
void RMS2INT(vrms,nt,dt,dz,nz,vint)
float *vrms,*vint;
float *dt,*dz;
int *nt,*nz;
    {
    rms2int_t2d(vrms,*nt,*dt,*dz,*nz,vint);
    }
/*###################################################################*/
void rms2int_t2d(vrms,nt,dt,dz,nz,vint)
float *vrms,*vint;
float dt,dz;
int   nt,nz;
    {
    float *depth,*vel,vel2,buffer[4086];
    int i,it,ix,iz;

/*
    depth = (float *)malloc(8*nt*(sizeof(float)));
    vel = (float *)malloc(8*nt*(sizeof(float)));
*/
    depth = buffer;
    vel = &buffer[nt+1];

    vel[0] = vrms[0];
    depth[0] = 0;


    for(it=1; it<nt; it++){
        vel2= vrms[it]*(vrms[it])*it-
                vrms[it-1]*(vrms[it-1])*(it-1);
        
        vel[it]= (vel2>0)? sqrt(vel2): 0;
        depth[it]= vel[it]*dt*.5+depth[it-1];
        }

    linterp(vel,depth,nt,vint,dz,nz);

/*
    free(depth);
    free(vel);
*/

    }
/*###################################################################*/
void PSMOOTH(v,n)
float *v;
int *n;
    {   
    psmooth(v,*n);
    }   
/*#####################################################################*/
void psmooth(y,n)
float *y;
int n;
    {   
    float *ynew,x0,y0,x1,y1,c0,c1,a;
    int i,k;
 
    ynew = (float *)malloc(n*sizeof(float));

    i=0;
    x1=0;
    y1=y[0];

    while(x1<n-2){
        x0=x1;
        y0=y1;
      
        while(y[i]==y[i+1] && i<n-1 
              )i++;
        y1=(i<n-1)? (y[i]+y[i+1])/2.: y[n-1];
        x1 = i+.5;

        for(k=x0; k<x1; k++){
            if((x1-x0)!=0){
                c0=(x1-k)/(x1-x0);
                c1=(k-x0)/(x1-x0);
                }
            else{
                c0=.5;
                c1=.5;
                }
            ynew[k]=c0*y0+c1*y1;
            }
        i++;
        }
 
    for(i=0; i<n; i++){
        y[i] = ynew[i];
        }
 
    free(ynew);
    }  
            
              
/*#####################################################################*/
void int2rms(vint,nt,dt,dz,nz,vrms)
float *vrms,*vint;
float dt,dz;
int nt,nz;
    {
    int it;
    float vrms2;

    vrms2=0;
    vrms[0]=vint[0];
    for(it=1; it<nt; it++){
        vrms2 += vint[it]*vint[it];
        vrms[it] = sqrt(vrms2/it);
        }
    }

/*####################################################################*/
void LINTERP(vel,depth,ndepth,newvel,dz,nz)
float *vel,*depth,*newvel,*dz;
int *ndepth,*nz;
    {
    linterp(vel,depth,*ndepth,newvel,*dz,*nz);
    }
/*####################################################################
   linterp interpolates a nonuniformly sampled interval velocity
   function in depth into a evenly sampled interval velocity function

   INPUT:
   vel[]  = interval velocity function.
   depth[] = depths corresponding to vel[].
   ndepth = integer number of samples in vel[] and depth[]
   dz = output depth sample interval.
   nz = output number of depth samples.
   
   OUTPUT:
   newvel[] = interval velocity array containing nz samples and 
          sampled at a depth interval of dz.
####################################################################*/
void linterp(vel,depth,ndepth,newvel,dz,nz)
float *vel,*depth,*newvel,dz;
int ndepth,nz;
    {  
    int i,iz; 
    float z1,z2,c1,c2; 
 
    iz=0;
    while(iz*dz < depth[0]){ newvel[iz]=vel[0];
        iz++;}



    for(i=0; i<ndepth-1; i++){ 
        z1 = depth[i]; 
        z2 = depth[i+1]; 
        while(iz*dz<z2 && iz<nz && iz*dz>=z1){         
            c1 = (z2-iz*dz)/(z2-z1); 
            c2 = (iz*dz-z1)/(z2-z1); 
            newvel[iz]=c1*vel[i] + c2*vel[i+1]; 
            iz++; 
            }   
        }  

    while(iz<nz){ newvel[iz]=vel[ndepth-1]; 
        iz++;}
             
    } 

/*####################################################################*/
void NL2L(vel,time,ntime,newvel,dt,nt)
float *vel,*time,*newvel,*dt;
int *ntime,*nt;
    {
    nl2l(vel,time,*ntime,newvel,*dt,*nt);
    }
/*####################################################################
   nl2l converts a nonuniformly sampled interval velocity
   vs. time/depth function into a evenly sampled interval velocity
   vs. time/depth function by prorating the nonuniform intervals into
   the uniform intervals which preserves the travel time over
   the new interval, unlike linterp which has absolutely no
   regard for travel times and should be executed by firing squad

   INPUT:
   vel[]  = interval velocity function.
   time[] = times/depths corresponding to vel[].
   ntime = integer number of samples in vel[] and time[]
   dt = output time/depth sample interval.
   nt = output number of time/depth samples.
   
   OUTPUT:
   newvel[] = interval velocity array containing nt samples and 
          sampled at a time/depth interval of dt.
####################################################################*/
void nl2l(vel,time,ntime,newvel,dt,nt)
float *vel,*time,*newvel,dt;
int ntime,nt;
{  
  int i,it,top=1,bot=1;
  float t1,t2,velweight;
 
  newvel[0]=vel[0];

  for (it=1; it<nt; it++) {
    t1=(it-1)*dt;
    t2=it*dt;
    if (t2 > time[ntime-1]) {
      newvel[it]=vel[ntime-1];
    } else {
      for (i=top; time[i]<t1 && i<ntime; i++) top=bot=i+1;
      for (i=top; time[i]<t2 && i<ntime; i++) bot=i+1;
      if (top == bot) {
        newvel[it]=vel[top];
      } else {
        velweight=vel[top]*(time[top]-t1);
        for (i=top+1; i<bot; i++) {
          velweight=velweight+(vel[i]*(time[i]-time[i-1]));
        }
        velweight=velweight+(vel[bot]*(t2-time[bot-1]));
        newvel[it]=velweight/dt;
      }
    }
  }
} 

/*####################################################################
   int2avg converts an array of interval velocities into average
   velocities at the same depth locations.
   
   INPUT:
   vint[]  = array of interval velocities.
   depth[] = array of depths corresponding to vint[] and vavg[].
   ndepth  = integer number of samples of depth[], vint[] and vavg[].

   OUTPUT:
   vavg[]  = array of average velocities.
####################################################################*/

void int2avg(vint,depth,ndepth,vavg)
float *vint,*vavg,*depth;
int ndepth;
    {
    int i,it,ix,iz;
    float *time,v;

    time = (float *) malloc(ndepth*sizeof(float));

    /*----------calculate average velocity-------------------------*/
    time[0]=(vint[0]!=0)? 2.*depth[0]/vint[0]: 0;
    vavg[0]=vint[0];
    for(iz=1; iz<ndepth; iz++){
        if (vint[iz]!=0) v=vint[iz];
        time[iz]=time[iz-1] + 2.*(depth[iz]-depth[iz-1])/v;

        if(time[iz]==0){
            vavg[iz]=vavg[iz-1];
            }  
        else{
            vavg[iz] = (v*(time[iz]-time[iz-1])+vavg[iz-1]*time[iz-1])/
                                      time[iz];
            }
        }
    }
/*####################################################################
   avg2int converts an array of average velocities into an array           
   of interval velocities at the same depth locations.

   INPUT:
   vavg[]  = array of average velocities.
   depth[] = array of depths corresponding to vint[] and vavg[].
   ndepth  = integer number of samples of depth[], vint[] and vavg[].

   OUTPUT:
   vint[]  = array of interval velocities.
####################################################################*/
void AVG2INT(vavg,depth,ndepth,vint)
float *vavg,*depth,*vint;
int *ndepth;
    {
    avg2int(vavg,depth,*ndepth,vint);
    }
/*********************************************************************/
void avg2int(vavg,depth,ndepth,vint)
float *vint,*vavg,*depth;
int ndepth;
    {
    int i,it,ix,iz;
    float time;

    /*--------convert depth axis to a time axis-----------*/
    time = (vavg[0]!=0)? 2*depth[0]/vavg[0]: 0;
    vint[0]=vavg[0];

    for(iz=1; iz<ndepth; iz++){
        if(depth[iz]!=vavg[iz]*time*.5)
            vint[iz]= (depth[iz]-depth[iz-1])*vavg[iz]/
                   (depth[iz]-vavg[iz]*time*.5);
        else{
            vint[iz]=vint[iz-1];
            }

        if(vint[iz]!=0)
            time += 2*(depth[iz]-depth[iz-1])/vint[iz];

        }
    }

/*####################################################################*/
void TLP2TINT(vlp,nt,vint,p)
float *vlp,*vint,*p;
int *nt;
    {
    tlp2tint(vlp,*nt,vint,*p);
    }
/*####################################################################
  tlp2tint converts an array of lp norm velocities as a function
  of time into interval velocities as a function of time.
  
  INPUT:
  vlp[]   = array of lp norm velocities.
  nt      = number of elements in vlp[] and vint[]

  OUTPUT:
  vint[]  = array of interval velocities.
###################################################################*/
void tlp2tint(vlp,nt,vint,p)
float *vlp,*vint,p;
int nt;
    {
    int it;
    float temp,pinv;

    pinv = 1./p;

    vint[0]=vlp[0];
    for(it=1; it<nt; it++){
        temp = it*pow(vlp[it],p)- (it-1)*pow(vlp[it-1],p);
        if(temp<0)temp =0;
        vint[it]= pow(temp,pinv);
        }
    }

/*####################################################################*/
void NLTLP2NLTINT(vlp,time,nt,vint,p)
float *vlp,*time,*vint,*p;
int *nt;
    {
    nltlp2nltint(vlp,time,*nt,vint,*p);
    }
/*####################################################################
  nltlp2nltint converts an array of lp norm velocities as a function
  of time into interval velocities as a function of time, just like
  tlp2tint, except the sampling is non-linear.
  
  INPUT:
  vlp[]  = array of lp norm velocities.
  time[] = array of times/depth
  nt     = number of elements in vlp[], time[] and vint[]

  OUTPUT:
  vint[]  = array of interval velocities.
###################################################################*/
void nltlp2nltint(vlp,time,nt,vint,p)
float *vlp,*time,*vint,p;
int nt;
    {
    int it;
    float temp,pinv;

    pinv = 1./p;

    vint[0]=vlp[0];
    for(it=1; it<nt; it++){
        temp=(time[it]*pow(vlp[it],p)-time[it-1]*pow(vlp[it-1],p)) /
             (time[it]-time[it-1]);
        if(temp<0)temp=0;
        vint[it]= pow(temp,pinv);
        }
    }

/*####################################################################*/
void TINT2TLP(vint,nt,vavg,p)
float *vint,*vavg,*p;
int *nt;
    {
    tint2tlp(vint,*nt,vavg,*p);
    }
/*####################################################################
  tint2tlp converts an array of interval velocities as a function
  of time into average velocities as a function of time.
  
  INPUT:
  vint[]  = array of interval velocities.
  nt      = number of elements in vavg[] and vint[]

  OUTPUT:
  vavg[]  = array of average velocities.
###################################################################*/
void tint2tlp(vint,nt,vavg,p)
float *vavg,*vint,p;
int nt;
    {
    int it;
    float pinv;

    pinv = 1./p;

    vavg[0]=vint[0];
    for(it=1; it<nt; it++){
        vavg[it]=pow((( pow(vavg[it-1],p)*(it-1)+pow(vint[it],p))/it),pinv);

        }
    }
/*##############################################################*/
void D2NLT(vint,n,dz,time)
float *vint,*dz,*time;
int *n;
    {
    d2nlt(vint,*n,*dz,time);
    }

/*##############################################################*/
void d2nlt(vint,n,dz,time)
float *vint,dz,*time;
int n;
    {
    float v;
    int i;

    v=0;
    time[0]=0;
    
    for(i=1; i<n; i++){
        if(vint[i]!=0) v=vint[i];
        time[i]=(v!=0)? time[i-1]+2.*dz/v: time[i-1];
        }
    }
/*##############################################################*/
void NLD2NLT(vint,depth,ndepth,time)
float *vint,*depth,*time;
int *ndepth;
    {
    nld2nlt(vint,depth,*ndepth,time);
    }
/*##############################################################
   nld2nlt takes an interval velocity array and a corresponding depth
   array and determines a corresponding time array.
   INPUT:
   vint[]  = array of interval velocities.
   depth[] = array of depths corresponding to vint[].
   ndepth  = integer number of values in vint[], depth[] and time[].
  
   OUTPUT:
   time[]  = array of time values corresponding to depth[] and vint[].   
##############################################################*/
void nld2nlt(vint,depth,ndepth,time)
float *vint,*depth,*time;
int ndepth;
{
  int i;
    
  time[0]= (vint[0]!=0)? 2.*depth[0]/vint[0]: 0;
  for(i=1; i<ndepth; i++)
     time[i] = (vint[i]!=0)? time[i-1]+2.*(depth[i]-depth[i-1])/vint[i]:0;
}  
/*################################################################*/
void T2NLD(vint,nt,dt,depth)
float *vint,*depth,*dt;
int *nt;
    {
    t2nld(vint,*nt,*dt,depth);
    }
/*################################################################
  t2nld takes an array of interval velocities, vint, as a function
  of a linear time scale of sample interval, dt, and converts it to
  a depth scale for the same intervals.  The depth scale is then
  nonlinear.
####################################################################*/
void t2nld(vint,nt,dt,depth) 
float *vint,*depth,dt;
int nt;
    {
    int i;
  

    depth[0] = 0;
    for(i=1; i<nt; i++){
         depth[i] = depth[i-1]+vint[i]*dt*.5;
         }
    }
/*######################################################################
  indx2lp  takes an array of indexes which represent trace numbers of
  picks from a cube of data.  It also uses 
  INPUT:
  index[] = Indexes of trace numbers which represent picks versus time
  nt      = number of values in index[] and vavg[]
  np      = number of possible index values for picks.  (The number
            of perturbations of the velocity function)
  vavgpert[] = packed 2 dimensional array containing np different
               velocity function perurbations each of length nt.
  vavg[]  = average velocity as a function of a linear time scale of
####################################################################*/
void indx2lp(index,nt,np,vavgpert,vavg,p)
float *index,*vavgpert,*vavg,p;
int nt,np;
    {
    int it,i1,i2,k1,k2;
    float pinv,temp;

    pinv = 1./p;

    for(it=0; it<nt; it++){
        i1 = index[it];
        i2 = i1 + 1;
        k1 = (i1-1)*nt + it;
        k2 = (i2-1)*nt + it;
        vavg[it] = (index[it]-i1)*vavgpert[k2]+(i2-index[it])*vavgpert[k1];
        }
    }
/*########################################################################
  pturbvint uses an array of interval velocities on a linear scale and
  calculates np perturbations of the interval velocity function.

  INPUT:
  vint[]   = array of interval velocities of length nv
  nv       = number of alements in vint[] and each perturbed velocity
             function in vintpert[].
  np       = number of perturbations to calculate.
  begp     = beginning percentage of vint[] for first perturbation.
  dp       = Incremental percentage amount between each perturbation.
  OUTPUT:
  vintpert[] = packed 2 dimensional array of np velocity function
             perturbations, each of length nv.
######################################################################*/
void pturbvint(vint,nv,np,begp,dp,vintpert)
float *vint,begp,dp,*vintpert;
int nv,np;
    {
    int i,k,ip;
    float a;
    
    for(k=0; k<np; k++){
        a = (begp + k*dp)/100.;
        for(i=0; i<nv; i++){
            ip = k*nv + i;
            vintpert[ip] = vint[i]*a;
            }
        }
    }

/*#########################################################################*/
/*#########################################################################*/
void EXTVINTZ(velz,depth,ndepth,np,begptrb,dptrb,index,nx,nt
          ,dt,nz,dz,vintz,vavgpertt,
         trweight,ntrsmoo,zweight,nzsmoo,outtype,p)
float *velz,*depth,*begptrb,*dptrb,*index,*dt,*dz,*vintz,*vavgpertt;
int *ndepth,*np,*nx,*nt,*nz;
float *trweight,*zweight,*p;
int *ntrsmoo,*nzsmoo,*outtype;
    {
    int j,ix,k;

    if(vavgpertt==0)printf("malloc error , vavgpertt\n");

    velpert(velz,depth,*ndepth,*np,*begptrb,*dptrb,*dt,*nt,vavgpertt,*p);

/*
for(ix=0; ix<(*nx); ix++){
for(j=0; j<(*nz); j++){vintz[ix*(*nz)+j]=vavgpertt[ix*(*nt)+j];}
    }
*/

    extractvintz(velz,depth,*ndepth,vavgpertt,*np,index,*nx,
                 *nt,*dt,*nz,*dz,vintz
                 ,*trweight,*ntrsmoo,*zweight,*nzsmoo,*outtype,*p);

    }
/*#########################################################################*/
/*#########################################################################*/
void VELEXTR(nlvavgpert,time,nzseg,np,vavgpertt,index,nx,nt
          ,dt,nz,dz,vintz,
         trweight,ntrsmoo,zweight,nzsmoo,outtype,p)
float *index,*dt,*dz,*vintz,*nlvavgpert,*vavgpertt,*time;
int *np,*nx,*nt,*nz,*nzseg;
float *trweight,*zweight,*p;
int *ntrsmoo,*nzsmoo,*outtype;
    {


    velextract(nlvavgpert,time,*nzseg,*np,vavgpertt,index,*nx,
                 *nt,*dt,*nz,*dz,vintz
                 ,*trweight,*ntrsmoo,*zweight,*nzsmoo,*outtype,*p);


    }
/*#####################################################*/



void velextract(nlvavgpert,time,nzseg,np,vavgpertt,index,nx,nt,dt,nz,dz,vintz,
       trweight,ntrsmoo,zweight,nzsmoo,outtype,p)
float *nlvavgpert,*vavgpertt,*index,dt,dz,*vintz,*time;
int nzseg,np,nx,nt,nz;
float trweight,zweight,p;
int ntrsmoo,nzsmoo,outtype;
{
    float *vavgt,*vintt,*z;
    int ip,i,ip1,ip2,ip3,ix;
    int j,it,ix0,one=1,nxnt=nx*nt;

/* zdwn04 18 oct 94, the vavgpertt array might not be big enough to do this
    vavgt=&vavgpertt[np*nt];
    vintt=&vavgpertt[nt*(np+1)];
    z=&vavgpertt[nt*(np+2)];
*/
    vavgt=(float *)malloc(nt*sizeof(float));
    vintt=(float *)malloc(nt*sizeof(float));
    z    =(float *)malloc(nt*sizeof(float));

    /*......resample average velocity model to a linear time scale......*/
    /*......which corresponds to the time scale of index[]..............*/
    for (ip=0; ip<np; ip++)
        linterp(&nlvavgpert[ip*nzseg],time,nzseg,&vavgpertt[ip*nt],dt,nt);

    for (ix=0; ix<nx; ix++) {
        indx2lp(&index[ix*nt],nt,np,vavgpertt,vavgt,p);
        if(nzsmoo>0) {
            VRECIP(vavgt,&one,vavgt,&one,&nt);
            tsmooth(vavgt,nt,zweight,nzsmoo);
            VRECIP(vavgt,&one,vavgt,&one,&nt);
        }
        for (it=0; it<nt; it++)
            index[ix*nt+it]=vavgt[it];
        /*.......output average velocity vs time.......*/
        if(outtype==3)
            for (it=0; it<nt; it++)
                vintz[ix*nt+it]=vavgt[it];
    }
            
    if (ntrsmoo>0) {
      VRECIP(index,&one,index,&one,&nxnt);
      xsmooth(index,nx,nt,trweight,ntrsmoo);
      VRECIP(index,&one,index,&one,&nxnt);
    }

    for (ix=0; ix<nx; ix++) {
        tlp2tint(&index[ix*nt],nt,vintt,p);
        t2nld(vintt,nt,dt,z);
        if (outtype==2)
            /*......output average velocity vs depth......*/
            linterp(&index[ix*nt],z,nt,&vintz[ix*nz],dz,nz);
        if (outtype==0)
            /*......output interval velocity vs depth....*/
            nl2l(vintt,z,nt,&vintz[ix*nz],dz,nz);
        if (outtype==1)
            /*......output interval velocity vs time......*/
            for (it=0; it<nt; it++)
                vintz[ix*nt+it]=vintt[it];
    }

/*
    for (ix=0; ix<nx; ix++)
        for (it=0; it<nt; it++)
            printf("%3d %3d %4d\n",ix,it,(int)vintz[ix*nt+it]);
*/

    free(vavgt);
    free(vintt);
    free(z);
}

/*#####################################################*/
/*#####################################################################*/
/*###########################################################*/
void velpert(velz,depth,ndepth,np,begptrb,dptrb,dt,nt,vavgpertt,p)
float *velz,*depth,begptrb,dptrb,dt,*vavgpertt,p;
int np,nt,ndepth;
    {
    int ip,i;
    float *vintpertnlz,*vintpertt,*time;

    vintpertnlz = &vavgpertt[np*nt];
    vintpertt = &vavgpertt[(nt+ndepth)*(np)];
    time = &vavgpertt[(nt*2+ndepth)*(np)];

    pturbvint(velz,ndepth,np,begptrb,dptrb,vintpertnlz);

    for(ip=0; ip<np; ip++){
        nld2nlt(&vintpertnlz[ip*ndepth],depth,ndepth,time);
 
        linterp(&vintpertnlz[ip*ndepth],time,ndepth,
                       vintpertt,dt,nt);

 
        tint2tlp(vintpertt,nt,&vavgpertt[ip*nt],p);
        }
      
    }

/*#####################################################*/
void extractvintz(velz,depth,ndepth,vavgpertt,np,index,nx,nt,dt,nz,dz,vintz,
       trweight,ntrsmoo,zweight,nzsmoo,outtype,p)
float *velz,*depth,*vavgpertt,*index,dt,dz,*vintz;
int ndepth,np,nx,nt,nz;
float trweight,zweight,p;
int ntrsmoo,nzsmoo,outtype;
    {
    float *vavgt,*vintt,*z;
    int ip,i,ip1,ip2,ip3,ix;
    int j,it;

printf("extractvintz- trweight=%f ntrsmoo=%d zweight=%f nzsmoo=%d\n",
        trweight,ntrsmoo,zweight,nzsmoo);

    vavgt = &vavgpertt[np*nt];
    vintt = &vavgpertt[nt*(np+1)];
    z = &vavgpertt[nt*(np+2)];


    for(ix=0; ix<nx; ix++){
        indx2lp(&index[ix*nt],nt,np,vavgpertt,vavgt,p);
        if(nzsmoo>0) tsmooth(vavgt,nt,zweight,nzsmoo);
        for(it=0; it<nt; it++)index[ix*nt+it]=vavgt[it];
        /*.......output average velocity vs time.......*/
        if(outtype==3){
            for(it=0; it<nt; it++)vintz[ix*nt+it]=vavgt[it];
            }
        }
            

    if(ntrsmoo>0) xsmooth(index,nx,nt,trweight,ntrsmoo);



   
    for(ix=0; ix<nx; ix++){

        tlp2tint(&index[ix*nt],nt,vintt,p);

        t2nld(vintt,nt,dt,z);
      
        if(outtype==2){
            /*......output average velocity vs depth......*/
            linterp(&index[ix*nt],z,nt,&vintz[ix*nz],dz,nz);
            }
        if(outtype==1){
            /*......output interval velocity vs time......*/
            for(it=0; it<nt; it++){
                vintz[ix*nt+it]=vintt[it];
                }
            }
        if (outtype==0){
            /*......output interval velocity vs depth....*/
            linterp(vintt,z,nt,&vintz[ix*nz],dz,nz);
            }

        }

    }
/*#####################################################*/
/*#####################################################################*/
void XSMOOTH(a,nx,nt,weight,nsmooth)
float *a,*weight;
int *nx,*nt,*nsmooth;
    {
    xsmooth(a,*nx,*nt,*weight,*nsmooth);
    }
/*######################################################################*/
void xsmooth(a,nx,nt,weight,nsmooth)
float *a,weight;
int nx,nt,nsmooth;
    {
    float a1,a2,a3,c1,c2,c3;
    int ia1,ia2,ia3,i,ismooth,ix,it;
    float smooth,alpha,k1,k2,k3;  

    c1=(1-weight)/2.;
    c2=weight;
    c3=c1;


    ismooth = 0;

    for(it=0; it<nt; it++){
        ismooth = it*nsmooth/nt;
        for(i=0; i<ismooth; i++){
            ia2 = it;
            ia3 = it+nt;
            a2 = a[ia2];
            a3 = a[ia3];
            a[ia2] = c1*a2 + c2*a2 + c3*a3;

            for(ix=1; ix<nx-1; ix++){
                a1=a2;
                a2=a3;
                ia2 = ia3;
                ia3 = it + (ix+1)*nt;
                a3 = a[ia3];
                a[ia2] = c1*a1 + c2*a2 + c3*a3;
                }

            a1=a2;
            a2=a3;
            ia2 = ia3;
            a[ia2] = c1*a1 + c2*a2 + c3*a2;
            }


            smooth = (1.*it*nsmooth)/(1.*nt);
            alpha = smooth - ismooth;
            k2 = c2*alpha + 1.-alpha; 
            k1 = (1-k2)/2.;
            k3 = k1;

            ia2 = it;
            ia3 = it+nt;
            a2 = a[ia2];
            a3 = a[ia3];
            a[ia2] = k1*a2 + k2*a2 + k3*a3;

            for(ix=1; ix<nx-1; ix++){
                a1=a2;
                a2=a3;
                ia2 = ia3;
                ia3 = it + (ix+1)*nt;
                a3 = a[ia3];
                a[ia2] = k1*a1 + k2*a2 + k3*a3;
                }

            a1=a2;
            a2=a3;
            ia2 = ia3;
            a[ia2] = k1*a1 + k2*a2 + k3*a2;
        }    
    }    

/*######################################################################*/
void xsmoothold(a,nx,nt,weight,nsmooth)
float *a,weight;
int nx,nt,nsmooth;
    {
    float a1,a2,a3,c1,c2,c3;
    int ia1,ia2,ia3,i,ismooth,ix,it;

    c1=(1-weight)/2.;
    c2=weight;
    c3=c1;

    for(it=0; it<nt; it++){
        ismooth = it*nsmooth/nt;
        for(i=0; i<ismooth; i++){
            ia2 = it;
            ia3 = it+nt;
            a2 = 1./a[ia2];
            a3 = 1./a[ia3];
            a[ia2] = 1./(c1*a3 + c2*a2 + c3*a3);

            for(ix=1; ix<nx-1; ix++){
                a1=a2;
                a2=a3;
                ia2 = ia3;
                ia3 = it + (ix+1)*nt;
                a3 = 1./a[ia3];
                a[ia2] = 1./(c1*a1 + c2*a2 + c3*a3);
                }

            a1=a2;
            a2=a3;
            ia2 = ia3;
            a[ia2] = 1./(c1*a1 + c2*a2 + c3*a1);
            }
        }    
    }    

/*######################################################################*/
void TSMOOTH(a,nt,weight,nsmooth)
float *a,*weight;
int *nt,*nsmooth;
    {
    tsmooth(a,*nt,*weight,*nsmooth);
    }
/*######################################################################*/
void tsmooth(a,nt,weight,nsmooth)
float *a,weight;
int nt,nsmooth;
    {
    float a1,a2,a3,c1,c2,c3;
    int ia1,ia2,ia3,i,ismooth,it,itbeg;

    c1=(1-weight)/2.;
    c2=weight;
    c3=c1;

    for(i=0; i<nsmooth; i++){
        itbeg = nt*i/nsmooth;
        ia3 = itbeg;
        a2 = (itbeg>0)? 1./a[itbeg-1]: 1./a[1];
        a3 = 1./a[itbeg];
        for(it=itbeg; it<nt-1; it++) {
            a1=a2;
            a2=a3;
            ia2 = ia3;
            ia3++;
            a3 = 1./a[ia3];
            a[ia2] = 1./(c1*a1 + c2*a2 + c3*a3);
        }
        /*........handle last point separately.......*/  
        a1=a2;
        a2=a3;
        ia2 = ia3;
  /*      a[ia2] = 1./(c1*a1 + c2*a2 + c3*a1);*/
        }
    }
/*############################################################*/
/*############################################################*/
/*  INFILL will replace and zero values in array a and fill   */
/*  them by linear interpolation.  The interpolation weights  */
/*  are based upon relative values in depth() array.          */
/*############################################################*/
void INFILL(a,depth,n)
float *a,*depth;
int *n;
    {
    infill(a,depth,*n);
    }
/*############################################################*/
void infill(a,depth,n)
float *a,*depth;
int n;
    {
    int i,ilast,inext;
    float anext,alast,c1,c2,deltaz;
    
    i=0;
    ilast=0;
    inext = 0;

    while(a[ilast]==0. && (ilast++)<n);
    inext=ilast;

    while(i<n){
        if(inext<=i){
            ilast = inext;
            
            while(++inext<n && a[inext]==0.);
            if(a[inext]==0.) inext = ilast;
            }
        if(a[i]==0.){
            deltaz = depth[inext]-depth[ilast];
           
            c1 = (deltaz!=0.)? (depth[inext]-depth[i])/deltaz: .5;
            c2 = (deltaz!=0.)? (depth[i]-depth[ilast])/deltaz: .5;
            a[i]=c1*a[ilast] + c2*a[inext];
            }

        i++;
        }

    }
/*####################################################################
  subroutine velblock takes the input array, vel[], and blocks it 
  so that adjacent values in the array are the same whenever it can
  be done such that the difference between the new value and the
  old value is no greater than dv.  When a segment is blocked it is 
  replaced with the slowness average of the unblocked values.
####################################################################*/
void VELBLOCK(vel,n,dv)
float *vel,*dv;
int *n;
    {
    velblock(vel,*n,*dv);
    }
/*####################################################################*/

void velblock(vel,n,dv)
float *vel,dv;
int n;
    {
    float vmin,vmax,vavg,vsum;
    int i,j,ibeg,iend,nsum;

    i=0;

    while(i<n){
        vmin = vel[i];
        vmax = vel[i];
        vsum = 1./vel[i];
        nsum = 1;
        ibeg = i;
       
        while(vmax-vmin<dv && i<n){
            

            iend = i;
            i++;
            if(vel[i]<vmin) vmin = vel[i];
            if(vel[i]>vmax) vmax = vel[i];
            if(vmax-vmin<dv && i<n && vel[i]!=0){
                vsum += 1./vel[i];
                nsum++;
                }

            }

        vavg = (vsum!=0)? (1.*nsum)/vsum: 0;

        for(j=ibeg; j<=iend; j++){
            vel[j] = vavg;
            }
        }
    }
