C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c Read a velocity field and find the water bottom depth,
c   and calculate the obg according to Marty Traugott's methods,
c   and calculate the densities from the obg's
c   and output either obg field and a density field if desired
c   the obg field can be a pipe out, but the density may not
c ppobgmt.F
c
c Mary Ann Thornton                 V: 1.0              October 05, 1994
c**********************************************************************c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
c-----conv is conversion factor for meters to feet conversion
c-----convg=conversion factor for converting to grams/cubic_centimeter
c-----ppgal is conversion factor for pounds/gallon from psi/ft
      parameter   (conv = 3.280833, convg = .433, ppgal = 0.052)
      integer     lhed (SZLNHD)
      integer * 2 itr  (SZLNHD)
      real        head (SZLNHD)
      real        vel  (SZLNHD), den(SZLNHD), obg(SZLNHD)
      real        watvel, scale, expo, nhg, spsi

      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, luout2, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne

      character   ntap * 100, otap * 100, otap2 * 100, name*7, version*4
      logical     verbos, hlp, query, met, eng
      integer     argis
      equivalence (itr( 1), lhed (1), head(1))
      data lbytes/ 0 /, nbytes/ 0 /, name/'PPOBGMT'/, version/' 1.0'/ 
 
      met = .false.
      eng = .false.
c-----
c     read program parameters from command line card image file
      query = ( argis ( '-?' ) .gt. 0 )
      hlp = ( argis ( '-h' ) .gt. 0 )
      if ( query .or. hlp )then
            call help()
            stop
      endif
c-----
c     open printout
#include <f77/mbsopen.h>
c-----
c     read command line arguments
      call gcmdln(ntap,otap,otap2,ns,ne,irs,ire,met,
     &            eng,scale,spsi,expo,watvel,nhg,verbos)
c-----
c     open input and output files
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)
      if(otap2.ne.' ')then
         call getln(luout2, otap2,'w', 1)
      endif
c-----
c     read line header
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'ppobgmt: no line header read from unit ',luin
         write(LOT,*)'JOB TERMINATED ABNORMALLY'
         stop
      endif
c------
c     save values from line header
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'Dx1000', idx,   LINHED)
      call saver(itr, 'Dz1000', idz,   LINHED)
      dx = idx/1000.
      dz = idz/1000.
      if(dz.le.0.0)then
        write(LOT,*)' Dz1000 in the line header cannot be zero'
        write(LOT,*)' when used for the pore pressure prediction'
        write(LOT,*)' programs.'
        write(LOT,*)' JOB TERMINATED ABNORMALLY'
      endif
c------
c     print line header into printer listing
      call hlhprt(itr, lbytes, name, 7, LERR)
c-----
c     check validity of these arguments
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
      nrecc = ire - irs + 1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr   = ne - ns + 1
      call savew(itr, 'NumTrc', jtr  , LINHED)
c----------------------
c     determine obytes as the number of bytes in output trace
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
c     save command line arguments in the historical part of line header
      call savhlh(itr,lbytes,lbyout)
c----------------------
c     write the output line header
      call wrtape(luout, itr, lbyout )
      if(otap2.ne.' ')then
         call wrtape(luout2, itr, lbyout )
      endif
c-----
      if( verbos ) then
         call verbal(nsamp, nsi, ntrc, nrec, iform,ntap,otap,otap2,
     &               ns,ne,irs,ire,met,eng,watvel,scale,expo,nhg,
     &               spsi,verbos)
      endif

c-----
c--------------------------------------------------
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------

c-----met=metric
c-----conv=conversion factor 
      if(met)then
         convert = conv
      else
         convert = 1.0
      endif
      watvel = watvel*convert
      dz = dz * convert
c-----skip unwanted records
      call recskp(1,irs-1,luin,ntrc,itr)

c-----RECORD LOOP
      do 1000 jj = irs, ire
         call trcskp(jj,1,ns-1,luin,ntrc,itr)
c--------TRACE LOOP
         do 1004  nn = ns, ne
c           read the traces into itr then move
            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:',luin
               write(LERR,*)'  rec= ',jj,'  trace= ',nn
               go to 999
            endif
            call vmov(lhed(ITHWP1),1,vel,1,nsamp)
c-----------SAMPLE LOOP
c----       if velocity is in metric units, convert to feet
c----       Find water bottom depth and set obg = nhg above water bottom
            wd = 0.0
            do 1001 kk = 1,nsamp
               if( (vel(kk)*convert).le.watvel)then
                  wd=(kk-1)*dz
                  obg(kk) = nhg
               endif
 1001       continue
            nstrt = wd / dz + 1.0
c----       compute overburden gradient 
            do 1002 kk = nstrt,nsamp
               z = kk * dz
               obg(kk) =  (( (z-wd)*(spsi + scale*(z-wd/2)**expo)) +
     &                        nhg * wd ) / z
 1002       continue
c----       write the obg trace
            call vmov (obg, 1, lhed(ITHWP1), 1, nsamp)
            call wrtape (luout, itr, obytes)

c----       compute density
            if(otap2.eq.' ')go to 1004
            den(1) = obg(1)/convg
            if(vel(1)*convert .le. watvel) den(1) = 1.03
            do 1003 kk = 2,nsamp 
               den(kk) = (kk*obg(kk) - (kk-1) * obg(kk-1))/convg
               if(vel(kk)*convert .le. watvel) den(kk) = 1.03
 1003       continue
c----       
            if(otap2.ne.' ')then
               call vmov (den, 1, lhed(ITHWP1), 1, nsamp)
               call wrtape (luout2, itr, obytes)
            endif
 1004    continue
c-----
c        skip to end of record
         call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
 1000 continue

c--------------------------------------------------
c--------------------------------------------------
c-----
c     END PROCESSING
c-----
c--------------------------------------------------
c--------------------------------------------------
 
  999 continue
 
c-----
c     close data files
      call lbclos(luin)
      call lbclos(luout)
      if(otap2.ne.' ')call lbclos(luout2)
c-----
      write(LERR,*)'end of ppobgmt, processed ',nrec,' record(s)',
     &               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)'ppobgmt reads a velocity field and outputs ' 
        write(LER,*)'a density field using Gardner equation.'
#ifdef CRAYSYSTEM
        write(LER,*)'See manual pages by typing:   tman ppobgmt '
#else
        write(LER,*)'See manual pages by typing:   man ppobgmt ',
     &              'or:   mman ppobgmt'
#endif
        write(LER,*)'See pattern file by typing:   catpat ppobgmt'
        write(LER,*)'Build an executable script by typing: ',
     &              '     catpat ppobgmt > ppobgmt.job'
        write(LER,*)'Execute ppobgmt by typing ppobgmt and the ',
     &              'program parameters.'
        write(LER,*)'Note that each parameter is proceeded by -a ',
     &              'where "a" is a character(s) corresponding ',
     &              'to some parameter.'
        write(LER,*)'Users enter the following parameters, or use ',
     &              'the default values.'
        write(LER,*)' '
        write(LER,*)'-VEL[ntap] : input velocity field'    
        write(LER,*)'-OBG[otap] : output overburden gradient field'  
        write(LER,*)'-DEN[otap2]: output density field       (Optional)'  
        write(LER,*)'-ns[ns]    : start trace number        (default=1)'    
        write(LER,*)'-ne[ne]    : end trace number        (default=all)'
        write(LER,*)'-rs[irs]   : start record number       (default=1)'
        write(LER,*)'-re[ire]   : end record number       (default=all)'
        write(LER,*)'-wv[watvel]: velocity of water      (default=1480)'
        write(LER,*)'-sp[spsi]  : surface psi            (default=.844)'
        write(LER,*)'-nhg[nhg]  : normal hydrostatic gradient'
        write(LER,*)'                (default for Gulf Coast USA =.465)'
        write(LER,*)'-a [scale ]: Scalar for formula  (default=.000415)'
        write(LER,*)'-b [expo  ]: Exponent for formula  (default=   .6)'
        write(LER,*)'-met       : velocity is in metric units'
        write(LER,*)'-eng       : velocity is in english units'
        write(LER,*)'-V  include on command line for verbose printout'
        write(LER,*)' '
        write(LER,*)'usage:  '
        write(LER,*)'ppobgmt -VEL[ntap] -OBG[otap] -DEN[otap2]'
        write(LER,*)'        -ns[ns] -ne[ne] -rs[irs] -re[ire]'
        write(LER,*)'        -wv[watvel] -sp[spsi] -nhg[nhg]'
        write(LER,*)'        -a[scale] -b[expo] [-met -eng -V] '
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine gcmdln(ntap,otap,otap2,ns,ne,irs,ire,met,
     &            eng,scale,spsi,expo,watvel,nhg,verbos)
c-----
c     get command arguments

c     ntap  - C*100    input file name
c     otap  - C*100    output file name (overburden)
c     otap2 - C*100    output file name (density)
c     ns    - int      starting trace index
c     ne    - int      ending trace index
c     irs   - int      starting record index
c     ire   - int      ending record index
c     spsi  - real     surface psi
c     scale - real     scale factor for formula
c     expo  - real     formula exponent
c     watvel- real     velocity of water  
c     nhg   - real     normal hydrostatic gradient
c     met   - L        flag indicating metric units
c     eng   - L        flag indicating english units
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), otap2*(*)
      integer     ns, ne, irs, ire
      logical     verbos, met,eng
      integer     argis
      real        watvel,scale,expo,spsi,nhg
 
c-------
c     last 2 arguments are values used when:
c     (1) if ONLY the key is present (no value attached to it)
c     (2) if NO key & no value are present
c-------
            call argstr('-VEL', ntap  , ' '   , ' '   )
            call argstr('-OBG', otap  , ' '   , ' '   )
            call argstr('-DEN', otap2 , ' '   , ' '   )
            call argi4 ('-ns' , ns    ,     1 ,      1)
            call argi4 ('-ne' , ne    ,     0 ,      0)
            call argi4 ('-rs' , irs   ,     1 ,      1)
            call argi4 ('-re' , ire   ,     0 ,      0)
            call argr4 ('-sp' , spsi  ,   .844,   .844)
            call argr4 ('-nhg', nhg   ,   .465,   .465)
            call argr4 ('-wv' , watvel,  1480.,  1480.)
            call argr4 ('-a'  , scale ,.000415,.000415)
            call argr4 ('-b'  , expo  ,     .6,     .6)

            verbos =   (argis('-V') .gt. 0)
            met    =   (argis('-met') .gt. 0)
            eng    =   (argis('-eng') .gt. 0)
           
            if(.not.met .and. .not.eng)then
              write(LOT,*) 'Units must be specified' 
              write(LOT,*) 'JOB TERMINATED'
              stop
            endif
c-------
      return
      end
 
C***********************************************************************
C***********************************************************************
C***********************************************************************
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,ntap,otap,otap2,
     &                  ns,ne,irs,ire,met,eng,watvel,scale,expo,nhg,
     &                  spsi,verbos)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c     otap2 - C*100   output file name
c     ns    - Int     starting record
c     ne    - Int     ending record
c     irs   - Int     starting trace
c     ire   - Int     ending trace
c     scale - real    Gardner's scaling constant
c     expo  - real    Gardner's exponenet
c     watvel- real    velocity of water  
c     spsi  - real    surface psi
c     nhg   - real    normal hydtrostatic gradient
c     met   - L       flag indicating metric units
c     eng   - L       flag indicating english units
c-----
#include <f77/iounit.h>
 
      integer     nsamp, nsi, ntrc, nrec,ns,ne,irs,ire
      logical     eng,met
      character   ntap*100, otap*100, otap2*100
      real        watvel,scale,expo,spsi,nhg
 
      write(LERR,*)' '
      write(LERR,*)' line header values after default check '
      write(LERR,*)' # of samples/trace          =  ', nsamp
      write(LERR,*)' sample interval             =  ', nsi
      write(LERR,*)' traces per record           =  ', ntrc
      write(LERR,*)' records per line            =  ', nrec
      write(LERR,*)' format of data              =  ', iform
      write(LERR,*)' input data set name         =  ', ntap
      write(LERR,*)' output overburden gradient  =  ', otap
      write(LERR,*)' output density (optional)   =  ', otap2
      write(LERR,*)' starting record             =  ', irs
      write(LERR,*)' ending record               =  ', ire
      write(LERR,*)' starting trace              =  ', ns
      write(LERR,*)' ending trace                =  ', ne
      write(LERR,*)' water velocity              =  ', watvel
      write(LERR,*)' normal hydrostatic gradient =  ', nhg
      write(LERR,*)' scaling constant            =  ', scale
      write(LERR,*)' exponent in formula         =  ', expo
      if(met)then
         write(LERR,*)' Units are metric '
      endif
      if(eng)then
        write(LERR,*)' Units are English '
      endif
      write(LERR,*)' '
 
      return
      end
 
