C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c-------------------------------------------------------------------------c
c            STACKC -- Perform multicomponent stack on seismic trace data c
c                                                                         c
c     Author:                                                             c
c         P. R. Gutowski
c                                                                         c
c     Description:                                                        c
c         Based on the input values taken from the command line, this     c
c         program will sum all common components within a gather          c
c         to make a single summed trace for the record.  The program      c
c         will continue through all selected records and will produce     c
c         an output file containing a record or records (based on the     c
c         input options) of stacked traces for each component             c
c                                                                         c
c     Variable list:                                                      c
c                 (integers -- all i*4)                                   c
c           LUIN  Logical unit number of input                            c
c          LUOUT  Logical unit number of output                           c
c         NBYTES  Number of bytes in trace record                         c
c         LBYTES  Number of bytes in lineheader record                    c
c          NSAMP  Number of samples in each trace                         c
c            NSI  Sampling interval                                       c
c           NTRC  Number of traces per record                             c
c           NREC  Number of records per line                              c
c            IRS  First record to process                                 c
c            IRE  Last record to process                                  c
c             NS  Starting trace number                                   c
c             NE  Ending trace number                                     c
c          NSKTR  Number of traces to skip                                c
c          IFORM  Input data format                                       c
c         NREOUT  Number of output records                                c
c         NTROUT  Number of output traces                                 c
c                 (logical)                                               c
c         VERBOS  If true, print verbos messages                          c
c          QUERY  If true, print query loop and end program               c
c                 (character)                                             c
c           NTAP  chr*100  Input file name from command line flag         c
c           OTAP  chr*100  Output file name from command line flag        c
c           NAME  chr*4    File name passed to HLH subroutine             c
c                                                                         c
c     Array list:                                                         c
c          ITR (8320)  i*2    Input data character stream                 c
c         LHED (1500)  i*4    Line header information                     c
c          TRI (4096)  r*4    Data trace                                  c
c         STRI (4096)  r*4    Stacked data trace                          c
c                                                                         c
c     Calls:                                                              c
c         (subroutines)                                                   c
c         RTAPE, LBOPEN, HLH, WRTAPE, SAVE, ARGSTR, ARGI4, ERROR          c
c                                                                         c
c         (functions)                                                     c
c         ARGIS                                                           c
c                                                                         c
c-------------------------------------------------------------------------c
c     program stack
c---------------------------------
c        declare variables
c---------------------------------
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/sisdef.h>
#include <f77/lhdrsz.h>
      integer     ITR (SZLNHD), itr0 (SZLNHD)
      integer     ARGIS
      integer     si, gi, di,isrc,igi,idi,lbyout,idists
      integer     LUIN, LUOUT, NBYTES, LBYTES, NSAMP, NSI, NTRC, NREC
      integer     IRS, IRE, NS, NE, NREOUT, NTROUT, IFORM, NSKTR
      real        iexp
c
      real        TRI (SZSMPM), STRI (SZSMPM)
c
      character    name * 6, ntap * 256, otap * 256
#include <f77/pid.h>
c
      logical      verbos, query, norm
c
c
c     equivalence (itr (129), tri (1))
c     equivalence (itr (1), lhed (1))
      data  itr0/SZLNHD*0/, name/'STACKC'/
c------------------------------------
c        initialize variables
c------------------------------------
      nbytes = 0
      lbytes = 0
c-------------------------------------------------------------------
c        If '-?' flag is used in command line, execute query
c        loop and end program.
c-------------------------------------------------------------------
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query ) then
         call help()
         stop
      end if
c  open message file
#include <f77/open.h>
c-----------------------------------------------------------------------------
c        Read program input parameters from command line argument flags
c------------------------------------------------------------------------------
      call cmdln(ntap,otap,ns,ne,irs,ire,d2neg,d1neg,d1pos,d2pos,ntpr,
     1           iexp,igath,norm,verbos)
c     if ( verbos ) then
         write(LERR,*) ' Values read from command line'
         write(LERR,*) ' Input data set name     =  ', ntap
         write(LERR,*) ' Output data set name    =  ', otap
         write(LERR,*) ' First record to process =  ', irs
         write(LERR,*) ' Last record to process  =  ', ire
         write(LERR,*) ' Starting trace number   =  ', ns
         write(LERR,*) ' Ending trace number     =  ', ne
         write(LERR,*) ' Output traces/rec       =  ', ntpr
         write(LERR,*) ' Nth root power          =  ', iexp
         write(LERR,*) ' Negative side spread: far distance =  ',d2neg
         write(LERR,*) ' Negative side spread: near distance =  ',d1neg
         write(LERR,*) ' Positive side spread: far distance =  ',d2pos
         write(LERR,*) ' Positive side spread: near distance =  ',d1pos
c     end if
c-------------------------------------------
c        Open input and output files
c-------------------------------------------
      call getln( luin, ntap, 'r', 0)
      call getln(luout, otap, 'w', 1)
c----------------------------------------------------
c        Read line header and save parameters
c----------------------------------------------------
      write(LERR,*)'luin= ',luin,' luout= ',luout
      lbytes=0
      call RTAPE (luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LERR,*)'STACK: no header read on unit ',luin
         write(LERR,*)'FATAL'
         stop
      endif
      call HLHprt(itr, lbytes, name, 6, LERR)
      call saver(itr, 'NumSmp', nsamp , LINHED)
      call saver(itr, 'SmpInt', nsi   , LINHED)
      call saver(itr, 'NumTrc', ntrc  , LINHED)
      call saver(itr, 'NumRec', nrec  , LINHED)
      call saver(itr, 'Format', iform , LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('InTrCn',ifmt_InTrCn,l_InTrCn,ln_InTrCn,TRACEHEADER)

c
      if(igath .eq. 0) igath = ntrc
      if ( verbos ) then
         write(LERR,*) ' Values read from input data set lineheader'
         write(LERR,*) ' Number of Samples/Trace =  ', nsamp
         write(LERR,*) ' Sample Interval         =  ', nsi
         write(LERR,*) ' Traces per Record       =  ', ntrc
         write(LERR,*) ' Records per Line        =  ', nrec
         write(LERR,*) ' Format of Data          =  ', iform
         write(LERR,*) ' Traces/gather           =  ', igath
      end if
c   fix up null trace
      call savew( itr0, 'StaCor', 30000 , TRCHED)
c---------------------------------------------------------------------------
c        Check bounds of input parameters and set values to defaults
c---------------------------------------------------------------------------
      call cmdchk(ns, ne, irs, ire, ntrc, nrec)
c--------------------------------------------------
c        Modify line header for output file
c--------------------------------------------------
      call savhlh( itr, lbytes, lbyout)
      if(igath .ne. ntrc) then
         if(irs.ne.1 .and. ire.ne. nrec .and. 
     &         ns.ne.1 .and. ne.ne.ntrc) then
            write(LERR,*)'If igath not = input gather size, start/end'
            write(LERR,*)'records & traces must be left to default'
            write(LERR,*)'Rerun with these defaults'
            stop
         endif
         itot = ntrc*nrec
         xre = float(itot)/float(igath)
         ire = xre
         meft = itot - igath*ire
         ire = xre + .99
         write(LERR,*)'xre= ',xre,' ire= ',ire,' meft= ',meft
         irs = 1
         ns = 1
         ne = igath
      endif
      ntrout = ire - irs + 1
      nreout = ne - ns + 1
c---------------------------------------------------------------
c        get number components (src & rcvr) from ntuple entry
c---------------------------------------------------------------
      call saver( itr, 'RATTrc',  ntuple , LINHED)
      call savew( itr, 'RATFld',    0    , LINHED)
      if(igath .ne. ntrc) then
        ntuple = 1
        if(ntpr .gt. ntrout) ntpr = ntrout
        eps = .99
      else
        eps = .99
        ntrc = ntrc/ntuple
        ntrout = ntrout*ntuple
        if(ntpr .le. 1) then
          ntpr = ntuple
        else
          if(mod(ntpr,ntuple) .ne. 0) then
             write(LERR,*)'Number traces/rec not a multiple of number'
             write(LERR,*)'of components  -- rerun with new -tr'
             stop
          endif
        endif
      endif
      if(ntpr .gt. ntrout) ntpr = ntrout
c   number output records
      write(LERR,*)' ntrout= ',ntrout,' nreout= ',nreout,' ntuple= ',
     1 ntuple
      xreco = float(ntrout)/float(ntpr)
      nreco = xreco + eps
      ireco = xreco 
      left = ntrout - ireco*ntpr
      ipad = ntpr - left
      write(LERR,*)'xreco= ',xreco,' ireco= ',ireco,' ipad= ',ipad
      write(LERR,*)'left= ',left,' ntrout= ',ntrout,' ntpr= ',ntpr,
     1             ' nreco= ',nreco
      xleft = igath
      igath = igath/ntuple
      xleft = float(igath)*float(ntuple) - xleft
      write(LERR,*)'xleft= ',xleft,' igath= ',igath
      if(xleft .ne. 0.) then
        write(LERR,*)'Number traces in stacking gather not a multiple'
        write(LERR,*)'of number of components (ntuple)'
        write(LERR,*)'Rerun with new igath or use default igath'
        stop
      endif
      call savew(itr, 'NumTrc' , ntpr  , LINHED)
      call savew(itr, 'NumRec' , nreco , LINHED)
      call WRTAPE (luout, itr, lbyout)
c---------------------------------------------------------------------
c        Skip unwanted records if necessary.  Skipping is done
c        in groups of ntrc traces per record.
c---------------------------------------------------------------------
      call recskp(1,irs-1,luin,ntrc,itr)
c----------------------------------------------------------------------
c                    Main processing loop
c
c        From the first selected record (irs) to last selected
c           record (ire) do the following:
c           (1)  Zero the storage array (stri).
c           (2)  Sum from trace (ns) to trace (ne) all selected
c                traces in each record.
c           (3)  Write the summed trace of that record to the
c                output file.
c----------------------------------------------------------------------
      anth = 1. / iexp
      IC=1
      icc = 0
      irec = 1
      do 200 i = irs, ire
c
c   if no. traces in gather is not input no. traces/gath we may have some traces in input
c   left over, i.e. partial gather
      if(igath .ne. ntrc .and. i .eq. ire) igath = meft
c        ------------------
         do 800 j=1,ntuple

c        ------------------
c        Zero storage array
c        ------------------
            call vclr (stri,1,nsamp)

c            -------------------
c            Sum selected traces
c            -------------------
            inorm = 0
            do 400 K = 1, igath
               nbytes = 0
               call RTAPE (luin, itr, nbytes)
               if(nbytes .eq. 0) then
                  write(LERR,*)'End of file on input:'
                  write(LERR,*)'  rec= ',i,'  trace= ',k
                  go to 999
               endif
               call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

               call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                     istatic, TRACEHEADER)
               call saver2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                     idi    , TRACEHEADER)
               call saver2(itr,ifmt_RecInd,l_RecInd, ln_RecInd,
     1                     igi    , TRACEHEADER)
               call saver2(itr,ifmt_SrcLoc,l_SrcLoc, ln_SrcLoc,
     1                     isi    , TRACEHEADER)
               call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                     idists , TRACEHEADER)
               isrc = isi/10
               if(idi .ne. 0) di = idi
               if(igi .ne. 0) gi = igi
               if(isrc .ne. 0) si = isrc

               if (K .ge. ns .and. K .le. ne) then
                if(istatic .ne. 30000) then
                  dist=idists
c-------------------
c  filter distances
c-------------------
                  if(dist .ge. d2neg .and. dist .le. d1neg .OR.
     1               dist .ge. d1pos .and. dist .le. d2pos) then
                   inorm = inorm + 1

                   if(verbos) then
               call saver2(itr,ifmt_ToStUn,l_ToStUn, ln_ToStUn,
     1                     i15   , TRACEHEADER)
               call saver2(itr,ifmt_ToTmAU,l_ToTmAU, ln_ToTmAU,
     1                     i16   , TRACEHEADER)
               call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                     irec  , TRACEHEADER)
               call saver2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                     itrc  , TRACEHEADER)
                     write(LERR,*)'stacking dist= ',dist,' rec/trc= ',
     1               irec,itrc,' comps= ',i15,i16,'  gath= ',j
                   endif

                      do 500 L = 1, nsamp
                         ampl = tri(L)
                         ampl = sign(1.0,ampl) * (abs(ampl)) ** anth
                         stri (L) = ampl + stri (L)
c                        stri (L) = tri (L) + stri (L)
  500                 continue
                  endif
                endif
               endif
  400       continue

c        ----------------------------------
c        raise stacked trace to nth pwr
c        ----------------------------------
                  do 501 L = 1, nsamp
                     ampl = stri(L)
                     ampl = sign(1.0,ampl) * (abs(ampl)) ** iexp
                     stri (L) = ampl
  501             continue

c        ----------------------------------
c        Write stacked trace to output file
c        ----------------------------------
            if(norm) then
              if(inorm .eq. 0) inorm = 1
              do 600 M = 1, nsamp
                 tri (M) = stri (M)/float(inorm)
  600         continue
            else
              do 601 M = 1, nsamp
                 tri (M) = stri (M)
  601         continue
            endif
          tri(1)=0.
          tri(2)=0.
         icc = icc + 1
          call savew2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                irec , TRACEHEADER)
          call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                icc  , TRACEHEADER)
c         call savew( itr, 'InTrCn', ic , TRCHED)
          call savew2(itr,ifmt_InTrCn,l_InTrCn, ln_InTrCn,
     1                ic     , TRACEHEADER)
c        itr(39)=ic
         if(mod(icc,ntpr) .eq. 0) then
           irec = irec + 1
           icc = 0
         endif
c   put something in incremental trace counter
c         call savew( itr, 'InTrCn', ic , TRCHED)
          call savew2(itr,ifmt_InTrCn,l_InTrCn, ln_InTrCn,
     1                ic     , TRACEHEADER)
c        itr(39)=0
c        itr(40)=ic
         ic=ic+1
c       fix up a nondead stacked gather that happens to end up with a flagged
c       dead trace header (picked up from the last rtape() in the gather
         if(ic .ge. 1 .and. istatic .eq. 30000) istatic=0
               idi = di
               igi = gi
               isi = 10 * si
               call savew2(itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                     idi    , TRACEHEADER)
               call savew2(itr,ifmt_RecInd,l_RecInd, ln_RecInd,
     1                     igi    , TRACEHEADER)
               call savew2(itr,ifmt_SrcLoc,l_SrcLoc, ln_SrcLoc,
     1                     isi    , TRACEHEADER)
               call savew2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                     istatic, TRACEHEADER)
         if(verbos) then
            write(LERR,*)'writing ntuple ',j,' from rec ',i
         endif
         call vmov   (tri, 1, itr(ITHWP1), 1, nsamp)
         call WRTAPE (luout, itr, nbytes)
  800    continue
  200 continue
c  pad the last record if necessary
      if(left .ne. 0) then
        do 700 l = 1, ipad
          icc = icc + 1
               call savew2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                     irec, TRACEHEADER)
               call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                     icc , TRACEHEADER)
          call wrtape(luout,itr0,nbytes)
  700   continue
      endif
c-------------------------------------------
c        Close files and end program
c-------------------------------------------
      call LBCLOS ( luin )
      call LBCLOS ( luout )
      write(LERR,*)'STACK: normal end'
      go to 1000
  999 continue
 1000 continue
      END

      subroutine error(mesgs)
      character*(*) mesgs
#include <f77/iounit.h>
      write(LERR,*)mesgs
      stop
      end
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     1 '***************************************************************'
         write(LER,*)
     1 'Run this program by typing: stackc and the following arguments'
         write(LER,*)
     1 ' -N[ntap]    (no default)      : Input data file name'
         write(LER,*)
     1 ' -O[otap]    (no default)      : Output data file name'
         write(LER,*)
     1 ' -dmin[dmin] starting distance to use'
         write(LER,*)
     1 ' -dmax[dmax] ending distance to use'
         write(LER,*)
     1 ' -rs[irs] starting record (default = 1)'
         write(LER,*)
     1 ' -re[ire] ending record (default = all)'
         write(LER,*)
     1 ' -ns[ns] starting trace (default = 1)'
         write(LER,*)
     1 ' -ne[ne] ending trace   (default = all traces)'
         write(LER,*)
     1 ' -tr[ntpr] output number traces/rec (default = # components)'
         write(LER,*)
     1 ' -pw[iexp] nth root power (default = 1)'
         write(LER,*)
     1 ' -gath[igath] force number of input traces/gather'
         write(LER,*)
     1 ' -S  scale stacked gather by 1/(number live traces)'
         write(LER,*)
     1 ' -V  Verbose mode.  All command line and lineheader parameters'
         write(LER,*)
     1 '                    printed to standard error output'
         write(LER,*)
     : 'USAGE:  ',
     : 'stackc -N[ntap] -O[otap] -rs[irs] -re[ire] -ns[ns] -ne[ne] -V',
     : '       -dmin[] -dmax[] -pw[]'
         write(LER,*)
     1 '***************************************************************'
      return
      end
      subroutine cmdln(ntap,otap,ns,ne,rs,re,d2neg,d1neg,d1pos,d2pos,
     1                 ntpr,iexp,igath,norm,verbos)
c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c     otap  - C*100  output file name
c      ns   - I      start trace
c      ne   - I      stop end trace
c      rs   - I      start record
c      re   - I      stop end record
c    d2neg  - R      far distance to stack: negative side of spread
c    d1neg  - R      near distance to stack: negative side of spread
c    d2pos  - R      far distance to stack: positive side of spread
c    d1pos  - R      near distance to stack: positive side of spread
c    ntpr   - I      number traces per rec on output
c    igath  - I      force number traces in gather to stack
c     iexp  - R      take nth nth root, stack, then nth power
c    norm   - L      normalize stacked trace by # live traces
c    verbos - L      verbose output or not
c-----
      integer     ns,ne,rs,re,ntpr,igath, argis
      real        iexp
      character   ntap*256, otap*256
      logical     verbos, norm
c------------------------------------------------------------------------------
c        ARGXXX has parameters
c             ( flag, variable name, default value, format error value )
c------------------------------------------------------------------------------
      call ARGSTR ('-N', ntap, ' ', ' ' )
      call ARGSTR ('-O', otap, ' ', ' ' )
      call ARGR4  ('-d2neg', d2neg, -99999., -99999.)
      call ARGR4  ('-d1neg', d1neg, -00000., -00000.)
      call ARGR4  ('-d1pos', d1pos,  00000.,  00000.)
      call ARGR4  ('-d2pos', d2pos,  99999.,  99999.)
      call ARGI4  ('-ns', ns, 0, 0)
      call ARGI4  ('-ne', ne, 0, 0)
      call ARGI4  ('-rs', irs, 0, 0)
      call ARGI4  ('-re', ire, 0, 0)
      call ARGI4  ('-tr', ntpr, 1, 1)
      call ARGI4  ('-gath', igath, 0, 0)
      call ARGR4  ('-pw', iexp, 1.0, 1.0)
c
      norm = ( argis ('-S') .gt. 0 )
      verbos = ( argis ('-V') .gt. 0 )
c------------------------------------------------------
c     chech range limiting parameters
c------------------------------------------------------
      if (iexp .eq. 0.0) iexp = 1.0
      if(d1neg .lt. d2neg) then
         write(LERR,*)'d1neg should be near range, negative side of spre
     1ad'
         write(LERR,*)'d2neg should be far  range, negative side of spre
     1ad'
         write(LERR,*)'Check command line arguments & rerun'
         stop
      endif
      if(d2pos .lt. d1pos) then
         write(LERR,*)'d1pos should be near range, positive side of spre
     1ad'
         write(LERR,*)'d2pos should be far  range, positive side of spre
     1ad'
         write(LERR,*)'Check command line arguments & rerun'
         stop
      endif

      return
      end
