C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************

c  routine to build a properly sampled velocity trace at a spatial location
c  between 2 control points and between an upper and lower horizon boundary.
c  Kh is the icurrent LOWER horz.  vel1 & vel2 are the two control velocity
c  functions equally sampled to sample interval "si" (which is our fundamental
c  time sampling.
c---
      subroutine vbuild (s1, s2, si, nsamp, nhor, ndi, nmax, kh, JJ,
     1                   thor1, thort, thor2, vhor1, vhort, vhor2,
     2                   tim1, timt, tim2, vel1, velt, vel2,
     3                   wrk1, wrkt, wrk2, vlast, iz, zz)

      integer  nsamp, nhor, nmax, kh, JJ
      real     thor1 (nhor+2), thort (nhor+2), thor2 (nhor+2)
      real     vhor1 (nhor+2), vhort (nhor+2), vhor2 (nhor+2)
      real     tim1 (nmax), timt (nmax), tim2 (nmax)
      real     vel1 (nmax), velt (nmax), vel2 (nmax)
      real     wrk1 (nmax), wrkt (nmax), wrk2 (nmax)
      real     s1, s2, si

      real        zz(4*nsamp)
      integer     iz(nsamp)

      ts1 = thor1 (kh-1)
      te1 = thor1 (kh)
      vs1 = vhor1 (kh-1)
      ve1 = vhor1 (kh)
      ts2 = thor2 (kh-1)
      te2 = thor2 (kh)
      vs2 = vhor2 (kh-1)
      ve2 = vhor2 (kh)
      delt1 = te1 - ts1
      delt2 = te2 - ts2
      delv1 = ve1 - vs1
      delv2 = ve2 - vs2

c     write(0,*)'Hor,ts1,te1,vs1,ve1,ts2,te2,vs2,ve2= ',
c    1kh,ts1,te1,vs1,ve1,ts2,te2,vs2,ve2
c     write(0,*)'delt1,delt2,delv1,delv2= ',delt1,delt2,delv1,delv2

c----
c   First interpolate very first velocities
c----
      velt (1) = s1 * vel1 (1) + s2 * vel2 (1)


c----
c   upper & lower horizons converge (or are parallel) going left to right. We
c   always choose the SMALLER "delt" as the one sampled with "si". The LARGER
c   "delt" will be resampled to a coarser s.i. as will the vector at the
c   desired output location (defined by the horizontal interpolation factors
c   "s1" & "s2"
c   The picture for this case looks like:

c   < ---------   horozintal location ------>             [ * -- Horz (K-1) ]
c                                                         [ # -- Horz (K)   ]
c        J-1   s1    JJ          s2               J    
c      _|_|...........|...........................||    _             
c  n    | ....*.......|............................|    .             
c  e   _|_............|.*..........................|    .    b        
c  w    | ............|..........*.................|    .    a        
c      _|_............|...................*........|    .    s        
c  s    | ............|...........................*|    .    i       
c  a   _|_............|............................|    .    c        
c  m    | ............|............................|    .             
c  p   _|_............|............................|    .    s        
c  l    | ............|............................|    .    a        
c  i   _|_............|............................|    .    m        
c  n    | ............|...........................#|    .    p       
c  g   _|_............|...................#........|    .    l        
c       | ............|........#...................|    .    e        
c      -|-#..........#|............................|    _             
c----
      IF (delt1 .ge. delt2) THEN


         icinit = 1
         dt2 = si
         dt1 = si * delt1 / delt2
         is2 = nint (ts2 / si)
         ie2 = nint (te2 / si)
         if (is2 .eq. 0) is2 = 1
         ns2 = ie2 - is2 + 1
         is1 = nint (ts1 / si)
         ie1 = nint (te1 / si)
         if (is1 .eq. 0) is1 = 1
         ns1 = ie1 - is1 + 1

         dtt = s1 * dt1 + s2 * dt2
         tst = s1 * ts1 + s2 * ts2
         tet = s1 * te1 + s2 * te2
         is  = nint (tst / dtt)
         if (is .eq. 0) is = 1
         ie  = nint (tet / dtt)
         dts = si / dtt
         ist = nint ( tst / si)
         if (ist .eq. 0) ist = 1
         iet = nint ( tet / si)
         nt  = ie  - is  + 1
         ntt = iet - ist + 1
         if (ntt .gt. ns1) ns1 = ntt

c----
c   build new time sampling for larger "delt". We count from time "ts1" starting
c   from zero and inrementing by the coarse s.i. "dt1" until we exceed te1
c----
         do  i = 1, ns1
             tim1 (i)  = dt1 * float(i-1)
             tim2 (i)  = dt2 * float(i-1)
             wrk1 (i) = 0.
             wrkt (i) = 0.
             wrk2 (i) = 0.
         enddo

c----
c   do cubic spline interpolation of vel1 on to new coarse sampled vector "wrk1".
c   Note that there will be the same number of sampled points on wrk1 as in vel2
c   between the upper and lower horizons.
c----

         call fcuint (tim2, vel1(is1), ns1, tim1, wrk1, ns2,
     1                iz, zz, icinit)

c----
c   Interpolate to find new "coarse" sample interval at output location.
c   also interpolate vectors of equal numbers of velocities on to coarsely
c   sampled output velocity vector. Then do cubic spline interpolation back 
c   onto finer grid.
c----
         do  i = 1, ns2
             wrkt (i) = s1 * wrk1 (i) + s2 * vel2 (i+is2-1)
         enddo


         do  i = 1, ns1
             timt (i)  = dtt * float(i-1)
         enddo

c----
c   Take the interpolated velocity at the current location on the coarse grid
c   and resample back to the "si" grid.
c----
         call fcuint (timt, wrkt, ns2, tim2, velt(ist), ntt,
     1                iz, zz, icinit)


c----
c   upper & lower horizons diverge going left to right
c----
      ELSEIF (delt1 .lt. delt2) THEN

         icinit = 1

         dt1 = si
         dt2 = si * delt2 / delt1
         is1 = nint (ts1 / si)
         ie1 = nint (te1 / si)
         if (is1 .eq. 0) is1 = 1
         ns1 = ie1 - is1 + 1
         is2 = nint (ts2 / si)
         ie2 = nint (te2 / si)
         if (is2 .eq. 0) is2 = 1
         ns2 = ie2 - is2 + 1

         dtt = s1 * dt1 + s2 * dt2
         tst = s1 * ts1 + s2 * ts2
         tet = s1 * te1 + s2 * te2
         is  = nint (tst / dtt)
         if (is .eq. 0) is = 1
         ie  = nint (tet / dtt)
         dts = si / dtt
         ist = nint ( tst / si)
         if (ist .eq. 0) ist = 1
         iet = nint ( tet / si)
         nt  = ie  - is  + 1
         ntt = iet - ist + 1
         if (ntt .gt. ns2) ns2 = ntt
 

c----
c   build new time sampling for larger "delt". We count from time "ts2" starting
c   from zero and inrementing by the coarse s.i. "dt2" until we exceed te2
c----
         do  i = 1, ns2
             tim1 (i) = dt1 * float (i-1)
             tim2 (i) = dt2 * float(i-1)
             wrk1 (i) = 0.
             wrkt (i) = 0.
             wrk2 (i) = 0.
         enddo
 
c----
c   do cubic spline interpolation of vel1 on to new coarse sampled vector "wrk2".
c   Note that there will be the same number of sampled points on wrk1 as in vel1
c   between the upper and lower horizons.
c----


         call fcuint (tim1, vel2(is2), ns2, tim2, wrk2, ns1,
     1                iz, zz, icinit)
 
c----
c   Interpolate to find new "coarse" sample interval at output location.
c   also interpolate vectors of equal numbers of velocities on to coarsely
c   sampled output velocity vector. Then do cubic spline interpolation back 
c   onto finer grid.
c----
         do  i = 1, ns1
             wrkt (i) = s1 * vel1 (i+is1-1) + s2 * wrk2 (i)
         enddo


         do  i = 1, nmax
             timt (i) = 0.
         enddo

         do  i = 1, ns2
             timt (i)  = dtt * float(i-1)
         enddo

c----
c   Take the interpolated velocity at the current location on the coarse grid
c   and resample back to the "si" grid.
c----
         call fcuint (timt, wrkt, ns1, tim1, velt(ist), ntt,
     1                iz, zz, icinit)

 
      ENDIF

c----
c   we keep an average of the velocity at the bottom of the last horizon
c   and the velocity at the top of the current
c----
      if (vlast .gt. 0.) then
          velt (ist) = .5 * (velt(ist) + vlast)
      endif
      vlast = velt (iet)

      return
      end
