C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C
C
C      PROGRAM MODULE taupmig: tau-p migration (post critical)
C
C**********************************************************************C
C      
C   THIS PROGRAM ESSENTIALLY PARALLELS THE DOWNWARD CONTINUATION
C   PAPER BY CLAYTON & MCMECHAN (GEOPH. V.46 P.860-868) FOR REFRACTION
C   WAVEFIELDS.  WE CHOOSE TO IMAGE THE REFLECTED WAVEFIELD USING
C   THE EXPRESSION FOR THE DOWNGOING WAVE DERIVED FROM GEOMETRICAL
C   OPTICS:
C                                 Z
C                   U(Z) = EXP(-I S K(Z) DZ)
C                                 0
C
C   (SEE ALSO TREITEL & ROBINSON, F81-E-6, P. 70)
C
C   WHERE,          K(Z) = W COS(GAMMA(Z))/V(Z)
C
C   W - FREQUENCY IN RAD/SEC
C   GAMMA(Z) - ANGLE OF EMERGENCE AT DEPTH Z, DERIVED FROM SNELL'S LAW
C   V(Z) - GIVEN VELOCITY-DEPTH FUNCTION
C**********************************************************************C

#include <localsys.h>
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      INTEGER * 2 ITR ( SZLNHD )
      INTEGER     LHED( SZLNHD ), LUIN, NBYTES, OBYTES
      INTEGER     NSAMP, NSI, NTRC, NREC, IFORM, LBYTES
      real        vref, angle
      REAL        TRI ( SZLNHD )
      CHARACTER   NAME * 7, ntap * 120, otap * 120
#include <f77/pid.h>
      logical     verbos, logsc, slowness
      logical     query, heap, slnt
      integer     argis
      integer     irhdr
      real        traces, image
      pointer     (wkaddr, traces (1))
      pointer     (wkvels, image  (1))
      pointer     (wkhdr , itrhdr (1))
      real        tmul, dmul
      real        dis (SZLNHD)
      real        ang (SZLNHD)
      real        cee (SZLNHD)
      real        wt  (SZLNHD)
      integer     nvel
      integer     ns,ne
      real        vmin,vmax

c     EQUIVALENCE ( ITR(129), TRI (1) )
      EQUIVALENCE ( ITR(  1), LHED(1) )
      DATA NAME /'TAUPMIG'/, NBYTES / 0 /, LBYTES / 0 /

      data pi/3.14159265/

      deg = 180. / pi
      rad = 1. / deg
C*********************************************************************C
c  get online help if necessary
C*********************************************************************C
      query = ( argis ('-?').gt.0 .or. argis('-h').gt.0 )
      if ( query ) then
           call help ()
           stop
      endif

C*********************************************************************C
c  open printout file
C*********************************************************************C
#include <f77/open.h>

C**********************************************************************C
C     READ PROGRAM PARAMETERS FROM COMMAND LINE
C**********************************************************************C
      call cmdln (ntap,otap,zmax,dz,dmul,tmul,vrefi,xmax,
     &               ns,ne,irs,ire,nvel,vmin,vmax,nz,ist,iend,
     &               logsc, slowness, slnt)

C**********************************************************************C
C     READ AND UPDATE LINE HEADER,
C     WRITE LINE HEADER, SAVE KEY PARAMETERS.
C**********************************************************************C
      call getln ( luin, ntap, 'r', 0 )
      call getln (luout, otap, 'w', 1 )

      lbytes = 0
      CALL RTAPE  ( LUIN, itr, LBYTES           )
      if(lbytes .eq. 0) then
         write(LERR,*)'taupmig: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif
      CALL HLHprt ( ITR , LBYTES, name, 7, LERR         )

c-----------------------------------------
c  save key header values
#include <f77/saveh.h>

      if (.not. slnt) then
         call saver (itr, 'MxRSEL' , ixmax , LINHED)
         call saver (itr, 'MxTrSt' , itmax , LINHED)
         call saver (itr, 'MnTrSt' , itmin , LINHED)
         xmax = ixmax
         tmax = amax1 (float(itmax), float(iabs(itmin)))
         if (tmax .eq. 0.) then
            vref = 0.
         else
            vref = 1000. * xmax / tmax
         endif
      else
         vref = vrefi
         tmax = 1000. * xmax / vref
      endif

      si = nsi
      ist  = ist  / si
      iend = iend / si
      if (ist .lt. 2) ist = 2
      if (iend .lt. 1) iend = nsamp
      if (iend .gt. nsamp) iend = nsamp

         write(LERR,*)' '
         write(LERR,*)'radonf input:'
         write(LERR,*)'Max distance      =  ',xmax
         write(LERR,*)'Max time (ms)     =  ',tmax
         write(LERR,*)'Reference velocity=  ',vref
         write(LERR,*)' '

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SGRNum',ifmt_SGRNum,l_SGRNum,ln_SGRNum,TRACEHEADER)
 

         write(LERR,*)' '
         write(LERR,*)' Values read from input data set lineheader'
         write(LERR,*)' # of Samples/Trace =  ',nsamp
         write(LERR,*)' Sample Interval    =  ',nsi
         write(LERR,*)' Traces per Record  =  ',ntrc
         write(LERR,*)' Records per Line   =  ',nrec
         write(LERR,*)' Format of Data     =  ',iform
         write(LERR,*)' Start Time (samps) =  ',ist
         write(LERR,*)' End Time (samps)   =  ',iend
         write(LERR,*)' Input from SLNT?   =  ',slnt
         write(LERR,*)' '

C**********************************************************************C
C     CHECK DEFAULT VALUES AND SET PARAMETERS
C**********************************************************************C
      if (nsi .le. 32) then
         dt = nsi / 1000.
      else
         dt = nsi / 1000000.
      endif
      si = nsi

      if (nz .eq. 0) nz = nsamp
      zmax = nz * dz

      if(ire.lt.1.or.ire.lt.irs.or.ire.gt.nrec)ire = nrec
      if(ns.le.0 )ns = 1
      if(ne.le.0 .or. ne.gt.ntrc)ne=ntrc

C**********************************************************************C
         write(LERR,*)' Values read from file command line'
         write(LERR,*)' Max depth (ft,m)                      =  ',zmax
         write(LERR,*)' Depth increment (ft,m)                =  ',dz
         write(LERR,*)' Number depth samples                  =  ',nz
         write(LERR,*)' Number of velocity inputs             =  ',nvel
         write(LERR,*)' Minimum velocity                      =  ',vmin
         write(LERR,*)' Maximum velocity                      =  ',vmax
         write(LERR,*)' Distance multiplied by a factor       =  ',dmul
         write(LERR,*)' Time multiplied by a factor           =  ',tmul
         write(LERR,*)' First record to process               =  ',irs
         write(LERR,*)' Last record to process                =  ',ire
         write(LERR,*)' Starting trace number                 =  ',ns
         write(LERR,*)' Ending   trace number                 =  ',ne
         if(slnt)
     1   write(LERR,*)' Input data comes from program slnt'
C**********************************************************************C


c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.

      nvel  = ntrc
      itemi = ntrc * ITRWRD
      items = ntrc * nsamp
      itemv = nvel * nsamp
      call galloc (wkhdr , itemi*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.
      call galloc (wkaddr, items*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.
      call galloc (wkvels, itemv*SZSMPD, errcod, abort)
      if (errcod .ne. 0) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi*SZSMPD,'  bytes'
         write(LERR,*) items*SZSMPD,'  bytes'
         write(LERR,*) itemv*SZSMPD,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi*SZSMPD,'  bytes'
         write(LERR,*) items*SZHFWD,'  bytes'
         write(LERR,*) itemv*SZSMPD,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------
c
c--------------------------------------------------------
c  adjust line header, historical l.h., & output header
       dvel = (vmax - vmin)/float(nvel-1)
 
       call savew( itr, 'MnLnIn',  0   , LINHED)
       call savew( itr, 'ILClIn',  vmin, LINHED)
       call savew( itr, 'CLClIn',  dvel, LINHED)

       call savew( itr, 'NumTrc', nvel , LINHED)
       call savew( itr, 'NumSmp', NZ   , LINHED)
       obytes = SZTRHD + SZSMPD * NZ
       call savhlh ( itr, lbytes, lbyout )
       call wrtape(luout, itr, lbyout )

       nrmp = .1 * ntrc
       call vfill (1.0, wt, 1, ntrc)
       do  i = 1, nrmp

           scl = i / nrmp
           wt (i) = scl
           wt (ntrc-i+1) = scl
       enddo

c---------------------------------------
c  skip to start record
      call recskp(1,irs-1,luin,ntrc,itr)

C**********************************************************************C
C
C      READ TRACES, COMPUTE SPECTRA, WRITE RESULTS
C
C**********************************************************************C
        scl = 1. / float(nvel)
        ist=0

        DO 5001 JJ = irs, ire

c-----------------------------------------------
c  skip to start trace of current record
            call trcskp(jj,1,ns-1,luin,ntrc,itr)

           ist=ist+1
           k=0

                 call vclr (traces, 1, ntrc*nsamp)
                 call vclr (image , 1, nvel*nsamp)
c----------------------------
c  read selected data
           DO 5002 KK = ns, ne

                 NBYTES = 0
                 CALL RTAPE  ( LUIN, ITR, NBYTES )
                 if(nbytes .eq. 0) then
                    write(LERR,*)'End of file on input:'
                    write(LERR,*)'  rec= ',jj,'  trace= ',kk
                    go to 999
                 endif
                 call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)

                 call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                       istatic, TRACEHEADER)

                 IF (istatic .ne. 30000) THEN

                     call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                           irec , TRACEHEADER)
                     k=k+1
                     IF (slnt) THEN

                        call saver2(lhed,ifmt_DstUsg,l_DstUsg,ln_DstUsg,
     1                               ivref , TRACEHEADER)
                        vref = float( ivref )

                        call saver2(lhed,ifmt_DstSgn,l_DstSgn,ln_DstSgn,
     1                              irayp , TRACEHEADER)
                        angle = pi * float( irayp )/180.
                        dis(k) = sin( angle )/vref
                        ang(k) = angle

                     ELSE

                        call saver2(lhed,ifmt_DstUsg,l_DstUsg,ln_DstUsg,
     1                              irayp , TRACEHEADER)
                        dis(k) = dmul * float(irayp)/10000000.
                        fac = dis(k)
                        cee (k) = 1. / fac
c-------
c  ray param **2
                        ang(k) = fac * fac

                     ENDIF

                     rmp = wt(KK)
                     istrc  = (k-1) * nsamp

                     do 5004 LL= 1, nsamp
                           traces(istrc + LL) = tri(LL) * rmp
 5004                continue

                     ishdr = (k-1) * ITRWRD
                     call vmov (lhed,1, itrhdr(ishdr+1),1,ITRWRD)

                 ENDIF

 5002      CONTINUE

c----------------------------------
c  do post-critical migration

      call migsub (nsamp, nz, ntrc, nvel, dz, dt, vref,
     1             traces, image, ang, cee, ist, iend,
     2             irs, irec, vmin, vmax, logsc, slowness)

      do  KK = 1, nvel
 
          kr = nvel - KK + 1
          istrc  = (kr-1) * nsamp
          call vsmul   (image(istrc+1), 1, scl, tri, 1, nz)
          ishdr = (kr-1) * ITRWRD
          call vmov (itrhdr(ishdr+1),1,lhed,1,ITRWRD)
          ivel = cee(kr)

          call savew2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                ivel  , TRACEHEADER)
c         call savew2(lhed,ifmt_DstUsg,l_DstUsg, ln_DstUsg,
c    1                ivel  , TRACEHEADER)

          call savew2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                  KK  , TRACEHEADER)
          call savew2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                 0   , TRACEHEADER)
          call savew2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                irec , TRACEHEADER)
          call vmov (tri, 1, lhed(ITHWP1), 1, NZ)
          call wrtape(luout, itr, obytes)

      enddo


      write(LER,*)'Completed Record= ',JJ

 5001 CONTINUE

  999 continue

      call lbclos( luin)
      call lbclos(luout)

      END
C**********************************************************************C

c-----------------------------------------
c  online help section
c-----------------------------------------
       subroutine  help
#include <f77/iounit.h>

         write(LER,*)
     :'***************************************************************'
      write(LER,*)' '
      write(LER,*)
     :'              taupmig - velocity spectra from tau-p data'
      write(LER,*)' '
      write(LER,*)
     :'Execute program by typing taupmig and the following arguments'
      write(LER,*)
     :' -N [ntap]     (default stdin)    : Input data file name'
      write(LER,*)
     :' -O [otap]     (default stdout)   : Output data file'
c     write(LER,*)
c    :' -zmax [zmax] (default = none)    : Max depth (ft,m)'
      write(LER,*)
     :' -nz [nz]   (def= input samps)    : Number depth samples'
      write(LER,*)
     :' -dz [dz]  (default = none)       : Depth increment (ft,m)'
      write(LER,*)
     :' -is [ist],    (default=1)        : start time (ms)'
      write(LER,*)
     :' -ie [iend],    (default=last)    : end time (ms)'
      write(LER,*)
     :' -rs [irs],    (default=1)        : beginning record number'
      write(LER,*)
     :' -re [ire],    (default=last)     : ending record number'
      write(LER,*)
     :' -ns[ns],      (default=1)        : starting trace number'
      write(LER,*)
     :' -ne[ne]       (default=last)     : ending trace number'
      write(LER,*)
     :' -nvel[nvel]   (default =100)     : number of velocities'
      write(LER,*)
     :' -dmul [dmul]  (default =1.0)     : distance multiplier'
      write(LER,*)
     :' -tmul [tmul]  (default=1.0)      : sampling interval multiplier'
      write(LER,*)
     :' -vmin[vmin](default 1000)  : minimum velocity (ft/sec) for plot'
      write(LER,*)
     :' -vmax[vmax](default 10000) : maximum velocity(ft/sec)for plot'
      write(LER,*) ' '
      write(LER,*)
     :' -V       (default = false) : verbose output'
      write(LER,*)
     :' -slnt   ( default false )  : input data from program slnt'
      write(LER,*)
     :' -S      ( default false )  : horizontal axis is slowness**2'
      write(LER,*)
     :' -log    ( default false )  : velocity (or slowness scale) is'
      write(LER,*)
     :'                              logarithmic'
      write(LER,*) ' '
      write(LER,*)
     :'USAGE: taupmig -N[ntap] -re[re] -rs[rs] -ns[ns] -ne[ne] '
      write(LER,*)
     :'                -vmin[vmin] -vmax[vmax]'
      write(LER,*)
     :'                -nvel[nvel] -dz[dz] -nz[nz]'
      write(LER,*)
     :'                -tmul[tmul] -dmul[dmul]  [-S -log -slnt -V]'
         write(LER,*)
     :'***************************************************************'

      return
      end

c-----
c     get command arguments
c
c     ntap  - C*100  input file name
c     otap  - C*100  output file name
c      ns   - I      start trace
c      ne   - I      stop end trace
c     irs   - I      start record
c     ire   - I      stop end record
c   start   - R      start time of analysis
c    stop   - R      stop time
c    dmul   - R      distance multiplier
c    tmul   - R      sample interval multiplier
c    nvel   - I      number velocities
c    vmin   - R      min velocity
c    vmax   - R      max velocity
c    abslut - L      use absolute semblance
c    slnt   - L      input data from program slnt
c    radon  - L      input data from program radon
c    verbos - L      verbose output or not
c-----
      subroutine cmdln (ntap,otap,zmax,dz,dmul,tmul,vrefi,xmax,
     &               ns,ne,irs,ire,nvel,vmin,vmax,nz,ist,iend,
     &               logsc, slowness, slnt)

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
      character  ntap*(*), otap*(*)
      integer    argis,ns,ne,irs,ire,nvel
      real       dmul,tmul,vmin,vmax
      logical    verbos, slnt, logsc,slowness

         call argstr('-N', ntap, ' ', ' ' )
         call argstr('-O', otap, ' ', ' ' )
         call argr4 ('-zmax', zmax , 0., 0.)
         call argr4 ('-dz', dz , 0., 0.)
         call argr4('-dmul',dmul,1.0,1.0)
         call argr4('-tmul',tmul,1.0,1.0)
         call argi4 ('-nz',nz,0,0)
         call argi4 ('-is',ist,1,1)
         call argi4 ('-ie',iend,0,0)
         call argi4 ('-rs',irs,1,1)
         call argi4 ('-re',ire,0,0)
         call argi4 ('-ns',ns,1,1)
         call argi4 ('-ne',ne,0,0)
         call argi4 ('-nvel',nvel,100,100)
         call argr4('-vmin',vmin,1000.,1000.)
         call argr4('-vmax',vmax,10000.0,10000.0)
         verbos   = ( argis ('-V') .gt. 0 )
         slnt     = ( argis ('-slnt') .gt. 0 )
         logsc    = ( argis ('-log') .gt. 0 )
         slowness = ( argis ('-S') .gt. 0 )
         if (dz .eq. 0.) then
            write(LERR,*)'taupmig command line error:'
            write(LERR,*)'Must enter depth increment using -dz[]'
            stop
         endif
c        if (zmax .eq. 0.) then
c           write(LERR,*)'taupmig command line error:'
c           write(LERR,*)'Must enter max depth using -zmax[]'
c           stop
c        endif
         if(nvel.gt.SZSPRD) then
            write(LERR,*)'Number velocities too high'
            write(LERR,*)'Must be less than ',SZSPRD
            stop
         endif

      return
      end
