C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------------bdstat-----------------------------------------72
c
c Author Paul Garossino TRC:3932
c
c bdstat reads data in USP format one trace at a time. Passes a user 
c defined window of data to subroutine bdstat_sub which returns the
c Mean, Median and Average Absolute Amplitudes of the passed data.  The
c data window is controlled by a user defined start time retrieved from
c a user defined header location.  A window end or a constant window
c are also supplied.  Any header value of padval will cause output for
c that trace to be skipped.  
c
c<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
c
c     declare variables
c
c ----- get machine dependent parameters -----
c

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

c
c ----- dimension standard USP variables -----
c

      integer   itr( SZLNHD )
      integer     nsamp,nsi,ntrc,nrec,iform
      integer     luin,lbytes,nbytes
      integer     argis,ist,iend,irs,ire,ns,ne
      integer     JJ,KK,errcd1,abort1
      integer     le1

      real        tri(SZLNHD)

      character   name*6, ntap*255, otap*255

      logical     verbos,query


c ----- integer USP variables -----
c
c	itr   array: trace plus header from rtape
c       nsamp      : number of samples of input trace
c       nsi        : input sample interval
c       ntrc       : input traces/record
c       nrec       : input number of records
c       iform      : format of data
c       luin       : input device
c       lbytes     :
c       nbytes     :
c       obytes     :
c       argis      : 
c       irs        : record start
c       ire        : record end
c       ns         : trace start
c       ne         : trace end
c       JJ,KK      : loop counters
c       errcd1     : error flag from galloc
c       abort1     : abort flag from galloc
c
c ----- real USP variables -----
c
c	tri array  : working trace   
c
c ----- character USP variables -----
c
c	name       : for print file identification
c	ntap       : input file name 
c       otap       : output file name
c
c ----- logical USP variables -----
c
c	verbos     : printout verbosity flag
c	query      : online help flag
c
c
c ----- dimension program specific variables -----
c

      integer     start, stop, const, nsampo
      integer     luout, luout1, luout2, padval
      real        work,xmean,xmed,xaaa,shot_point
      integer     ihdw0, ifmt_hdw0, l_hdw0, ln_hdw0
      integer     ihdw1, ifmt_hdw1, l_hdw1, ln_hdw1
      integer     SrcLoc, ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc
      integer     le0, le1

      character   hdw0*6, hdw1*6

      pointer     (wkadr1, work(2000000))


c ----- integer program variables -----
c
c       start : window start time
c       stop : window stop time
c       const : window length in ms
c       hdw0 : window start/stop header mnemonic
c       hdw1 : window start/stop header mnemonic
c       nsampo : output number of samples in window
c       luout(1,2,...) : logical output units
c       padval : value used in lmpicks to designate no horizon picked
c
c ----- real program variables -----
c
c	work array : array to hold input record (malloc space when needed)
c
c ----- pointer program variables -----
c
c	wkadr1     : points to location of 1st element of work array
c
c ----- set up useful hooks -----
c
 
c
c ----- initialize necessary variables -----
c

      data name/'BDSTAT'/
      data luin/1/
      data lbytes/0/
      data luout/29/
      data luout1/30/
      data luout2/31/
      data stop/0/


c
c ----- get online help if necessary -----
c

      query = (argis('-?').gt.0 .or. argis('-h').gt.0)

      if ( query ) then

           call help ()
           stop

      endif

#include <f77/open.h>

c
c ----- get command line parameters -----
c

      call cmdln (ntap,otap,ist,iend,irs,ire,ns,ne,hdw0,hdw1,const,
     :le1,padval,verbos)

c
c ----- get logical units -----
c

      call getln(luin,ntap,'r',0)

c
c ----- open output ascii file -----
c

      if (le1 .gt. 0) then
        open(luout,file=otap(1:le1)//'.mean',status='unknown',err=990)
        open(luout1,file =otap(1:le1)//'.med',status='unknown',err=991)
        open(luout2,file =otap(1:le1)//'.aaa',status='unknown',err=992) 
      else
        open(luout,file='bdstat'//'.mean',status='unknown',err=990)
        open(luout1,file ='bdstat'//'.med',status='unknown',err=991)
        open(luout2,file ='bdstat'//'.aaa',status='unknown',err=992) 
      endif

      le0=lenth(ntap)
      if (le0 .gt. 0) then
        write(LERR,*)'Input unit # is ',luin,' for DSN= ',ntap
      else
        write(LERR,*)'Input unit # is ',luin,' for stdin'
      endif
      if (le1 .gt. 0) then
        write(LERR,*)'Output unit # is ',luout,' for DSN= ',
     *	  otap(1:le1),'.mean'
        write(LERR,*)'Output unit # is ',luout1,' for DSN= ',
     *	  otap(1:le1),'.med'
        write(LERR,*)'Output unit # is ',luout2,' for DSN= ',
     *	  otap(1:le1),'.aaa'
      else
        write(LERR,*)'Output unit # is ',luout,' for DSN= ',
     *	  'bdstat.mean'
        write(LERR,*)'Output unit # is ',luout1,' for DSN= ',
     *	  'bdstat.med'
        write(LERR,*)'Output unit # is ',luout2,' for DSN= ',
     *	  'bdstat.aaa'
      endif

c build pointers to header entries

      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu(hdw0,ifmt_hdw0, l_hdw0, ln_hdw0,TRACEHEADER)
      call savelu(hdw1,ifmt_hdw1,l_hdw1,ln_hdw1,TRACEHEADER)

      write(LERR,*)'ifmt_hdw0, l_hdw0, ln_hdw0= ',
     1ifmt_hdw0, l_hdw0, ln_hdw0
      write(LERR,*)'ifmt_hdw1, l_hdw1, ln_hdw1= ',
     1ifmt_hdw1, l_hdw1, ln_hdw1

c
c ----- read line header, check to see if input empty -----
c

      lbytes = 0
      call rtape(luin,itr,lbytes)
      write(LERR,*)'lbytes= ',lbytes

      if(lbytes .eq. 0) then
         write(LERR,*)'bdstat: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         stop
      endif

c
c ----- alter line header -----
c

#include <f77/saveh.h>

c------
c     to get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c
c     see saver/w manual pages

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform , LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      if(nsamp .gt. SZLNHD) nsamp=SZLNHD

c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records)

      call cmdchk(ns,ne,irs,ire,ntrc,nrec)


c
c ----- set record start and end defaults -----
c

      if(irs .eq. 0) irs=1
      if(ire .eq. 0) ire=nrec

c
c ----- printout -----
c

      call verbal(nsamp,nsi,ntrc,nrec,ns,ne,irs,ire,hdw0,hdw1,
     :padval,const)

c
c ----- malloc only space we're going to use -----
c

      item = nsamp * SZSMPD

      call galloc(wkadr1,item,errcd1,abort1)

      if (errcd1 .ne. 0) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) item,'  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) item,'  bytes'
         write(LER,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) item,'  bytes'
         write(LERR,*)' '
      endif

c
c ----- skip to start record -----
c

      call recskp ( 1, irs-1, luin, ntrc, itr )

      DO 100 JJ = irs, ire
c
c ----- skip to desired trace -----
c

         call trcskp(jj,1,ns-1,luin,ntrc,itr)

         DO 99 KK = ns,ne

            nbytes = 0
            call rtape(luin,itr,nbytes)

            if(nbytes .eq. 0) then

               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',jj,'  trace= ',KK
               go to 999

            endif
            call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

c
c ----- check if hdw0 is defined.  If not then exit. -----
c

            IF ( hdw0 .ne. ' ' ) then
c
c ----- at this point hdw0 id defined.  Check to see if value in -----
c       header is padval  If so skip this trace (this value will
c       be put there by lmpicks if no horizon i present for this trace
c

               call saver2( itr, ifmt_hdw0, l_hdw0, ln_hdw0, ihdw0, 1)

               if(ihdw0 .eq. padval )goto 99

c
c ----- header entry ihdw0 should always be negative comming from lmpicks -----
c       check for this case.  If not then advise user to rerun lmpicks
c       using a datum of zero.
c

               if(ihdw0 .gt. 0)then

               write(LERR,*)' '
               write(LERR,*)' FATAL: You have asked for a window start '
               write(LERR,*)'        time of ',ihdw0,' on '
               write(LERR,*)'        trace ',KK,' of record ',JJ
               write(LERR,*)' '
               write(LERR,*)'        Try re-running lmpicks using a '
               write(LERR,*)'        Datum entry of zero. '  
               write(LERR,*)' '

                  stop

               endif

               
c
c
c ----- hdw0 is ok go ahead and assign window start time -----
c
                    
               start = iabs( ihdw0 / nsi )

c
c ----- define window stop if constant is defined -----
c

               if(const.gt.0)then

                  stop = start + const/nsi

               elseif(const.lt.0)then

                  stop = start
                  start = stop + const/nsi
 
               endif
             
            ELSE

c
c ----- define window start by ist from the command line -----
c
                    
               start = ist/nsi
c
c ----- define window stop if constant is defined -----
c

               if(const.gt.0)then

                  stop = start + const/nsi
               else
                  stop = nsamp
               endif
                  
            ENDIF

c
c ----- determine if hdw1 is used for window stop time -----
c       It is expected that itr(hdw1) will be greater
c       than itr(hdw0).  If not trace will be skipped.
c
            IF( hdw1 .ne. ' ' .and. stop .lt. 1 )then

c ----- at this point hdw0 id defined.  Check to see if value in -----
c       header is padval  If so skip this trace (this value will
c       be put there by lmpicks if no horizon is present for this trace)
c
               call saver2 ( itr, ifmt_hdw1, l_hdw1, ln_hdw1, ihdw1, 
     :              TRACEHEADER )

               if(ihdw1 .eq. padval)goto 99

               if(ihdw1 .gt. 0)then

               write(LERR,*)' '
               write(LERR,*)' FATAL: You have asked for a window end '
               write(LERR,*)'        time of ',ihdw1,' on '
               write(LERR,*)'        trace ',KK,' of record ',JJ
               write(LERR,*)' '
               write(LERR,*)'        Try re-running lmpicks using a '
               write(LERR,*)'        Datum entry of zero. '  
               write(LERR,*)' '

                  stop

               endif

c
c ----- hdw0 is ok go ahead and assign window start time -----
c

               stop = iabs(ihdw1/nsi)

c
c ----- if stop is less than start then skip this trace -----
c

               if((stop - start).le.2)goto 99

            ELSE

c
c ----- if neither hdw1 nor constant are defined then crash -----
c       out and warn user.
c

               if( hdw1 .eq. ' ' .and. const.eq.0 .and. stop.lt.1)then

                  write(LERR,*)' '
                  write(LERR,*)' FATAL: In addition to hdw0 '
                  write(LERR,*)' you must define either hdw1 '  
                  write(LERR,*)' or const to fully define' 
                  write(LERR,*)' your analysis window'
                  write(LERR,*)' ' 

                  stop

               endif

            ENDIF

c 
c ----- extract shot point data from header -----
c

            call saver2 ( itr, ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc, SrcLoc,
     :           TRACEHEADER )

            shot_point = float(SrcLoc)/10.

c
c ----- clear work array -----
c

            call vclr(work,1,nsamp)

c
c ----- determine number of samples in window -----
c

            nsampo = stop - start + 1

c
c ----- verbose output -----
c

            if(verbos)then

               write(LERR,*)' Window on record ',JJ,' trace ',KK,' is ',
     :start*nsi,' to ',stop*nsi,' ms.'

            endif

c
c ----- load work array with windowed data -----
c
            call vmov(tri(start),1,work(1),1,nsampo)
            
c
c ----- calculate statistics required -----
c

            call bdstat_sub(work,nsampo,xmean,xmed,xaaa,verbos)

c
c ----- write shot point, mean, median, aaa to output file -----
c

            write(luout,*)shot_point,xmean 
            write(luout1,*)shot_point,xmed
            write(luout2,*)shot_point,xaaa 
c
c ----- go to next trace -----
c

 99      CONTINUE

c
c ----- skip to end of record -----
c

         call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)

 100  CONTINUE

 999  continue

      call lbclos(luin)
      close(luout)

      stop

 990  continue

      write(LERR,*)' '
      write(LERR,*)' FATAL:Cannot open output file ',otap(1:le1),'.mean'
      write(LERR,*)' check bdstat command line entries and try again'
      write(LERR,*)' '
      stop
 991  continue

      write(LERR,*)' '
      write(LERR,*)' FATAL:Cannot open output file ',otap(1:le1),'.med'
      write(LERR,*)' check bdstat command line entries and try again'
      write(LERR,*)' '
      stop
 992  continue

      write(LERR,*)' '
      write(LERR,*)' FATAL:Cannot open output file ',otap(1:le1),'.aaa'
      write(LERR,*)' check bdstat command line entries and try again'
      write(LERR,*)' '
      stop

      end

c
c ----- online help section -----
c

      subroutine  help

#include <f77/iounit.h>

        write(LER,*)' '
        write(LER,*)'Command Line Arguments for bdstat'
        write(LER,*)' '
        write(LER,*)'Input....................................... (def)'
        write(LER,*)' '
        write(LER,*)'-N[ntap]   -- input data set'
        write(LER,*)'-O[otap]   -- output data set'
        write(LER,*)'-ns[ns]  -- start trace             (first trace)'
        write(LER,*)'-ne[ne]  -- end trace                (last trace)'
        write(LER,*)'-rs[irs]  -- start record                 (first)'
        write(LER,*)'-re[ire]  -- end record                    (last)'
        write(LER,*)'-s[start]  -- start time (ms)                 (1)'
        write(LER,*)'-e[end]  -- end time (ms)                   (end)'
        write(LER,*)'-hdw0[hdw0] - header mnemonic first event     ( )'
        write(LER,*)'-hdw1[hdw1] - header mnemonic last event      ( )'
        write(LER,*)'-const[const]  -- time below hdw0             (0)'
        write(LER,*)'-p[padval]  -- pad value from lmpicks    (-15999)'
        write(LER,*)'-V         -- verbos printout'
        write(LER,*)' '
        write(LER,*)'Usage:'
      write(LER,*)'bdstat -N[] -O[] -ns[] -ne[] -rs[] -re[]'
      write(LER,*)'       -hdw0[] -hdw1[] -const[] -p[] -V'
        write(LER,*)' '

      return
      end

c
c ----- command line parsing subroutine -----
c

      subroutine cmdln (ntap,otap,ist,iend,irs,ire,ns,ne,hdw0,hdw1,
     :const,le1,padval,verbos)

#include <f77/iounit.h>

      integer    argis,ist,iend,irs,ire,le1,const,padval
     
      character  ntap*(*), otap*(*), hdw0*6, hdw1*6

      logical    verbos

          call argi4('-const',const,0,0)
          call argi4('-e',iend,0,0)
          call argstr('-hwd0',hdw0,' ',' ')
          call argstr('-hwd1',hdw1,' ',' ')
          call argstr('-N',ntap,' ',' ') 
          call argi4('-ns',ns,0,0)
          call argi4('-ne',ne,0,0)
          call argstr('-O',otap,' ',' ') 
          le1 = lenth(otap)
          call argi4('-p',padval,-15999,-15999)

          call argi4('-rs',irs,1,1)
          call argi4('-re',ire,0,0)
          call argi4('-s',ist,1,1)

c
c ----- fix padval for the fact that lmpicks will have installed -----
c       0.0 - padval into the header
c

          padval = -padval

          verbos = (argis('-V') .gt. 0)

      return
      end

      subroutine verbal(nsamp,nsi,ntrc,nrec,ns,ne,irs,ire,hdw0,hdw1
     :,padval,const)

#include <f77/iounit.h>

      integer nsamp,nsi,ntrc,nrec,ns,ne,irs,ire,const
      integer padval

      character hdw0*6, hdw1*6

        write(LERR,*)
        write(LERR,*)' Values read from input data set line header'
        write(LERR,*)
        write(LERR,*) ' # of Samples/Trace =  ', nsamp
        write(LERR,*) ' Sample Interval    =  ', nsi
        write(LERR,*) ' Input Traces per Record  =  ', ntrc
        write(LERR,*) ' Records per Line   =  ', nrec
        write(LERR,*) ' trace start       =  ', ns
        write(LERR,*) ' trace end       =  ', ne
        write(LERR,*) ' record start       =  ', irs
        write(LERR,*) ' record end         =  ', ire
        write(LERR,*) ' header word for window start =  ', hdw0

        if(const.eq.0)then

           write(LERR,*) ' header word for window end =  ', hdw1

        else           

           write(LERR,*) ' constant added to window start =  ', const

        endif
        write(LERR,*) ' pad value used in lmpicks  =  ', -padval

        write(LERR,*) ' '

      return
      end
