/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  Modified by:  Cory Hoelting
  Original code by:  Chi-hsin Wu
  Date last modified:  August 13, 1996
  Purpose:  Perform segmentation of a 2D data set using
            disparity values from the program disparity
            or the program combine, random neighbors for 
            the program randomgraph, and possibly threshold
            values based on coherency from the program
            cogeom. 
*/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <stdlib.h>

/* USP stuff */
/* Used for command line processing */
/* This code may be converted to work with the */
/* USP file format in the future. */
#include "iounit.h"
#include "lhdrsz.h"
#include "save_defs.h"
#include "cu_defs.h"
#include "sisio.h"

#define      Q4      4   /* no. of neighbors    */
 
 
/* The constant BORDER should become a variable.  Its value */
/* should be the same as the value assigned to the variable 'stp' */
/* in the function findneighbor() */
#define      BORDER  15  /* using wider matrix to store interior[][] values */
#define      SQR(a) ((a)*(a))
#define      ESMALL  2   /* Block sz of SQR*(2*ESMALL+1) to test small region */
/*#define      NSEED   379658*/
#define        NSEED   85694332
#define      HALF_WIDTH 10 /* constant used in the channel following */
                           /* penalty function */


void bethe2d_help();
void allocation();
void initialization();
void BetheX();
int central();
void findx();
void copy();
int TooSmall();
int TooThin();
int ver_channel_follower();
int hor_channel_follower();
double periph();
double expHtotal();
void findneighbor();
void read_neighbor_sys();
void read_disparity();
void read_coherency();
void writeintlattice();

/* external functions */
extern void srand48();
extern double drand48();
extern long lrand48();

/* global variables */
int **interiorlabel;
int idelta[Q4]={1,0,-1,0};  
int jdelta[Q4]={0,1,0,-1}; 
int serial;       /* set to 1, if update is serial */
int parallel;     /* set to 1, if update is parallel */
int st_flag;      /* set to 1, if using the penalty functions */
                  /* TooSmall and TooThin */
int ch_flag;      /* set to 1, if using the penalty functions */
                  /* ver_channel_follower and hor_channel_follower */ 
int COARSE;       /* inverse resolution */
int LROW,LCOL;    /* number of rows and columns in the label data set */
int LN;           /* total number of label sites, i.e., LN=LROW*LCOL */
int HWW;          /* half window width for the c.d.f. calculation */
int WW;           /* window width for the c.d.f. calculation, including */
                  /* the center point */
int WS;           /* the window width squared, i.e. size of the window */
int NN;           /* number of neighbors, near and random */ 
int P;            /* number of partitions */
double J;         /* interaction strength */
int  targetdepth; /* depth of Bethe tree */
int *forward;     /* forward indexing to randomly ordered label lattice */
int *backward;    /* backward indexing to original label lattice */
int ***dispar;    /* the disparity values calculated by the */
                  /* program disparity */ 
int **lattice;    /* label lattice */
double ****x;     /* values used in the bethe tree approximation of the */
                  /* marginal site p.d.f.'s */
double ****y;     /* temp storage for x, if using parallel update */
int coh_flag;     /* set to 1, if coherency is used as a weight on the */
                  /* near neighbor terms */
double ***coh_dat; /* coherency values between near neighbors */


/* global variable for program name, used in error messages */
char prog_name[] = "texture";

void main(argc,argv)
int argc;
char *argv[];
{
 char file_name[256];

 FILE *lam_fp;   /* file pointer for the lambda values */
 int ifp,   /* file descriptor for the disparity values */
     ffp,   /* file descriptor for the forward index */
     bfp,   /* file descriptor for the backward index */
     lfp,   /* file descriptor for the lambda values */
     cfp,   /* file descriptor for the coherency weights */
     ofp;   /* output file pointer for the partition labels */
 int ipipe, fpipe, bpipe, lpipe, cpipe, opipe;
 int one;
 int ierr = 0;
 char host[10], blank9[10];
 int ikp;                  /* flag set to 1 if running under ikp */

 float j_temp;

 /* USP data */
 int usp_lineheader[SZLNHD];
 int usp_lineheader_bytes, usp_trace_bytes;

  /* Help? */
  if (C_ARGIS("-?", &argc, argv) ||
      C_ARGIS("-h", &argc, argv) ||
      C_ARGIS("-help", &argc, argv))
    {
     bethe2d_help();
     exit(0);
    }

  /* check to see if this program is being run under ikp */
  strcpy(host, "         ");
  strcpy(blank9, "         ");
  ikpchk_(host, strlen(host));
  if (strcmp(host, blank9))
     ikp = 1;
  else
     ikp = 0;

  /* display usage part of help if there was */
  /* only one argument on the command line */
  if (argc == 1 && !ikp)
    {
     fprintf(stderr,
            "Usage:  bethe2d [-N <dis_file>] -fmat <fifn> -bmat <bifn>\n\
                -lambda <lbdfn> [-O <output_file>] [-s | -p]\n\
                [-coarse <coarseness>] [-st | -ch] [-halfwin <half_win_size>]\n\
                [-par <num_par>] [-j <j>] [-tree_depth <td>] [-coh <coh_file>]\n\
                [-h -? -help]\n");
     exit(0);
    }

  /* set up the default file descriptors for ikp */
  ipipe = 0;
  fpipe = 3;
  bpipe = 4;
  lpipe = 5;
  cpipe = 6;
  opipe = 1;
  one = 1;

  /* process the command line arguments */
  C_ARGSTR("-N", file_name, "", "", &argc, argv);
  if (strlen(file_name) >= 1)
     lbopen(&ifp, file_name, "r");
  else
    {
     if (ikp)
       {
        if (pipcnt_(&ipipe, &one) > 0)
           sisfdfit_(&ifp, &ipipe);
       }
     else
       {
        ifp = fileno(stdin);
/*
        fprintf(stderr, "%s: error:  -N <dis_file> is required, for help use ",
                prog_name);
        fprintf(stderr, "bethe2d -h\n");
        ierr = 1; 
*/
       }
    }

  C_ARGSTR("-fmat", file_name, "", "", &argc, argv);
  if (strlen(file_name) >= 1)
    {
     ffp = open(file_name, O_RDONLY);
     if (ffp == -1)
       {
        fprintf(stderr, "%s: error:  cannot open the forward index file %s\n",
                prog_name, file_name);
        ierr = 1; 
       }
    }
  else
    {
     if (!ikp)
       {
        fprintf(stderr, "%s: error:  -fmat <fifn> is required, for help use ",
                prog_name);
        fprintf(stderr, "bethe2d -h\n");
        ierr = 1; 
       }
     else  /* running in ikp */
        ffp = fpipe; 
    }
 
  C_ARGSTR("-bmat", file_name, "", "", &argc, argv);
  if (strlen(file_name) >= 1)
    {
     bfp = open(file_name, O_RDONLY);
     if (bfp == -1)
       {
        fprintf(stderr, "%s: error:  cannot open the backward index file %s\n",
                prog_name, file_name);
        ierr = 1; 
       }
    }
  else
    {
     if (!ikp)
       {
        fprintf(stderr, "%s: error:  -bmat <bifn> is required, for help use ",
                prog_name);
        fprintf(stderr, "bethe2d -h\n");
        ierr = 1; 
       }
     else  /* running in ikp */
        bfp = bpipe;
    }

  C_ARGSTR("-lambda", file_name, "", "", &argc, argv);
  if (strlen(file_name) >= 1)
    {
     lfp = open(file_name, O_RDONLY);
     lam_fp = fdopen(lfp, "r");
     if (lam_fp == NULL)
       {
        fprintf(stderr, "%s: error:  cannot open the lambda schedule file %s\n",
                prog_name, file_name);
        ierr = 1; 
       }
    }
  else
    {
     if (!ikp)
       {
        fprintf(stderr, "%s: error:  -lambda <lbdfn> is required, for help ",
                prog_name);
        fprintf(stderr, "use bethe2d -h\n");
        ierr = 1; 
       }
     else  /* running in ikp */
       {
        lfp = lpipe;
        lam_fp = fdopen(lfp, "r");
       }
    }

  C_ARGSTR("-O", file_name, "", "", &argc, argv);
  if (strlen(file_name) < 1)
    {
     if (!ikp)
       {
        ofp = fileno(stdout);
/*
        fprintf(stderr, "%s: error:  -O <output_file> is required, for ",
                prog_name);
        fprintf(stderr, "help use bethe2d -h\n");
        ierr = 1; 
*/
       }
     else  /* running in ikp */
        if (pipcnt_(&opipe, &one) > 0)
           sisfdfit_(&ofp, &opipe);
    }
  else
     lbopen(&ofp, file_name, "w");

  C_ARGSTR("-coh", file_name, "", "", &argc, argv);
  if (strlen(file_name) < 1)
    {
     if (!ikp) {
        coh_flag = 0;
     }
     else  /* running in ikp */
       {
        coh_flag = 1;
        if (pipcnt_(&cpipe, &one) > 0)
           sisfdfit_(&cfp, &cpipe);
       }
    }
  else
    {
     coh_flag = 1; 
     lbopen(&cfp, file_name, "r");
    }

  serial = C_ARGIS("-s", &argc, argv);
  parallel = C_ARGIS("-p", &argc, argv);
  if (serial && parallel)
    {
     fprintf(stderr, "%s: error:  the flags -s and -p cannot both be set ",
             prog_name);
     fprintf(stderr, "on the same commandline.\n");
     ierr = 1;
    }
  else if( (!serial) && (!parallel) )
     serial = 1;  /* serial update by default */

  C_ARGI4("-coarse", &COARSE, 9, 9, &argc, argv);
  if (COARSE < 1)
    {
     fprintf(stderr, "%s: error:  the command line argument -coarse ",
             prog_name);
     fprintf(stderr, "<coarseness> must be positive.\n");
     ierr = 1;
    }

  st_flag = C_ARGIS("-st", &argc, argv);
  ch_flag = C_ARGIS("-ch", &argc, argv);
  if (st_flag && ch_flag)
    {
     fprintf(stderr, "%s: error:  the flags -st and -ch cannot both be set ",
             prog_name);
     fprintf(stderr, "on the same commandline.\n");
     ierr = 1; 
    }
  else if( (!st_flag) && (!ch_flag) )
     st_flag = 1;  /* TooSmall and TooThin penalty functions by default */

  C_ARGI4("-halfwin", &HWW, 7, 7, &argc, argv);
  if (HWW < 1)
    {
     fprintf(stderr, "%s: error:  the command line argument -halfwin ",
             prog_name);
     fprintf(stderr, "<half_win_size> must be positive.\n");
     ierr = 1; 
    }

  /* exit if the error condition has been set */
  if (ierr) exit(1);

  C_ARGI4("-par", &P, 2, 2, &argc, argv);
  C_ARGR4("-j", &j_temp, 1.0, 1.0, &argc, argv);
  J = (double) j_temp;
  C_ARGI4("-tree_depth", &targetdepth, 1, 1, &argc, argv);

  /* Read in the USP lineheader from the disparity file */
  rtape(ifp, usp_lineheader, &usp_lineheader_bytes);

  /* Grab values from the USP lineheader */
  /* The disparity file has LCOL samples per trace, NN traces */
  /* per record, and LCOL records.  So we can get these values */
  /* from the header. */ 
  saver(usp_lineheader, "NumRec", &LROW, LINEHEADER);
  saver(usp_lineheader, "NumTrc", &NN, LINEHEADER);
  saver(usp_lineheader, "NumSmp", &LCOL, LINEHEADER);

  WW = (2 * HWW) + 1;
  WS=SQR(WW);
  LN=LROW*LCOL;

  usp_trace_bytes = SZDTHD + (LCOL * SZSAMP);

  /* update the line header so it is correct for the label lattice */
  /* (output) file.  We take one input file with LROW records, */
  /* NN traces per record, and LCOL samples per trace. The label */
  /* lattice will have 1 record, LROW traces per record, and LCOL */
  /* samples per trace. So we need to change the NumTrc and NumRec */
  /* fields in the lineheader */
  savew(usp_lineheader, "NumTrc", LROW, LINEHEADER);
  savew(usp_lineheader, "NumRec", 1, LINEHEADER);

  /* Update the historical line header */
  /* savhlh_(usp_lineheader, &usp_lineheader_bytes, &usp_lineheader_bytes); */

  /* write out the line header */
  wrtape(ofp, usp_lineheader, usp_lineheader_bytes);

  allocation();

  read_neighbor_sys(forward, ffp, LN);
  if (!ikp)
     close(ffp);
  read_neighbor_sys(backward, bfp, LN);
  if (!ikp)
     close(bfp);
  read_disparity(dispar, ifp, usp_trace_bytes);
  if (!ikp)
     lbclos(ifp);
  /* Use coherency as a threshold */
  if (coh_flag)
    {
     read_coherency(coh_dat, cfp, usp_trace_bytes);
     if (!ikp)
        lbclos(cfp);
    }

  initialization();
  BetheX(lam_fp);
  if (!ikp)
     fclose(lam_fp);
  writeintlattice(lattice, ofp, usp_trace_bytes);
  if (!ikp)
     lbclos(ofp);

  /* Normal completion */
  fprintf(stderr, "%s:  normal completion\n", prog_name); 

}/*main*/


/* allocate memory */
void allocation()
{
  int i,j,nn,p,fr;
  double ***pccc, **pcc, *pc;
  int    ***ipccc, **ipcc, *ipc;

  interiorlabel =(int **)malloc((LROW+2*BORDER)*sizeof(int *));
  interiorlabel += BORDER;
  ipc=(int *)malloc((LROW+2*BORDER)*(LCOL+2*BORDER)*sizeof(int));
  ipc += BORDER*(LCOL+2*BORDER)+BORDER;
  for (i=-BORDER; i<LROW+BORDER; i++)
    interiorlabel[i]=ipc+i*(LCOL+2*BORDER);
  for (i=-BORDER;i<LROW+BORDER;i++)  /*initialize interior[][]*/
    for (j=-BORDER;j<LCOL+BORDER;j++) {
      if ( (i >0) && (i < LROW-1) && (j >0) && (j < LCOL-1) )
        interiorlabel[i][j]=4;
      else if (i<0 || j<0 || i>=LROW || j>=LCOL)
        interiorlabel[i][j]=0;
      else if ((i==0 && (j==0 || j==LCOL-1)) || (i==LROW-1 && (j==0 || j==LCOL-1)))
        interiorlabel[i][j]=2; 
      else 
        interiorlabel[i][j]=3;
    }/*for j*/

  forward= (int*)malloc(LN*sizeof(int));
  backward=(int*)malloc(LN*sizeof(int));

  /* allocate space for the coherency values, if coherency is being used */
  if (coh_flag)
    {
     coh_dat=(double ***)malloc(LROW*sizeof(double **));
     pcc=(double **)malloc(LROW*LCOL*sizeof(double *));
     for (i=0; i<LROW; i++)
        coh_dat[i]=pcc+i*LCOL;
     pc=(double *)malloc(LROW*LCOL*Q4*sizeof(double));
     for (i=0; i<LROW; i++)
        for (j=0; j<LCOL; j++)
           coh_dat[i][j]=pc+((i*LCOL)+j)*Q4;
    }

/*3D array of resulting disparity*/
  dispar=(int ***)malloc(LROW*sizeof(int **));
  ipcc=(int **)malloc(LROW*LCOL*sizeof(int *));
  for (i=0; i<LROW; i++) 
    dispar[i]=ipcc+i*LCOL;
  ipc=(int *)malloc(LROW*LCOL*NN*sizeof(int));
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++)
      dispar[i][j]=ipc+((i*LCOL)+j)*NN;

  lattice =(int **)malloc((LROW+2*BORDER)*sizeof(int *));
  lattice += BORDER;
  ipc=(int *)malloc((LROW+2*BORDER)*(LCOL+2*BORDER)*sizeof(int));
  ipc += BORDER*(LCOL+2*BORDER)+BORDER;
  for (i=-BORDER; i<LROW+BORDER; i++)
    lattice[i]=ipc+i*(LCOL+2*BORDER);

  x=(double ****)malloc((LROW)*sizeof(double ***));
  pccc=(double ***)malloc((LROW)*(LCOL)*sizeof(double **));
  for (i=0; i<LROW; i++)
    x[i]=pccc+i*(LCOL);
  pcc=(double **)malloc((LROW)*(LCOL)*NN*sizeof(double *));
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++) {
      x[i][j]=pcc+((i)*(LCOL)+(j))*NN;
    }
  pc=(double *)malloc((LROW)*(LCOL)*NN*P*sizeof(double));
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++) 
      for(nn=0; nn < NN; nn++) {
        x[i][j][nn]=pc+(((i)*(LCOL)+(j))*NN+nn)*P;
      }

  if (!serial)
    {
     /* Only needed when using parallel update */

     y=(double ****)malloc((LROW)*sizeof(double ***));
     pccc=(double ***)malloc((LROW)*(LCOL)*sizeof(double **));
     for (i=0; i<LROW; i++)
       y[i]=pccc+i*(LCOL);
     pcc=(double **)malloc((LROW)*(LCOL)*NN*sizeof(double *));
     for (i=0; i<LROW; i++)
       for (j=0; j<LCOL; j++) {
         y[i][j]=pcc+((i)*(LCOL)+(j))*NN;
       }
     pc=(double *)malloc((LROW)*(LCOL)*NN*P*sizeof(double));
     for (i=0; i<LROW; i++)
       for (j=0; j<LCOL; j++) 
         for(nn=0; nn < NN; nn++) {
           y[i][j][nn]=pc+(((i)*(LCOL)+(j))*NN+nn)*P;
         }
    }

/*Initialization...*/

  for (i=0;i<LROW;i++)
    for (j=0;j<LCOL;j++) 
      for(nn=0; nn<NN ; nn++) 
          for (p=0; p<P; p++) {
            x[i][j][nn][p]=1.0;
      }

}/*allocation*/


/* use NSEED as the seed value and generate initial label lattice */
void initialization()
{
  int i,j;
  
  srand48(NSEED);
  
  for (i=0; i<LROW; i++)
    for (j=0; j<LCOL; j++) 
      lattice[i][j]=lrand48()%P;
      
}/*initialization*/


/*calculate the meanfield of lattice using the Ising Model on Bethe tree */
void BetheX(lam_fp)
FILE *lam_fp;
{
  double lambda;  /* constraint relaxation parameter */
  int i,j;        /* central point */
  int iteration;

  iteration = 1;
  while ( (fscanf(lam_fp, "%lf",&lambda) != EOF) ) {

    fprintf(stderr, "%s: performing iteration %d with J = %f, lambda = %f\n",
            prog_name, iteration, J, lambda);

    findx(lambda);
    for (i=0; i<LROW; i++)
      for(j=0; j<LCOL; j++) {
   
        /* added 5/20/96 -- CJH */
        /* 
        fprintf(stderr, "in partition -- i = %d, j = %d\n", i, j);
        */

        lattice[i][j]=central(i,j, lambda);
      }

   iteration++;
  }/*while fscanf*/

}/*Bethe*/


/* find x[][][] recursively with depth=targetdepth */
void findx(lambda)
double lambda;
{
  int i,j,m,n,nn,p;
  int depth,ns;
  double prod;

  prod=1.0;
  depth=0;/* Initial depth at center should be 0 */

  if (serial) 
    for (i=0; i<LROW; i++)
      for(j=0; j<LCOL; j++)   
        for (nn=0; nn<NN; nn++) {
          findneighbor(i,j,nn,&m,&n);
          if (interiorlabel[m][n]) {
            for (p=0; p<P; p++)
              x[i][j][nn][p] = periph(i,j,depth,nn,m,n,p, lambda);
            }
          }
  else {
    for (i=0; i<LROW; i++)
      for(j=0; j<LCOL; j++)   
        for (nn=0; nn<NN; nn++) {
          findneighbor(i,j,nn,&m,&n);
          if (interiorlabel[m][n]) {
            for (p=0; p<P; p++)
              y[i][j][nn][p] = periph(i,j,depth,nn,m,n,p, lambda);
            }
          }
    copy(x,y);
  }
  
}/*findx*/


/* Copy the b array into the a array */ 
void copy(a,b)
double ****a;
double ****b;
{
  int i,j,nn,p;

  for (i=0;i<LROW;i++)
    for (j=0;j<LCOL;j++) 
      for (nn=0; nn<NN; nn++) 
        for (p=0; p<P; p++) 
            a[i][j][nn][p]=b[i][j][nn][p];

}


/* find the nn_th neighbor of (i,j) */
void findneighbor(i,j,nn,pm,pn)
int i,j,nn,*pm,*pn;
{
  static int allocated = 0;  /* once twos has been allocated set to 1 */
  static int *twos;
  int ln,rn,qn,ind,hb,nd,nbr,lb,stp;
  int ND,RN,ENN;

  /* ND=ceil(log2((double)LN)); */
  ND=ceil(log10((double)LN) / log10(2.0));
  RN=NN-Q4;
  /* ENN=(int)log2((double)(RN)); */
  ENN=(int)( log10((double)(RN)) / log10(2.0) );
  if (!allocated)
    {
     twos=(int *)malloc(ND*sizeof(int));
     allocated = 1;
    }


  if (nn<Q4) {        /*4 near neighbors*/
    stp=((int)ceil((double)WW/COARSE));
    *pm=i+idelta[nn]*stp; 
    *pn=j+jdelta[nn]*stp;
  }
  else {              /*random neighbors*/
    ln=i*LCOL+j;
    rn=forward[ln];

    qn=rn;            /*base-2 representation*/
    for(nd=0; nd<ND; nd++) {
      twos[nd]=qn%2;
      qn=qn/2;
    }/*for nd*/

    if (nn>=Q4 && nn<Q4+RN/2 ) {  /*high-bit random neighbor, i.e., xxxbbbbb*/
      ind=hb=nn-Q4;
      for (nd=ND-ENN-1; nd>=0; nd--)
        ind=ind*2+twos[nd+ENN];
      nbr=backward[ind];
      *pm=nbr/LCOL; *pn=nbr%LCOL;
    }/*if nn*/

    else {                        /*low-bit random neighbor, i.e., bbbbbxxx*/
      lb=nn-(Q4+RN/2);
      ind=0;
      for (nd=ND-ENN-1; nd>=0; nd--)
        ind=ind*2+twos[nd];
      ind=ind*RN/2+lb;
      nbr=backward[ind];
      *pm=nbr/LCOL; *pn=nbr%LCOL;
    }/*else*/
  
  }/*else*/

}/*findneighbor*/


/*calculate meanfield value at site i,j */
int central(i,j, lambda)
int i,j;      /* central point              */
double lambda;
{  
  int nn,m,n;
  int depth;
  int p,label;
  double prod,exphd,maxval,W1;

  label=0;
  maxval=0;

  for (p=0; p<P; p++) {
    prod=1.0;
    if (p != 0)
      for (nn=0; nn<NN; nn++) {
        findneighbor(i,j,nn,&m,&n);
        if (interiorlabel[m][n])
          prod *= x[i][j][nn][p];
      }
    /* Choose the penalty function */
    if (st_flag)
       W1=lambda*(TooSmall(i,j,p)+TooThin(i,j,p));
    else
       W1=lambda*(ver_channel_follower(i,j,p) + hor_channel_follower(i,j,p));
    exphd=exp(-W1)*prod;
    if (exphd>=maxval) { label=p; maxval=exphd; }
  }

  return (label);

}/*central*/


/* calculate the x(s(i,j)) of the meanfield at site s(i,j) */
double periph(i,j,depth,nn,m,n,p, lambda)
int i,j;               /* central point                                     */
int depth;             /* depth counted from center which is depth 0        */
int nn;                /* 4 near nbr. or random neighbor                    */
int m,n;               /* neighbor of (i,j)                                 */
int p;                 /* partition                                         */
double lambda;
{
  int k,l,nnn;
  int pp;
  double prod,den,nm;

  if (depth==targetdepth) return(x[i][j][nn][p]);

  nm=0;
  den=0;
  depth++;

  for (pp=0; pp<P; pp++) {
    prod=1.0;
    if (pp != 0)
      for (nnn=0; nnn<NN; nnn++) {
        findneighbor(m,n,nnn,&k,&l);
        if ( !((k==i)&&(l==j)) && (interiorlabel[k][l]) )       
          prod *= periph(m,n,depth,nnn,k,l,pp, lambda);
      }
    nm += expHtotal(i,j,p,nn,m,n,pp, lambda)*prod;
    den += expHtotal(i,j,0,nn,m,n,pp, lambda)*prod;

  }

  return(nm/den);

}/*periph*/


/* compute exp(-Htotal)                                                   */
double expHtotal(i,j,p,nn,m,n,pp, lambda)
int i,j,p,nn,m,n,pp;
double lambda;
{
  double U,V;

  U= (p==pp)*dispar[i][j][nn];

  /* If the -coh option was specified on the commandline, use */
  /* coherency as a weight on the near neighbor terms, i.e. on */
  /* the U terms where nn is less than Q4 */
  if (coh_flag)
     if (nn < Q4)
        U *= coh_dat[i][j][nn];

  if (st_flag)
     V= TooSmall(m,n,pp)+TooThin(m,n,pp);
  else
     V= ver_channel_follower(m,n,pp) 
        + hor_channel_follower(m,n,pp);

  return exp(-(J*U+lambda*V));

}/*expHtotal*/


/*find if the block is too small, if # of label value in block agree with value
 *at (i,j) < SQR(ESMALL+1), then define this block as too-small               */
int TooSmall(i,j,label)
int i,j,label;
{
  int m,n;
  double sum_agree;

  sum_agree=0.0;

  for (m=-ESMALL+i; m<ESMALL+i; m++)
    for (n=-ESMALL+j; n<ESMALL+j; n++) {
      if (interiorlabel[m][n])
        sum_agree += (label==lattice[m][n]);
    }

  return (sum_agree<SQR(ESMALL+1));

}/*TooSmall*/ 


/*find if the block is too thin                                               */
int TooThin(i,j,label)
int i,j,label;
{
  int hm,hn,hk,hl,vm,vn,vk,vl;
  int penalty;

  hm=i+idelta[0];hn=j+jdelta[0];hk=i+idelta[2];hl=j+jdelta[2];
  vm=i+idelta[1];vn=j+jdelta[1];vk=i+idelta[3];vl=j+jdelta[3];

  penalty= (label!=lattice[hm][hn] && label!=lattice[hk][hl])
          +(label!=lattice[vm][vn] && label!=lattice[vk][vl]);

  return (double)penalty;

}/*TooThin*/  


int ver_channel_follower(i, j, label)
 int i, j, label;
 {
  int m, n, count;
  int sum_agree, sum_agree_i_minus, sum_agree_i_plus;
 
  sum_agree = 0;
  sum_agree_i_minus = 0;
  sum_agree_i_plus = 0;
 
  count = 0;
  m = i;
  for(n = -HALF_WIDTH + j; n <= HALF_WIDTH + j; n++)
     if (interiorlabel[m][n])
       {
        sum_agree += (label == lattice[m][n]);
        count++;
       }
 
  if ( ((double) sum_agree) > (0.75 * ((double) count)) )
    {
     m = i - 1;
     for(n = -((int) 1.5*HALF_WIDTH) + j; n <= ((int) 1.5*HALF_WIDTH) + j; n++)
        if (interiorlabel[m][n])
           sum_agree_i_minus += (label == lattice[m][n]);
 
     m = i + 1;
     for(n = -((int) 1.5*HALF_WIDTH) + j; n <= ((int) 1.5*HALF_WIDTH) + j; n++)
        if (interiorlabel[m][n])
           sum_agree_i_plus += (label == lattice[m][n]);
 
     if ( ((double) sum_agree_i_minus) > (0.75 * ((double) sum_agree)) &&
          ((double) sum_agree_i_plus)  > (0.75 * ((double) sum_agree)))
        return (0);
     else
        return (1);
    }
  else
     return (0);
 }


int hor_channel_follower(i, j, label)
 int i, j, label;
 {
  int m, n, count;
  int sum_agree, sum_agree_j_minus, sum_agree_j_plus;
 
  sum_agree = 0;
  sum_agree_j_minus = 0;
  sum_agree_j_plus = 0;
 
  count = 0;
  n = j;
  for(m = -HALF_WIDTH + i; m <= HALF_WIDTH + i; m++)
     if (interiorlabel[m][n])
       {
        sum_agree += (label == lattice[m][n]);
        count++;
       }
 
  if ( ((double) sum_agree) > (0.75 * ((double) count)) )
    {
     n = j - 1;
     for(m = -((int) 1.5*HALF_WIDTH) + i; m <= ((int) 1.5*HALF_WIDTH) + i; m++)
        if (interiorlabel[m][n])
           sum_agree_j_minus += (label == lattice[m][n]);
 
     n = j + 1;
     for(m = -((int) 1.5*HALF_WIDTH) + i; m <= ((int) 1.5*HALF_WIDTH) + i; m++)
        if (interiorlabel[m][n])
           sum_agree_j_plus += (label == lattice[m][n]);
 
     if ( ((double) sum_agree_j_minus) > (0.75 * ((double) sum_agree)) &&
          ((double) sum_agree_j_plus)  > (0.75 * ((double) sum_agree)))
        return (0);
     else
        return (1);
    }
  else
     return (0);
 }


/* read index from the file fp into the array img */
void read_neighbor_sys(img, fp, LN)
int *img;
int fp;  /* input file descriptor */
int LN;  /* size of img */
{
  read(fp, (char *) img, sizeof(int) * LN);

}/*read_neighbor_sys*/


/* read the disparity values from the file fp into the array img */
/* the img array is LROW X LCOL X NN, but the input data file */
/* has LROW records, NN traces per record, and LCOL samples per */
/* trace.  This function places each trace from the USP file into */
/* the correct elements of the img array. */ 
void read_disparity(img, fp, usp_trace_bytes)
int ***img;
int fp;     /* input file descriptor */
int usp_trace_bytes;    /* size of the input and output usp traces */
{
  float *usp_input_trace;
  int i, j, k, nn;

  usp_input_trace = (float *) malloc(usp_trace_bytes); 

  /* loop over records */
  for(i = 0; i < LROW; i++)
    {
     /* loop over traces */
     for(nn = 0; nn < NN; nn++)
       {
        rtape(fp, usp_input_trace, &usp_trace_bytes);
        /* loop over samples */
        for(j = 0, k = ITRWRD; j < LCOL; j++, k++)
           img[i][j][nn] = (int) usp_input_trace[k];
       }
    }

  free(usp_input_trace);
}


/* read the coherency values from the file fp into the array img */
/* the img array is LROW X LCOL X Q4, but the input data file */
/* has LROW records, Q4 traces per record, and LCOL samples per */
/* trace.  This function places each trace from the USP file into */
/* the correct elements of the img array. */
void read_coherency(img, fp, usp_trace_bytes)
 double ***img;
 int fp;        /* input file descriptor */
 int usp_trace_bytes;     /* size of the input trace */
 {
  float *usp_input_trace;
  int i, j, k, nn;

  usp_input_trace = (float *) malloc(usp_trace_bytes);

  /* loop over records */
  for(i = 0; i < LROW; i++)
    {
     /* loop over traces */
     for(nn = 0; nn < Q4; nn++)
       {
        rtape(fp, usp_input_trace, &usp_trace_bytes);
        /* loop over samples */
        for(j = 0, k = ITRWRD; j < LCOL; j++, k++)
           img[i][j][nn] = (double) usp_input_trace[k];
       }
    }

  free(usp_input_trace);
 }


/* write the label lattice to the file fp as a one record USP file */
/* the ouput file will have 1 record, LROW traces per record, and */
/* LCOL samples per trace.  The trace numbers will have the */
/* same trace number as the trace of data values (in the input to */
/* texture) to which the trace of label values maps, */
/* i.e. i * COARSE + HWW */
/* NOTE:  if bethe is segmenting the data set in into n partitions, */
/* it uses labels 0...n-1, but the program brightsize expects the */
/* labels to be in the range 1...n.  So one is added to all the */
/* lables before they are placed into usp_output_trace. */
void writeintlattice(img, fp, usp_trace_bytes)
int **img;
int fp;     /* output file descriptor */
int usp_trace_bytes;    /* size of the input and output usp traces */
{
  float *usp_output_trace;
  int i, j, k;

  usp_output_trace = (float *) malloc(usp_trace_bytes);

  /* Set the Static Correction field in the traceheader */
  savew(usp_output_trace, "StaCor", 0, TRACEHEADER);
  /* set the RecNum field in the traceheader */
  savew(usp_output_trace, "RecNum", 1, TRACEHEADER);

  /* loop over traces */
  for(i = 0; i < LROW; i++)
    {
     /* set the TrcNum field in the traceheader */ 
     savew(usp_output_trace, "TrcNum", i * COARSE + HWW, TRACEHEADER);
     /* loop over samples */
     for(j = 0, k = ITRWRD; j < LCOL; j++, k++)
        usp_output_trace[k] = (float) (img[i][j] + 1);
     wrtape(fp, usp_output_trace, usp_trace_bytes);
    }

  free(usp_output_trace);
}


void bethe2d_help()
 {
  fprintf(stderr, "\
  This program uses the bethe tree approximation of a\n\
  Markov Random Field to perform texture segmentation.\n\
  For more information on the bethe tree approximation\n\
  technique refer to (Wu and Doerschuk).  The estimator\n\
  used is the maximizer of the posterior marginals (MPM),\n\
  see (Marroquin, et al.).  The choice of penalty functions\n\
  can be tailored to the data set and to the textures of\n\
  interest.  For large, blocky textures the penalty functions\n\
  TooSmall and TooThin work well.  The penalty functions\n\
  ver_channel_follower and hor_channel_follower are\n\
  designed to look for channels. \n\
  \n\
  REFERENCES:  Wu, C. and Doerschuk, P., 'Texture-Based Segmentation\n\
               Using Markov Random Field Models and Approximate Bayesian\n\
               Estimators Based on Trees,' Jour. of Math. Imaging and\n\
               Vision, vol. 5, (1995), pp. 277-286.\n\
  \n\
               Marroquin, J., Mitter, S., and Poggio, T., 'Probabilistic\n\
               Solution of Ill-posed Problems in Computer Vision,' Jour.\n\
               of the American Statistical Assoc., March 1987, vol. 82,\n\
               no. 397, pp. 76-89.\n\
  \n\
  Usage:  bethe2d [-N <dis_file>] -fmat <fifn> -bmat <bifn> -lambda <lbdfn>\n\
                  [-O <output_file>] [-s | -p] [-coarse <coarseness>]\n\
                  [-st | -ch]\n\
                  [-halfwin <half_win_size>] [-par <num_par>] [-j <j>]\n\
                  [-tree_depth <td>] [-coh <coh_file>]\n\
                  [-h -? -help]\n\
  \n\
  -N <dis_file> - the input file containing the disparity\n\
                  values between all neighboring label sites.\n\
                  This file is generated by the program\n\
                  disparity or the program combine.\n\
                  (Default:  stdin)\n\
  \n\
  -fmat <fifn> - the input file containing the forward index used\n\
                 for locating random neighbors, generated by the\n\
                 program randomgraph\n\
  \n\
  -bmat <bifn> - the input file containing the backward index used\n\
                 for locating random neighbors, generated by the\n\
                 program randomgraph\n\
  \n\
  -lambda <lbdfn> - the input file containing the lambda schedule.\n\
                    The i-th lambda value is the weight on the\n\
                    penalty functions during the i-th iteration\n\
                    (see Wu & Doerschuk)\n\
  \n\
  -O <output_file> - the output file containing the segmented\n\
                     data.\n\
                     (Default:  stdout)\n\
  \n\
  [-s | -p] - the type of update scheme. Serial updating is used when\n\
              the -s flag is set. Parallel updating is used when the\n\
              -p flag is set (see Wu & Doerschuk).  Only one of these\n\
              flags can be specified on the command line.\n\
              (Default: -s)\n\
  \n\
  -coarse <coarseness> - inverse of the resolution at which the\n\
                         segmentation is performed.  That is, the ratio\n\
                         of the number of samples/traces in the data\n\
                         lattice to the number of samples/traces in the\n\
                         label lattice.  The data lattice is the input to\n\
                         the program texture, and the label lattice is the\n\
                         output from any segmentation optimizer, such as\n\
                         bethe2d.  For highest resolution set the coarseness\n\
                         to 1.\n\
                         (Default: 9)\n\
  \n\
  [-st | -ch] - the type of penalty function to use.  If the -st\n\
                flag is set the penalty functions TooSmall and\n\
                TooThin are used.  If the -ch flag is set the\n\
                penalty functions ver_channel_follower and\n\
                hor_channel_follower are used.  Only one of\n\
                these flags can be specified on the command\n\
                line.\n\
                (Default:  -st)\n\
  \n\
  -halfwin <half_win_size> - half window width used to determine\n\
                             the width of the texture analysis window\n\
                             over which the cdf's are computed in the\n\
                             program disparity.  The width of the texture\n\
                             analysis window is given by\n\
                             2 * <half_win_size> + 1.\n\
                             (Default: 7)\n\
  \n\
  -par <num_par> - the number of partitions to segment the data\n\
                   set into.  After segmentation, labels with the\n\
                   same value belong to a partition.  Each partition\n\
                   represents a specific texture class.\n\
                   (Default:  2)\n\
  \n\
  -j <j> - the weight on the cost function (see Wu & Doerschuk).\n\
           (Default:  1)\n\
  \n\
  -tree_depth <td> - the depth at which the bethe approximation\n\
                     is terminated (see Wu & Doerschuk).\n\
                     (Default:  1)\n\
  \n\
  -coh <coh_file> - when this option is specified on the command line\n\
                    the coherence measures in the file <coh_file>\n\
                    are used as weights on the near neighbor\n\
                    terms of the cost function, i.e. the coherence\n\
                    between two sites, which are near neighbors,\n\
                    determines the contribution of the corresponding\n\
                    near neighbor term in the cost function.  The file\n\
                    <coh_file> is produced by the program cogeom.  When\n\
                    this option is not specified on the command line\n\
                    weighting is not applied to the near neighbor terms.\n\
  \n\
  -h, -?, -help - Help.\n\
");
}
