C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************

c     program module davc
c
c**********************************************************************c
c
c davc reads seismic trace data from an input file,
c applies a user-specified agc, and
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>
 
      parameter   (mmax = 100)
 
      integer     itr ( 2*SZLNHD )
      integer     lhed( 2*SZLNHD )
      real        head( 2*SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, lw, nwind
      integer     luin , luout, lbytes, nbytes, lbyout,lbyte
      integer     irs,ire,ns,ne, ist, iend
      integer     recnum, trcnum, static, poly
      real        amp, scal1
      real        tri ( 2*SZLNHD ), gain ( 2*SZLNHD )
      real        sig ( 2*SZLNHD ), time ( 2*SZLNHD )
      real        covar (mmax), A(mmax)
      integer     lista(mmax)
      character   ntap * 255, otap * 255, name*4, gtap * 255
      logical     verbos, agc, med, rms, mute
      integer     argis, itype, idec, pipe
 
c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1), head(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'DAVC'/
      data amp  / 307.05 /
      data pipe/3/
 
c-----
c     read program parameters from command line card image file
c-----
      if ( argis ( '-?' ) .gt. 0 .or.
     :     argis ( '-h' ) .gt. 0 .or.
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif
 
c-----
c     open printout files
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,ns,ne,irs,ire,scal1,nwind,agc,verbos,
     1            poly, itype, med, idec, rms, ist, iste, iend, vel,
     2            gtap,mute)
 
      if (poly*poly .gt. mmax) then
         write(LERR,*)'Order of polynomial ',poly,' too high'
         amax = mmax
         imax = sqrt(amax)
         write(LERR,*)'Cannot be greater than ',imax
         stop
      endif
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)
 
c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape   ( luin, itr, lbyte)
      if(lbyte .eq. 0) then
         write(LERR,*)'DAVC: no header read from unit ',luin
         write(LERR,*)'FATAL'
         stop
      endif
 
c------
c     save certain parameters
 
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif
 
c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
 
 
c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
c-----
      call hlhprt (itr, lbyte, name, 4, LERR)
      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr=ne-ns+1
      call savew(itr, 'NumTrc', jtr  , LINHED)
 
      call savhlh(itr,lbyte,lbyout)
      call wrtape ( luout, itr, lbyout                 )
 
         dt = real (nsi) * unitsc
	 lw = ( nwind / nsi)
 
      ist = ist / nsi
      if (ist .lt. 1) ist = 1
      iste = iste / nsi
      if (iste .lt. 1) iste = 1
      iend = iend / nsi
      if (iend .lt. 1) iend = nsamp
      iscl = iend - ist + 1
      veldt = vel * dt
 
c-----
c     get correct window length ensuring an odd number of samples
c-----
      if(scal1 .ne. 0.0)then
            amp = 0.01 * scal1 * 2047.
      endif
      if(lw .le. 0 .or. lw .gt. nsamp)then
            lw = nsamp / 2
      endif
      lw = lw + ( mod (lw,2) - 1 )
c-----
c     generate vector of 2-way times
c-----
      do  10  i = 1, nsamp
 
          time (i) = float ( i ) * dt
          sig (i) = 1.0
10    continue
 
c-----
c     find limits
c-----
      call maxmgv (time, 1, xmax, indx, nsamp)
      call minmgv (time, 1, xmin, indx, nsamp)
 
c-----
c     generate list of parameters
c-----
      do  11  i = 1, mmax
 
          lista (i) = i
11    continue
 
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
c     if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform, lw,
     1                  scal1,amp,nwind,ntap,otap,agc,poly,itype,med,
     2                  idec, rms, ist,iste, iend, vel, mute)
c     endif
      idec = idec / nsi
      if (idec .lt. 1) idec = 1
c-----
c     BEGIN PROCESSING
c     read trace, do agc, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
c-----
c     process desired trace records
c-----
      ir = 0
      DO  1000  JJ = irs, ire
 
            istr = ir * (iste - ist)/float(nrecc)
            ir = ir + 1
c----------------
c  skip to start
c  of record
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c----------------
 
            do  1001  KK = ns, ne
 
                  nbytes = 0
                  call rtape ( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
 
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        idist  , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)
                  dist  = iabs( idist )
 
                  if (mute) then
                      call detmut ( tri, is, nsamp)
                      call bd_detmut ( tri, ie, nsamp, 1)
                  else
                      is     = istr + ist  + dist/veldt
                      ie     = istr + iend + dist/veldt
                  endif
                  if (is .lt. 1) is = 1
                  if (ie .gt. nsamp ) ie = nsamp
                  nsr    = ie - is + 1
 
                  call vfill (1.0, gain, 1, 2*nsamp)
 
                  IF(static .ne. 30000)then
 
                     if     (med) then
                        call dagcm(nsr,lw,amp,tri(is),gain(is),idec)
                     elseif (rms) then
                        call dagcsq (tri(is),gain(is),nsr,lw,amp)
                     else
                        call dagcab (tri(is),gain(is),nsr,lw,amp)
                     endif
 
                        if (poly .ne. 0) then
                           call lfit (time, gain, sig, nsamp, A, poly,
     1                                lista, poly, covar, poly, chisq,
     2                                xmin, xmax, itype)
                        endif
 
                        if (ie .lt. nsamp) then
                           do  19  ii = ie, nsamp
                               gain(ii) = 1.0
19                         continue
                        endif
 
                        if (agc) then
                            call vmov (gain,1,tri,1,nsamp)
                        else
                            call vmult(nsamp, tri, gain)
                        endif
 
                  ELSE
 
                        call vclr(tri,1,nsamp)
 
                  ENDIF
 
                  call vmov ( tri, 1, lhed(ITHWP1), 1, nsamp)
                  call wrtape( luout, itr, nbytes)
 
 1001             continue
 
c----------------
c  skip to end of
c  current record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c----------------
            if(verbos) write(LERR,*)'davc:  ri ',recnum
 
 1000       CONTINUE
c-----
c     close data files
c-----
  999 continue
      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'end of davc, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       HELP                                                 *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  ???                                                       *
C  ENTRY POINTS:                                                       *
C      HELP                                                            *
C  ARGUMENTS:     NONE                                                 *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 96/04/08  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 96/04/08  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:                                                              *
C      LER  ( OUTPUT SEQUENTIAL ) -                                    *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
      write(LER,*)
     :'execute davc by typing davc and a list of program parameters.'
      write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
      write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
      write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)       : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)       : output data file name'
c       write(LER,*)
c    :' -G [gtap]    (no output)        : optional gain file name'
      write(LER,*)' '
        write(LER,*)
     :' -ns[ns]      (default = first)  : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)   : end trace number'
        write(LER,*)
     :' -rs[irs]      (default = first) : start record number'
        write(LER,*)
     :' -ne[ire]      (default = last)  : end record number'
        write(LER,*)
     :' -w [window]  (default = 500ms)  : agc window in ms'
        write(LER,*)
     :' -s [scale]   (default = 15%)    : scale ave amp to s % of 2047'
        write(LER,*)
     :' -ts [ist]    (default = 0ms)    : scaling start time (ms) at rec
     : irs'
        write(LER,*)
     :' -te [iste]   (default = ist)    : scaling start time (ms) at rec
     : ire'
        write(LER,*)
     :' -mute                           : start time follows mute line'
        write(LER,*)
     :' (must have applied previous on-mute; other start time options ig
     :nored)'
        write(LER,*)
     :' -u [iend]  (default = last samp): scaling end time (ms) at rec i
     :rs'
        write(LER,*)
     :' -v [vel]     (default = flat)   : start time linear velocity'
        write(LER,*)
     :' -p [poly]    (default = 0)      : fit nth order poly to gain'
        write(LER,*)
     :' -f [itype]   (default = 0)      : type of function to fit:'
        write(LER,*)
     :'      -f0 {nth order polynomial in x: a0 +a1 X +a2 X**2 + ...}'
        write(LER,*)
     :'      -f1 {nth order gaussian polynomial in exp(x**2)}'
      write(LER,*)' '
        write(LER,*)
     :' -agc                            : output agc curves only'
        write(LER,*)
     :' -rms                            : compute rms curve, otherwise'
        write(LER,*)
     :'                                   compute average value curve'
        write(LER,*)
     :' -M                              : use median gain calculation'
        write(LER,*)
     :' -d [idec]   (default = 20ms)    : window increment for median'
        write(LER,*)
     :' -V                              : verbose printout'
      write(LER,*)' '
         write(LER,*)
     :'usage:   davc -N[ntap] -O[otap] -w[window] -s[scale] -p[poly]'
         write(LER,*)
     :'              -ns[ns] -ne[ne] -rs[irs] -re[ire] -f[]'
         write(LER,*)
     :'              [ -ts[] -te[] -u[] -v[] -mute]'
         write(LER,*)
     :'              [ [ -agc -G[] ] -rms -M -d[] -V]'
         write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       GCMDLN                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  ???                                                       *
C  ENTRY POINTS:                                                       *
C      GCMDLN  (NTAP,OTAP,NS,NE,IRS,IRE,SCAL1,NWIND,AGC,VERBOS,POLY,   *
C               ITYPE,MED,IDEC,RMS,IST,ISTE,IEND,VEL,GTAP,MUTE)        *
C  ARGUMENTS:                                                          *
C      NTAP    CHAR*(*)  ??IOU* -                                      *
C      OTAP    CHAR*(*)  ??IOU* -                                      *
C      NS      INTEGER   ??IOU* -                                      *
C      NE      INTEGER   ??IOU* -                                      *
C      IRS     INTEGER   ??IOU* -                                      *
C      IRE     INTEGER   ??IOU* -                                      *
C      SCAL1   INTEGER   ??IOU* -                                      *
C      NWIND   INTEGER   ??IOU* -                                      *
C      AGC     LOGICAL   ??IOU* -                                      *
C      VERBOS  LOGICAL   ??IOU* -                                      *
C      POLY    INTEGER   ??IOU* -                                      *
C      ITYPE   INTEGER   ??IOU* -                                      *
C      MED     LOGICAL   ??IOU* -                                      *
C      IDEC    INTEGER   ??IOU* -                                      *
C      RMS     LOGICAL   ??IOU* -                                      *
C      IST     INTEGER   ??IOU* -                                      *
C      ISTE    INTEGER   ??IOU* -                                      *
C      IEND    INTEGER   ??IOU* -                                      *
C      VEL     REAL      ??IOU* -                                      *
C      GTAP    CHAR*(*)  ??IOU* -                                      *
C      MUTE    LOGICAL   ??IOU* -                                      *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 96/04/08  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 96/04/08  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:                                                    *
C      ARGSTR          -                                               *
C      ARGI4           -                                               *
C      ARGR4           -                                               *
C      ARGIS   INTEGER -                                               *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      IFIX    INTEGER -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,scal1,nwind,agc,verbos,
     1            poly, itype, med, idec,rms, ist, iste, iend, vel,
     2             gtap,mute)
c-----
c     get command arguments
c
c     ntap  - c*255     input file name
c     otap  - c*255     output file name
c     ns    - i*4 starting trace index
c     ne    - i*4 ending trace index
c     irs   - i*4 starting record index
c     ire   - i*4 ending record index
c     scal1 - i*4 trace amplitude is held near scal1 % of 2047
c     ist     i*4 scaling start time
c     iend    i*4 scaling end time
c     vel     r*4 scaling start time velocity
c     nwind - i*4 agc window in ms
c     poly    i*4 fit nth order polynomial to gain curve
c     itype   i*4 type of function to fit
c     idec    i*4 for median calculation sample window every idec samples
c     med         - L   use median gain calculation
c     agc         - L   output agc curves only
c     rms         - L   rms curve, else average curve
c     verbos      - L   verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), gtap*(*)
      integer     ns, ne, irs, ire, nwind
      integer     ist, iend
      real        vel, scal1
      logical     verbos, agc, med, rms, mute
      integer     argis, itype, poly, idec
 
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argstr( '-G', gtap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argi4 ( '-ts', ist ,   0  ,  0    )
            call argi4 ( '-te', iste , -999  , -999   )
            if (ist .eq. 0 .and. iste .eq. -999) then
               call argi4 ( '-t', ist ,   0  ,  0    )
            endif
            if (iste .eq. -999) iste = ist
            call argi4 ( '-u', iend ,   0  ,  0    )
            call argr4 ( '-v', vel ,   9999999.  ,  9999999.    )
            call argr4( '-s', scal, 0., 0. )
            call argr4( '-w', wind, 500., 500. )
            call argi4( '-p', poly, 0, 0 )
            call argi4( '-f', itype, 0, 0 )
            scal1 = scal
            nwind = ifix(wind)
            med    = ( argis( '-M' ) .gt. 0 )
            if (med)
     1      call argi4 ( '-d', idec, 20, 20)
            agc    = ( argis( '-agc' ) .gt. 0 )
            rms    = ( argis( '-rms' ) .gt. 0 )
            mute   = ( argis( '-mute' ) .gt. 0 )
            verbos = ( argis( '-V' ) .gt. 0 )
 
      return
      end
 
C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       VERBAL                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:  ???                                                       *
C  ENTRY POINTS:                                                       *
C      VERBAL  (NSAMP,NSI,NTRC,NREC,IFORM,LW,SCAL1,AMP,NWIND,NTAP,     *
C               OTAP,AGC,POLY,ITYPE,MED,IDEC,RMS,IST,ISTE,IEND,VEL,    *
C               MUTE)                                                  *
C  ARGUMENTS:                                                          *
C      NSAMP   INTEGER   ??IOU* -                                      *
C      NSI     INTEGER   ??IOU* -                                      *
C      NTRC    INTEGER   ??IOU* -                                      *
C      NREC    INTEGER   ??IOU* -                                      *
C      IFORM   INTEGER   ??IOU* -                                      *
C      LW      INTEGER   ??IOU* -                                      *
C      SCAL1   INTEGER   ??IOU* -                                      *
C      AMP     REAL      ??IOU* -                                      *
C      NWIND   INTEGER   ??IOU* -                                      *
C      NTAP    CHAR*(*)  ??IOU* -                                      *
C      OTAP    CHAR*(*)  ??IOU* -                                      *
C      AGC     LOGICAL   ??IOU* -                                      *
C      POLY    INTEGER   ??IOU* -                                      *
C      ITYPE   INTEGER   ??IOU* -                                      *
C      MED     LOGICAL   ??IOU* -                                      *
C      IDEC    INTEGER   ??IOU* -                                      *
C      RMS     LOGICAL   ??IOU* -                                      *
C      IST     INTEGER   ??IOU* -                                      *
C      ISTE    INTEGER   ??IOU* -                                      *
C      IEND    INTEGER   ??IOU* -                                      *
C      VEL     REAL      ??IOU* -                                      *
C      MUTE    LOGICAL   ??IOU* -                                      *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 96/04/08  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 96/04/08  *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
C  FILES:                                                              *
C      LERR  ( OUTPUT SEQUENTIAL ) -                                   *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform, lw,
     1            scal1,amp,nwind,ntap,otap,agc,poly,itype,med,
     2                  idec,rms, ist,iste, iend, vel, mute)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c     lw    - I*4 agc filter length
c     scal1 - I*4 percentage scaling factor
c     amp   - R*4 default peak amplitude
c     poly    i*4 fit nth order polynomial to gain curve
c     itype   i*4 type of function to fit
c     idec    i*4 for median calculation sample window every idec samples
c     nwind - I*4 window length in milliseconds
c     ntap  - C*255     input file name
c     otap  - C*255     output file name
c     agc         - L   output agc curves only
c     rms         - L   rms curve, else average curve
c     med         - L   use median gain calculation
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec, iform, lw, nwind, poly
      integer     itype, idec, ist, iend
      real        amp, vel, scal1
      logical     agc, med, rms, mute
      character   ntap*(*), otap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' agc filter length  =  ', lw
 
            if (mute) then
            write(LERR,*) ' start time follows on-mute'
            else
            write(LERR,*) ' scaling start time =  ', ist,' samps'
            write(LERR,*) ' ... at start record'
            write(LERR,*) ' scaling start time =  ', iste,' samps'
            write(LERR,*) ' ... at end record'
            write(LERR,*) ' scaling end time   =  ', iend,' samps'
            write(LERR,*) ' ... at start record'
            write(LERR,*) ' start time velocity=  ', vel
            endif
 
            if(scal1 .ne. 0.0) then
                  write(LERR,*) ' scale %             =  ', scal1
            else
                  write(LERR,*)  ' amplitude factor    =  ', amp
            endif
            write(LERR,*) ' window length in ms =  ', nwind
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            if (rms) then
                write(LERR,*)'Compute rms curve'
            else
                write(LERR,*)'Compute average value curve'
            endif
            if (agc)
     1      write(LERR,*)'Output AGC curves'
            if (med) then
            write(LERR,*)'Using median gain curve calculation'
            write(LERR,*)'Sampling every ',idec,' samples'
            endif
            if (poly .ne. 0) then
            write(LERR,*)' Order of polynomial to fit to gain= ',poly
            write(LERR,*)' Function type= ', itype
            endif
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
