C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c----------------------------------------------------------------------
c     USP Routine dwt2d
c
c     Author: Paul G. A. Garossino   [Feb:94]
c----------------------------------------------------------------------
c
c     routine to perform a forward or reverse discrete 2D wavelet 
c     transform of input seismic record.  Based on following routines 
c     from  Numerical Recipes, Press  et.al.  2nd Edition, 
c     pp. 589,590,595
c
c     pwtset(), pwt(), wtn()

#include <save_defs.h>
#include <localsys.h>
#include <f77/iounit.h>
#include <f77/sisdef.h>
#include <f77/lhdrsz.h>

c Standard USP variables

      integer lhed(SZLNHD)
      integer luin, luout, nbytes, obytes, argis
      integer nsamp, nsi, nrec, ntrc, iform, lbytes
      integer nsampo, ntrco
      integer irs, ire, ns, ne, start, end
      
      character   name * 5, ntap * 255, otap * 255

      logical verbos

c variables used with dynamic memory allocation

      integer errcd1, errcd2, abort, itemHeader, itemData
      integer headers

      real data

      pointer ( wkadr1, headers(200000) )
      pointer ( wkadr2, data(200000) )

c program dependant variables

      integer NextPowerOf2, size, JJ, KK
      integer trndx, hdrndx, itotal, level_req
      integer ordfft, nsampR, ntrcR

      real    tfact, pi

      logical remove, peak

c Press et. al. variables

      integer NumCoefs, ioff, joff, nn(SZLNHD)

      real cc(SZLNHD), cr(SZLNHD)

c initialize variables

      data  nbytes / 0 /
      data  lbytes / 0 /
      data  abort / 0 /
      data  name/'DWT2D'/
      data  peak/.false./
      pi =  4. * atan(1.0)

C get command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0  .or. 
     :     argis ( '-help' ) .gt. 0  )then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

C read command line parameters

      call cmdln ( ntap, otap, irs, ire, ns, ne, start, end, remove, 
     :     NumCoefs, level_req, peak, verbos )

c open input and output datasets

      call getln ( luin, ntap, 'r', 0 )
      call getln ( luout, otap, 'w', 1 )

c input dataset verification

      lbytes=0
      call rtape  ( luin, lhed, lbytes )
      if(lbytes .eq. 0) then
         write(LERR,*)'DWT2D: no header read on unit ',ntap
         write(LERR,*)'       Check existence/permissions & rerun'
         write(LERR,*)'FATAL'
         stop
      endif

c save required input line header descriptors, the forward ntrc and nsamp
c are stored in OrNTRC and OrNSMP

      if ( remove ) then
         call saver(lhed, 'OrNSMP', nsampR, LINHED)
         call saver(lhed, 'OrNTRC', ntrcR , LINHED)
      endif

      call saver(lhed, 'NumSmp', nsamp, LINHED)
      call saver(lhed, 'NumTrc', ntrc , LINHED)
      call saver(lhed, 'SmpInt', nsi  , LINHED)
      call saver(lhed, 'Format', iform, LINHED)
      call saver(lhed, 'NumRec', nrec , LINHED)
      call saver(lhed, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(lhed, 'UnitSc', unitsc, LINHED)
      endif

c handle defaults for rec, trc, sample

      if ( irs .eq. 0 ) irs = 1
      if ( ire .eq. 0 ) ire = nrec
      nreco = ire - irs + 1

      if ( ns .eq. 0 ) ns = 1
      if ( ne .eq. 0 ) ne = ntrc

c handle microseconds

      tfact = unitsc

      if ( start .eq. 0 ) then
         start = 1
      else
         start = nint ( float(start) / float(nsi) )
      endif

      if ( end .eq. 0 ) then
         end = nsamp
      else
         end = nint( float(end) / float(nsi) )
      endif
      
      nsamp = end - start + 1

      NextPowerOf2 = ordfft(nsamp)
      nsampo = 2 ** NextPowerOf2

      NextPowerOf2 = ordfft(ntrc)
      ntrco = 2 ** NextPowerOf2

      if(nsampo .gt. 8192) then
         write(LERR,*) 'DWT2D: Trace length too long to transform'
         write(LERR,*) '          Be sure next power of 2 is less than'
         write(LERR,*) '          8192 or contact the USP guys at APR'
         write(LERR,*) '          for a program change'
         write(LERR,*) 'Fatal'  
         write(LERR,*) ' '
      
         write(LER,*) 'DWT2D: Trace length too long to transform'
         write(LER,*) '          Be sure next power of 2 is less than'
         write(LER,*) '          8192 or contact the USP guys at APR'
         write(LER,*) '          for a program change'
         write(LER,*) 'Fatal'        
         write(LER,*) ' '
         stop
      endif
      
c load nn array for wtn[] routine

      nn(1) = nsampo
      nn(2) = ntrco
      itotal = nsampo * ntrco
      ndim = 2

c dynamic memory allocation

      itemHeader = ntrco * ITRWRD * SZSMPD
      itemData = ntrco * nsampo * SZSMPD

c make sure there is enough memory to contain the larger of the input or
c output dataset.

      if ( ntrco .ge. ntrc ) then
         if ( nsampo .lt. nsamp ) itemData = ntrco * nsamp * SZSMPD
      else 
         if ( nsampo .ge. nsamp ) then
            itemHeader = ntrc * ITRWRD * SZSMPD
            itemData = ntrc * nsampo * SZSMPD
         else
            itemHeader = ntrc * ITRWRD * SZSMPD
            itemData = ntrc * nsamp * SZSMPD
         endif
      endif  
    
      call galloc (wkadr1, itemHeader, errcd1, abort)
      call galloc (wkadr2, itemData, errcd2, abort)

      if ( errcd1 .ne. 0 .or. errcd2 .ne. 0 ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemHeader,' bytes'
         write(LERR,*) itemData,' bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemHeader,' bytes'
         write(LERR,*) itemData,' bytes'
         write(LERR,*)' '
      endif

c update HLH

      call hlhprt ( lhed , lbytes, name, 5, LERR )
  
c modify header as needed

      if ( .not. remove ) then

c forward transform, store original ntrc and nsamp in
c line header 

         call savew( lhed, 'OrNTRC', ntrc, LINHED)
         call savew( lhed, 'OrNSMP', nsamp, LINHED)

c load next power of 2 stuff to regular nsamp,ntrc locations

         call savew( lhed, 'NumSmp', nsampo, LINHED)
         call savew ( lhed, 'NumTrc', ntrco, LINHED )

      else

c reverse transform, clear forward storage locations and store
c original nsamp,ntrc info to output line header

         call savew( lhed, 'OrNTRC', 0, LINHED)
         call savew( lhed, 'OrNSMP', 0, LINHED)
         call savew( lhed, 'NumSmp', nsampR, LINHED)
         call savew ( lhed, 'NumTrc', ntrcR, LINHED )

      endif 
     
      call savew ( lhed, 'NumRec', nreco, LINHED )

c add the command line to the HLH

      call savhlh ( lhed, lbytes, lbyout )

C write out modified line header

      call wrtape(luout, lhed, lbyout)

C input data and do dwt2d

      size = nsampo
      if ( nsamp .gt. nsampo) size = nsamp
      if ( ntrc .gt. ntrco ) then
         size = size * ntrc
      else
         size = size * ntrco
      endif

c initialize wavelet to use

      call pwtset ( NumCoefs, cc, cr, ioff, joff, peak )

c define number of output bytes

      if (remove) then
         obytes = SZTRHD + SZSMPD * nsampR
      else
         obytes = SZTRHD + SZSMPD * nsampo
      endif

c write parameters in use to printout file

         call verbal ( ntap, nsi, ntrc, nrec, iform, nsamp, irs, ire, 
     :        ns, ne, start, end, otap, ntrco, nreco, nsampo, NumCoefs, 
     :        remove, nsampR, ntrcR, peak, verbos )

c skip unwanted records

      call recskp(1,irs-1,luin,ntrc,lhed)

c start processing 

      DO JJ = irs, ire

c clear workspace

         call vclr ( data, 1, size )

c skip unwanted traces

         call trcskp(jj,1,ns-1,luin,ntrc,lhed)

         do KK = ns, ne
            trndx = (KK-1) * nsampo + 1
            hdrndx = (KK-1) * ITRWRD + 1

            nbytes = 0
            call rtape ( luin, lhed, nbytes )
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

            call vmov ( lhed(ITHWP1 + start -1), 1, data(trndx), 1, 
     :           nsamp )
            call vmov ( lhed, 1, headers(hdrndx), 1, ITRWRD )
         enddo
            
         if ( .not. remove ) then

c  whole record now loaded, do the forward 2D DWT

            call wtn ( data, nn, ndim, 1, cc, cr, NumCoefs, ioff, joff, 
     :           itotal, level_req )

         else
         
c Inverse 2D DWT

            call wtn ( data, nn, ndim, -1, cc, cr, NumCoefs, ioff, 
     :           joff, itotal, level_req )

         endif

         hdrndx = 1 - ITRWRD
         trndx = 1 - nsampo

         if ( .not. remove ) then

c output the forward transformed data

            do KK = 1, ntrco
               trndx = trndx + nsampo
               hdrndx = hdrndx + ITRWRD
               call vmov ( headers(hdrndx), 1, lhed, 1, ITRWRD )
               call vmov ( data(trndx), 1, lhed(ITHWP1), 1, nsampo )
               call wrtape( luout, lhed, obytes)
            enddo

         else

c output the reverse transformed data

            do KK = 1, ntrcR
               trndx = trndx + nsampo
               hdrndx = hdrndx + ITRWRD
               call vmov ( headers(hdrndx), 1, lhed, 1, ITRWRD )
               call vmov ( data(trndx), 1, lhed(ITHWP1), 1, nsampR )
               call wrtape( luout, lhed, obytes)
            enddo

         endif

c  skip to end of record

         call trcskp(jj,ne+1,ntrc,luin,ntrc,lhed)

      ENDDO

c finished processing, close all attached files and terminate

      call lbclos ( luin )
      call lbclos ( luout )

      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Normal Termination: processed ',nreco, 'records of 
     :',ntrco,' traces.'
      write(LER,*)'dwt2d: Normal Termination'
      stop

 999  continue

c something abnormal occured, usually premature eof

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)' Abormal Termination'
      write(LER,*)'dwt2d: Abnormal Termination'
      stop
      END
