/*
 *	Copyright (c) 1990 by Columbia University.
 */

#include <stdio.h>

#include "ikp_defines.h"
#include "ikp_externs.h"
#include "globalE.h"

extern void prc_draw(Process *p);
extern void prc_erase();

Net *net_create()
{
  Net *new;
  
  if ((new= (Net *) calloc(1, sizeof(Net))) == (Net *) NULL)
    ikp_err("Memory allocation error.", FATAL);
  new->net_prclist= (Process *) NULL;
  new->net_npipes= 0;
  new->net_pst= (Pipeslot *) NULL;
  new->net_next= (Net *) NULL;
  new->name=NULL;
  return(new);
}

/*
   Unlink a Net.
*/
int net_unlink(Net *nt)
{
  Net *mt;
  Pipeslot *pst;
  int i;
  
  if (nt == (Net *) NULL)
    return(IKP_FAILURE);
  
  /* close any pipe descriptors inadvertently left open */
  if (nt->net_pst != (Pipeslot *) NULL) {
    for (i= 0, pst= nt->net_pst; i < nt->net_npipes; i++, pst++) {
      if (pst->pst_flags & PST_RDACTIVE) {
	if (close(pst->pst_fd[0]) == -1)
	  ikp_err("Close error.", NONFATAL);
	pst->pst_flags&= ~PST_RDACTIVE;
      }
      if (pst->pst_flags & PST_WRACTIVE) {
	if (close(pst->pst_fd[1]) == -1)
	  ikp_err("Close error.",  NONFATAL);
	pst->pst_flags&= ~PST_WRACTIVE;
      }
    }
  }
  
  /* unlink from net list */
  if (nt == nets)
    nets= nt->net_next;
  else {
    for (mt= nets; mt->net_next != (Net *) NULL; mt= mt->net_next) {
      if (nt == mt->net_next)
	break;
    }
    if (nt == mt->net_next)
      mt->net_next= nt->net_next;
  }
  
  if (curr_net == nt)
    curr_net= (Net *) NULL;
  return(IKP_SUCCESS);
}

/*
   Destroy a Net and free all associated memory.
   The Net must already be unlinked, and all Processes
   must already be unlinked and freed.
*/
int net_free(Net *nt)
{
  if (nt == (Net *) NULL)
    return(IKP_FAILURE);
  if (nt->net_pst != (Pipeslot *) NULL)
    cfree((char *) nt->net_pst);
  cfree((char *) nt);
  
  return(IKP_SUCCESS);
}

void net_draw(Net *nt)
{
  Process *p;
  
  if (nt == (Net *) NULL)
    return;
  if (nt->net_prclist == (Process *) NULL)
    return;
  
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next)
    prc_draw(p);
  return;
}

void net_erase(Net *nt)
{
  Process *p;
  
  if (nt == (Net *) NULL)
    return;
  if (nt->net_prclist == (Process *) NULL)
    return;
  
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next)
    prc_erase(p);
  return;
}

/*
   Splits a potentially disconnected Net into
   connected subnets and returns the number of
   new Nets which have been created. 
*/
int net_split(Net *nt)
{
  int i;
  
  for (i= 0; net_bldsubnet(nt) == TRUE; i++);
  return(i);
}

/*
   Does a depth-first search of all Processes connected
   to the first Process in the net Process list. If this
   is a subset of said list, it is split off into a new
   Net and the function returns TRUE. If all Processes
   in the net are connected to each other, the function
   returns FALSE.
*/
boolean net_bldsubnet(Net *old)
{
  Net *new;
  Process *p, *q;
  void net_srchconnect();
  
  if (old == (Net *) NULL)
    return(FALSE);
  if (old->net_prclist == (Process *) NULL)
    return(FALSE);
  
  for (p= old->net_prclist; p != (Process *) NULL; p= p->prc_next)
    p->prc_flags&= ~PRC_NETMEMBER;
  
  net_srchconnect(old->net_prclist);
  
  /* all Processes still connected to each other? */
  for (p= old->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (!(p->prc_flags & PRC_NETMEMBER))
      break;
  }
  if (p == (Process *) NULL)
    return(FALSE);
  
  /* create new Net and fill it with all
     Processes flagged by PRC_NETMEMBER */
  if ((new= net_create()) == (Net *) NULL)
    return(FALSE);
  new->net_next= nets;
  nets= new;
  for (p= old->net_prclist; p != (Process *) NULL; p= q) {
    if (!(p->prc_flags & PRC_NETMEMBER))
      q= p->prc_next;
    else {
      if (p == old->net_prclist)
	old->net_prclist= p->prc_next;
      else {
	for (q= old->net_prclist; q->prc_next != p; q= q->prc_next);
	q->prc_next= p->prc_next;
      }
      q= p->prc_next;
      p->prc_next= new->net_prclist;
      new->net_prclist= p;
      p->prc_net= (caddr_t) new;
    }
  }
  return(TRUE);
}

/*
   Recursively mark p and any Processes connected to it
   via Connectors (either directly or indirectly) by
   setting the PRC_NETMEMBER flag bit.
*/
void net_srchconnect(Process *p)
{
  Connector *c;
  Process *q;
  
  if (p == (Process *) NULL)
    return;
  
  p->prc_flags|= PRC_NETMEMBER;
  for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
    if (c->cnt_conn != (Connector *) NULL) {
      q= (Process *) c->cnt_conn->cnt_process;
      if (!(q->prc_flags & PRC_NETMEMBER))
	net_srchconnect((Process *) c->cnt_conn->cnt_process);
    }
  }
  for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
    if (c->cnt_conn != (Connector *) NULL) {
      q= (Process *) c->cnt_conn->cnt_process;
      if (!(q->prc_flags & PRC_NETMEMBER))
	net_srchconnect((Process *) c->cnt_conn->cnt_process);
    }
  }
  for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
    if (c->cnt_conn != (Connector *) NULL) {
      q= (Process *) c->cnt_conn->cnt_process;
      if (!(q->prc_flags & PRC_NETMEMBER))
	net_srchconnect((Process *) c->cnt_conn->cnt_process);
    }
  }
  return;
}

void net_highlight(Net *nt)
{
  Process *p;
  
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
  }
  return;
}

Net *net_copy(Net *nt,
	      int x,int y)
{
  Net *newnet;
  Process *newprc;
  Process *p, *q;
  int ntx, nty;
  Process *prc_copy();
  void net_mktags();
  
  if ((newnet= net_create()) == (Net *) NULL)
    return((Net *) NULL);
  
  net_mktags(nt);
  
  /* compute origin of smallest bounding rectangle
     containing all process icons of existing net */
  if (nt->net_prclist != (Process *) NULL) {
    ntx= nt->net_prclist->prc_x;
    nty= nt->net_prclist->prc_y;
  }
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (ntx > p->prc_x)
      ntx= p->prc_x;
    if (nty > p->prc_y)
      nty= p->prc_y;
  }
  
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if ((newprc= prc_copy(p, x+p->prc_x-ntx, y+p->prc_y-nty, PRC_NETCONNECT)) == (Process *) NULL) {
      for (p= newnet->net_prclist; p != (Process *) NULL; p= p->prc_next) {
	prc_unlink(p);
	prc_free(p, PRC_RETAINNET);
      }
      net_free(newnet);
      return((Net *) NULL);
    }
    newprc->prc_net= (caddr_t) newnet;
    newprc->prc_next= (Process *) NULL;
    
    if (newnet->net_prclist == (Process *) NULL)
      newnet->net_prclist= newprc;
    else {
      for (q= newnet->net_prclist; q->prc_next != (Process *) NULL; q= q->prc_next);
      q->prc_next= newprc;
    }
    
  }
  
  /* connect all connectors, build segpaths */
  if (net_rdtags(newnet) != IKP_SUCCESS) {
    for (p= newnet->net_prclist; p != (Process *) NULL; p= p->prc_next) {
      prc_unlink(p);
      prc_free(p, PRC_RETAINNET);
    }
    net_free(newnet);
    return((Net *) NULL);
  }
  
  /* link into net list */
  newnet->net_next= nets;
  nets= newnet;
  
  return(newnet);
}

boolean net_editopen(Net *n)
{
  Process *p;
  Connector *c;
  
  for (p= n->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (p->prc_flags & PRC_EDITOPEN)
      return(TRUE);
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_flags & CNT_EDITOPEN)
	return(TRUE);
    }
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_flags & CNT_EDITOPEN)
	return(TRUE);
    }
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_flags & CNT_EDITOPEN)
	return(TRUE);
    }
  }
  return(FALSE);
}
