/*
 *	Copyright (c) 1990 by Columbia University.
 */

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>
#include <sys/wait.h>

#include "ikp_defines.h"
#include "ikp_externs.h"
#include "ikp_remote.h"
#include "globalE.h"

void net_chkchldprocs();
void net_chkremprocs();
void net_stop(Net *);
void net_restart(Net *);
void net_chkchldprocs_abortCB(Widget w,
			      XtPointer client_data,
			      XtPointer call_data);
void net_chkchldprocs_restartCB(Widget w,
				XtPointer client_data,
				XtPointer call_data);
void net_chkchldprocs_ignoreCB(Widget w,
			       XtPointer client_data,
			       XtPointer call_data);
void net_remchldprocs_restartCB(Widget w,
				XtPointer client_data,
				XtPointer call_data);
void net_poll_chkremprocs(Process *);

Process *p_chk;
int pid_chk;

void net_monitor()
{
  boolean chld_active, rem_active;
  Process *p;

  if (run_net == (Net *) NULL)
    return;
  
  chld_active= rem_active= FALSE;
  for (p= run_net->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (!(p->prc_flags & PRC_REMOTE) && !(p->prc_flags & PRC_EXITED))
      chld_active= TRUE;
    if ((p->prc_flags & PRC_REMOTE) && (p->prc_flags & PRC_SOCKOPEN))
      rem_active= TRUE;
  }
  if (!chld_active && !rem_active) {
    run_net= (Net *) NULL;
    if (usage_mode == INTERACTIVE)
      XDefineCursor(display,top_window, 
		    get_cursor(LXC_CCURSOR));
    return;
  }
  
  if (chld_active)
    net_chkchldprocs();
  
  /* the following could happen if
     net_chkchldprocs() calls net_abortfork() */
  if (run_net == (Net *) NULL)
    return;
  
  if (rem_active)
    net_chkremprocs();
  return;
}

void net_chkchldprocs()
{
  int status;
  int pid, sigtype, exitcode;
  int val,root_x,root_y;
  Process *p;
  char childnm[40], termmsg1[120], termmsg2[120];
  
  if (run_net == (Net *) NULL)
    return;

  if ((pid= waitpid(0,&status, WNOHANG | WUNTRACED)) < 0) {
    ikp_err("waitpid error.", NONFATAL);
    return;
  }
  
  /* no change in status of any child */
  if (pid == 0)
    return;
  
  for (p= run_net->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (!(p->prc_flags & PRC_REMOTE) && (p->prc_pid == pid))
      break;
  }
  
  /* pid may be associated with a process which was terminated
     during an earlier net_abortfork() and never wait()ed for */
  if (p == (Process *) NULL) {
    if (wtp_delete(pid) != IKP_SUCCESS) {
      (void) sprintf(ikp_errbuf, "Unrecognized child %d exited.", pid);
      ikp_err(ikp_errbuf, NONFATAL);
      return;
    }
    return;
  }

/*
  bcopy((char *) &status, (char *) &(p->prc_wstatus), sizeof(int));
*/
  memcpy((char *) &(p->prc_wstatus), (char *) &status, sizeof(int));
  
				/* Store values for callbacks. */
  p_chk=p;
  pid_chk=pid;

  /* process stopped by a signal sent by parent */
  if (WIFSTOPPED(status) && (p->prc_flags & PRC_IKPSTOPPED)) { 
    if (usage_mode == INTERACTIVE)
      prc_showactive(p);
    p->prc_flags&= ~PRC_RUNNING;
    p->prc_flags|= PRC_STOPPED;
    if (usage_mode == INTERACTIVE)
      prc_showactive(p);
  /* process stopped by a signal sent by someone other than the parent */
  } else 
    if (WIFSTOPPED(status)) {
      sigtype= (status >> 8) & 0377;
      if (usage_mode == INTERACTIVE) {
	prc_showactive(p);
	draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
      }
      p->prc_flags&= ~PRC_RUNNING;
      p->prc_flags|= PRC_STOPPED;
      prc_bldlabel(p, childnm, 39);
      
      switch (sigtype) {
      case SIGSTOP:
      case SIGTSTP:
	switch (usage_mode) {
	case INTERACTIVE:
	  locate_cursor(&root_x,&root_y);
	  sprintf(buf1,"Process '%s' (PID %d)\nstopped by signal %d.\nDo you wish to abort all processes?",
		  childnm,pid,sigtype);
	  XBell(display,100);
	  XtManageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
	  XtVaSetValues(confirm,
			XmNdefaultPosition,False,
			XmNdialogTitle,XMstr("Process Stopped"),
			XmNmessageString,XMstr(buf1),
			XmNokLabelString,XMstr("Abort"),
			XmNokCallback,CBl(net_chkchldprocs_abortCB,0),
			XmNcancelLabelString,XMstr("Restart"),
			XmNcancelCallback,CBl(net_chkchldprocs_restartCB,0),
			XmNhelpLabelString,XMstr("Ignore"),
			XmNhelpCallback,CBl(net_chkchldprocs_ignoreCB,0),
			XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
			XmNx,root_x,
			XmNy,root_y,
			NULL);
	  XtManageChild(confirm);
	  break;
	case BATCH:
	  (void) sprintf(termmsg1, "Process '%s' (PID %d) stopped by signal %d.", childnm, pid, sigtype);
	  ikp_err(termmsg1, NONFATAL);
	  if (!ignore_batcherrs)
	    net_abortfork(run_net);
	  break;
	}
	break;
      case SIGTTIN:
      case SIGTTOU:
      default:
	switch (usage_mode) {
	case INTERACTIVE:
	  locate_cursor(&root_x,&root_y);
	  sprintf(buf1,"Process '%s' (PID %d)\nwaiting on tty i/o stopped by signal %d.\nDo you wish to abort all processes?",
		  childnm,pid,sigtype);
	  XBell(display,100);
	  XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
	  XtVaSetValues(confirm,
			XmNdefaultPosition,False,
			XmNdialogTitle,XMstr("Process Stopped"),
			XmNmessageString,XMstr(buf1),
			XmNokLabelString,XMstr("Abort"),
			XmNokCallback,CBl(net_chkchldprocs_abortCB,0),
			XmNcancelLabelString,XMstr("Ignore"),
			XmNcancelCallback,CBl(net_chkchldprocs_ignoreCB,0),
			XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
			XmNx,root_x,
			XmNy,root_y,
			NULL);
	  XtManageChild(confirm);
	  break;
	case BATCH:
	  (void) sprintf(termmsg1, "Process '%s' (PID %d) waiting on tty i/o stopped by signal %d.", childnm, pid, sigtype);
	  ikp_err(termmsg1, NONFATAL);
	  if (!ignore_batcherrs)
	    net_abortfork(run_net);
	  break;
	}
	break;
      }
      /* process terminated by a signal sent from parent */
    } else 
      if (WIFSIGNALED(status) && (p->prc_flags & PRC_IKPKILLED)) {
	if (usage_mode == INTERACTIVE)
	  prc_showactive(p);
	p->prc_flags&= ~PRC_RUNNING;
	p->prc_flags&= ~PRC_STOPPED;
	p->prc_flags|= PRC_EXITED;
      }
  /* process terminated by a signal sent from someone other than the parent */
      else 
	if (WIFSIGNALED(status)) {
	  if (usage_mode == INTERACTIVE) {
	    prc_showactive(p);
	    draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
	  }
	  p->prc_flags&= ~PRC_RUNNING;
	  p->prc_flags&= ~PRC_STOPPED;
	  p->prc_flags|= PRC_EXITED;
	  sigtype= status & 0177;
	  prc_bldlabel(p, childnm, 39);
	  
	  switch (usage_mode) {
	  case INTERACTIVE:
	    locate_cursor(&root_x,&root_y);
	    sprintf(buf1,"Process '%s' (PID %d)\nterminated by signal %d.\nDo you wish to abort all processes?",
		    childnm,pid,sigtype);
	    XBell(display,100);
	    XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
	    XtVaSetValues(confirm,
			  XmNdefaultPosition,False,
			  XmNdialogTitle,XMstr("Process Stopped"),
			  XmNmessageString,XMstr(buf1),
			  XmNokLabelString,XMstr("Abort"),
			  XmNokCallback,CBl(net_chkchldprocs_abortCB,0),
			  XmNcancelLabelString,XMstr("Ignore"),
			  XmNcancelCallback,CBl(NULL,0),
			  XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
			  XmNx,root_x,
			  XmNy,root_y,
			  NULL);
	    XtManageChild(confirm);
	    break;
	  case BATCH:
fprintf(stderr,"Mark1\n");
	    (void) sprintf(termmsg1, "Process '%s' (PID %d) terminated by signal %d.", childnm, pid, sigtype);
	    ikp_err(termmsg1, NONFATAL);
	    if (!ignore_batcherrs)
	      net_abortfork(run_net);
	    break;
	  }
	}
  
  /* process exited */
	else 
	  if (WIFEXITED(status)) {
	    exitcode= status >> 8;
	    
	    /* perform sign extension if exit code appears negative */
	    if (exitcode & 0x80)
	      exitcode|= 0xffffff80;
	    
	    if (usage_mode == INTERACTIVE)
	      prc_showactive(p);
	    p->prc_flags&= ~PRC_RUNNING;
	    p->prc_flags&= ~PRC_STOPPED;
	    p->prc_flags|= PRC_EXITED;
	    if (exitcode != 0) {
	      prc_bldlabel(p, childnm, 39);
	      
	      switch (usage_mode) {
	      case INTERACTIVE:
		draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
		locate_cursor(&root_x,&root_y);
		sprintf(buf1,"Process '%s' (PID %d)\nexited with status %d.\nDo you wish to abort all processes?",
			childnm,pid,exitcode);
		XBell(display,100);
		XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
		XtVaSetValues(confirm,
			      XmNdefaultPosition,False,
			      XmNdialogTitle,XMstr("Process Stopped"),
			      XmNmessageString,XMstr(buf1),
			      XmNokLabelString,XMstr("Abort"),
			      XmNokCallback,CBl(net_chkchldprocs_abortCB,0),
			      XmNcancelLabelString,XMstr("Ignore"),
			      XmNcancelCallback,CBl(NULL,0),
			      XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
			      XmNx,root_x,
			      XmNy,root_y,
			      NULL);
		XtManageChild(confirm);
		break;
	      case BATCH:
		(void) sprintf(termmsg1, "Process '%s' (PID %d) exited with status %d.", childnm, pid, exitcode);
		ikp_err(termmsg1, NONFATAL);
		if (!ignore_batcherrs)
		  net_abortfork(run_net);
		break;
	      }
	    }
	  }
  
  /* since the main event loop is in a polling race
     alternating between XCheckIfEvent() and wait3(),
     pause after reaping each local child to share processing
     time with any remaining child processes */

  sleep(2);
  poll_count= 0;
  
  return;
}

void net_chkchldprocs_abortCB(Widget w,
			      XtPointer client_data,
			      XtPointer call_data)
{
  XDefineCursor(display,top_window,
		get_cursor(LXC_CCURSOR));

  draw_fill_rectangle(net_igc, p_chk->prc_x-CNT_LIP, p_chk->prc_y-CNT_LIP, p_chk->prc_w+(2*CNT_LIP), p_chk->prc_h+(2*CNT_LIP));
	
  net_abortfork(run_net);

  if ((int)client_data==1) 
    net_poll_chkremprocs(p_chk->prc_next);
}

void net_chkchldprocs_restartCB(Widget w,
				XtPointer client_data,
				XtPointer call_data)
{
  draw_fill_rectangle(net_igc, p_chk->prc_x-CNT_LIP, p_chk->prc_y-CNT_LIP, p_chk->prc_w+(2*CNT_LIP), p_chk->prc_h+(2*CNT_LIP));
	
  if (kill(pid_chk, SIGCONT) < 0)
    ikp_err("Restart failed -- process assumed dead.", NONFATAL);
	  
	  /* assume successful restart (this
	     may be a dangerous assumption!) */
  else {
    p_chk->prc_flags&= ~PRC_STOPPED;
    p_chk->prc_flags|= PRC_RUNNING;
    prc_showactive(p_chk);
  }
}

void net_remchldprocs_restartCB(Widget w,
				XtPointer client_data,
				XtPointer call_data)
{
  draw_fill_rectangle(net_igc, p_chk->prc_x-CNT_LIP, p_chk->prc_y-CNT_LIP, p_chk->prc_w+(2*CNT_LIP), p_chk->prc_h+(2*CNT_LIP));
	
  if (rp_cont(p_chk) != IKP_SUCCESS) {
    ikp_err("Restart failed -- net execution continuing.", NONFATAL);
    (void) rp_closedown(p_chk);
  } else {
    p_chk->prc_flags&= ~PRC_STOPPED;
    p_chk->prc_flags|= PRC_RUNNING;
    prc_showactive(p_chk);
  }

  if ((int)client_data==1) 
    net_poll_chkremprocs(p_chk->prc_next);
}

void net_chkchldprocs_ignoreCB(Widget w,
			       XtPointer client_data,
			       XtPointer call_data)
{
  XtUnmanageChild(w);

  draw_fill_rectangle(net_igc, p_chk->prc_x-CNT_LIP, p_chk->prc_y-CNT_LIP, p_chk->prc_w+(2*CNT_LIP), p_chk->prc_h+(2*CNT_LIP));
	
  prc_showactive(p_chk);

  if ((int)client_data==1) 
    net_poll_chkremprocs(p_chk->prc_next);
}


/*
   Polls all active remote process control sockets
   for remote child status change information.
*/
void net_chkremprocs()
{
  char status;
  int pid, code;
  int val,root_x,root_y;
  Process *p;
  char childnm[40], termmsg1[120], termmsg2[120];
  
  if (run_net == (Net *) NULL)
    return;
  
  net_poll_chkremprocs(run_net->net_prclist);
}

/*
   Polls all active remote process control sockets
   for remote child status change information.
*/
void net_poll_chkremprocs(Process *p)
{
  char status;
  int code,root_x,root_y,val;
  int dialog_flag=0;
  char childnm[40], termmsg1[120], termmsg2[120];

  if (p==NULL)
    return;

  p_chk=p;

  if (!(p->prc_flags & PRC_REMOTE) || !(p->prc_flags & PRC_SOCKOPEN)) {
    net_poll_chkremprocs(p->prc_next);
    return;
  }
    
  if (rp_status(p, &status, &code) != IKP_SUCCESS) {
    (void) rp_closedown(p);
    p->prc_flags&= ~PRC_RUNNING;
    p->prc_flags&= ~PRC_STOPPED;
    p->prc_flags|= PRC_EXITED | PRC_IKPKILLED;
/*
   this is has been changed to provide a more meaningful error exit when
   a child isn't responding to polls.

    net_poll_chkremprocs(p->prc_next);
*/
    switch (usage_mode) {
    case INTERACTIVE:
      prc_bldlabel(p, childnm, 39);
      draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
      locate_cursor(&root_x,&root_y);
      dialog_flag=1;
      sprintf(buf1,"Process '%s' (PID %s:%d)\nis no longer communicating.\nDo you wish to abort all processes?",
	      childnm,p_chk->prc_host,pid_chk);
      XBell(display,100);
      XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
      XtVaSetValues(confirm,
		    XmNdefaultPosition,False,
		    XmNdialogTitle,XMstr("Process Query Failure"),
		    XmNmessageString,XMstr(buf1),
		    XmNokLabelString,XMstr("Abort"),
		    XmNokCallback,CBl(net_chkchldprocs_abortCB,1),
		    XmNcancelLabelString,XMstr("Ignore"),
		    XmNcancelCallback,CBl(net_chkchldprocs_ignoreCB,1),
		    XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
		    XmNx,root_x,
		    XmNy,root_y,
		    NULL);
      XtManageChild(confirm);
      break;
    case BATCH:
      (void) sprintf(termmsg1, "Process '%s' (PID %s:%d) failed to answer a status query.", childnm, p->prc_host, p->prc_pid);
      ikp_err(termmsg1, NONFATAL);
      if (!ignore_batcherrs)
	net_abortfork(run_net);
      break;
      }
    }
/* end of new error exit code. */
  
  else switch (status) {
    
  case IKPRP_SETUP:
    
    /* this should never happen */
    (void) rp_closedown(p);
    p->prc_flags&= ~PRC_RUNNING;
    p->prc_flags&= ~PRC_STOPPED;
    p->prc_flags|= PRC_EXITED | PRC_IKPKILLED;
    ikp_err("Remote execution error.", NONFATAL);
    break;
    
  case IKPRP_RUNNING:
    
    /* remote process still running */
    break;
    
  case IKPRP_STOPPED:
    
    /* remote process stopped by external signal */
    if (!(p->prc_flags & PRC_RUNNING))
      break;
    
    if (usage_mode == INTERACTIVE) {
      prc_showactive(p);
      draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
    }
    p->prc_flags&= ~PRC_RUNNING;
    p->prc_flags|= PRC_STOPPED;
    prc_bldlabel(p, childnm, 39);
    
    switch (code) {
    case SIGSTOP:
    case SIGTSTP:
      switch (usage_mode) {
      case INTERACTIVE:
	locate_cursor(&root_x,&root_y);
	dialog_flag=1;
	sprintf(buf1,"Process '%s' (PID %s:%d)\nstopped by signal %d.\nDo you wish to abort all processes?",
		childnm,p_chk->prc_host,pid_chk,code);
	XBell(display,100);
	XtManageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
	XtVaSetValues(confirm,
		      XmNdefaultPosition,False,
		      XmNdialogTitle,XMstr("Process Stopped"),
		      XmNmessageString,XMstr(buf1),
		      XmNokLabelString,XMstr("Abort"),
		      XmNokCallback,CBl(net_chkchldprocs_abortCB,1),
		      XmNcancelLabelString,XMstr("Restart"),
		      XmNcancelCallback,CBl(net_remchldprocs_restartCB,1),
		      XmNhelpLabelString,XMstr("Ignore"),
		      XmNhelpCallback,CBl(net_chkchldprocs_ignoreCB,1),
		      XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
		      XmNx,root_x,
		      XmNy,root_y,
		      NULL);
	XtManageChild(confirm);
	break;
      case BATCH:
	(void) sprintf(termmsg1, "Process '%s' (PID %s:%d) stopped by signal %d.", childnm, p->prc_host, p->prc_pid, code);
	ikp_err(termmsg1, NONFATAL);
	if (!ignore_batcherrs)
	  net_abortfork(run_net);
	break;
      }
      break;
    case SIGTTIN:
    case SIGTTOU:
    default:
      switch (usage_mode) {
      case INTERACTIVE:
	locate_cursor(&root_x,&root_y);
	dialog_flag=1;
	sprintf(buf1,"Process '%s' (PID %s:%d)\nwaiting on tty i/o stopped by signal %d.\nDo you wish to abort all processes?",
		childnm,p_chk->prc_host,pid_chk,code);
	XBell(display,100);
	XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
	XtVaSetValues(confirm,
		      XmNdefaultPosition,False,
		      XmNdialogTitle,XMstr("Process Stopped"),
		      XmNmessageString,XMstr(buf1),
		      XmNokLabelString,XMstr("Abort"),
		      XmNokCallback,CBl(net_chkchldprocs_abortCB,1),
		      XmNcancelLabelString,XMstr("Ignore"),
		      XmNcancelCallback,CBl(net_chkchldprocs_ignoreCB,1),
		      XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
		      XmNx,root_x,
		      XmNy,root_y,
		      NULL);
	XtManageChild(confirm);
	break;
      case BATCH:
	(void) sprintf(termmsg1, "Process '%s' (PID %s:%d) waiting on tty i/o stopped by signal %d.", childnm, p->prc_host, p->prc_pid, code);
	ikp_err(termmsg1, NONFATAL);
	if (!ignore_batcherrs)
	  net_abortfork(run_net);
	break;
      }
      break;
    }
    break;
    
  case IKPRP_KILLED:
    
    if (p->prc_flags & PRC_EXITED)
      break;
    
    /* remote process terminated (perhaps by us) */
    if (usage_mode == INTERACTIVE) {
      prc_showactive(p);
      draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
    }
    p->prc_flags&= ~PRC_RUNNING;
    p->prc_flags&= ~PRC_STOPPED;
    p->prc_flags|= PRC_EXITED;
    (void) rp_closedown(p);
    
    prc_bldlabel(p, childnm, 39);
    
    switch (usage_mode) {
    case INTERACTIVE:
      locate_cursor(&root_x,&root_y);
      dialog_flag=1;
      sprintf(buf1,"Process '%s' (PID %s:%d)\nterminated by signal %d.\nDo you wish to abort all processes?",
	      childnm,p_chk->prc_host,pid_chk,code);
      XBell(display,100);
      XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
      XtVaSetValues(confirm,
		    XmNdefaultPosition,False,
		    XmNdialogTitle,XMstr("Process Stopped"),
		    XmNmessageString,XMstr(buf1),
		    XmNokLabelString,XMstr("Abort"),
		    XmNokCallback,CBl(net_chkchldprocs_abortCB,1),
		    XmNcancelLabelString,XMstr("Ignore"),
		    XmNcancelCallback,CBl(net_chkchldprocs_ignoreCB,1),
		    XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
		    XmNx,root_x,
		    XmNy,root_y,
		    NULL);
      XtManageChild(confirm);
      break;
    case BATCH:
      (void) sprintf(termmsg1, "Process '%s' (PID %s:%d) terminated by signal %d.", childnm, p->prc_host, p->prc_pid, code);
      ikp_err(termmsg1, NONFATAL);
      if (!ignore_batcherrs)
	net_abortfork(run_net);
      break;
    }
    
    break;
    
  case IKPRP_EXITED:
    
    if (p->prc_flags & PRC_EXITED)
      break;
    
    /* remote process exit()ed on its own */
    if (usage_mode == INTERACTIVE)
      prc_showactive(p);
    p->prc_flags&= ~PRC_RUNNING;
    p->prc_flags&= ~PRC_STOPPED;
    p->prc_flags|= PRC_EXITED;
    (void) rp_closedown(p);
    
    if (code != 0) {
      prc_bldlabel(p, childnm, 39);
      
      switch (usage_mode) {
      case INTERACTIVE:
	draw_fill_rectangle(net_igc, p->prc_x-CNT_LIP, p->prc_y-CNT_LIP, p->prc_w+(2*CNT_LIP), p->prc_h+(2*CNT_LIP));
	locate_cursor(&root_x,&root_y);
	dialog_flag=1;
	sprintf(buf1,"Process '%s' (PID %s:%d)\nexited with status %d.\nDo you wish to abort all processes?",
		childnm,p_chk->prc_host,pid_chk,code);
	XBell(display,100);
	XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
	XtVaSetValues(confirm,
		      XmNdefaultPosition,False,
		      XmNdialogTitle,XMstr("Process Stopped"),
		      XmNmessageString,XMstr(buf1),
		      XmNokLabelString,XMstr("Abort"),
		      XmNokCallback,CBl(net_chkchldprocs_abortCB,1),
		      XmNcancelLabelString,XMstr("Ignore"),
		      XmNcancelCallback,CBl(net_chkchldprocs_ignoreCB,1),
		      XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
		      XmNx,root_x,
		      XmNy,root_y,
		      NULL);
	XtManageChild(confirm);
	break;
      case BATCH:
	(void) sprintf(termmsg1, "Process '%s' (PID %s:%d) exited with status %d.", childnm, p->prc_host, p->prc_pid, code);
	ikp_err(termmsg1, NONFATAL);
	if (!ignore_batcherrs)
	  net_abortfork(run_net);
	break;
      }
    }
    break;
    
  default:
    sprintf(ikp_errbuf, "Illegal command byte %d sent by %s:%d.\n", status, p->prc_host, p->prc_pid);
    ikp_err(ikp_errbuf, NONFATAL);
    break;
  }
  if (dialog_flag==0)
    net_poll_chkremprocs(p->prc_next);

  return;
}

void net_stop(Net *nt)
{
  Process *p;
  int count;
  char childnm[40];
  
  count= 0;
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (!(p->prc_flags & PRC_RUNNING))
      continue;
    
    count++;
    if (p->prc_flags & PRC_REMOTE) {
      if (rp_stop(p) != IKP_SUCCESS) {
	prc_bldlabel(p, childnm, 39);
	(void) sprintf(ikp_errbuf, "Process '%s' (PID %s:%d) cannot be stopped.", childnm, p->prc_host, p->prc_pid);
	ikp_err(ikp_errbuf, NONFATAL);
      }
      else {
	prc_showactive(p);
	p->prc_flags&= ~PRC_RUNNING;
	p->prc_flags|= PRC_STOPPED;
	prc_showactive(p);
      }
    }
    else {
      if (kill(p->prc_pid, SIGSTOP) < 0) {
	prc_bldlabel(p, childnm, 39);
	(void) sprintf(ikp_errbuf, "Process '%s' (PID %d) cannot be stopped.", childnm, p->prc_pid);
	ikp_err(ikp_errbuf, NONFATAL);
      }
      
      /* don't change PRC_STOPPED or PRC_RUNNING
	 flags -- this is done in net_chkchldprocs() */
      else
	p->prc_flags|= PRC_IKPSTOPPED;
    }
  }
  
  if (count == 0)
    ikp_err("There are no running processes.", NONFATAL);

  return;
}

void net_restart(Net *nt)
{
  Process *p;
  int count;
  char childnm[40];
  
  count= 0;
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (!(p->prc_flags & PRC_STOPPED))
      continue;
    
    count++;
    if (p->prc_flags & PRC_REMOTE) {
      if (rp_cont(p) != IKP_SUCCESS) {
	prc_bldlabel(p, childnm, 39);
	(void) sprintf(ikp_errbuf, "Process '%s' (PID %s:%d) cannot be restarted.", childnm, p->prc_host, p->prc_pid);
	ikp_err(ikp_errbuf, NONFATAL);
	continue;
      }
    }
    else {
      if (kill(p->prc_pid, SIGCONT) < 0) {
	prc_bldlabel(p, childnm, 39);
	(void) sprintf(ikp_errbuf, "Process '%s' (PID %d) cannot be restarted.", childnm, p->prc_pid);
	ikp_err(ikp_errbuf, NONFATAL);
	continue;
      }
    }
    
    /* net_chk{chld, rem}procs() can't sense when a process is
       restarted, so change PRC_STOPPED and PRC_RUNNING flags here */
    if (usage_mode == INTERACTIVE)
      prc_showactive(p);
    p->prc_flags&= ~PRC_STOPPED;
    p->prc_flags|= PRC_RUNNING;
    p->prc_flags&= ~PRC_IKPSTOPPED;
    if (usage_mode == INTERACTIVE)
      prc_showactive(p);
  }
  
  if (count == 0)
    ikp_err("There are no stopped processes.", NONFATAL);
  
  return;
}

