/*
 *	Copyright (c) 1990 by Columbia University.
 */

#include <stdio.h>
#include <pwd.h>

#include "ikp_defines.h"
#include "ikp_externs.h"
#include "ikp_remote.h"
#include "globalE.h"

#define MAX_HOSTS 100

extern char *host_list[MAX_HOSTS];
extern int host_count,current_host;
extern int sort_table[MAX_HOSTS];

extern char **environ;
static char diagbuf[DIAGBUFSZ];

/*
   Perform all remote process initialization
   except for data socket connections.
*/
int rp_setup(Process *p)
{
  struct sockaddr_in caddr;
  struct hostent *hp;
  char command, **cv;
  u_long ul;
  int user_id, cvcount;
  char *remuser, *wkdir;
  struct passwd *pw = NULL;
  Connector *cnt;
  struct hostent *gethostbyname();
/* - caused problems on the convex - should be declared in pwd.h anyways - 
  struct passwd *getpwuid();
*/
  char *log_flag = ":debug";
  int ikpd_debug;

  if (p == (Process *) NULL) {
    ikp_err("Null process.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (p->prc_host == (char *) NULL) {
    ikp_err("Null hostname.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (strlen(p->prc_host) == 0) {
    ikp_err("Null hostname.", NONFATAL);
    return(IKP_FAILURE);
  }

  if (strcmp(p->prc_host,"*")==0) {
    if (p->prc_tmp_host!=NULL)
      free(p->prc_tmp_host);
    p->prc_tmp_host=strdup(host_list[sort_table[current_host]]);
    current_host++;
    if (current_host>host_count)
      current_host=0;

    if (verbose)
      fprintf(stderr,"Replace * with host %s\n",p->prc_tmp_host);
  } else {
    if (p->prc_tmp_host!=NULL)
      free(p->prc_tmp_host);
    p->prc_tmp_host=strdup(p->prc_host);
  }

/* Added back-door debugging feature activated by appending ":debug" to 
   the remote host name. This must be stripped before doing the actual
   connection, though.				- joe m. wade 1/19/96
*/
   ikpd_debug = 0;
   if (strstr(p->prc_tmp_host,log_flag) != NULL) {
        ikpd_debug = 1;
        *(p->prc_tmp_host+strlen(p->prc_tmp_host)-strlen(log_flag)) = '\0';
	}

  /* build address of remote ikpd socket */
  if ((hp= gethostbyname(p->prc_tmp_host)) == (struct hostent *) NULL) {
    sprintf(ikp_errbuf, "Host '%s' unknown.", p->prc_tmp_host);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
  }

  memset((char *) &caddr, 0, sizeof(caddr));
  memcpy((char *) &caddr.sin_addr, hp->h_addr, hp->h_length);
  caddr.sin_family= hp->h_addrtype;
  caddr.sin_port= ikpd_port;
  
  /* open control connection to remote ikpd */
  if ((p->prc_socket= socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    ikp_err("Socket open error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (connect(p->prc_socket, (struct sockaddr *) &caddr, sizeof(caddr)) < 0) {
    sprintf(ikp_errbuf, "Cannot connect to host '%s'.", p->prc_tmp_host);
    ikp_err(ikp_errbuf, NONFATAL);
    if (close(p->prc_socket) < 0)
      ikp_err("Socket close error.", NONFATAL);
    return(IKP_FAILURE);
  }
  p->prc_flags|= PRC_SOCKOPEN;
  if (verbose >= 5)
    fprintf(stderr, "rem cntl %d\n", p->prc_socket);
  
  /* declare target machine name (needed for alternate addresses ie. fddi) */

  command= IKPRP_HOSTNAME;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (ikpd_debug == 1)
	*(p->prc_tmp_host + strlen(p->prc_tmp_host)) = *(char *)log_flag;
  ul= htonl((u_long) strlen(p->prc_tmp_host));
  if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_write(p->prc_socket, p->prc_tmp_host, strlen(p->prc_tmp_host)) != strlen(p->prc_tmp_host)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    sprintf(ikp_errbuf, "Hostname error on host '%s'", p->prc_tmp_host);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
  }

  /* declare user name */
  command= IKPRP_USERNAME;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  user_id= getuid();
  if ((pw= getpwuid(user_id)) == (struct passwd *) NULL) {
    ikp_err("Unknown user ID.", NONFATAL);
    return(IKP_FAILURE);
  }
  ul= htonl((u_long) strlen(pw->pw_name));
  if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_write(p->prc_socket, pw->pw_name, strlen(pw->pw_name)) != strlen(pw->pw_name)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (p->prc_user == (char *) NULL)
    remuser= pw->pw_name;
  else if (strlen(p->prc_user) == 0)
    remuser= pw->pw_name;
  else
    remuser= p->prc_user;
  ul= htonl((u_long) strlen(remuser));
  if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_write(p->prc_socket, remuser, strlen(remuser)) != strlen(remuser)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    sprintf(ikp_errbuf, "User authentication error on host '%s'.", p->prc_tmp_host);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
  }
  
  /* pass argument list */
  command= IKPRP_ARGV;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  for (cv= p->prc_argv, cvcount= 0; *cv != (char *) NULL; cv++, cvcount++);
  ul= htonl((u_long) cvcount);
  if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  for (cv= p->prc_argv; *cv != (char *) NULL; cv++) {
    ul= htonl((u_long) strlen(*cv));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_write(p->prc_socket, *cv, strlen(*cv)) != strlen(*cv)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    ikp_err("Remote argument setup error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  /* pass environment */
  command= IKPRP_ENVV;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  for (cv= environ, cvcount= 0; *cv != (char *) NULL; cv++, cvcount++);
  ul= htonl((u_long) cvcount);
  if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  for (cv= environ; *cv != (char *) NULL; cv++) {
    ul= htonl((u_long) strlen(*cv));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_write(p->prc_socket, *cv, strlen(*cv)) != strlen(*cv)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    ikp_err("Remote environment setup error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  /* set working directory */
  command= IKPRP_DIR;
  if (p->prc_type == PRC_LIBRARY)
    wkdir= p->prc_module->mdl_dir;
  else
    wkdir= p->prc_dir;
  if (wkdir == (char *) NULL)
    wkdir= dirname;
  else if (strlen(wkdir) == 0)
    wkdir= dirname;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  ul= htonl((u_long) strlen(wkdir));
  if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_write(p->prc_socket, wkdir, strlen(wkdir)) != strlen(wkdir)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    sprintf(ikp_errbuf, "Directory access error on host '%s'.", p->prc_tmp_host);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
    }
  
  /* lock all remote connector descriptors */
  for (cnt= p->prc_input; cnt != (Connector *) NULL; cnt= cnt->cnt_next) {
    command= IKPRP_LOCKFD;
    if (buf_write(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) (cnt->cnt_desc));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_read(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (command != IKPRP_SUCCESS) {
      sprintf(ikp_errbuf, "Connector lock error on host '%s'.", p->prc_tmp_host);
      ikp_err(ikp_errbuf, NONFATAL);
      return(IKP_FAILURE);
    }
  }
  for (cnt= p->prc_output; cnt != (Connector *) NULL; cnt= cnt->cnt_next) {
    command= IKPRP_LOCKFD;
    if (buf_write(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) (cnt->cnt_desc));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_read(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (command != IKPRP_SUCCESS) {
      sprintf(ikp_errbuf, "Connector lock error on host '%s'.", p->prc_tmp_host);
      ikp_err(ikp_errbuf, NONFATAL);
      return(IKP_FAILURE);
    }
  }
  for (cnt= p->prc_diag; cnt != (Connector *) NULL; cnt= cnt->cnt_next) {
    command= IKPRP_LOCKFD;
    if (buf_write(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) (cnt->cnt_desc));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_read(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (command != IKPRP_SUCCESS) {
      sprintf(ikp_errbuf, "Connector lock error on host '%s'.", p->prc_tmp_host);
      ikp_err(ikp_errbuf, NONFATAL);
      return(IKP_FAILURE);
    }
  }
  
  /* open all remote files */
  for (cnt= p->prc_input; cnt != (Connector *) NULL; cnt= cnt->cnt_next) {
    
    /* connected to another process */
    if (!(cnt->cnt_flags & CNT_FILE))
      continue;
    
    /* connected to a file */
    command= IKPRP_FILERD;
    if (buf_write(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) (cnt->cnt_desc));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) strlen(cnt->cnt_filename));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_write(p->prc_socket, cnt->cnt_filename, strlen(cnt->cnt_filename)) != strlen(cnt->cnt_filename)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_read(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (command != IKPRP_SUCCESS) {
      sprintf(ikp_errbuf, "File access error on host '%s'.", p->prc_tmp_host);
      ikp_err(ikp_errbuf, NONFATAL);
      return(IKP_FAILURE);
    }
  }
  for (cnt= p->prc_output; cnt != (Connector *) NULL; cnt= cnt->cnt_next) {

    /* connected to another process */
    if (!(cnt->cnt_flags & CNT_FILE))
      continue;
    
    /* connected to a file */
    if (cnt->cnt_flags & CNT_APPEND)
      command= IKPRP_FILEWRAPP;
    else
			command= IKPRP_FILEWRTRUNC;
    if (buf_write(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) (cnt->cnt_desc));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
		ul= htonl((u_long) strlen(cnt->cnt_filename));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_write(p->prc_socket, cnt->cnt_filename, strlen(cnt->cnt_filename)) != strlen(cnt->cnt_filename)) {
			ikp_err("Socket write error.", NONFATAL);
			return(IKP_FAILURE);
		      }
    if (buf_read(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (command != IKPRP_SUCCESS) {
      sprintf(ikp_errbuf, "File access error on host '%s'.", p->prc_tmp_host);
      ikp_err(ikp_errbuf, NONFATAL);
      return(IKP_FAILURE);
    }
  }
  for (cnt= p->prc_diag; cnt != (Connector *) NULL; cnt= cnt->cnt_next) {

    /* unconnected */
    if (!(cnt->cnt_flags & CNT_FILE) && (cnt->cnt_conn == (Connector *) NULL)) {
      command= IKPRP_DIAG;
      if (buf_write(p->prc_socket, &command, 1) != 1) {
	ikp_err("Socket write error.", NONFATAL);
				return(IKP_FAILURE);
      }
      ul= htonl((u_long) (cnt->cnt_desc));
      if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
	ikp_err("Socket write error.", NONFATAL);
	return(IKP_FAILURE);
      }
      if (buf_read(p->prc_socket, &command, 1) != 1) {
	ikp_err("Socket read error.", NONFATAL);
	return(IKP_FAILURE);
      }
      if (command != IKPRP_SUCCESS) {
	sprintf(ikp_errbuf, "Remote diagnostic redirect error on host '%s'.", p->prc_tmp_host);
				ikp_err(ikp_errbuf, NONFATAL);
	return(IKP_FAILURE);
      }
      continue;
    }
    
    /* connected to another process */
    if (!(cnt->cnt_flags & CNT_FILE))
      continue;
    
    /* connected to a file */
    if (cnt->cnt_flags & CNT_APPEND)
			command= IKPRP_FILEWRAPP;
    else
      command= IKPRP_FILEWRTRUNC;
    if (buf_write(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) (cnt->cnt_desc));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    ul= htonl((u_long) strlen(cnt->cnt_filename));
    if (buf_write(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_write(p->prc_socket, cnt->cnt_filename, strlen(cnt->cnt_filename)) != strlen(cnt->cnt_filename)) {
      ikp_err("Socket write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (buf_read(p->prc_socket, &command, 1) != 1) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (command != IKPRP_SUCCESS) {
      sprintf(ikp_errbuf, "File access error on host '%s'.", p->prc_tmp_host);
      ikp_err(ikp_errbuf, NONFATAL);
      return(IKP_FAILURE);
    }
  }
  
  return(IKP_SUCCESS);
}

/*
   Execute remote process.
*/
int rp_exec(Process *p)
{
  u_long ul;
  char command;
  
  if (p == (Process *) NULL) {
    ikp_err("Null process.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  command= IKPRP_EXEC;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_PID) {
    sprintf(ikp_errbuf, "Remote execution error on host '%s'.", p->prc_tmp_host);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  p->prc_pid= (int) ntohl(ul);
  
  return(IKP_SUCCESS);
}

/*
   Get status of remote process.
   Copy any diagnostic message to stderr.
*/
int rp_status(Process *p,char *status,int *code)
{
  char command;
  u_long ul;
  int msglen;
  
  if (p == (Process *) NULL) {
    ikp_err("Null process.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  command= IKPRP_STATUS;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, status, sizeof(char)) != sizeof(char)) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  switch (*status) {
  case IKPRP_SETUP:
  case IKPRP_RUNNING:
    break;
  case IKPRP_EXITED:
  case IKPRP_STOPPED:
  case IKPRP_KILLED:
    if (buf_read(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    *code= (int) ntohl(ul);
    break;
    
  case IKPRP_DIAGMSG:
    
    /* retrieve diagnostic message */
    if (buf_read(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    msglen= (int) ntohl(ul);
    if (buf_read(p->prc_socket, diagbuf, msglen) != msglen) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    if (fwrite(diagbuf, 1, msglen, stderr) != msglen) {
      ikp_err("Diagnostic write error.", NONFATAL);
      return(IKP_FAILURE);
    }
    
    /* status info follows */
    if (buf_read(p->prc_socket, status, sizeof(char)) != sizeof(char)) {
      ikp_err("Socket read error.", NONFATAL);
      return(IKP_FAILURE);
    }
    
    switch (*status) {
    case IKPRP_SETUP:
    case IKPRP_RUNNING:
      break;
    case IKPRP_EXITED:
    case IKPRP_STOPPED:
    case IKPRP_KILLED:
      if (buf_read(p->prc_socket, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
	ikp_err("Socket read error.", NONFATAL);
	return(IKP_FAILURE);
      }
      *code= (int) ntohl(ul);
      break;
    }
    break;
  }
  return(IKP_SUCCESS);
}

/*
   Suspend child of remote execution daemon.
*/
int rp_stop(Process *p)
{
  char command;
  
  if (p == (Process *) NULL) {
    ikp_err("Null process.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  command= IKPRP_STOP;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    sprintf(ikp_errbuf, "Remote stop error on process %s:%1d.", p->prc_tmp_host, p->prc_pid);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
  }
  
  return(IKP_SUCCESS);
}

/*
   Restart child of remote execution daemon.
*/
int rp_cont(Process *p)
{
  char command;
  
  if (p == (Process *) NULL) {
    ikp_err("Null process.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  command= IKPRP_CONT;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    sprintf(ikp_errbuf, "Remote restart error on process %s:%1d.", p->prc_tmp_host, p->prc_pid);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
  }
  
  return(IKP_SUCCESS);
}

/*
   Terminate child of remote execution daemon.
*/
int rp_kill(Process *p)
{
  char command;
  
  if (p == (Process *) NULL) {
    ikp_err("Null process.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  command= IKPRP_KILL;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    sprintf(ikp_errbuf, "Remote abort error on process %s:%1d.", p->prc_tmp_host, p->prc_pid);
    ikp_err(ikp_errbuf, NONFATAL);
    return(IKP_FAILURE);
  }
  
  return(IKP_SUCCESS);
}

/*
   Terminate remote execution daemon (and its child, if still running).
   Attempt to close control socket regardless of errors.
*/
int rp_closedown(Process *p)
{
  char command;
  
  if (p == (Process *) NULL) {
    ikp_err("Null process.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  command= IKPRP_CLOSEDOWN;
  if (buf_write(p->prc_socket, &command, 1) != 1) {
    sprintf(ikp_errbuf, "Remote closedown error on process %s:%1d.", p->prc_tmp_host, p->prc_pid);
    ikp_err(ikp_errbuf, NONFATAL);
    (void) close(p->prc_socket);
    p->prc_flags&= ~PRC_SOCKOPEN;
    return(IKP_FAILURE);
  }
  if (buf_read(p->prc_socket, &command, 1) != 1) {
    sprintf(ikp_errbuf, "Remote closedown error on process %s:%1d.", p->prc_tmp_host, p->prc_pid);
    ikp_err(ikp_errbuf, NONFATAL);
    (void) close(p->prc_socket);
    p->prc_flags&= ~PRC_SOCKOPEN;
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    sprintf(ikp_errbuf, "Remote closedown error on process %s:%1d.", p->prc_tmp_host, p->prc_pid);
    ikp_err(ikp_errbuf, NONFATAL);
    (void) close(p->prc_socket);
    p->prc_flags&= ~PRC_SOCKOPEN;
    return(IKP_FAILURE);
  }
  
  if (close(p->prc_socket) < 0) {
    ikp_err("Control socket close error.", NONFATAL);
    return(IKP_FAILURE);
  }
  p->prc_flags&= ~PRC_SOCKOPEN;
  return(IKP_SUCCESS);
}

/*
   Connect one local and one remote process.
   Copy the socket to be used by the local process into *sd.
*/
int rp_localconn(Connector *ct,int *sd)
{
  int control;
  Process *p, *q;
  char command;
  int fd, addrlen;
  u_long ul;
  u_short port;
  char addrbuf[MAXSOCKADDRLEN];
  struct sockaddr_in saddr;
  
  *sd= NULL_FD;
  if (ct == (Connector *) NULL) {
    ikp_err("Null connector.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  p= (Process *) ct->cnt_process;
  q= (Process *) ct->cnt_conn->cnt_process;
  if (p->prc_flags & PRC_REMOTE) {
    control= p->prc_socket;
    fd= ct->cnt_desc;
  }
  else {
    control= q->prc_socket;
    fd= ct->cnt_conn->cnt_desc;
  }
  
  /* request remote daemon to open and bind data socket */
  command= IKPRP_SERVERSOCK;
  if (buf_write(control, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  ul= htonl((u_long) fd);
  if (buf_write(control, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  /* retrieve address of new socket from remote daemon */
  if (buf_read(control, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SOCKADDR) {
    ikp_err("Remote connection error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(control, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if ((addrlen= (int) ntohl(ul)) > MAXSOCKADDRLEN) {
    ikp_err("Remote connection error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(control, addrbuf, addrlen) != addrlen) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(control, (char *) &port, sizeof(u_short)) != sizeof(u_short)) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  /* open local socket and connect to remote data socket */

  memset((char *) &saddr, 0, sizeof(saddr));
  memcpy((char *) &saddr.sin_addr, addrbuf, addrlen);
  saddr.sin_family= AF_INET;
  saddr.sin_port= port;
  if ((*sd= socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    ikp_err("Remote connection error.", NONFATAL);
    return(IKP_SOCKET);
  }
  
  if (connect(*sd, (struct sockaddr *) &saddr, sizeof(saddr)) < 0) {
    ikp_err("Remote connection error.", NONFATAL);
    return(IKP_CONNSOCK);
  }

  {  /* added this to set traffic size - 3/15/96 - joe m. wade */
  int optval;
  int optval_length;
  char *streamsize;
  optval = 1;
  streamsize = getenv("IKP_SOCKETSIZE");
  if (streamsize != NULL) {
    sscanf(streamsize,"%d",&optval);
    setsockopt(*sd,SOL_SOCKET,SO_SNDBUF,&optval,sizeof(int));
    setsockopt(*sd,SOL_SOCKET,SO_RCVBUF,&optval,sizeof(int));
    }
   if (verbose) {
     optval_length = sizeof(int);
     getsockopt(*sd,SOL_SOCKET,SO_SNDBUF,&optval,&optval_length);
     fprintf(stderr,"SO_SNDBUF = %d on socket %d\n",optval,*sd);
     optval_length = sizeof(int);
     getsockopt(*sd,SOL_SOCKET,SO_RCVBUF,&optval,&optval_length);
     fprintf(stderr,"SO_RCVBUF = %d on socket %d\n",optval,*sd);
     }
  }

  if (verbose >= 5)
    fprintf(stderr, "rem conn %d\n", *sd);
  
  return(IKP_SUCCESS);
}

/*
   Connect two remote processes.
*/
int rp_remconn(Connector *ct)
{
  int scntl, ccntl;
  Process *sprc, *cprc;
  Connector *scnt, *ccnt;
  char command;
  int addrlen;
  u_long ul;
  u_short port;
  char addrbuf[MAXSOCKADDRLEN];
  
  if (ct == (Connector *) NULL) {
    ikp_err("Null connector.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  switch (ct->cnt_iodir) {
  case CNT_INPUT:
    ccnt= ct;
    cprc= (Process *) ct->cnt_process;
    scnt= ct->cnt_conn;
    sprc= (Process *) ct->cnt_conn->cnt_process;
    break;
  case CNT_OUTPUT:
  case CNT_DIAG:
    scnt= ct;
    sprc= (Process *) ct->cnt_process;
    ccnt= ct->cnt_conn;
    cprc= (Process *) ct->cnt_conn->cnt_process;
    break;
  }
  scntl= sprc->prc_socket;
  ccntl= cprc->prc_socket;
  
  /* request remote server to open and bind data socket */
  command= IKPRP_SERVERSOCK;
  if (buf_write(scntl, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  ul= htonl((u_long) scnt->cnt_desc);
  if (buf_write(scntl, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  /* retrieve address of new socket from remote server */
  if (buf_read(scntl, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SOCKADDR) {
    ikp_err("Remote connection error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(scntl, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if ((addrlen= (int) ntohl(ul)) > MAXSOCKADDRLEN) {
    ikp_err("Remote connection error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(scntl, addrbuf, addrlen) != addrlen) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(scntl, (char *) &port, sizeof(u_short)) != sizeof(u_short)) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  command= IKPRP_CLIENTSOCK;
  if (buf_write(ccntl, &command, 1) != 1) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  ul= htonl((u_long) ccnt->cnt_desc);
  if (buf_write(ccntl, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  ul= htonl((u_long) addrlen);
  if (buf_write(ccntl, (char *) &ul, sizeof(u_long)) != sizeof(u_long)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_write(ccntl, addrbuf, addrlen) != addrlen) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_write(ccntl, (char *) &port, sizeof(u_short)) != sizeof(u_short)) {
    ikp_err("Socket write error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (buf_read(ccntl, &command, 1) != 1) {
    ikp_err("Socket read error.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (command != IKPRP_SUCCESS) {
    ikp_err("Remote connection error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  return(IKP_SUCCESS);
}
