C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c instphz reads seismic trace data from an input file,
c performs some arcane geophysical process
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
      integer     itr ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne, npoints, npts, npts21
      real        fl, fh, dt
c------
c    this is necessary for buliding printout files
#include <f77/pid.h>
c------
      integer     recnum, trcnum, static
      real        tri ( SZSMPM ), triphz ( SZSMPM )
      character   ntap * 256, otap * 256, name*7
      logical     verbos, query, twopi, sqr, rect
      integer     argis
 
c     equivalence ( itr(129), tri (1) )
      data lbytes / 0 /, nbytes / 0 /, name/'INSTPHZ'/
 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,ns,ne,irs,ire,
     1             npoint,fl,fh,verbos,twopi,sqr,rect)

c-----
c     get logical unit numbers for input and output of seismic data
c     0 = default stdin
c     1 = default stdout
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'INSTPHZ: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif

c------
c     to get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

      call hlhprt (itr, lbytes, name, 7, LERR)
c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records)
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c-----
c     modify line header to reflect actual number of traces output
c-----
      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr=ne-ns+1
      call savew(itr, 'NumTrc', jtr  , LINHED)

c--------------------------------------------------
c  compute sample interval in secs
c  and power-of-2 # samples
      call pwrof2 (nsamp, ipw)
      npts = 2 ** ipw
      npts21 = npts/2 + 1
c     dt = float(nsi)/1000.
c     fnyq = 500./float(nsi)

      dt = real (nsi) * unitsc
      fnyq = 1./(2*dt)

      if (fh .eq. 0.) fh = fnyq/2.
c--------------------------------------------------

c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
c     if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  fl,fh,npoints,npts,ntap,otap,twopi)
c     end if


c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)

c-----
c     process desired trace records
c-----
      do 1000 jj = irs, ire
 
c----------------------
c  skip to start trace
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c----------------------
 
            do 1001 kk=ns,ne

                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

c------
c     for trace header values you can use the I*2 vector
c     rather than saver/savew (which you can use)
c     this is faster in general than saver/w
c------

                  call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        dist  , TRACEHEADER)
                  call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static, TRACEHEADER)
                  call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum, TRACEHEADER)
                  call saver2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum, TRACEHEADER)

                  IF(static .ne. 30000)then

c-----------------------
c  here's the meat...
c  compute inst phase

                     call iphase (tri, triphz, nsamp, npts, npts21,
     1                            npoint, dt, fl, fh, sqr, rect)

                     if (twopi) then
                        call piwrap (triphz, nsamp)
                     else
                        call drum (nsamp, triphz)
                     endif

                     call vmov (triphz, 1, itr(ITHWP1), 1, nsamp)
c-----------------------

                  ENDIF

c---------------------
c  write output data
                  call wrtape (luout, itr, obytes)

                  if(verbos)write(LERR,*)'ri ',recnum,' trace ',trcnum
 1001             continue
 
c----------------------
c  skip to end of record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c----------------------
 
 1000       continue

c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of instphz, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'instphz does dark and terrible things to seismic data:'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute instphz by typing instphz and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)         : output data file name'
        write(LER,*)
     :' -ns[ns]      (default = first)    : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)     : end trace number'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -ne[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -fl[fl]           (default = 5Hz) :  lo cut frequency'
        write(LER,*)
     :' -fh[fh]    (default = 1/2Nyquist) :  hi cut frequency'
        write(LER,*)
     :' -w[npoint]  (default = 10 pts)    :  length smoothing window'
        write(LER,*) ' '
        write(LER,*)
     :' -S  include on command line to square input data (with sign)'
        write(LER,*)
     :' -R  include on command line to square input data (without sign)'
        write(LER,*)
     :' -W  include on command line if output is to be wrapped 2pi'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   instphz -N[ntap] -O[otap] -ns[ns] -ne[ne] -rs[irs] '
        write(LER,*)
     :'                 -re[ire] -fl[] -fh[] -w[] [-W -S -R -V]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,
     1                  npoint,fl,fh,verbos,twopi,sqr,rect)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c   npoint  - R*4      length of smoothing window
c     fl    - R*4      low cut frequency
c     fh    - R*4      hi cut frequency
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     twopi   L        wrap the output around after 2 * pi
c     sqr     L        square input data before phz calc
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire, npoint
      real        fl, fh
      logical     verbos, twopi, sqr, rect
      integer     argis
 
c-------
c     see manual pages on the argument handler routines
c     for the meanings of these functions
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argr4 ( '-fl', fl, 5., 5.)
            call argr4 ( '-fh', fl, 0., 0.)
            call argi4 ( '-w', npoint ,  10  , 10    )
            sqr    =   (argis('-S') .gt. 0)
            rect   =   (argis('-R') .gt. 0)
            twopi  =   (argis('-W') .gt. 0)
            verbos =   (argis('-V') .gt. 0)

            if (rect .and. sqr) then
               write(LERR,*)'instphz  WARNING:'
               write(LERR,*)'both -S & -R options chosen.  Must choose'
               write(LERR,*)'only one so -S will be the default (this'
               write(LERR,*)'is to square the data with sign'
               rect = .false.
            endif
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  fl,fh,npoints,npts,ntap,otap,twopi)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c   npoint  - R*4      length of smoothing window
c     fl    - R*4      low cut frequency
c     fh    - R*4      hi cut frequency
c     npts  - I*4      number power of 2 points
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     twopi   L        wrap the output around after 2 * pi
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c-----
#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec, npoint, npts
      real        fl, fh
      logical     twopi
      character   ntap*(*), otap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' Lo cut frequency   =  ', fl,' Hz'
            write(LERR,*) ' Hi cut frequency   =  ', fh,' Hz'
            write(LERR,*) ' Length of smoother =  ', npoint,' pts'
            if (twopi)
     1      write(LERR,*)'Ouput will be wrapped at 2 * pi'
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
