C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c mixn reads seismic trace data from an input file,
c performs some arcane geophysical process
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
 
      integer     itr   ( SZLNHD )
      integer     itr0  ( SZLNHD )
      integer     lhed  ( SZLNHD )
      integer     lhed0 ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     ist, ied, ns, ne, irs, ire
#include <f77/pid.h>
      integer     recnum, trcnum
      real        tri ( SZLNHD ), wt ( SZLNHD ), work ( SZLNHD )
      real        tmp ( SZLNHD )
      character   ntap * 256, otap * 256, name*4
      logical     verbos, query, dead, med
      integer     argis
 
      equivalence ( itr(  1), lhed(1) )
      equivalence ( itr0(  1), lhed0(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'MIXN'/
      data itr0   /SZLNHD*0/

 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,amp,mix,verbos,dead,med,
     1            ist, ied, ns, ne, irs, ire)

c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'MIXN: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)


      call hlhprt (itr, lbytes, name, 4, LERR)

c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-------------------------
c  compute filter weights
      if (mod(mix,2) .eq. 0) then
         mix = mix + 1
         write(LERR,*)'Filter length not odd.  Changing to ',mix
      endif
      mo = 1 + (mix-1)/2
      wt(mo) = 1.0
      sum = wt(mo)
      do  10  i = 1, mo-1
              fact = amp ** i
              wt(mo+i) = fact
              sum = sum + fact
              wt(mo-i) = fact
              sum = sum + fact
10    continue
      call vsmul (wt, 1, 1./sum, wt, 1, mix)
c-------------------------

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  mix,wt,ntap,otap,med,
     2                  ist, ied, ns, ne, irs, ire)

c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary
 
      dt = real (nsi) * unitsc

      ist = ist / nsi
      ied = ied / nsi
      if (ist .le. 1) ist = 1
      if (ied .le. 1) ied = nsamp
      if (ied .gt. nsamp) ied = nsamp
      nsp = ied - ist + 1

c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c     read trace, mix data, write to output file
c-----
      mix2 = mix / 2 + 1
      mix0 = mix / 2
      mix1 = mix + 1
      m21  = mix + mix2
      nm   = 2*mix + nsp
      nmd  = mix + nsp

      nbytes = obytes
      call recrw (1,irs-1,luin,ntrc,itr,luout, nbytes)
      if (nbytes  .eq. 0) go to 999

      DO 1000 JJ = irs, ire
 
 
            ic = 0
            nbytes = obytes
            call trcrw (JJ, 1, ns-1, luin, ntrc, itr, luout, nbytes)
            if (nbytes  .eq. 0) go to 999

            do   kk = ns, ne
                  nbytes = 0
                  call rtape( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
c------
c     use previously derived pointers to trace header values
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                        trcnum , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        istatic, TRACEHEADER)

c----------------------
c  pack data into array
                  if (istatic .eq. 30000) then
                         call vclr (work, 1, nsamp)
                  else
                         if (med) then
                            call vmov  (tri(ist), 1, work(mix2), 1, nsp)
                            do  ii = 1, mix0
                                work (ii) = 0.
                                work (nsp+ii) = 0.
                            enddo
                            call medsm (mix, nmd, work, tmp)
                            call vmov  (tmp(mix2), 1, tri(ist), 1, nsp)
                            call vmov  (tri, 1, lhed(ITHWP1), 1, nsamp)
                         else
                            call vmov  (tri, 1, work(mix1), 1, nsp)
                            do  ii = 1, mix
                                work (ii) = tri (1)
                                work (nsp+mix+ii) = tri (nsp)
                            enddo
                            call fold  (mix,wt,nm,work,nf,tmp)
                            call vmov (tmp(m21), 1, tri(ist), 1, nsp)
                            call vmov (tri, 1, lhed(ITHWP1), 1, nsamp)
                         endif
                  endif

                  call wrtape (luout, itr, nbytes)

            enddo

c-----
c     skip to end of present record; go get next record
c-----
            nbytes = obytes
            call trcrw (JJ, ne+1, ntrc, luin, ntrc, itr, luout, nbytes)
            if (nbytes .eq. 0) go to 999
 
1000  CONTINUE

c------------------------
c  pass remainder of recs
      nbytes = obytes
      call recrw (ire+1, nrec, luin, ntrc, itr, luout, nbytes)
      if (nbytes .eq. 0) go to 999

c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of mixn, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
            write(LER ,*)'end of mixn, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'mixn mixes samples over a sliding window. To mix traces'
        write(LER,*)
     :'you must run the combo ... transp | mixn ... | transp ...'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute mixn by typing mixn and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
        write(LER,*)
     :' -m[mix]      (default = 3)     : number points in mix filter'
        write(LER,*)
     :' -w[amp]      (default = 0.5)   : weight of filter'
        write(LER,*) ' '
        write(LER,*)
     :' -s[ist]      (default = first) : start time (ms) of mix'
        write(LER,*)
     :' -e[ied]      (default = last)  : end time (ms) of mix'
        write(LER,*)
     :' -ns[ns]      (default = first) : start trace processing'
        write(LER,*)
     :' -ne[ne]      (default = last)  : end trace processing'
        write(LER,*)
     :' -rs[irs]     (default = first) : start record processing'
        write(LER,*)
     :' -re[ire]     (default = last)  : end record processing'
        write(LER,*) ' '
        write(LER,*)
     :' -M  include on command line if median smoothing is done'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   mixn -N[ntap] -O[otap] -m[mix] -w[amp] [ -M -V ]'
        write(LER,*)
     :'              [ -s[] -e[] -ns[] -ne[] -rs[] -re[] ]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,amp,mix,verbos,dead,med,
     1                  ist, ied, ns, ne, irs, ire)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c     amp   - R*4      filter weight
c     mix   - I*4      number points in mix filter
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      real        amp
      logical     verbos, dead, med
      integer     argis, mix, ist, ied, ns, ne, irs, ire
 
      med    = .false.
      dead   = .false.
      verbos = .false.

            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-m', mix ,   3  ,  3    )

            call argi4 ( '-s', ist ,   0  ,  0    )
            call argi4 ( '-e', ied ,   0  ,  0    )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,  0  ,  0    )
            call argi4 ( '-re', ire ,  0  ,  0    )

            call argr4( '-w', amp, 0.5, 0.5)
            if (mix .le. 0) then
                mix = 3
                write(LERR,*)'mix must be greater than 1'
                write(LERR,*)'Setting mix = ',mix,' (default)'
            endif
            if (amp .le. 0.) then
                amp = .5
                write(LERR,*)'amp must be in range 0.0 - 1.0'
                write(LERR,*)'Setting amp = ',amp,' (default)'
            endif
            if (amp .gt. 1.) then
                amp = amp/100.
                write(LERR,*)'amp assumed to be %'
                write(LERR,*)'Setting amp = ',amp
            endif
            med    = (argis('-M') .gt. 0)
            dead   = (argis('-D') .gt. 0)
            verbos = (argis('-V') .gt. 0)
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  mix,wt,ntap,otap,med,
     2                  ist, ied, ns, ne, irs, ire)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c      wt   - R*4      filter weights
c     mix   - I*4      number points in mix filter
c     ntap  - C*120   input file name
c     otap  - C*120   output file name
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec, mix
      integer     ist, ied, ns, ne, irs, ire
      real        wt(*)
      character   ntap*(*), otap*(*)
      logical     med
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' Start time of mix  =  ', ist
            write(LERR,*) ' End time of mix    =  ', ied
            write(LERR,*) ' Start trace of mix =  ', ns
            write(LERR,*) ' End trace of mix   =  ', ne
            write(LERR,*) ' Start record of mix=  ', irs
            write(LERR,*) ' End record of mix  =  ', ire
            write(LERR,*) ' Number points in filter = ',mix
            if (med) then
            write(LERR,*) ' Median smoothing done'
            else
            write(LERR,*) ' Filter weights  '
            write(LERR,*)' '
            write(LERR,100) (wt(i),i=1,mix)
100         format(5f10.3)
            write(LERR,*)' '
            endif
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
