C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c mixrec reads seismic trace data from an input file,
c performs some arcane geophysical process
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
 
      integer     itr   ( SZLNHD )
      integer     itr0  ( SZLNHD )
      integer     lhed  ( SZLNHD )
      integer     lhed0 ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
#include <f77/pid.h>
      integer     recnum, trcnum
      real        tri ( SZLNHD ), wt ( SZLNHD )
      integer     dedtrc ( SZLNHD )
      real        datai, temp1, temp2
      pointer     (wkdatai, datai(1))
      pointer     (wktemp1, temp1(1))
      pointer     (wktemp2, temp2(1))
      integer     itrhd
      pointer     (wkitrhd, itrhd(1))
      character   ntap * 256, otap * 256, name*6
      logical     verbos, heapi, heap1,heap2,heap3, dead
      integer     argis
 
      equivalence ( itr(  1), lhed(1) )
      equivalence ( itr0(  1), lhed0(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'MIXREC'/
      data itr0   /SZLNHD*0/
      data dedtrc /SZLNHD*0/

 
c-----
c     read program parameters from command line card image file
c-----
      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif
 
c-----
c     open printout files
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,amp,mix,verbos,dead,ist,ied,ns,ne,
     1            irs,ire)

c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'MIXREC: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)


      call hlhprt (itr, lbytes, name, 6, LERR)

c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

      ntrj = ne -ns + 1

c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-------------------------
c  compute filter weights
      if (mod(mix,2) .eq. 0) then
         mix = mix + 1
         write(LERR,*)'Filter length not odd.  Changing to ',mix
      endif
      mo = 1 + (mix-1)/2
      wt(mo) = 1.0
      sum = wt(mo)
      do  10  i = 1, mo-1
              fact = amp ** i
              wt(mo+i) = fact
              sum = sum + fact
              wt(mo-i) = fact
              sum = sum + fact
10    continue
      call vsmul (wt, 1, 1./sum, wt, 1, mix)
c-------------------------

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
c     if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  mix,wt,ntap,otap,ist,ied,ns,ne,
     1                  irs,ire)
c     end if

      heapi = .true.
      heap1 = .true.
      heap2 = .true.
      heap3 = .true.

      nt    = ntrj + 3*mix
      itemi = ntrj * ITRWRD * SZSMPD
      item1 = ntrj * nsamp  * SZSMPD
      item2 = nt            * SZSMPD
      item3 = nt            * SZSMPD

      call galloc (wkitrhd, itemi, errcdi, aborti)
      call galloc (wkdatai, item1, errcd1, abort1)
      call galloc (wktemp1, item2, errcd2, abort2)
      call galloc (wktemp2, item3, errcd3, abort3)

      if (errcdi .ne. 0.) heapi = .false.
      if (errcd1 .ne. 0.) heap1 = .false.
      if (errcd2 .ne. 0.) heap2 = .false.
      if (errcd3 .ne. 0.) heap3 = .false.
 
      if (.not. heap1 .or. .not. heap2 .or. .not. heapi .or.
     1    .not. heap3) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*)' '
         write(LER ,*)' '
         write(LER ,*)'Unable to allocate workspace:'
         write(LER ,*) itemi,'  bytes'
         write(LER ,*) item1,'  bytes'
         write(LER ,*) item2,'  bytes'
         write(LER ,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*)' '
      endif

c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary
 
      dt = real (nsi) * unitsc

      ist = ist / nsi
      ied = ied / nsi
      if (ist .le. 1) ist = 1
      if (ied .le. 1) ied = nsamp
      if (ied .gt. nsamp) ied = nsamp

c--------------------------------------------------
      call savew2(lhed0,ifmt_StaCor,l_StaCor, ln_StaCor,
     1            30000 , TRACEHEADER)

c-----
c     BEGIN PROCESSING
c     read trace, mix data, write to output file
c-----
      lag = mix / 2

      nbytes = obytes
      call recrw (1,irs-1,luin,ntrc,itr,luout, nbytes)
      if (nbytes  .eq. 0) go to 999

      DO 1000 JJ = irs, ire
 
         istrc = 1 - nsamp
         ishdr = 1 - ITRWRD

         ic = 0
         nbytes = obytes
         call trcrw (JJ, 1, ns-1, luin, ntrc, itr, luout, nbytes)
         if (nbytes  .eq. 0) go to 999

         do   kk = ns, ne
            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',jj,'  trace= ',kk
               go to 999
            endif
            call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
c------
c     use previously derived pointers to trace header values
            call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1           recnum , TRACEHEADER)
            call saver2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1           trcnum , TRACEHEADER)
            call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1           istatic, TRACEHEADER)

c----------------------
c  pack data into array
            IF (dead) THEN
               ic = ic + 1
               istrc = istrc + nsamp
               ishdr = ishdr + ITRWRD
               call vmov ( tri, 1, datai(istrc), 1, nsamp )
               call vmov ( lhed, 1, itrhd(ishdr), 1, ITRWRD )

            ELSE

c header stored regardless so as to preserve dead trace flag
c and ProMAX trace identifier

               ishdr = ishdr + ITRWRD
               call vmov (lhed,1, itrhd(ishdr),1,ITRWRD)

               if (istatic .ne. 30000) then
                  ic = ic + 1
                  istrc = istrc + nsamp
                  call vmov (tri,1, datai(istrc),1, nsamp)
                  dedtrc (kk) = 0
               else
                  dedtrc (kk) = 1
               endif
            ENDIF
            
         enddo
         jtr = ic

c-----------
c convolve input
c data with fltr
c weights
c-----------
         nw = jtr + mix - 1
         call xfold (jtr,ied,datai,ist,mix,wt,nsamp,temp1,temp2)

         ic = 0
         istrc = 1 - nsamp
         ishdr = 1 - ITRWRD

         do  kk = 1, ntrj

            IF (dedtrc (kk) .eq. 0) THEN
               ic = ic + 1
               istrc = istrc + nsamp
               ishdr = ishdr + ITRWRD
               call vmov (datai(istrc),1,lhed(ITHWP1),1, nsamp)
               call vmov (itrhd(ishdr),1,lhed,1,ITRWRD)
               call wrtape (luout, itr, obytes)
            ELSE
               ishdr = ishdr + ITRWRD
               call vmov ( itrhd(ishdr), 1, itr0, 1, ITRWRD )
               call wrtape (luout, itr0, obytes)
            ENDIF
         enddo
c-----
c     skip to end of present record; go get next record
c-----
         nbytes = obytes
         call trcrw (JJ, ne+1, ntrc, luin, ntrc, itr, luout, nbytes)
         if (nbytes .eq. 0) go to 999
 
 1000 CONTINUE

c------------------------
c  pass remainder of recs
      nbytes = obytes
      call recrw (ire+1, nrec, luin, ntrc, itr, luout, nbytes)
      if (nbytes .eq. 0) go to 999

c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of mixrec, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
            write(LER ,*)'end of mixrec, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'mixrec mixes records spatially'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute mixrec by typing mixrec and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
        write(LER,*)
     :' -m[mix]      (default = 3)     : number points in mix filter'
        write(LER,*)
     :' -w[amp]      (default = 0.5)   : weight of filter'
        write(LER,*)
     :' -s[ist]      (default = first) : start time (ms) of mix'
        write(LER,*)
     :' -e[ied]      (default = last)  : end time (ms) of mix'
        write(LER,*)
     :' -ns[ns]      (default = first) : start trace processing'
        write(LER,*)
     :' -ne[ne]      (default = last)  : end trace processing'
        write(LER,*)
     :' -rs[irs]     (default = first) : start record processing'
        write(LER,*)
     :' -re[ire]     (default = last)  : end record processing'
        write(LER,*) ' '
        write(LER,*)
     :' -D  process with dead traces embedded in record, else'
        write(LER,*)
     :'     process with dead trace stripped out first'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :' -h, -help, -?   include on command line for help'
        write(LER,*)
     :'usage:   mixrec -N[ntap] -O[otap] -m[] -w[] -s[] -e[]'
        write(LER,*)
     :'                 -ns[] -ne[] -rs[] -re[] [ -D -V -h -H -? ]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,amp,mix,verbos,dead,ist,ied,ns,ne,
     1                  irs,ire)
c-----
c     get command arguments
c
c     ntap  - C*256    input file name
c     otap  - C*256    output file name
c     amp   - R*4      filter weight
c     mix   - I*4      number points in mix filter
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      real        amp
      logical     verbos, dead
      integer     argis, mix
 
      dead   = .false.
      verbos = .false.

            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )

            call argi4 ( '-s', ist ,   0  ,  0    )
            call argi4 ( '-e', ied ,   0  ,  0    )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,  0  ,  0    )
            call argi4 ( '-re', ire ,  0  ,  0    )

            call argi4 ( '-m', mix ,   3  ,  3    )
            call argr4( '-w', amp, 0.5, 0.5)
            if (mix .le. 0) then
                mix = 3
                write(LERR,*)'mix must be greater than 1'
                write(LERR,*)'Setting mix = ',mix,' (default)'
            endif
            if (amp .le. 0.) then
                amp = .5
                write(LERR,*)'amp must be in range 0.0 - 1.0'
                write(LERR,*)'Setting amp = ',amp,' (default)'
            endif
            if (amp .gt. 1.) then
                amp = amp/100.
                write(LERR,*)'amp assumed to be %'
                write(LERR,*)'Setting amp = ',amp
            endif
            dead   = (argis('-D') .gt. 0)
            verbos = (argis('-V') .gt. 0)
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  mix,wt,ntap,otap,ist,ied,ns,ne,
     1                  irs,ire)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c      wt   - R*4      filter weights
c     mix   - I*4      number points in mix filter
c     ntap  - C*120   input file name
c     otap  - C*120   output file name
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec, mix, ist, ied
      real        wt(*)
      character   ntap*(*), otap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' Start time of mix  =  ', ist
            write(LERR,*) ' End time of mix    =  ', ied
            write(LERR,*) ' Start trace of mix =  ', ns
            write(LERR,*) ' End trace of mix   =  ', ne
            write(LERR,*) ' Start record of mix=  ', irs
            write(LERR,*) ' End record of mix  =  ', ire
            write(LERR,*) ' Number points in filter = ',mix
            write(LERR,*) ' Filter weights  '
            write(LERR,*)' '
            write(LERR,100) (wt(i),i=1,mix)
100         format(5f10.3)
            write(LERR,*)' '
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
