C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine bdsub(y,dt,nsamp,x,v_trace,remove,nmo_trace,top,bot,
     :     rnum,XY,luXY,yp1,ypn,t)

#include <f77/lhdrsz.h>
#include <f77/iounit.h>

c----------------------------------------------------------------
c       Subroutine Arguments 
c----------------------------------------------------------------

      integer nsamp,luXY

      real y(*),v_trace(*),nmo_trace(*),rnum(*),t(*),dt,x,yp1,ypn

      logical remove,top,bot,XY

c----------------------------------------------------------------
c      y()          : input data trace
c      v_trace()    : velocity trace (velocity vs sample)
c      dt           : sample interval
c      nsamp        : number samples/trace
c      x            : distance
c      remove       : if true remove nmo
c      nmo_trace()  : output trace
c      top          : use nmo map ahead of cross-over
c      bot          : use nmo map from cross-over to end of trace
c      rnum()       : random number array used to fill out zero data
c      XY           : flag for xgraph output
c      luXY         : logical unit for output of tvdnmo_fcn data
c      yp1          : value of second derivative at sample 1
c      ypn          : value of second derivative at nsamp
c      t()          : array containing sample abscissa
c----------------------------------------------------------------
c      Subroutine Internal Variables
c----------------------------------------------------------------

      integer cros_ndx,cros_ndx2,start_index,icount,j,Onset,OnsetOld
   
      real workx(2*SZSMPM),worky(2*SZSMPM),y2(2*SZSMPM),yj
      real xvi,xt,tnmo,dj,rj

c----------------------------------------------------------------
c    cros_ndx    : sample value at velocity cross-over (as measured from nsamp to 1)
c    cros_ndx2   : sample value at 2nd velocity cross-over (as measured from nsamp to 1)
c    start_index : sample value of first active output sample in nmo-map.
c    icount      : number of samples in nmo map for current trace
c
c
c  s  |              <--------- icount ------------------>
c  a  |
c  m  |
c  p  |          start_index                              *
c  l  |              |                                  * |
c  e  |              |                                *   |
c     |              |                              *     |
c  i  |              |   *                        *       |
c  n  |                * |  *                  *          |
c     |              *   |     *            *       [start_index + icount -1]
c     |                  |           *
c     |             cros_ndx2        |      
c     |                          cros_ndx    
c     |
c     |
c     |
c     |_ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _
c               sample out
c
c    j         : index variables
c    workx()   : working array of times (tnmo or 1,2,3,...,nsamp)
c    worky()   : working array of tnmo or amp (forward or remove)
c    y2()      : array of second derivatives (used in cubic spline subroutine)
c    yj        : interpolated value from splint subroutine
c    dj        : fractional portion of tnmo
c    rj        : floating point index used as argument to splint subroutine
c    xt        : distance/sample interval
c    xvi       : xt/velocity
c    tnmo      : floating point sample required for nmo
c    Onset     : sample value of first non hard zero sample (used for mute restore)
c    OnsetOld  : sample value of first non hard zero sample (used for mute restore)
c----------------------------------------------------------------

c
c ----- initialization of variables ----- 
c

      Onset = 1
      OnsetOld = 1

      if(remove)call vclr(workx,1,nsamp)

      DO i=1,nsamp

         nmo_trace(i) = 0.0
         worky(i) = 0.0

c
c ----- the spline routine hates consecutive zeros in the trace data -----
c       which often occur in the mute zone or in any pad. This logic
c       replaces zeros with very small random numbers so that ieee errors of
c       underflow are not encountered in spline
c

         if(abs(y(i)).lt.1.e-30)then

c
c ----- determine onset of data for mute restore when mute start -----
c       is deeper into the data than the cross-over.  This was 
c       prompted by Bill Felinski who didn't like the small rand
c       numbers in his data.  It caused the mute restore option in 
c       rmmult to fail.  Now I will keep track of the first non-hard-zero
c       sample and not allow output above this samples output location
c

            if(i.gt.1)then

               if(abs(y(i)-(y(i-1)-rnum(i-1))).lt.1.e-30
     :              .and.Onset.eq.(i-1))Onset = i
            endif

            y(i)=y(i)+rnum(i)

         endif

      ENDDO

      OnsetOld = Onset

c
c ----- reduce x by delta t so I can work in samples -----
c

      xt = x/dt

c
c ----- reset the tnmo function element counter -----
c

      icount = 0

c
c ----- reset the cross-over and start indices to zero -----
c

      cros_ndx = 0
      cros_ndx2 = 0
      start_index = 0

c
c ----- BUILD NMO MAP -----
c

      DO 100 i = 1,nsamp

c
c ----- calculate input sample whose amplitude -----
c       is associated with this output sample
c

         xvi=xt/v_trace(i)
            
         tnmo =  sqrt (abs((i-1)**2 + xvi**2 ))+1.
         j= tnmo
         dj= tnmo - j

c
c ----- make sure that this sample exists on the input trace -----
c

         if (j .ge. 1 .and. j .le. (nsamp-1)) then

c
c ----- if it does then flag which output sample this occured at -----
c       This is the start of the map for this trace as shown above
c

            if(start_index.eq.0)start_index = i

c
c ----- REVERSE NMO MAP -----
c

            IF (remove) then 

               workx(i) = tnmo
               worky(i) = y(i)

c
c ----- keep track of number of entries in tnmo table -----
c       This will prevent zeros at the start of the function
c       which, as mentioned earlier, kills the spline interpolator
c

               icount = icount + 1

c
c ----- output nmo map for requested trace -----
c

c
c ----- assign onset of output nmo removed trace -----
c       

               if(Onset.eq.float(i).and.Onset.eq.OnsetOld
     :              .and.Onset.ne.1)then

                  Onset = nint(workx(i))

               endif


               if(XY)then

                  write(luXY,6666)float(i),workx(i)

               endif

            ELSE

c
c ----- FORWARD NMO MAP -----
c

               worky(i) = tnmo
               icount = icount + 1

c
c ----- output nmo map for requested trace -----
c

               if(XY) then

                  write(luXY,6666)float(i),worky(i)
 6666             format(f10.4,1x,f10.4)

               endif

            ENDIF

         endif

 100  CONTINUE

c
c ----- CROSS-OVER DETERMINATION -----
c

c
c ----- examine tnmo from sample nsamp to sample 1 -----
c       looking for cross-over point.  Tag that point.
c       The following is expensive in terms of number of
c       lines of code, however, it runs faster.  When
c       removing nmo search workx(), when applying nmo
c       search worky().
c

      IF(remove)then

         DO j=icount+start_index-1,start_index+1,-1

            if(cros_ndx.eq.0)then

               if(workx(j).lt.workx(j-1))then

                  cros_ndx = j

               endif

            else

               if(workx(j).gt.workx(j-1))then

                  cros_ndx2 = j
                  goto 2500

               endif

            endif

         ENDDO

      ENDIF

      IF(.not.remove) then

         DO j = start_index+icount-1,start_index+1,-1

            if(cros_ndx.eq.0) then

               if(worky(j).lt.worky(j-1))then

                  cros_ndx = j

               endif

            else
            
               if(worky(j).gt.worky(j-1).and.worky(j-1).gt.1.e-8)then

                  cros_ndx2 = j
                  goto 2500

               endif

            endif

         ENDDO

      ENDIF

c
c ----- if removing nmo and using only the bottom of the -----
c       nmo map then the number of entries in the nmo map
c       will be the total number (icount) minus the number
c       of samples before the cross-over (cros_ndx-start_index)
c

 2500 if(remove.and.bot.and.cros_ndx.ne.0)
     :     icount=icount-(cros_ndx-start_index) 

c
c ----- when removing nmo using the top of the nmo map -----
c       this routine will only recover data above the 
c       first cross-over point (cros_ndx) if it exists and
c       below the second (cros_ndx2).  To do more would
c       be non-linear and force me to decide which moved
c       out sample should be honoured (Canadian spelling)
c       at the single output sample location.  Obviously
c       this is an ongoing struggle in the world of nmo and
c       may end up being handled by some type of curve fitting
c       to the nmo map to smooth out the cross-overs (first
c       contemplated by the Hon. Paul Gutowski).  For now though
c       what you see is what you get.
c
  
      if(cros_ndx2.eq.0)cros_ndx2 = 2

c
c ----- when removing nmo using the top option, if there is -----
c       no cross-over (ie:cros_ndx = 0) then there is no data
c       associated with the top of the map for this offset.
c       In this case return a zero trace, otherwise reset 
c       icount to be the cross-over point as we are not interested
c       in any data from beyond that point.
c

      if(top) then

         if(cros_ndx.lt.1)return
         icount = cros_ndx

      endif

c
c ----- if forward and bot and no cross-over then set cros_ndx to 1 ----
c       This stops workx(cros_ndx) from being workx(0) which is 
c       undefined in this routine.
c

      if(.not.remove.and.bot.and.cros_ndx.lt.1)cros_ndx=1

c
c ----- if bot then move data from below cross-over to top of array -----
c       again to prevent zero values from corrupting the second 
c       derivative calculation used in the spline interpolator, 
c       else just continue with new icount from above
c

      IF(remove.and.bot)then

         if(cros_ndx.gt.0) then

            DO j=1,icount

               workx(j) = workx(cros_ndx + j - 1)
               worky(j) = worky(cros_ndx + j - 1)
c
c ----- if 2 consecutive sample indices are equal the second derivative  -----
c       calculation packs it in with a zero divide.. the following code
c       covers for the shitty interpolator.  Note only near the 
c       cross-over do values attempt this miraculous feat
c

               if(j.gt.1)then

                  if(abs(workx(j)-workx(j-1)).lt.1.e-3)then

                     workx(j)=workx(j)+1.e-3

                  endif

               endif

            ENDDO

         endif

      ELSEIF(remove.and..not.bot)then

         DO j=1,icount-start_index+1

            if(cros_ndx.gt.0) then

               workx(j) = workx(start_index + j - 1)
               worky(j) = worky(start_index + j - 1)

            endif

         ENDDO

         icount = icount - start_index + 1

      ENDIF

c+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c
c      REMOVE OPTION
c
c+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      IF(remove)then

c
c ----- interpolate new output trace -----
c

         if(.not.bot)then

c
c ----- sort tnmo,amplitude arrays to increasing order.  The interpolation -----
c       routines don't work on decreasing order datasets.  Make sure to
c       keep the correct amplitude associated with the correct tnmo.  This
c       is done by sorting both arrays simultaneously.
c

            call sort(icount,workx,worky)

         endif

c
c ----- if 2 consecutive sample indices are equal the second derivative  -----
c       calculation packs it in with a zero divide.. the following code
c       covers for the shitty interpolator.  Note only near the 
c       cross-over do values attempt this miraculous feat
c

        DO j=icount,2,-1

           if(abs(workx(j)-workx(j-1)).lt.1.e-3)workx(j)=workx(j)+1.e-3

        ENDDO

c
c ----- if cross-over occurs within 4 samples of the bottom of -----
c       the trace then just zero the trace and call it a day.
c       The cubic spline second derivative calculations don't work
c       with less than 4 samples.
c

         if(icount.lt.4)then

            write(LERR,*)' less than 4 input samples used in '
            write(LERR,*)' moveout map.  Trace at offset ',x
            write(LERR,*)' will be deleted.'
            write(LERR,*)' '
            return

         endif 

c
c ----- create array of second derivatives (y2) -----
c

         call spline(workx,worky,icount,yp1,ypn,y2)


c
c ----- INTERPOLATE OUTPUT TRACE FOR REMOVE OPTION -----
c

         DO 5000 j=1,nsamp

            rj=float(j)

            if(bot) then

               if((rj.lt.workx(2)).or.(rj.gt.workx(icount))
     :              .or.(rj.lt.float(Onset)))then

c
c ----- if j outside the range of data in transfer function  -----
c       set amplitude to zero, if it didn't go forward it can't
c       come back.
c

                  nmo_trace(j) = 0.

               else

c
c ----- interpolate amplitude at j from tnmo,input amplitude data
c

                  call splint(workx,worky,y2,icount,rj,yj) 
                  nmo_trace(j) = yj

               endif

            endif

c
c ----- this next bit handles the top map between cros_ndx2 and cros_ndx -----
c       which icount is now equal to.
c

            if(.not.bot) then

               if((rj.lt.workx(cros_ndx2)).or.(rj.gt.workx(icount))
     :              .or.(rj.lt.float(Onset)))then

c
c ----- if j outside the range of data in transfer function  -----
c       set amplitude to zero, if it didn't go forward it can't
c       come back.
c

                  nmo_trace(j) = 0.

               else

c
c ----- interpolate amplitude at j from tnmo,input amplitude data
c

                  call splint(workx,worky,y2,icount,rj,yj)
                  nmo_trace(j) = yj

               endif

            endif

 5000    CONTINUE 
 
c++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c
c     FORWARD NMO
c
c++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      ELSE

c
c ----- if the trace has less than 4 samples then just zero -----
c       the trace and call it a day.  The cubic spline second
c       derivative calculations don't work with less than 4 samples.
c

      if(nsamp.lt.4)then

         write(LERR,*)' less than 4 input samples used in '
         write(LERR,*)' moveout map.  Trace at offset ',x
         write(LERR,*)' will be deleted.'
         write(LERR,*)' '
         return

      endif  

c
c ----- form array of second derivatives -----
c    

      call spline(t,y,nsamp,yp1,ypn,y2)

c
c ----- INTERPOLATE OUTPUT TRACE FOR APPLICATION OF NMO  OPTION -----
c

      DO 6000 j=1,nsamp 

c
c ----- determine if there is an amplitude associated with the current -----
c       output sample.  If not then leave a zero entry, if so then
c       interpolate a value.
c

         if(top)then
            
            if((j.lt.ifix(t(1))).or.(j.gt.ifix(t(cros_ndx))))
     :           goto 6000 

         endif

         if(bot)then

            if((j.lt.ifix(t(cros_ndx))).or.(j.gt.ifix(t(icount+
     :           start_index-1))))goto 6000

         endif

         if(.not.top.and..not.bot)then 

            if((j.lt.ifix(t(1))).or.(j.gt.ifix(t(icount+
     :           start_index-1))))goto 6000

         endif

         call splint(t,y,y2,nsamp,worky(j),yj)

c
c ----- handle onset of data -----
c
 
         if(nint(worky(j)).lt.Onset)then
            
            nmo_trace(j) = 0.0

         else

            nmo_trace(j) = yj

         endif
 
 6000 CONTINUE

      ENDIF

c
c ----- return to calling routine -----
c

      XY = .false.
      return
      end
