C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
c princomp:  PRINcipal COMPonents analysis for seismic data.
c writes the results to an output file
 
c It is assumed that the development location has afp - Amoco Fortran
c Preprocessor
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters
c    these includes are picked up and expanded into the code by afp
c    and are found in ~usp/include/f77
 

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
c-----
c    SZLNHD is a value obtained from lhdrsz.h
c-----
      integer     itr ( SZLNHD )
      integer     lhed( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne
      integer     static
 
c------
c  dynamic memory allocation for big arrays, eg whole records
      integer     itrh
      real        datai, prcomp, covar, eigvec
      pointer     (wkitrh,  itrh(1))
      pointer     (wkadri, datai(1))
      pointer     (wkadrp, prcomp(1))
      pointer     (wkadrc, covar(1))
      pointer     (wkadre, eigvec(1))
c------
 
c------
c    this is necessary for buliding printout files
#include <f77/pid.h>
c------
      real        gain(SZLNHD), toteng, epsiln, sum
      integer     ibiggest,kodepc, codepc(SZLNHD), irec
c-----
c    SZSMPM is a value obtained from lhdrsz.h
c-----
      real        tri(SZLNHD), vmean(SZLNHD), eigval(SZLNHD)
      real        cumsum(0:SZLNHD)
      character   ntap * 256, otap * 256, name*8, sfile * 256
      logical     verbos, query, heap, incld, rms, norm
      integer     argis
 
c-----
c    we acces the floating point data through an equivalence statement
c    that starts the reals at 1/2-word 129
c-----
c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'PRINCOMP'/
      data epsiln/1.e-30/
 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/open.h>

      call gcmdln(ntap,otap,sfile,ns,ne,irs,ire,irec,
     1              ibiggest,toteng,kodepc,incld,verbos,rms,norm,
     2              itop)
 
c-----
c     get logical unit numbers for input and output of seismic data
 
c     input values are strings ntap & otap (i.e. names of files), and the
c     read 'r' or write 'w' designations of these files, and the default
c     logical unit numbers if ntap & otap are blank strings, i.e. ntap = " "
c     in which case:
c     0 = default stdin
c     1 = default stdout
c     Note: default values other that 0 & 1 may be used
 
c     output values are the logical unit numbers accessing these disk files. if
c     these values are less than 0 it means there was a fatal error in trying to
c     open these files
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

      if (sfile(1:1) .ne. ' ') then
         open (lun, file = sfile, status = 'unknown', err=990)
         write(LERR,*)'Opened output file for fractional energy'
         go to 991
990      write(LERR,*)'Could not open file for fractional energy- FATAL'
         write(LERR,*)'Seek professional help'
         stop
      endif
991   continue

 
c-----
c     read line header of inputa DSN (rtape reads data into vector "itr"
c     lbytes is the number of bytes actually read
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'PRINCOMP: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
 
c------
c     save certain parameters
 
c     to get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
 
c     LINHED is a value in the include file <f77/sisdef.h> that refers to the
c     lineheader
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

 
c------
c     hlhprt prints out the historical line header of length lbytes AND
 
c     hlhprt takes "name", in this case 4 characters long and stuffs this
c     into the modified historical line header and returns the NEW length
c     of the line header in lbytes
c------
      call hlhprt (itr, lbytes, name, 8, LERR)
 
c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
 
c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.
 
c--------------------------
c  note: these don't
c  have to be the same size
 
      itemi = (ne-ns+1) * ITRWRD * SZSMPD
      item1 = (ne-ns+1) * nsamp  * SZSMPD
      item2 = (ne-ns+1) * (ne-ns+1)  * SZSMPD
 
c  note also SZSMPD is the native
c  size of a float or int in bytes
c--------------------------
 
c--------
c  galloc - general allocation (machine independent since it uses C
c  malloc internally
c  inputs to galloc are pointer, number of bytes to allocate
c  outputs are error codes:
c     errcod = 1  (allocation succeeded)
c     errcod = 0  (allocation failed)
c--------
 
c----
c   wkitrh - headers          (ntr x ITRWRD) : itrh
c   wkadri - input data       (ntr x nsamp)  : datai
c   wkadrp - princ components (ntr x nsamp)  : prcomp
c   wkadrc - covar matrix     (ntr x ntr)    : covar
c   wkadre - eienvect         (ntr x ntr)    : eigvec
c----
      call galloc (wkitrh, itemi, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (wkadri, item1, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
 
      call galloc (wkadrp, item1, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (wkadrc, item2, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (wkadre, item2, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
 
      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*) item2,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------
 
 
c-----
c     modify line header to reflect actual number of traces output
c-----
      nrecc = ire - irs+1
      call savew(itr, 'NumRec', nrec , LINHED)
      jtr   = ne-ns+1
      call savew(itr, 'NumTrc', ntrc , LINHED)
 
c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
 
c----------------------
c  inject command line into
c  historical LH:
c  inputs are current line header in vector "itr" and length lbytes (bytes)
c  outputs are line header modified by insertion of the command line into
c  the historical LH and the modified length of the new LH (lbyout)
 
      call savhlh(itr,lbytes,lbyout)
c----------------------
 
c------
c     write to unit number luout lbyout bytes contained in vector itr
c------
      call wrtape ( luout, itr, lbyout                 )
 
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
            call verbal(nsamp, nsi, ntrc, nrec, iform, irec,
     1                  ibiggest,toteng,kodepc,ntap,otap,rms,norm,
     2                  ns,ne,irs,ire,itop)
 
c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary
 
         dt = real (nsi) * unitsc
 
c--------------------------------------------------
c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     skip unwanted records
c-----
c     call recskp(1,irs-1,luin,ntrc,itr)
      call recrw (1,irs-1,luin,ntrc,itr,luout, nbytes)
 
c-----
c     process desired trace records
c-----
      do 1000 jj = irs, ire
 
c----------------------
c  skip to start trace
c           call trcskp(jj,1,ns-1,luin,ntrc,itr)
            call trcrw (JJ, 1, ns-1, luin, ntrc, itr, luout, nbytes)
c----------------------
 
            call vclr (datai, 1, jtr*nsamp)
            ic = 0
            do 1001  kk = ns, ne
 
                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)

                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)

 
                  if (static .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif
 
c----------------------
c  save headers
c  pack data into array
c  compute gain
                  ic = ic + 1
                  istrc = (ic-1) * nsamp
                  ishdr = (ic-1) * ITRWRD

                  if (norm .or. rms) then
                     call vabs (tri, 1, vmean, 1, nsamp)
                     if (rms) then
                        call vsqrt (vmean, 1, vmean, 1, nsamp)
                     endif
                     call sve (vmean, 1, sum, nsamp)
                     if (sum .lt. 1.e-30) sum = 1.0
                     if (rms) sum = sum * sum
                     gain(ic) = sum / float(nsamp)
                     call vsdiv (tri, 1, gain(ic), tri, 1, nsamp)
                  endif
                  call vmov (lhed,1, itrh(ishdr+1),1,ITRWRD)
                  call vmov (tri,1, datai(istrc+1),1,nsamp)
 
1001        continue
 
c----------------------
c  skip to end of record
c           call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c----------------------

 
c-----------------------
c  here's the meat...
 
                   call pccalc (sfile,jj,irs,verbos,nsamp, jtr, datai,
     1                          ibiggest,prcomp, covar, eigvec, vmean,
     2                          eigval, cumsum, codepc, kodepc, epsiln,
     3                          toteng, sumval, kmxvec, iprsgn, iprcom,
     4                          irec, incld, itop)
 
c-----------------------
 
c---------------------
c  extract traces from
c  output array, restore
c  headers and
c  write output data
            do 1002 kk = 1, jtr
 
                  istrc = (kk-1) * nsamp
                  ishdr = (kk-1) * ITRWRD
c                 call move (1,itr(1),itrh(1,kk),SZTRHD )
                  call vmov (itrh(ishdr+1),1,lhed,1,ITRWRD)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)
                  call vmov (datai(istrc+1),1,tri,1,nsamp)
                  if (norm .or. rms) then
                     call vsmul (tri, 1, gain(ic), tri, 1, nsamp)
                  endif
                  if (static .eq. 30000) call vclr (tri, 1, nsamp)
                  call vmov (tri, 1, lhed(ITHWP1), 1, nsamp)
                  call wrtape (luout, itr, obytes)
 
 
 1002             continue
 
            call trcrw (JJ, ne+1, ntrc, luin, ntrc, itr, luout, nbytes)


            if(verbos)write(LERR,*)'ri ',itr(106)
 
 1000       continue
 
            call recrw (ire+1, nrec, luin, ntrc, itr, luout, nbytes)
  999 continue
 
c-----
c     close data files
c     flush data left
c     in output buffer
c     NOTE: if the output buffer is not closed you can sometimes end up
c     with missing data
c-----
      if (sfile(1:1) .ne. ' ') close (lun)
      call lbclos ( luin )
      call lbclos ( luout )
 
            write(LERR,*)'end of princomp, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'princomp does dark and terrible things to seismic data:'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute by typing princomp and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
        write(LER,*)
     :' -S [sfile]   (none)            : optional energy fraction file'
        write(LER,*)
     :' -ir [irec]   (none)            : output sfile for every ir recs'
        write(LER,*)
     :'                                  suitable for xgraph display'
        write(LER,*)
     :' -ns[ns]      (default = first) : start process trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)  : end process trace number'
        write(LER,*)
     :' -rs[irs]     (default = first) : start process record number'
        write(LER,*)
     :' -re[ire]     (default = last)  : end process record number'
        write(LER,*)
     :' all other data will be passed unprocessed from input to output'
        write(LER,*) ' '
        write(LER,*)
     :' -itp[itop] (default = none)    :  reconstruct using top % ev'
        write(LER,*)
     :' -epc[pc] (default = 100)       :  % total energy to output'
        write(LER,*)
     :' -kd[kd] (default = 0) :  principal component binary code'
        write(LER,*)
     :' -ib[ib] (default = 0) :  use ib largest eig. vals in recon'
        write(LER,*)
     :'                          this entry overrides -kd[]'
        write(LER,*) ' '
        write(LER,*)
     :' -norm include on command line if traces are energy normed'
        write(LER,*)
     :' -rms  include on cmd line if traces are rms amplitude normed'
        write(LER,*)
     :'       in both norm cases it is undone before data are output'
        write(LER,*)
     :' -I    include on cmd line for -kd[] if princ. comp. included'
        write(LER,*)
     :' -V    include on cmd line if verbose printout is desired'
        write(LER,*) ' '
        write(LER,*)
     :'usage:   princomp -N[] -O[] -S[] -ir[] -rs[] -re[] -ns[] -ne[]'
        write(LER,*)
     :'                   -itp[] -epc[] -ib[] -kd[] [-norm -rms -I -V]'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,sfile,ns,ne,irs,ire,irec,
     1                  ibiggest,toteng,kodepc,incld,verbos,rms,norm,
     2                  itop)
c-----
c     get command arguments
c
c     ntap  - C*100    input file name
c     otap  - C*100    output file name
c     sfile - C*100    file name for xgraph fractional energy 
c     toteng- R*4      % total output energy
c     kodepc- I*4      binary code for dominant principal components
c   ibiggest- I*4      take ibiggest largest e.vals in recon
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     ir    - I*4      output diagnostic sfile every ir recs
c     norm    L        normalize with abs averge ampl before calculations
c     norm    L        normalize with averge rms ampl before calculations
c     incld   L        include p.c.
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), sfile*(*)
      integer     ns, ne, irs, ire
      real        toteng
      integer     ibiggest,kodepc, irec
      logical     verbos, incld,rms,norm
      integer     argis
 
c-------
c     import values from the command line using keys, e.g. -N
c     to which are immediately attached the users values.
 
c     For example program princomp might be invoked in the following way:
 
c     princomp  -Nxyz -Oabc
 
c     in which case xyz is a string (the name of the input data set)
c     which will be imported into princomp and associated with the variable
c     "ntap"
 
c     see manual pages on the argument handler routines
c     for the meanings of these functions.  Briefly though
c     we can import from the command line strings, integers, reals,
c     double precision reals, and check the existence of command line
c     keys.  The last 2 fields are the value to be assigned the variable
c     (1) if ONLY the key is present (no value attached to it), or
c     (2) if NO key & no value are present
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argstr( '-S', sfile, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argi4 ( '-kd', kodepc ,   0  ,  0    )
            call argi4 ( '-ib', ibiggest ,   0  ,  0    )
            call argi4 ( '-ir', irec ,   1  ,  1    )
            call argi4 ( '-itp', itop ,   0  ,  0    )
            call argr4 ( '-epc', toteng, 100., 100. )

            if (itop .ne. 0) ibiggest = 0

            rms    =   (argis('-rms') .gt. 0)
            norm   =   (argis('-norm') .gt. 0)
            incld =   (argis('-I') .gt. 0)
            verbos =   (argis('-V') .gt. 0)
 
            if (ibiggest .ne. 0) kodepc = 0
c-------
c     Note:  argis() = 1 if the key is present
c            argis() = 0 if it isn't
 
      return
      end
 
C***********************************************************************
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform, irec,
     1                  ibiggest,toteng,kodepc,ntap,otap,rms,norm,
     2                  ns,ne,irs,ire,itop)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ns    - I*4     starting trace index
c     ne    - I*4     ending trace index
c     irs   - I*4     starting record index
c     ire   - I*4     ending record index
c     ntap  - C*100   input file name
c     otap  - C*100   output file name
c-----
#include <f77/iounit.h>
 
      integer     ibiggest,nsamp, nsi, ntrc, nrec, irec
      character   ntap*(*), otap*(*)
      logical     rms,norm
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            if (itop .ne. 0) then
            write(LERR,*) ' Reconstruct using top ',itop,' % evs'
            else
            write(LERR,*) ' % total energy     =  ',toteng
            write(LERR,*) ' Principal com code =  ',kodepc
            write(LERR,*) ' # largest eig vals =  ',ibiggest
            endif
            write(LERR,*) ' Traces ave abs ampl normed=  ',norm
            write(LERR,*) ' Traces rms ampl normed    =  ',rms
            write(LERR,*) ' Start process trace =  ', ns
            write(LERR,*) ' End process trace   =  ', ne
            write(LERR,*) ' Start process record=  ', irs
            write(LERR,*) ' End process record  =  ', ire
            write(LERR,*) ' Outputting diagnostics every ',irec,' recs'
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
