/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* 
   This program is designed to perform one or more time slices on a 
   USP format dataset. It is designed to gain a speed advantage by
   reading the input data sequentially and maintaining buffers for the
   output data until a full set of traces are ready to be written. If
   space is insufficient, the input data may be "windowed", where multiple
   passes across the input data are required. In this mode, the input
   data must reside on disk. Output dataset are always required to be on
   disk due to the nature of the process.

   Note that regular tape i/o is NOT used in this program. This allows
   the program to avoid the time penalties associated with data reformatting
   on some machines. Since this process is basically a reordering of the
   dataset and nothing else, this is possible. It does provide for some
   interesting code when modifying the line and trace headers, though.

						- Joe M. Wade 11/23/94 

   Added -restart option for picking up where a job left off due to system
   crash, disk errors, etc. ( ..not too uncommon an occurrence, unfortunately)
						- Joe M. Wade 03/15/96 
*/
#include <stdio.h>
#include <unistd.h>
#include <localsys.h>
#include <size_defs.h>
#include <sisio_c.h>
#include <slice.h>		/* resides in ~usp/include */
#include <ut_defs.h>
#include <cu_defs.h>
#include <sys/types.h>
#include <usp_headers.h>
#ifdef TERAFILE
#include <file64.h>
#endif
#ifndef FSEEKO
#define fseeko fseek
#endif

char ntap[256],otap[256];
FILE *fdin,*fdout,*prtfile;

extern int usp_argc;
extern char **usp_argv;

#if ( BYTE_ORDER == LITTLE_ENDIAN )
#define CONVERT2(word) (short)(((long)word<<8)&0x0000ff00)|(((long)word>>8)&0x000000ff)
#define CONVERT4(word) (((word<<24)&0xff000000)|((word<<8)&0x00ff0000)|((word>>8)&0x0000ff00)|((word>>24)&0x000000ff))
#endif

main(argc,argv)
int argc;
char **argv;
{
	char *staging;
	char *staged_sample;
	char prtfilename[32];
	int smpwin,max_smpwin;	 /* no. of samples in input window to buffer */
	int orig_smpwin;
	int i,j,k;
	int piped_input,piped_output;
	int revers,restart,verbos,is_a_pipe();
	int first_slice,last_slice,increment,slice;
	int first_slice_time, slice_interval;
	int record_number,trace_number;
	float first_time;
	int trace_size;
	char **green_trace,**trace,**sample;
	char *header;
	char *junk;		/* space for storing skipped data when piping */
	char *ikppid;
	int hdr_byte_count;
	off_t rec_skip_size;
	off_t skip_size,initial_skip_size,normal_skip_size,sample_skip_size;
#ifdef CONVEXSYSTEM
	off64_t offset,location;
	off64_t saved_pos;
#else
	off_t offset,location;
	off_t saved_pos;
#endif
	char hostname[9];
	char *prgmname = "qdslice";
	size_t old_hdr_byte_count,new_hdr_byte_count;
	int entry_length;
	line_header old,new;
	trace_header trace_hdr;
	void read_err(),write_err();
	int get_full_word(),get_half_word();
	void put_full_word(),put_half_word();
	float get_real_full_word();
	int C_IN_IKP();
        time_t current_time, start_time, estimated_time;
	int start_slice,time_est;
	void help();

	if (C_ARGIS("-h",&argc,argv) != 0) help();
	if (C_ARGIS("-?",&argc,argv) != 0) help();

	sprintf(prtfilename,"QDSLICE.%d",getpid());

	if (C_IN_IKP() != 0) {
	  if (gethostname(hostname,9) == 0) {
	    hostname[8] = '\0';
	    strcat(prtfilename,".");
	    strcat(prtfilename,hostname);
	    }
	  }

	prtfile = fopen(prtfilename,"w");
	if (prtfile == NULL) prtfile = stderr;

	fprintf(prtfile,"QDSLICE:\n");

	save_cmd_args(argc,argv);
	verbos = C_ARGIS("-V",&argc,argv);
	revers = C_ARGIS("-R",&argc,argv);
	restart = C_ARGIS("-restart",&argc,argv);
	time_est = C_ARGIS("-VT",&argc,argv);
	C_ARGSTR("-N",ntap,"","",&argc,argv);
	C_ARGSTR("-O",otap,"","",&argc,argv);

	if (strlen(ntap) != 0) {
	  fprintf(prtfile,"input data set -  %s\n",ntap);
#ifdef CONVEXSYSTEM
	  fdin = fopen(ntap,"rl");
#else
	  fdin = fopen(ntap,"r");
#endif
	  if (fdin == NULL) {
	    fprintf(stderr,"ERROR: unable to open input data set - ");
	    fprintf(stderr,"please check data set\n");
	    fprintf(prtfile,"ERROR: unable to open input data set - ");
	    fprintf(prtfile,"please check data set\n");
	    fclose(prtfile);
	    exit(100);
	    }
	  }
	else {
	    fprintf(prtfile,"input data set -  stdin\n");
	    fdin = stdin;
	  }

	if (restart != 0) {
	  fprintf(prtfile,"restart option selected\n");
	  if (revers != 0) {
	    fprintf(stderr,"QDSLICE: restart option currently unavailable ");
	    fprintf(stderr,"with reverse option; aborting\n");
	    fprintf(stderr,"( A separate qdslice and a gather of the two data");
	    fprintf(stderr,"sets may be used to construct the output data.)\n");
	    fprintf(prtfile,"ERROR: restart option currently unavailable ");
	    fprintf(prtfile,"with reverse option; aborting\n");
	    fprintf(prtfile,"( A separate qdslice and a gather of the two data");
	    fprintf(prtfile,"sets may be used to construct the output data.)\n");
	    exit(1);
	    }
	  }

	if (strlen(otap) != 0) {
	  fprintf(prtfile,"output data set -  %s\n",otap);
#ifdef CONVEXSYSTEM
	  if (restart == 0) 
	    fdout = fopen(otap,"wl");
	  else
	    fdout = fopen(otap,"rl+");
#else
	  if (restart == 0) 
	    fdout = fopen(otap,"w");
	  else {
	    fdout = fopen(otap,"r+");
	    }
#endif
	  if (fdout == NULL) {
	    fprintf(stderr,"ERROR: unable to open output data set - ");
	    fprintf(stderr,"please verify permissions\n");
	    fprintf(prtfile,"ERROR: unable to open output data set - ");
	    fprintf(prtfile,"please verify permissions\n");
	    fclose(prtfile);
	    exit(100);
	    }
	  }
	else {
	    fprintf(prtfile,"output data set -  stdout\n");
	    fdout = stdout;
	  }
	
	if (revers == 0)
	  setbuf(fdout,NULL);
/*
	else {
	  setbuf(fdin,NULL);
	  }
*/

	buffer.value = 0;
	if (fread(&buffer.fullword.value[0],SZGRWD,1,fdin) == 0) read_err();

#if ( BYTE_ORDER == LITTLE_ENDIAN )
	old_hdr_byte_count = CONVERT4(buffer.value);
#else
	old_hdr_byte_count = buffer.value;
#endif

	if (restart != 0) {
	  buffer.value = 0;
	  if (fread(&buffer.fullword.value[0],SZGRWD,1,fdout) == 0) read_err();
	  new_hdr_byte_count = buffer.value;
	  }
	else {
	  new_hdr_byte_count = old_hdr_byte_count + strlen(prgmname) + SZDHWD;
	
	  for (i=1; i<usp_argc; i++)
	    new_hdr_byte_count = new_hdr_byte_count + strlen(usp_argv[i]) + 1;
	  }

	header = (char *) malloc(new_hdr_byte_count);
	if (fread(header,old_hdr_byte_count,(size_t)1,fdin) == 0) read_err();

	if (restart == 0) {	/* build new HLH */
	put_half_word(header+old_hdr_byte_count,new_hdr_byte_count - old_hdr_byte_count - SZDHWD);
	new_hdr_byte_count = old_hdr_byte_count;
	new_hdr_byte_count += 2;
	memcpy(header+new_hdr_byte_count,prgmname,strlen(prgmname));
	new_hdr_byte_count += strlen(prgmname);

	for (i=1; i<usp_argc; i++) {
	  memcpy(header+new_hdr_byte_count," ",1);
	  new_hdr_byte_count += 1;
	  memcpy(header+new_hdr_byte_count,
		usp_argv[i],strlen(usp_argv[i]));
	  new_hdr_byte_count += strlen(usp_argv[i]);
	  }
	  }

	old.Number_of_Traces  = get_full_word(header+NUMBER_OF_TRACES_OFFSET);
	old.Number_of_Records = get_full_word(header+NUMBER_OF_RECORDS_OFFSET);
	old.Sample_Interval   = get_full_word(header+SAMPLE_INTERVAL_OFFSET);
	old.Number_of_Samples = get_full_word(header+NUMBER_OF_SAMPLES_OFFSET);
	old.Format 	      = get_half_word(header+FORMAT_OFFSET);
	old.Hlh_Entries	      = get_half_word(header+HLHENT_OFFSET);
	old.Hlh_Bytes	      = get_half_word(header+HLHBYT_OFFSET);
	first_time	      = get_real_full_word(header+
					FIRST_SAMPLE_TIME_OFFSET);

	fprintf(prtfile,"\ninput:\n");
	fprintf(prtfile,"\t\tformat = %d\n",		old.Format);
	fprintf(prtfile,"\t\tno. of records = %d\n",	old.Number_of_Records);
	fprintf(prtfile,"\t\tno. of traces/record = %d\n",old.Number_of_Traces);
	fprintf(prtfile,"\t\tno. of samples = %d\n",	old.Number_of_Samples);
	fprintf(prtfile,"\t\tsample interval = %d\n",	old.Sample_Interval);

	new.Format		= old.Format;
	if (revers == 0) {
	  new.Number_of_Samples	= old.Number_of_Traces;
	  new.Number_of_Traces	= old.Number_of_Records;
	  }
	else {
	  new.Number_of_Samples	= old.Number_of_Records;
	  new.Number_of_Records	= old.Number_of_Traces;
	  }
	new.Sample_Interval	= 1;
	new.Hlh_Entries		= old.Hlh_Entries + 1;
	new.Hlh_Bytes		= old.Hlh_Bytes + new_hdr_byte_count -
				  old_hdr_byte_count;

/*  look for sample parms first, then by time; if neither, use default */

	C_ARGI4("-ss",&first_slice,0,-1,&argc,argv);
	if (first_slice == -1) {
	  C_ARGI4("-s",&first_slice,0,(int)first_time,&argc,argv);
	  first_slice = (((float)first_slice - first_time) /
		old.Sample_Interval) + 1;
	  }
	if (first_slice <  1) first_slice = 1;
	if (first_slice > old.Number_of_Samples)
		first_slice = old.Number_of_Samples;

	C_ARGI4("-es",&last_slice,old.Number_of_Samples,-1,&argc,argv);
	if (last_slice == -1) {
	  C_ARGI4("-e",&last_slice,-1,-1,&argc,argv);
	  if (last_slice == -1)
	    last_slice = old.Number_of_Samples;
	  else
	    last_slice = (((float)last_slice - first_time) /
		old.Sample_Interval) + 1;
	  }
	if (last_slice < 1) last_slice = 1;
	if (last_slice > old.Number_of_Samples)
		last_slice = old.Number_of_Samples;

	C_ARGI4("-is",&increment,1,-1,&argc,argv);
	if (increment == -1) {
	  C_ARGI4("-i",&increment,old.Sample_Interval,old.Sample_Interval,
		&argc,argv);
	  increment = (int) (increment / old.Sample_Interval);
	  }
	if (increment < 0) increment = 1;

	max_smpwin = (last_slice + increment - first_slice) / increment;
	C_ARGI4("-ns",&smpwin,max_smpwin,max_smpwin,&argc,argv);
	if (smpwin > max_smpwin) smpwin = max_smpwin;

	fprintf(prtfile,"\nsampling:\n");
	fprintf(prtfile,"\t\tfirst slice = %.2f ms (sample %d)\n",
		first_time+((first_slice-1)*old.Sample_Interval),first_slice);
	fprintf(prtfile,"\t\tlast slice = %.2f ms (sample %d)\n",
		first_time+((last_slice-1)*old.Sample_Interval),last_slice);
	fprintf(prtfile,"\t\tsampling increment = %d ms ( %d samples )\n",
		increment*old.Sample_Interval,increment);
	fprintf(prtfile,"\t\tsampling window = %d samples\n",smpwin);

	if (increment != 0.) {
	  if (revers == 0)
	    new.Number_of_Records =
		(int) ((last_slice - first_slice) / increment) + 1;
	  else
	    new.Number_of_Traces =
		(int) ((last_slice - first_slice) / increment) + 1;
	  }
	else {
	  if (first_slice != last_slice) {
	    fprintf(stderr,"ERROR: An increment of zero has been ");
	    fprintf(stderr,"specified although start\nand end slices differ;");
	    fprintf(stderr,"please correct and resubmit.\n");
	    fprintf(prtfile,"ERROR: An increment of zero has been ");
	    fprintf(prtfile,"specified although start\nand end slices differ;");
	    fprintf(prtfile,"please correct and resubmit.\n");
	    fclose(prtfile);
	    exit(100);
	    }
	  if (revers == 0)
	    new.Number_of_Records = 1;
	  else
	    new.Number_of_Traces = 1;
	  increment = 1; /* this will ensure that we get out of our loop! */
	  }

	piped_input  = is_a_pipe(fdin);
	piped_output = is_a_pipe(fdout);

	if ( revers && piped_input) {
	    fprintf(stderr,"ERROR: piped input not allowed\n");
	    fprintf(prtfile,"ERROR: piped input not allowed\n");
	    fclose(prtfile);
	    exit(100);
	  }

	if ( !revers && piped_output) {
	    fprintf(stderr,"ERROR: piped output not allowed\n");
	    fprintf(prtfile,"ERROR: piped output not allowed\n");
	    fclose(prtfile);
	    exit(100);
	    }

	fprintf(prtfile,"\noutput:\n");
	fprintf(prtfile,"\t\tformat = %d\n",		new.Format);
	fprintf(prtfile,"\t\tno. of records = %d\n",	new.Number_of_Records);
	fprintf(prtfile,"\t\tno. of traces/record = %d\n",
		new.Number_of_Traces);
	fprintf(prtfile,"\t\tno. of samples = %d\n",	new.Number_of_Samples);
	fprintf(prtfile,"\t\tsample interval = %d\n\n",new.Sample_Interval);

/* for reverse rotation, pull the slice interval out of the input header
   and save as sample interval
*/
	if ( revers ) {
	  slice_interval = get_full_word( header+SLICE_INTERVAL_OFFSET );
	  put_full_word(header+SAMPLE_INTERVAL_OFFSET,slice_interval);
	  put_full_word(header+SLICE_INTERVAL_OFFSET,0);
	  put_full_word(header+FIRST_SLICE_OFFSET,0);
	  }

	first_slice_time = ( first_slice - 1 ) * old.Sample_Interval;
	slice_interval   = increment * old.Sample_Interval;

/* if restarting, we've already taken care of the line header
*/
	if (!restart) {
	  put_full_word(header+NUMBER_OF_TRACES_OFFSET,new.Number_of_Traces);
	  put_full_word(header+NUMBER_OF_RECORDS_OFFSET,new.Number_of_Records);
	  put_full_word(header+NUMBER_OF_SAMPLES_OFFSET,new.Number_of_Samples);
	  put_half_word(header+FORMAT_OFFSET,new.Format);

	  if ( ! revers ) {
	    put_full_word(header+SAMPLE_INTERVAL_OFFSET,new.Sample_Interval);
	    put_full_word(header+FIRST_SLICE_OFFSET,first_slice_time);
	    put_full_word(header+SLICE_INTERVAL_OFFSET,slice_interval);
	    }

	  put_half_word(header+HLHENT_OFFSET,new.Hlh_Entries);
	  put_half_word(header+HLHBYT_OFFSET,new.Hlh_Bytes);

#if ( BYTE_ORDER == LITTLE_ENDIAN )
	  buffer.value = CONVERT4(new_hdr_byte_count);
#else
	  buffer.value = new_hdr_byte_count;
#endif
	  if (fwrite(buffer.fullword.value,SZGRWD,1,fdout) == 0) write_err();
	  if (fwrite(header,new_hdr_byte_count,1,fdout) == 0) write_err();
	  }

	buffer.value = 0;
	if (fread(buffer.fullword.value,sizeof(char),SZGRWD,fdin) == 0)
		read_err();

	header = (char *) realloc(header,TRACE_HDR_SIZE);
	if (fread(header,TRACE_HDR_SIZE,1,fdin) == 0) read_err();


	trace_size = TRACE_HDR_SIZE + (new.Number_of_Samples * SZSAMP);

	if (revers == 0) {
	  rec_skip_size = (new.Number_of_Traces-1)*(trace_size+SZGRWD);
	  }

	sample_skip_size = (increment-1)*SZSAMP;

	green_trace = (char **) malloc(smpwin*sizeof(char *));
	trace	    = (char **) malloc(smpwin*sizeof(char *));
	sample	    = (char **) malloc(smpwin*sizeof(char *));

	if ((green_trace == NULL) || (trace == NULL) ||
		(sample == NULL)) {
	  fprintf(stderr,"QDSLICE: ERROR - unable to allocate enough memory\n");
	  fprintf(stderr,"Resubmit using a smaller sampling window ('-ns')\n");
	  fprintf(prtfile,"QDSLICE: ERROR - unable to allocate enough memory\n");
	  fprintf(prtfile,"Resubmit using a smaller sampling window ('-ns')\n");
	  fclose(prtfile);
	  exit(100);
	  }

	for (i=0; i<smpwin; i++) {
	 if (revers == 0) {
	   trace_hdr.Record_Number = i+1;
	   trace_hdr.Trace_Number  = 1;
	   }
	 else {
	   trace_hdr.Record_Number = 1;
	   trace_hdr.Trace_Number = i+1;
	   }

	 green_trace[i] = (char *) calloc( trace_size + SZGRWD, sizeof(char));

	 if (green_trace[i] == NULL) {
	   fprintf(stderr,"QDSLICE: ERROR - unable to allocate enough memory\n");
	   fprintf(stderr,"Resubmit using a smaller sampling window ('-ns')\n");
	   fprintf(prtfile,"QDSLICE: ERROR - unable to allocate enough memory\n");
	   fprintf(prtfile,"Resubmit using a smaller sampling window ('-ns')\n");
	   fclose(prtfile);
	   exit(100);
	   }
	 trace[i] = green_trace[i] + SZGRWD;
	 memset(trace[i],0,TRACE_HDR_SIZE);
	 put_full_word(green_trace[i],trace_size);
	 put_half_word(trace[i]+RECORD_NUMBER_OFFSET,trace_hdr.Record_Number);
	 put_half_word(trace[i]+TRACE_NUMBER_OFFSET,trace_hdr.Trace_Number);
	 }

	if (revers == 0)
	  normal_skip_size = (off_t)
		((old.Number_of_Samples - (smpwin * increment)) * SZSAMP)
		+ SZGRWD + SZDTHD;
	else
/*
	  normal_skip_size = (long int)
		(old.Number_of_Traces - 1) *
			((old.Number_of_Samples * SZSAMP) + SZGRWD + SZDTHD);
*/
	  normal_skip_size = (off_t)
		(old.Number_of_Traces *
			((old.Number_of_Samples * SZSAMP) + SZGRWD + SZDTHD)
		 - (smpwin * increment) * SZSAMP);

/*
   We'll use the same junk buffer multiple times, so allocate for worst case
*/
	junk = (char *) malloc(SZGRWD + SZDTHD +
		(old.Number_of_Samples * SZSAMP));

	if (junk == NULL) {
	  fprintf(stderr,"QDSLICE: ERROR - unable to allocate enough memory ");
	  fprintf(stderr,"for needed buffering.\n");
	  fprintf(prtfile,"QDSLICE: ERROR - unable to allocate enough memory ");
	  fprintf(prtfile,"for needed buffering.\n");
	  fclose(prtfile);
	  exit(100);
	  }

	if (revers == 0) {

	  int Trace_Number;

	  fprintf(prtfile,"txy -> xyt rotation selected\n");

	  if ( restart != 0 ) {
	    fprintf(prtfile,"restart option selected\n");
	    if (time_est != 0)
	      fprintf(stderr,"QDSLICE - restart option selected\n");
	    /* find first slice */
  
#ifdef CONVEXSYSTEM
	   saved_pos = (off64_t) (new_hdr_byte_count+SZGRWD);
	   if ( fseek64(fdout,saved_pos,SEEK_SET) != 0) write_err();
#else
	   saved_pos = new_hdr_byte_count+SZGRWD;
	   if ( fseeko(fdout,saved_pos,SEEK_SET) != 0) write_err();
#endif
	   skip_size = ((old.Number_of_Samples * SZSAMP) + SZGRWD + SZDTHD);

	   for (i=0; i< new.Number_of_Traces; i++) {
#ifdef CONVEXSYSTEM
	     saved_pos = fseek64(fdout,(off64_t)0,SEEK_CUR);
#else
	     saved_pos = fseeko(fdout,0,SEEK_CUR);
#endif
	     if (fread(green_trace[0],trace_size+SZGRWD,1,fdout) == 0) write_err();
	     Trace_Number= get_half_word(trace[0]+TRACE_NUMBER_OFFSET);
	     if (Trace_Number != 9999) {
/*
   read a record from the input dataset. we keep one record behind we'll 
   actually start with the last slice that's go a valid trace header in 
   the output. (slice was probably never completed)
*/
	       if (i > 0)
	         for (j=0; j < old.Number_of_Traces; j++) {
		   if (piped_input) {
	             if (fread(junk,skip_size,1,fdin) == 0) read_err();
		     }
		   else {
	             if (fseeko(fdin,skip_size,SEEK_CUR) != 0) read_err();
		     }
		   }
	       }
	     else {
	       fprintf(prtfile,"restart location detected; trace = %d\n",i);
	       trace_hdr.Trace_Number = i;
	       i=new.Number_of_Traces;
	       }
	     }
#ifdef CONVEXSYSTEM
	   if ( fseek64(fdout,saved_pos,SEEK_SET) != 0) write_err();
#else
	   if ( fseeko(fdout,saved_pos,SEEK_SET) != 0) write_err();
#endif
	   }
	 else {
/* new stuff to pre-allocate space ; required for restart capabilities */
	   Trace_Number = 9999;
	   put_half_word(trace[0]+TRACE_NUMBER_OFFSET,Trace_Number);
	   if (time_est != 0)
		fprintf(stderr,"QDSLICE - pre-allocating disk space");
	   for (i=0;
		i< new.Number_of_Records*new.Number_of_Traces; i++) {
	      if ((time_est != 0) &&
	         (/* remainder */fmod((double)i,(double)new.Number_of_Traces) == 0.))
		fprintf(stderr,".");
	      if (fwrite(green_trace[0],trace_size+SZGRWD,1,fdout) == 0)
		write_err();
	     }
	   if (time_est != 0) fprintf(stderr,"\n");
	   }

	if ( time_est != 0 ) {
	  start_time = time(NULL);
	  start_slice = trace_hdr.Trace_Number;
	  if (time_est != 0)
	    fprintf(stderr,"QDSLICE - data processing begins - time: %s",
		ctime(&start_time));
	  }

/* forward rotation ( txy -> xyt */
	for (slice=first_slice;
		slice<=last_slice && smpwin>0; slice+=smpwin*increment) {

	 initial_skip_size = (slice -1) * SZSAMP;

	 for (i=trace_hdr.Trace_Number-1,skip_size=initial_skip_size;
		i< new.Number_of_Traces;
		  i++) {
	  for(k=0;k<smpwin;k++){sample[k]=(trace[k]+TRACE_HDR_SIZE);}
	  for (j=0;
		j < new.Number_of_Samples;
		  j++,skip_size=normal_skip_size) {

	    if (skip_size != 0)  {
	      if (piped_input) {
	        if (fread(junk,skip_size,1,fdin) == 0) read_err();
		}
	      else  {
	        if (fseeko(fdin,skip_size,SEEK_CUR) != 0) read_err();
		}
	      }

	    for (k=0; k<smpwin; k++) {
	      if (fread(sample[k],SZSAMP,1,fdin) == 0) {
		read_err();
		}
	      if (sample_skip_size != 0) {
	        if (fread(junk,sample_skip_size,1,fdin) == 0) {
/* could be last data sample - more elaborate test is possible */
		  if ((k+1) != smpwin) read_err();
		  }
	 	}
	      sample[k]+=SZSAMP;
	      }
	    }
  
#ifdef CONVEXSYSTEM
	  saved_pos = (off64_t) (new_hdr_byte_count+SZGRWD+
		((off64_t)(trace_hdr.Record_Number-smpwin) *
		(off64_t)new.Number_of_Traces +
		(off64_t)(trace_hdr.Trace_Number-1)) *
		(off64_t)(trace_size+SZGRWD));
	  if ( fseek64(fdout,saved_pos,SEEK_SET) != 0) write_err();
#else
	  saved_pos = (off_t) new_hdr_byte_count+SZGRWD+
		(((off_t)(trace_hdr.Record_Number-smpwin) *
		new.Number_of_Traces) + (off_t)(trace_hdr.Trace_Number-1)) *
		(off_t)(trace_size+SZGRWD);
	  if ( fseeko(fdout,saved_pos,SEEK_SET) != 0) write_err();
#endif

/* put in timing stuff */
	    if ((trace_hdr.Record_Number == new.Number_of_Records)
		&& (time_est != 0)) {
	      current_time = time(NULL);
	      fprintf(stderr,"QDSLICE - %d of %d: ",
		trace_hdr.Trace_Number,new.Number_of_Traces);
	      estimated_time = (time_t) (((current_time - start_time) *
		(new.Number_of_Traces-start_slice))  /
		(trace_hdr.Trace_Number+1-start_slice));
	      estimated_time += start_time;
	      fprintf(stderr,"est. time of completion: %s",
	        ctime(&estimated_time));
	      }
/* put in timing stuff */
	  for (k=0; k<smpwin; k++) {
	    put_half_word(trace[k]+TRACE_NUMBER_OFFSET,trace_hdr.Trace_Number);

	    if (fwrite(green_trace[k],trace_size+SZGRWD,1,fdout) == 0)
		write_err();
	    if ( fseeko(fdout,rec_skip_size,SEEK_CUR) != 0) read_err();
	    }
	  trace_hdr.Trace_Number++;
	  }
	 trace_hdr.Trace_Number = 1;
	 trace_hdr.Record_Number++;
	 for (k=0; k<smpwin; k++) {
	   if (trace_hdr.Record_Number > new.Number_of_Records) {
	     slice += (smpwin-k)*increment;	/* fool the math */
	     smpwin = k;
	     normal_skip_size = (off_t)
		((old.Number_of_Samples - (smpwin * increment)) * SZSAMP)
		+ SZGRWD + SZDTHD;
	     }
	   else {
	     put_half_word(trace[k]+RECORD_NUMBER_OFFSET,
			trace_hdr.Record_Number);
	     put_half_word(trace[k]+TRACE_NUMBER_OFFSET,trace_hdr.Trace_Number);
	     trace_hdr.Record_Number++;
	     }
	   }
	 trace_hdr.Record_Number--;
	 if (smpwin != 0) {
	  if (fseeko(fdin,
	   old_hdr_byte_count+TRACE_HDR_SIZE+(2*SZGRWD),SEEK_SET) != 0) {
		fprintf(prtfile,"eof skipping past trace hdr\n");
		read_err();
	   	}
	  }
	  if ( time_est != 0) {
	    current_time = time(NULL);
	    fprintf(stderr,"QDSLICE - Processing completed: %s",
		ctime(&current_time));
	    }
	 }
	}
	else {
	  fprintf(prtfile,"xyt -> txy rotation selected\n");
	  if ( time_est != 0 ) {
	    start_time = time(NULL);
	    start_slice = trace_hdr.Record_Number;
	    fprintf(stderr,"QDSLICE - data processing begins - time: %s",
		ctime(&start_time));
	    }
/* reverse rotation ( xyt -> txy ) */
	  orig_smpwin = smpwin;
	  staging = (char *) malloc(orig_smpwin*increment*SZSAMP);
	  if (staging == NULL) {
	    fprintf(stderr,
		"QDSLICE: ERROR - unable to allocate enough memory ");
	    fprintf(stderr,"for needed buffering.\n");
	    fprintf(prtfile,
		"QDSLICE: ERROR - unable to allocate enough memory ");
	    fprintf(prtfile,"for needed buffering.\n");
	    fclose(prtfile);
	    exit(100);
	    }

	  for (i=0;
		i< new.Number_of_Records;
		  i++) {
	    smpwin = orig_smpwin;

	    for (slice=first_slice;
		slice<=last_slice && smpwin>0; slice+=smpwin*increment) {

	      normal_skip_size = (off_t)
		(old.Number_of_Traces *
			((old.Number_of_Samples * SZSAMP) + SZGRWD + SZDTHD)
		 - (smpwin * increment) * SZSAMP);

	      for(k=0;k<smpwin;k++){sample[k]=(trace[k]+TRACE_HDR_SIZE);}

	      for (j=0,skip_size=0;
		j < new.Number_of_Samples;
		  j++,skip_size=normal_skip_size) {

	        if (skip_size != 0)  {
	          if (fseeko(fdin,skip_size,SEEK_CUR) != 0) read_err();
	          }

	        if (fread(staging,SZSAMP,smpwin*increment,fdin) == 0) {
		  read_err();
		  }
		staged_sample = staging;

	        for (k=0; k<smpwin; k++) {
		  memcpy(sample[k],staged_sample,SZSAMP);
		  staged_sample+=sample_skip_size+SZSAMP;
	          sample[k]+=SZSAMP;
	          }
		}
	  
/* put in timing stuff */
	      if ((trace_hdr.Trace_Number == new.Number_of_Traces)
		&& (time_est != 0)) {
		current_time = time(NULL);
		fprintf(stderr,"QDSLICE - %d of %d: ",
		  trace_hdr.Record_Number,new.Number_of_Records);
		estimated_time = (time_t) (((current_time - start_time) *
		  (new.Number_of_Records - start_slice))  /
		  (trace_hdr.Record_Number+1-start_slice));
		estimated_time += start_time;
		fprintf(stderr,"est. time of completion: %s",
		ctime(&estimated_time));
		}
/* put in timing stuff */

	      for (k=0; k<smpwin; k++) {
		if (fwrite(green_trace[k],trace_size+SZGRWD,1,fdout) == 0)
			write_err();
		}

	    trace_hdr.Trace_Number++;

	    if (trace_hdr.Trace_Number > new.Number_of_Traces) {
		trace_hdr.Trace_Number = 1;
		trace_hdr.Record_Number++;
		}

	    if (trace_hdr.Trace_Number != 1) {
	    for (k=0; k<smpwin; k++) {
	      if (trace_hdr.Trace_Number > new.Number_of_Traces) {
		slice += (smpwin-k)*increment;	/* fool the math */
		smpwin = k;
	        trace_hdr.Trace_Number = 1;
	        trace_hdr.Record_Number++;
	        }
	      else {
		put_half_word(trace[k]+RECORD_NUMBER_OFFSET,
			trace_hdr.Record_Number);
		put_half_word(trace[k]+TRACE_NUMBER_OFFSET,
			trace_hdr.Trace_Number);
	        trace_hdr.Trace_Number++;
		}
	      }

#ifdef CONVEXSYSTEM
	      offset = (off64_t)
	          old_hdr_byte_count+2*SZGRWD+TRACE_HDR_SIZE+
		  (off64_t)(i*((old.Number_of_Samples * SZSAMP)+
		  SZGRWD+TRACE_HDR_SIZE))+
		  ((slice+(smpwin*increment)-1)*SZSAMP);
	      if (fseek64(fdin,offset,SEEK_SET) != 0) {
#else
	      offset = old_hdr_byte_count+2*SZGRWD+TRACE_HDR_SIZE+
		  (off_t)(i*((old.Number_of_Samples * SZSAMP)+
		  SZGRWD+TRACE_HDR_SIZE))+((slice+(smpwin*increment)-1)*SZSAMP);
	      if (fseeko(fdin,offset,SEEK_SET) != 0) {
#endif
		fprintf(prtfile,"eof skipping past trace hdr\n");
		read_err();
	   	}
	      }
	    else {
	      for (k=0; k<orig_smpwin; k++) {
		put_half_word(trace[k]+RECORD_NUMBER_OFFSET,
			trace_hdr.Record_Number);
		put_half_word(trace[k]+TRACE_NUMBER_OFFSET,
			trace_hdr.Trace_Number);
	        trace_hdr.Trace_Number++;
		}
#ifdef CONVEXSYSTEM
	      offset = (off64_t)(old_hdr_byte_count+2*SZGRWD+TRACE_HDR_SIZE+
		(off64_t)((i+1)*((old.Number_of_Samples * SZSAMP)+
		SZGRWD+TRACE_HDR_SIZE)));
	      if (fseek64(fdin, offset, SEEK_SET) != 0) {
#else
	      offset = (off_t)(old_hdr_byte_count+2*SZGRWD+TRACE_HDR_SIZE+
		(off_t)((i+1)*((old.Number_of_Samples * SZSAMP)+
		SZGRWD+TRACE_HDR_SIZE)));
	      if (fseeko(fdin, offset, SEEK_SET) != 0) {
#endif
		fprintf(prtfile,"eof skipping past trace hdr\n");
		read_err();
	   	}
	      }
	      trace_hdr.Trace_Number--;
	      }
	    }
	  if ( time_est != 0) {
	    current_time = time(NULL);
	    fprintf(stderr,"QDSLICE - Processing completed: %s",
		ctime(&current_time));
	    }
	  }

 	fprintf(prtfile,"normal completion\n");
	fclose(fdin);
	fclose(fdout);
	fclose(prtfile);

	exit(0);
}
void read_err()
{
 	fprintf(stderr,
	  "ERROR: qdslice - error reading input data\n");
 	fprintf(prtfile,
	  "ERROR: qdslice - error reading input data\n");
	fflush(fdout);
	fclose(fdout);
	fclose(prtfile);
	abort(100);
	exit(0);
}

void write_err()
{
 	fprintf(stderr,
	  "ERROR: qdslice - error writing output data\n");
 	fprintf(prtfile,
	  "ERROR: qdslice - error writing output data\n");
	fclose(fdout);
	fclose(prtfile);
	abort(100);
	exit(0);
}
int get_full_word(source)
char *source;
{
	buffer.value = 0;
	memcpy(buffer.fullword.value,source,SZDFWD);
#if ( BYTE_ORDER == LITTLE_ENDIAN )
	return(CONVERT4(buffer.value));
#else
	return(buffer.value);
#endif
}
float get_real_full_word(source)
char *source;
{
	float tmpval;
#ifdef CRAYSYSTEM
	int ier = 0;
	int zero = 0;
	int one = 1;
	int mode = 2;

/*	IESCTC(source,&one,&tmpval,&one); */
	ier = IEG2CRAY(&mode,&one,source,&zero,&tmpval);
        if (ier != 0) {
          fprintf(stderr,"QDSLICE: WARNING - unable to translate value for time of first sample; 0. assumed");
          tmpval = 0.;
          }

	return(*(float *)&tmpval);
#else
	buffer.value = 0;
	memcpy(buffer.fullword.value,source,SZDFWD);
	return(*(float *)&buffer.value);
#endif
}
int get_half_word(source)
char *source;
{
	int value = 0;
	buffer.value = 0;
#if ( BYTE_ORDER == LITTLE_ENDIAN )
	memcpy(&value,source,SZDHWD);
	buffer.value = CONVERT2(value);
#else
	memcpy(buffer.halfword.value,source,SZDHWD);
#endif
	return(buffer.value);
}
void put_full_word(source,value)
char *source;
int value;
{
#if ( BYTE_ORDER == LITTLE_ENDIAN )
	buffer.value = CONVERT4(value);
#else
	buffer.value = value;
#endif
	memcpy(source,buffer.fullword.value,SZDFWD);
	return;
}
void put_half_word(source,value)
char *source;
int value;
{
#if ( BYTE_ORDER == LITTLE_ENDIAN )
	buffer.value = CONVERT4(value);
#else
	buffer.value = value;
#endif
	memcpy(source,buffer.halfword.value,SZDHWD);
	return;
}
void help()
{
	fprintf(stderr,"Purpose: generate data slice(s) from USP data set\n");
	fprintf(stderr,"Usage: qdslice -N[ntap] -O[otap] -R -s[start_time]");
	fprintf(stderr,"\n\t\t -e[end_time] -i[time_increment] -ns[buffer in samples] [-restart]\n");
	fprintf(stderr,"       qdslice -N[ntap] -O[otap] -ss[begin_sample]");
	fprintf(stderr,"\n\t\t -es[end_sample] -is[increment_sample] -ns[buffer in samples] [-restart]\n\n");
	fprintf(stderr,"Options:\n");
	fprintf(stderr,"  -R  reverse rotation (xyt->txy) (default=txy->xyt)\n");
	fprintf(stderr,"  -s  starting time for slicing (default=0)\n");
	fprintf(stderr,"  -e  ending time for slicing ");
	fprintf(stderr,"(default=trace length)\n");
	fprintf(stderr,"  -i  time increment between slices ");
	fprintf(stderr,"(default=input sample interval)\n");
	fprintf(stderr,"  -ss  starting sample for slicing (default=0)\n");
	fprintf(stderr,"  -es  ending sample for slicing ");
	fprintf(stderr,"(default=trace length)\n");
	fprintf(stderr,"  -is  increment in samples between slices ");
	fprintf(stderr,"(default=1)\n");
	fprintf(stderr,"  -ns  number of samples/trace to buffer on input ");
	fprintf(stderr,"(default=NumSmp)\n");
	fprintf(stderr,"  -restart  restart option; output must be existing");
	fprintf(stderr,"disk dataset\n");
	exit(0);
}
