************************************************************************
*                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
************************************************************************
*  ROUTINE:       ATMEAN                                               *
*  ROUTINE TYPE:  SUBROUTINE                                           *
*  PURPOSE:  Does an Alpha Trim MEAN on a set of data.  Suggested by   *
*            Kurt J. Marfurt.                                          *
*  ENTRY POINTS:                                                       *
*      ATMEAN  (A,DROWS,ROWS,COLS,ALPHA,LOIDX,HIIDX)                   *
*  ARGUMENTS:                                                          *
*      A       REAL     Update  (DROWS,*) - 2-d matrix of amplitudes   *
*      DROWS   INTEGER  Input             - leading declared dimension *
*                                           of A (or 1 for 1-d array)  *
*      ROWS    INTEGER  Input             - number of rows in A; size  *
*                                           of NRMSUM                  *
*      COLS    INTEGER  Input             - number of columns in A     *
*      ALPHA   REAL     Input             - determines how many low/   *
*                                           high values to throw out   *
*                                           (range: 0..1)              *
*      LOIDX   INTEGER  Output            - Low index based on alpha   *
*      HIIDX   INTEGER  Output            - High index based on alpha  *
*       +------------------------------------------------------+       *
*       |               DEVELOPMENT INFORMATION                |       *
*       +------------------------------------------------------+       *
*  AUTHOR:   Kelly D. Crawford                  ORIGIN DATE: 93/05/06  *
*  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 93/05/06  *
*       +------------------------------------------------------+       *
*       |                 EXTERNAL ENVIRONMENT                 |       *
*       +------------------------------------------------------+       *
*  ROUTINES CALLED:                                                    *
*      PART    INTEGER - Partition function                            *
*  INTRINSIC FUNCTIONS CALLED:                                         *
*      REAL    REAL - Converts integer value to a real value           *
*  FILES:                                                              *
*      0  ( OUTPUT SEQUENTIAL ) - Warning message                      *
*  COMMON:           NONE                                              *
*  STOP CODES:       NONE                                              *
*       +------------------------------------------------------+       *
*       |             OTHER DOCUMENTATION DETAILS              |       *
*       +------------------------------------------------------+       *
*  ERROR HANDLING:  Warning message if alpha is too small or large.    *
*  GENERAL DESCRIPTION:  Normally, this operation would have been      *
*       done using a sort.  Sort the values, sum between lo and hi     *
*       indices and divide by (hi-lo)/2.  But we really don't need to  *
*       sort the values.  We can use the partitioning scheme from      *
*       quicksort to make sure that numbers in a list are partitioned  *
*       into high/medium/low bins.  Then we can sum up the medium bin  *
*       for our result.  A full-blown sort is overkill for this        *
*       operation.                                                     *
*  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
*       +------------------------------------------------------+       *
*       |                 ANALYSIS INFORMATION                 |       *
*       +------------------------------------------------------+       *
*  NONSTANDARD FEATURES:   NONE DETECTED                               *
********************   END OF DOCUMENTATION PACKAGE   ******************
      subroutine atmean(a,drows,rows,cols,alpha,loidx,hiidx,lerr)
      integer drows, rows, cols, loidx, hiidx
      real a(drows, *), alpha
 
      integer idx, row
      integer left, rite, loleft, lorite, hileft, hirite
 
      integer part
      external part
 
* What low and high positions are we partitioning around?
      loidx = int(cols * (alpha / 2.0)) + 1
      hiidx = cols - loidx + 1
*      print*,'loidx, hiidx = ', loidx, hiidx

* Quick error check.  loidx and hiidx better contain reasonable values.
      if (loidx .le. 1 .or. hiidx .ge. cols) then
*        The user probably doesn't understand what this routine does.
         loidx = 1
         hiidx = cols
         write(lerr,*) 'Warning -- alpha too small for call to atmean!'
         write(lerr,*) 'Computing simple sum of all elements, with no'
         write(lerr,*) 'trimming being done!  Alpha = ',alpha
         close(lerr)
         call exit(666)
      endif
 
 
* Partially sort each row such that the following is true:
*    A[row,1..loidx-1] < A[row,loidx..hiidx] < A[row,hiidx+1..cols]
      do 100 row = 1, rows
 
         left = 1
         rite = cols
 
* Find a partition between loidx and hiidx.
*         print*,'************** Row ',row,' ***************'
 10      idx = part(a, drows, left, rite, row)
         if (idx+1 .lt. loidx) then
*           Too far to the left...try again
            left = idx + 1
            goto 10
         else if (idx-1 .gt. hiidx) then
*           Too far to the right...try again
            rite = idx - 1
            goto 10
         else
*           In the middle!  Set hi and lo, left and right pointers.
            loleft = left
            lorite = idx - 1
            hileft = idx + 1
            hirite = rite
*            print*,'loleft,loidx,lorite = ',loleft,loidx,lorite
*            print*,'hileft,hiidx,hirite = ',hileft,hiidx,hirite
*            print*,'idx = ',idx
*            print 200, (int(a(row,i)),i=1,cols)
* 200        format(10(2x,I4))
         endif

* Partition around loidx between loleft and lorite.
         if (loleft .lt. loidx .and. loidx .le. lorite) then
 20         idx = part(a, drows, loleft, lorite, row)
*            print*,'loleft,loidx,lorite,idx = ',loleft,loidx,lorite,idx
*            print 200, (int(a(row,i)),i=1,cols)
            if (idx+1 .lt. loidx) then
               loleft = idx + 1
               goto 20
            else if (idx .gt. loidx) then
               lorite = idx - 1
               goto 20
            endif
         endif
 
* Partition around hiidx between hileft and hirite.
         if (hileft .le. hiidx .and. hiidx .lt. hirite) then
 30         idx = part(a, drows, hileft, hirite, row)
*            print*,'hileft,hiidx,hirite,idx = ',hileft,hiidx,hirite,idx
*            print 200, (int(a(row,i)),i=1,cols)
            if (idx .lt. hiidx) then
               hileft = idx + 1
               goto 30
            else if (idx-1 .gt. hiidx) then
               hirite = idx - 1
               goto 30
            endif
         endif
 
 100  continue
      end
************************************************************************
*                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
************************************************************************
*  ROUTINE:       ATMSUM                                               *
*  ROUTINE TYPE:  SUBROUTINE                                           *
*  PURPOSE:  Sums data on a row between a low and high index.  Skips   *
*            0 values.  Returns array of sums, one per row, normalized *
*            to n values.                                              *
*  ENTRY POINTS:                                                       *
*      ATMSUM  (A,SKIP,ROWS,COLS,NRMSUM,LOIDX,HIIDX)                   *
*  ARGUMENTS:                                                          *
*      A       REAL     Update  (*)       - 2-d matrix of amplitudes   *
*                                           (declared as 1-d array)    *
*      SKIP    INTEGER  Input             - leading declared dimension *
*                                           of A (or 1 for 1-d array)  *
*      ROWS    INTEGER  Input             - number of rows in A; size  *
*                                           of NRMSUM                  *
*      COLS    INTEGER  Input             - number of columns in A     *
*      NRMSUM  REAL     Output  (*)       - Array of DROWS elements to *
*                                           hold sums of the rows.     *
*      LOIDX   INTEGER  Output            - Low index based on alpha   *
*      HIIDX   INTEGER  Output            - High index based on alpha  *
*       +------------------------------------------------------+       *
*       |               DEVELOPMENT INFORMATION                |       *
*       +------------------------------------------------------+       *
*  AUTHOR:   Kelly D. Crawford                  ORIGIN DATE: 93/05/06  *
*  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 93/05/06  *
*       +------------------------------------------------------+       *
*       |                 EXTERNAL ENVIRONMENT                 |       *
*       +------------------------------------------------------+       *
*  ROUTINES CALLED:  NONE                                              *
*  INTRINSIC FUNCTIONS CALLED:                                         *
*      REAL    REAL - Converts integer value to a real value           *
*  FILES:  NONE                                                        *
*  COMMON:           NONE                                              *
*  STOP CODES:       NONE                                              *
*       +------------------------------------------------------+       *
*       |             OTHER DOCUMENTATION DETAILS              |       *
*       +------------------------------------------------------+       *
*  ERROR HANDLING:  None needed.                                       *
*  GENERAL DESCRIPTION:  The 2-d array a contains rows of data that    *
*       have been partially ordered.  For any row in a, the data is    *
*       arranged such that a(row,1:loidx-1) < a(row,loidx:hiidx) <     *
*       a(hiidx+1:cols).  We want to sum the data between loidx and    *
*       hiidx, inclusive, with the following exception.  If we find    *
*       any 0 value (dead or muted traces) we will ignore it and grab  *
*       a data value from the area before loidx or after hiidx.  When  *
*       done, the sum will be divided by the total number of items     *
*       summed, and then multiplied by cols to get a normalized sum.   *
*  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
*       +------------------------------------------------------+       *
*       |                 ANALYSIS INFORMATION                 |       *
*       +------------------------------------------------------+       *
*  NONSTANDARD FEATURES:   NONE DETECTED                               *
********************   END OF DOCUMENTATION PACKAGE   ******************
      subroutine atmsum(a, skip, rows, cols, nrmsum, loidx, hiidx)
      integer skip, rows, cols, loidx, hiidx
      real a(*), nrmsum(*)

      integer left, right, lo, hi, row, col
      real norm, s
      logical isleft

      norm = real(cols) / (hiidx - loidx + 1)

      do 20 row = 1, rows
         lo = ((loidx - 1) * skip) + row
         hi = ((hiidx - 1) * skip) + row
         isleft = .true.
         left = lo - skip
         right = hi + skip
         s = 0.0
         do 10 col = lo, hi, skip
            if (a(col) .ne. 0.0) then
               s = s + a(col)
            else if (isleft) then
               s = s + a(left)
               left = left - skip
               isleft = .not.isleft
            else
               s = s + a(right)
               right = right + skip
               isleft = .not.isleft
            endif
 10      continue
         nrmsum(row) = s * norm
 20   continue

      end
************************************************************************
*                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
************************************************************************
*  ROUTINE:       PART                                                 *
*  ROUTINE TYPE:  FUNCTION  INTEGER                                    *
*  PURPOSE:  Median-of-three partitioning routine used in quicksort.   *
*       Partitions an array around a chosen partition element.  Return *
*       value 'part' is an index into the array such that:             *
*          a(row,1..part-1) < a(row,part) <= a(row,part+1..N)          *
*  ENTRY POINTS:                                                       *
*      PART  INTEGER  (A,SKIP,LEFT,RIGHT,ROW)                          *
*  ARGUMENTS:                                                          *
*      A       REAL     Update  (*) - 1-d or 2-d array.  The current   *
*                                     row will be partitioned.         *
*      SKIP    INTEGER  Input       - Distance to skip for a 2-d array *
*                                     (leading dimension for a 2-d     *
*                                     array; 1 for a 1-d array)        *
*      LEFT    INTEGER  Input       - Leftmost position of the array   *
*                                     to partition                     *
*      RIGHT   INTEGER  Input       - Rightmost position of the array  *
*                                     to partition                     *
*      ROW     INTEGER  Input       - Current row being partitioned    *
*                                     (current leading index for 2-d   *
*                                     array; 1 for 1-d array)          *
*       +------------------------------------------------------+       *
*       |               DEVELOPMENT INFORMATION                |       *
*       +------------------------------------------------------+       *
*  AUTHOR:   Kelly D. Crawford                  ORIGIN DATE: 93/05/06  *
*  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 93/05/06  *
*       +------------------------------------------------------+       *
*       |                 EXTERNAL ENVIRONMENT                 |       *
*       +------------------------------------------------------+       *
*  ROUTINES CALLED:  NONE                                              *
*  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
*  FILES:            NONE                                              *
*  COMMON:           NONE                                              *
*  STOP CODES:       NONE                                              *
*       +------------------------------------------------------+       *
*       |             OTHER DOCUMENTATION DETAILS              |       *
*       +------------------------------------------------------+       *
*  ERROR HANDLING:  None needed.  However, the caller must make sure   *
*                   that left < right before calling this function!    *
*  GENERAL DESCRIPTION:  From Sedgewick, "Algorithms".  See below.     *
*  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
*       +------------------------------------------------------+       *
*       |                 ANALYSIS INFORMATION                 |       *
*       +------------------------------------------------------+       *
*  NONSTANDARD FEATURES:   NONE DETECTED                               *
********************   END OF DOCUMENTATION PACKAGE   ******************
************************************************************************
      integer function part(a, skip, left, right, row)
      integer skip, left, right, row
      real a(*)

* We may be passed a row of a 2-d array.  We want to handle this case
* efficiently, as well as handling the simple case of a 1-d array, so
* we use a skip value to convert 2-d addresses into 1-d for this routine.
 
      integer l, r, m, p, i, j
      real t, v

*     Convert column values to 2-d array offsets
      l = ((left  - 1) * skip) + row
      r = ((right - 1) * skip) + row
      m = (((right-left+1)/2) * skip) + row

* Use the median-of-three partitioning.  Select a median of three elements
* and move it to a(r).  The silly looking if block below, believe it or
* not, is the fastest way to find the median of three elements.  Note that
* at most, there will only be three comparisons.
      if (a(l) .ge. a(m)) then
         if (a(m) .ge. a(r)) then
            p = m
         else if (a(l) .ge. a(r)) then
            p = r
         else
            p = l
         endif
      else if (a(m) .ge. a(r)) then
         if (a(l) .lt. a(r)) then
            p = r
         else
            p = l
         endif
      else
         p = m
      endif
      if (r .ne. p) then
         t = a(r)
         a(r) = a(p)
         a(p) = t
      endif

      v = a(r)
      i = l
      j = r - skip

 10   if (i .le. j) then

 20      if (a(j) .ge. v .and. j .ge. l) then
            j = j - skip
            goto 20
         endif

 30      if (a(i) .lt. v) then
            i = i + skip
            goto 30
         endif

         if (i .lt. j) then
            t = a(i)
            a(i) = a(j)
            a(j) = t
            i = i + skip
            j = j - skip
         endif

         goto 10
      endif

      if (i .ne. r) then
         t = a(i)
         a(i) = a(r)
         a(r) = t
      endif

*     a(i) is now in place.  Return i, but convert to column first.
      part = ((i - row) / skip) + 1

      end
