C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C      DOME PETROLEUM
C
c      title: mclev - multi frequency Wiener Levinson algorithm to solve 
C                     complex hermetian system of equations.
c                     matrix a is toeplitz.
C
C      the system of equations this routine will solve is of the form:
C
C      |       a(1)     a(2)   ...   a(N)   | |u(1)|     |rtrans(1)|
C      |  CONJG(a(2))   a(1)   ...   a(N-1) | |  . |     |      .  |
C      |         .                          | |  . |  =  |      .  |
C      |         .                          | |  . |     |      .  |
C      |         .                          | |    |     |         |
C      |  CONJG(a(N))          ...     a(1) | |u(N)|     |rtrans(N)|
C
C
C      note that the main diagonal "a(1)" must be real.
C
C      NOTE THaT THE SYSTEM IS SCaLED BY 1./a(1) BEFORE IT IS SOLVED.
C      (THIS WILL BE NOTICED ON OUTPUT)
C
C
C      INPUT
C
c      n       =  order of system of equations
C
C      INPUT/OUTPUT
C
C      a       =  (n) first row/column of Toeplitz matrix.
C      rtrans  =  (n) rhs of system to solve
C
C      OUTPUT
C
C      u       =  (n) solution vector
C      s       =  (n) solution vector for spike rhs
C
C
C      HISTORY:
C
C
C      VERSION 01       OCTOBER 16, 1987         TONY SaMSOM
C         01) BUILD ORIGINaL
C
C      VERSION 02       OCTOBER 20, 1987         TONY SaMSOM
C         01) DOUBLE PRECISION
C
C      VERSION 03       OCTOBER 26, 1987         TONY SaMSOM
C         01) SaFETY REV
C
c      version 04       March 4, 1992            Kurt J. Marfurt
c         01) replace clev with mclev, a multi frequency complex
c                   Levinson recursion algorithm.
c         02) replace recursive loop over i by nonrecursive loop over ifreq.
c         03) replace inner products over i by outer products over ifreq. 
c         04) transpose matrix arguments to allow vectorization over fastest
c                   moving indices.
c         05) add appropriate temporary work vectors.
C
C      END
c_______________________________________________________________
      subroutine mclev(n,a,rtrans,u,s,ifl,ifh,
     1                 rnorm,v,r,e,ec,temp,rc)             
c - switched from and() to iand() for hp portability - j.m.wade 8/21/92
      integer  iand
c_______________________________________________________________
c     input matrix
c_______________________________________________________________
      complex    a(ifl:ifh,n)
c_______________________________________________________________
c     rhs
c_______________________________________________________________
      complex    rtrans(ifl:ifh,n)
c_______________________________________________________________
c     soln for unit rhs
c_______________________________________________________________
      complex    s(ifl:ifh,n)
c_______________________________________________________________
c     soln for full rhs
c_______________________________________________________________
      complex    u(ifl:ifh,n)
c_______________________________________________________________
c     temporary work arrays for vectorization across solutions.
c_______________________________________________________________
      real       rnorm(ifl:ifh)
      complex    v(ifl:ifh),r(ifl:ifh)
      complex    e(ifl:ifh),ec(ifl:ifh)
      complex    temp(ifl:ifh),rc(ifl:ifh)
c_______________________________________________________________
c     normalize matrix and rhs, zero other vectors
c     since [a] is Hermitian, a(1) and hence rnorm are real.
c_______________________________________________________________
      do 10000 ifreq=ifl,ifh
       rnorm(ifreq)=1./real(a(ifreq,1))
10000 continue
      do 1020 i=1,n
       do 11000 ifreq=ifl,ifh
        a(ifreq,i)=a(ifreq,i)*rnorm(ifreq)
        rtrans(ifreq,i)=rtrans(ifreq,i)*rnorm(ifreq)
        s(ifreq,i)=(0.,0.)
        u(ifreq,i)=(0.,0.)
11000  continue
1020  continue
c_______________________________________________________________
c     initialize recursion
c_______________________________________________________________
      do 20000 ifreq=ifl,ifh
       v(ifreq)=(1.,0.)
       s(ifreq,1)=(1.,0.)
       u(ifreq,1)=rtrans(ifreq,1)
20000 continue
c_______________________________________________________________
c     main recursion loop
c_______________________________________________________________
      do 99000 j=2,n
c_______________________________________________________________
c      error term for spike rhs
c_______________________________________________________________
       do 30000 ifreq=ifl,ifh
        e(ifreq)=(0.,0.)
30000  continue
       do 2020 i=2,j
        do 31000 ifreq=ifl,ifh
         e(ifreq)=e(ifreq)+conjg(a(ifreq,i))*s(ifreq,j-i+1)
31000   continue
2020   continue
c_______________________________________________________________
c      scale factor and new spike value
c_______________________________________________________________
       do 40000 ifreq=ifl,ifh
        r(ifreq)=e(ifreq)/conjg(v(ifreq))
40000  continue
       do 50000 ifreq=ifl,ifh
        v(ifreq)=v(ifreq)-r(ifreq)*conjg(e(ifreq))
50000  continue
c_______________________________________________________________
c      new solution vector for spike
c_______________________________________________________________
       do 2040 i=1,j/2
        do 60000 ifreq=ifl,ifh
         temp(ifreq)=s(ifreq,i)
         s(ifreq,i)=s(ifreq,i)-r(ifreq)*conjg(s(ifreq,j-i+1))
60000   continue
        do 70000 ifreq=ifl,ifh
         s(ifreq,j-i+1)=s(ifreq,j-i+1)-r(ifreq)*conjg(temp(ifreq))
70000   continue
2040   continue
c - switched from and() to iand() for hp portability - j.m.wade 8/21/92
       if(iand(j,1).ne.0) then
          do 80000 ifreq=ifl,ifh
           s(ifreq,j/2+1)=
     1         s(ifreq,j/2+1)-r(ifreq)*conjg(s(ifreq,j/2+1))
80000     continue
       endif
c_______________________________________________________________
c      error term for full rhs
c_______________________________________________________________
       do 81000 ifreq=ifl,ifh
        ec(ifreq)=(0.,0.)
81000  continue
       do 2120 i=2,j
        do 82000 ifreq=ifl,ifh
         ec(ifreq)=ec(ifreq)+conjg(a(ifreq,i))*u(ifreq,j-i+1)
82000   continue
2120   continue
c_______________________________________________________________
C      scale factor
c_______________________________________________________________
       do 83000 ifreq=ifl,ifh
        rc(ifreq)=(ec(ifreq)-rtrans(ifreq,j))/conjg(v(ifreq))
83000  continue
c_______________________________________________________________
c      new solution vector for spike
c_______________________________________________________________
       do 2140 i=1,j
        do 84000 ifreq=ifl,ifh
         u(ifreq,i)=u(ifreq,i)-rc(ifreq)*conjg(s(ifreq,j-i+1))
84000   continue
2140   continue
C
99000 continue
C
       return
       end
