C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c spacetrp reads seismic trace data from an input file,
c either interpolates or decimates the data to arbitrary sample interval
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
 
      integer     itr ( SZLNHD )
      integer     lhed( SZLNHD )
      real        head( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform,obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      real        sii,sio,xoff,sgn
      integer     nsampo
#include <f77/pid.h>
      integer     recnum, static
      real        tri ( 2*SZLNHD )
      real        tabl1 (SZLNHD), tabl2(SZLNHD), zz(4*SZLNHD)
      integer     iz(SZLNHD)
      real        tablh1 (SZLNHD), tablh2(SZLNHD), zzh(4*SZLNHD)
      integer     izh(SZLNHD)

c------
c  dynamic memory allocation for big arrays, eg whole records

      real        bigar1
      pointer     (wkadr1, bigar1(1))
      integer     itrhdr
      pointer     (wkadri, itrhdr(1))
c------


      character   ntap * 256, otap * 256, name*8
      logical     verbos, query, heap1, single
      integer     argis
 
c     equivalence ( itr(129), tri (1) )
      equivalence ( itr(  1), lhed(1), head(1) )
      data lbytes / 0 /, nbytes / 0 /, name/'SPACETRP'/
      data single/.false./
 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,sii,sio,xoff,sgn,live,verbos)
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'spacetrp: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      single = .false.
      nrecc = nrec
      if (ntrc .eq. 1 .AND. nrec .gt. 1) then
          ntrc  = ntrc * nrec
          nrecc = 1
          single = .true.
          write(LER,*)'WARNING from spacetrp:'
          write(LER,*)'input is composed of 1 trace records'
          write(LER,*)'will try to read data in as a single ',ntrc
          write(LER,*)'trace record - may not be able to hold this'
          write(LER,*)'in memory though.  If not try running on a'
          write(LER,*)'large memory server'
      elseif 
     1    (ntrc .eq. 1 .AND. nrec .eq. 1) then
          write(LER,*)'FATAL error in spacetrp:'
          write(LER,*)'input has only a single trace and so no'
          write(LER,*)'spatial interpolation is possible'
          stop
      endif

c     For trace header values we take mnemonics and build a
c     set of pointers to an I*2 array equivalenced to the
c     RTAPE  Integer array (headers + data)
c     TRACEHEADER is a value in the include file <sisdef.h> that
c     refers to the trace header
 
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)


      call hlhprt (itr, lbytes, name, 8, LERR)

      ntrco = nint (ntrc * sii / sio)

      write(LERR,*)'Number output traces= ',ntrco

c-----
c     modify line header to reflect actual number of traces output
c-----
      call savew(itr, 'NumTrc', ntrco , LINHED)
      call savew(itr, 'NumRec', nrecc , LINHED)

      obytes = SZTRHD + nsamp * SZSMPD
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-----
c     build interpolation tables
c-----

      do  300  j = 1, SZLNHD
         tabl1(j) = float( j ) * sii
300   continue

      do  301  j = 1, SZLNHD
         tabl2(j) = float( j ) * sio
301   continue

      do  302  j = 1, SZLNHD
          tablh1(j) =  float( j ) * sii
302   continue
      do  303  j = 1, SZLNHD
          tablh2(j) =  float( j ) * sio
303   continue

      sr = sio / sii
      write(LERR,*)'sio/sii = ',sr

      write(LERR,*)' '
      write(LERR,*)'Table 1:'
      write(LERR,*)(tabl1(j),j=1,ntrc)
      write(LERR,*)' '
      write(LERR,*)'Table 2:'
      write(LERR,*)(tabl2(j),j=1,ntrco)
      write(LERR,*)' '
c---------------------------------------------------
c  malloc only space we're going to use
      heap1 = .true.
 
c--------------------------
c  note: these don't
c  have to be the same size
 
      ntrcm = max(ntrc,ntrco)
      item1 = ntrcm  * nsamp  * SZSMPD
      itemi = ntrcm  * ITRWRD * SZSMPD
 
c  note also SZSMPD is the native
c  size of a float or int in bytes
c--------------------------
 
c--------
c  galloc - general allocation (machine independent since it uses C
c  malloc internally
c  inputs to galloc are pointer, number of bytes to allocate
c  outputs are error codes:
c     errcod = 1  (allocation succeeded)
c     errcod = 0  (allocation failed)
c--------
 
      call galloc (wkadr1, item1, errcd1, abort1)
      if (errcd1 .ne. 0.) heap1 = .false.

      call galloc (wkadri, itemi, errcdi, aborti)
      if (errcdi .ne. 0.) heap1 = .false.
 
      if (.not. heap1) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) item1,'  bytes'
         write(LERR,*) itemi,'  bytes'
         write(LERR,*)' '
      endif

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  nsampo,sii,sio,ntap,otap)
      end if

c-----
c     BEGIN PROCESSING
c     read trace, interpolate or decimate, write to output file
c-----
                     icinit = 1
c-----
c     process desired trace records
c-----
      do 1000 jj = 1, nrec
 
 
            call vclr (bigar1, 1,  ntrcm*nsamp)

            livtrc = 0
            do 1001 kk = 1, ntrc
                  nbytes = 0
                  call rtape( luin, itr, nbytes)
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (lhed(ITHWP1), 1, tri, 1, nsamp)
                  call saver2(lhed,ifmt_RecNum,l_RecNum, ln_RecNum,
     1                        recnum , TRACEHEADER)
                  call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        static , TRACEHEADER)

                  IF(static .eq. 30000) then
                    call vclr (tri, 1, nsamp)
                  ELSE
                    livtrc = livtrc + 1
                  ENDIF

                  istrc = (kk-1) * nsamp
                  call vmov (tri, 1, bigar1(istrc+1), 1, nsamp)
                  ishdr = (kk-1) * ITRWRD
                  call vmov (lhed, 1, itrhdr(ishdr+1),1,ITRWRD)

 1001             continue

c-----------
c do spatial 
c interpolation

            IF      (sr  .lt. 1.0) THEN
               call spctrp (ntrc, ntrco, ntrcm, nsamp, bigar1,
     1                      tabl1, tabl2, tablh1, tablh2, itrhdr,
     2                      zz, iz, zzh, izh,
     3                      icinit, sii, sio,l_StaCor,livtrc)
            ELSEIF (sr .gt. 1.0) THEN
               call untrp  (ntrc, ntrco, ntrcm, nsamp, bigar1,
     1                      tabl1, tabl2, tablh1, tablh2, itrhdr,
     2                      zz, iz, zzh, izh,
     3                      icinit, sr,l_StaCor)
            ENDIF
 
            il = 0
            do 1002 kk = 1, ntrco

               istrc = (kk-1) * nsamp
               call vmov (bigar1(istrc+1), 1, lhed(ITHWP1), 1, nsamp)
               ishdr = (kk-1) * ITRWRD
               call vmov (itrhdr(ishdr+1),1,lhed, 1,ITRWRD)
               call saver2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                     static , TRACEHEADER)
               if (static .ne. 30000) il = il + 1
               if (live .gt. 0) then
                  if (static .eq. 30000 .AND. il .le. live) then
                     call savew2(lhed,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                           0 , TRACEHEADER)
                     call vclr (lhed(ITHWP1), 1, nsamp)
                     il = il + 1
                  endif
               endif
               call savew2(lhed,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                     kk     , TRACEHEADER)

               call wrtape (luout, itr, obytes)

 1002       continue
 
            if(verbos)write(LER,*)'spacetrp: ri ',recnum,' processed'
 1000       continue
c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of spacetrp, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'spacetrp either interpolates or decimates seismic data'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute spacetrp by typing spacetrp and the program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
        write(LER,*) ' '
        write(LER,*)
     :' -i[sii] (default = input) :  input group interval'
        write(LER,*)
     :' -o[sio] (default = input) :  output group interval'
        write(LER,*)
     :' -l[live] (default = none) :  force # live traces output'
        write(LER,*) ' '
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*) ' '
        write(LER,*)
     :'usage:   spacetrp -N[] -O[] -i[] -o[] -l[] -V'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,sii,sio,xoff,sgn,live,verbos)
c-----
c     get command arguments
c
c     ntap  - C*100     input file name
c     otap  - C*100     output file name
c     sii   - R*4       input sample interval override
c     sio   - R*4       output sample interval
c     verbos  L         verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      real        sii, sio, xoff, sgn
      logical     verbos, far
      integer     argis
 
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argr4( '-i', sii, 0., 0. )
            call argr4( '-o', sio, 0., 0. )
            call argr4( '-x', xoff, 0., 0. )
            call argi4( '-l', live, 0, 0 )
            far    = (argis('-F') .gt. 0)
            verbos = (argis('-V') .gt. 0)

            if (far) then
               sgn = -1.
            else
               sgn = 1.
            endif

            if (sii .eq. 0.) then
               write(LERR,*)'Must supply input group interval -- FATAL'
               write(LERR,*)'Rerun using -i[] cmd line arg'
               stop
            endif
            if (sio .eq. 0.) then
               write(LERR,*)'Must supply output group interval -- FATAL'
               write(LERR,*)'Rerun using -o[] cmd line arg'
               stop
            endif
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  nsampo,sii,sio,ntap,otap)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c    nsampo - I*4 output number of samples in trace
c     sii   - R*4       input sample interval override
c     sio   - R*4  output sample interval
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c     ntap  - C*120     input file name
c     otap  - C*120     output file name
c-----
#include <f77/iounit.h>
      integer     nsamp, nsi, ntrc, nrec
      real        sio, sii
      character   ntap*(*), otap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' output samples/trace =  ', nsampo
            write(LERR,*) ' input sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' input sample interval   = ',sii
            write(LERR,*) ' output sample interval   = ',sio
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
