C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine tr2ti( rt, rv, nn, tape, SlopeAdjustmentFactor, vflag,
     :     Force)

c tr2ti converts an input RMS velocity function to INTERVAL velocity 
c using Dix Equation. 
c
c it is assumed that:
c
c  rt() = 2way TIME in seconds
c  rv() = RMS velocity in units/second
c
c on output rv() = INTERVAL velocity in units/second
c velocity is referenced to the bottom of the interval

#include <f77/iounit.h>
#include <f77/lhdrsz.h>

      integer nn, i, NewNumberOfPoints 

      real rt(*), rv(*), vint(SZLNHD), Numerator, SlopeAdjustmentFactor
      real IntervalTime(SZLNHD), temprv(SZLNHD), temprt(SZLNHD), Slope
      real NewSlope

      character switch*3

      logical tape, vflag, Force

c initialize memory
      
      call vclr( vint, 1, SZLNHD )
      call vclr( IntervalTime, 1, SZLNHD )
      call vclr( temprv, 1, SZLNHD )
      call vclr( temprt, 1, SZLNHD )

c assign first interval velocity

      vint(1) = rv(1)

c if TAPE then scan function for slope breaks greater than 
c         SlopeAdjustmentFactor * Slope
c glean rv() values at these locations only. 

      IF( tape .and. .not. vflag ) then
         switch = 'off'
         Slope = ( rv(2) - rv(1) ) / ( rt(2) - rt(1) )
         NewNumberOfPoints = 0

         DO i = 3, nn
            NewSlope = ( rv(i) - rv(i-1) ) / ( rt(i) - rt(i-1) )
 
            if ( ( abs( NewSlope - Slope ) ) .gt.
     :           ( SlopeAdjustmentFactor * abs( Slope ) ) ) then

               if ( switch .ne. 'on' ) then
                  switch = 'on'
                  NewNumberOfPoints = NewNumberOfPoints + 1
                  temprt(NewNumberOfPoints) = rt(i)
                  temprv(NewNumberOfPoints) = rv(i)
                  Slope = NewSlope
               else
                  switch = 'off'
                  Slope = NewSlope
               endif
            endif
         ENDDO

        do i=1,NewNumberOfPoints
           rt(i) = temprt(i)
           rv(i) = temprv(i)
        enddo

c if the max time is already used don't use it again, otherwise put 
c the max time and max vel at the end of the function

        if ( ( temprt(NewNumberOfPoints) - rt(nn) ) .gt. 1.e-30 ) then
           NewNumberOfPoints=NewNumberOfPoints+1
           rv(NewNumberOfPoints) = rv(nn)
           rt(NewNumberOfPoints) = rt(nn)
        endif

        nn = NewNumberOfPoints
      ENDIF

c assign IntervalTime[1]

      IntervalTime(1) = rt(1) / 2.

      DO i = 2, nn
         IntervalTime(i) = ( rt(i) - rt(i-1) ) / 2.

c form numerator

         Numerator = ( rv(i)**2 * rt(i) / 2. ) - 
     :        ( rv(i-1)**2 * rt(i-1) / 2. )

c POLICEMAN: check to see if numerator is negative (ie. rms velocity inversion)
c            if so, then stop and warn user of this problem and suggest
c            options......use interval velocities....supply maxtime rms velocity
c            use -force on command line and hope for the best.

         if ( Numerator .lt. 0. .and. .not. Force ) then
            write(LERR,*)' '
            write(LERR,*)'VELIN: You have rms velocity inversions '
            write(LERR,*)'in your input RMS velocity function.'
            write(LERR,*)'Since you have not supplied a function value'
            write(LERR,*)'at MAX TIME this routine is trying to make'
            write(LERR,*)'one by determining the last INTERVAL '
            write(LERR,*)'velocity and using that to the end of your'
            write(LERR,*)'input data.  Unfortunately this involves a'
            write(LERR,*)'square root calculation in Dix equation.'
            write(LERR,*)'In your case this would involve a negative'
            write(LERR,*)'number in the square root which is BAD.'
            write(LERR,*)' '
            write(LERR,*)'A solution to this problem is to supply an'
            write(LERR,*)'RMS velocity for the end of your dataset.'
            write(LERR,*)'Alternately, supply INTERVAL velocity'
            write(LERR,*)'to this routine and ask for RMS velocity as'
            write(LERR,*)'output. '
            write(LERR,*)' '
            write(LERR,*)'You may also flag the command option -force'
            write(LERR,*)'which will cause the interval velocity above'
            write(LERR,*)'to be used when a velocity inversion is '
            write(LERR,*)'detected.  This option is for the expert user'
            write(LERR,*)'only who is very cognisant of the disastor '
            write(LERR,*)'that may result from the use of this option.'
            write(LERR,*)' '
            write(LERR,*)'FATAL'
            stop
         elseif ( Numerator .lt. 0. .and. Force ) then
            vint(i) = vint(i-1)
         else
            vint(i) = sqrt ( Numerator / IntervalTime(i) )
         endif

c calculate vint

      ENDDO

c handle zero TIME entry, since interval velocity is referenced to the
c base of the interval the first entry [if zero] must contain the 
c same velocity as at the base of the first interval.

      IF ( rt(1) .gt. 1.0e-20 ) then

c there is no zero time entry, must manufacture one
c bump rt() values up one slot to make room for zero entry

         do i = nn + 1, 2, -1
            rt(i) = rt(i-1)
         enddo

c assign zero TIME rt() value

         rt(1) = 0.

c load INTERVAL velocities to rv()

         call vmov( vint, 1, rv(2), 1, nn )

c assign zero TIME rv[1] value

         rv(1) = rv(2)

c increase function count by one

         nn = nn + 1

      ELSE

c load INTERVAL velocities to rv()

         call vmov( vint, 1, rv, 1, nn )

      ENDIF   
         
      return
      end

