C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c velnotch reads seismic record data from an input file,
c performs pont source dghosting on the input record and
c writes the results to an output file
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
      integer itr ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne

c------
c  static memory allocation
c     complex     data(1024*1024),cop(1024*1024)
c------
c  dynamic memory allocation for big arrays, eg whole records
      complex     data, cop
      pointer     (datadr, data(1))
      pointer     (copadr, cop (1))
      integer     itrhdr
      pointer     (iaddrh, itrhdr(1))
c------
      complex      czero

c------
c    this is necessary for buliding printout files
#include <f77/pid.h>
c------
      integer     recnum, trcnum, static, mutes(SZSMPM)
      real        tri ( SZLNHD ), tpi(SZSMPM), tps(SZSMPM)
      character   ntap * 255, otap * 255, name*80
      logical     verbos, query, heap
      integer     argis
 
      data lbytes / 0 /, nbytes / 0 /, name/'VELNOTCH'/
 
      czero = cmplx (0.,0.)

c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 .or. argis( '-h' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c     this consists of a 2-character identifier, the process id number,
c     and the child process id number:
c     XY.nnnnn.mmmmm
c     this should be unique even for multiple occurences of the same
c     process in a pipeline
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,ns,ne,irs,ire,
     1            fh, vel, pct,thr,ipow,dx,verbos)

c-----
c     get logical unit numbers for input and output of seismic data
c     0 = default stdin
c     1 = default stdout
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'velnotch: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif

c------
c     to get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('TrcNum',ifmt,l_TrcNum,length,TRACEHEADER)
      call savelu('RecNum',ifmt,l_RecNum,length,TRACEHEADER)
      call savelu('SrcLoc',ifmt,l_SrcLoc,length,TRACEHEADER)
      call savelu('RecInd',ifmt,l_RecInd,length,TRACEHEADER)
      call savelu('DphInd',ifmt,l_DphInd,length,TRACEHEADER)
      call savelu('DstSgn',ifmt,l_DstSgn,length,TRACEHEADER)
      call savelu('StaCor',ifmt,l_StaCor,length,TRACEHEADER)


      call hlhprt (itr, lbytes, name, 80, LERR)

      write(LERR,*)' '
      write(LERR,*)'Global Line Parameters'
      write(LERR,*)' '
      write(LERR,*)'Number samples/trace        = ',nsamp
      write(LERR,*)'Sample interval (ms)        = ',nsi
      write(LERR,*)'Number traces/record        = ',ntrc
      write(LERR,*)'Number records/line         = ',nrec
      write(LERR,*)' '

c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records)
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary

      dt = real (nsi) * unitsc

c--------------------------------------------------
c-----
c     initialize dimensions, etc
c-----
      call dginit (dt,dx,vel,ntrc,nsamp,
     1             nsmpe,ntrce,ntro,d4,t4)


c---------------------------------------------------
c  malloc only space we're going to use
      heap = .true.

c--------------------------
c  note: these don't
c  have to be the same size

      itemh = ntrc * ITRWRD
      itemd = 2 * nsmpe * ntrce
      itemc = 2 * nsmpe * ntrce
      maxdim = max(ntrce,nsmpe)
      itemw = 2 * maxdim
      write(LERR,*)'items: ',itemd,itemc,itemw

c  note also SZSMPD is the 
c  size of an item in bytes
c--------------------------

      call galloc (iaddrh, itemh*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (datadr, itemd*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.
      call galloc (copadr, itemc*SZSMPD, errcd, abort)
      if (errcd .ne. 0.) heap = .false.

      if (.not. heap) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemh*SZSMPD,'  bytes'
         write(LERR,*) itemd*SZSMPD,'  bytes'
         write(LERR,*) itemc*SZSMPD,'  bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemh*SZSMPD,'  bytes'
         write(LERR,*) itemd*SZSMPD,'  bytes'
         write(LERR,*) itemc*SZSMPD,'  bytes'
         write(LERR,*)' '
      endif
c---------------------------------------------------


c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD
c----------------------
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  vel,pct,thr,ipow,dx,ntap,otap)
      end if

c-----
c     initialize dimensions, etc
c-----
      call cvfill (czero, cop, 2, nsmpe*ntrce)
      call tapers (dt,dx,vel,pct,thr,ipow,ntrc,nsamp,
     1             nsmpe,ntrce,ntro,d4,t4,
     2             fh,fac,tpi,tps,cop)

      fac = 1./fac


c-----
c     BEGIN PROCESSING
c     read trace, do terrible things to data, write to output file
c-----
c-----
c     pass unwanted records
c-----
      nbytes = obytes
      call recrw (1,irs-1,luin,ntrc,itr,luout, nbytes)
      if (nbytes  .eq. 0) go to 999

c-----
c     process desired trace records
c-----
      DO 1000 jj = irs, ire
 
            ic = 0
            call vclr (data,1, ntrce*nsmpe)
            do 1001  kk = 1, ntrc

                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
                  call detmut (tri, mutes(kk), nsamp)

c------
c     for trace header values you can use the I*2 vector
c     rather than saver/savew (which you can use)
c     this is faster in general than saver/w
c------
                  dist   = iabs (itr(l_DstSgn))
                  static = itr(l_StaCor)
                  recnum = itr(l_RecNum)
                  trcnum = itr(l_TrcNum)

                  if (static .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif

c----------------------
c  time/space taper &
c  pack data into array
                  ic = ic + 1
                  istrc = (ic-1+ntro) * nsmpe
                  ishdr = (ic-1) * ITRWRD
                  call vmov   (itr,1, itrhdr(ishdr+1),1,ITRWRD)
                  call vmul   (tri,1, tpi,1, tri,1, nsamp)
                  call vsmul  (tri,1, tps(kk), tri,1, nsamp)
                  call cvreal (tri,1, data(istrc+1),2, nsamp)

1001        continue

c-----------------------
c  here's the meat...
c  2-d fft
c  complex multiply deghost
c  operator with data

                  call cfft2d (data, nsmpe, ntrce, +1)
                  call cvmul  (data, 2, cop, 2, data, 2, ntrce*nsmpe, 1)
                  call cvsmul (data, 2, fac, data, 2, ntrce*nsmpe)
                  call cfft2d (data, nsmpe, ntrce, -1)

c-----------------------

c---------------------
c  extract traces from
c  output array and
c  write output data
            do 1002 kk = 1, ntrc

                  ishdr = (kk-1) * ITRWRD
                  istrc = (kk-1+ntro) * nsmpe
                  call vmov (data(istrc+1),2, tri, 1, nsamp)
                  call resmut (tri, mutes(kk), nsamp)
                  call vmov (tri, 1, itr(ITHWP1),1, nsamp)
                  call vmov (itrhdr(ishdr+1),1,itr,1,ITRWRD)

                  call wrtape (luout, itr, obytes)


 1002             continue
 
                  if(verbos)write(LERR,*)'processed rec ',recnum
 
1000        CONTINUE

c------------------------
c  pass remainder of recs
      nbytes = obytes
      call recrw (ire+1, nrec, luin, ntrc, itr, luout, nbytes)
      if (nbytes .eq. 0) go to 999

c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of velnotch, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      stop
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'velnotch removes the ghosting effect caused by src/rcvr depth'
        write(LER,*)
     :'see manual pages for details ( online by typing uman prg )'
        write(LER,*)' '
        write(LER,*)
     :'execute by typing velnotch and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)         : output data file name'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -re[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -v[vel] (default = none) :  source/receiver velocity (ft,m/s)'
        write(LER,*)
     :' -dx[dx] (default = none) :  input trace spacing (ft,m)'
        write(LER,*)
     :' -t[pct] (default = 2.5%) :  time/space taper'
        write(LER,*)
     :' -w[thr] (default = 10%)  :  wavenumber notch 1/2-width'
        write(LER,*)
     :' -p[ipow] (default = 4)   :  notch power factor (depth)'
        write(LER,*) ' '
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   velnotch -N[ntap] -O[otap] -rs[ns] -re[ne] -v[vel] '
        write(LER,*)
     :'                  -h[zd] -dx[dx] -t[pct] -w[] -p[] -V'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,
     1                  fh,vel,pct,thr,ipow,dx,verbos)
c-----
c     get command arguments
c
c     ntap  - C*255    input file name
c     otap  - C*255    output file name
c     vel   - R*4      src/rcvr velocity
c     pct   - R*4      taper %
c     dx    - R*4      grp int
c     fh    - R*4      hi-cut freq
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*)
      integer     ns, ne, irs, ire, ipow
      real        vel, thr, pct, dx, fh
      logical     verbos
      integer     argis
 
c-------
c     see manual pages on the argument handler routines
c     for the meanings of these functions
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argr4 ( '-v', vel, 0.0, 0.0)
            call argr4 ( '-dx', dx, 0.0, 0.0)
            call argr4 ( '-t', pct, 2.5, 2.5)
            call argr4 ( '-w', thr, 10., 10.)
            call argi4 ( '-p', ipow, 4, 4)
            verbos =   (argis('-V') .gt. 0)

      ns = 0
      ne = 0

      if (pct .gt. 50.0) then
         write(LERR,*)'Cannot have % taper > 50: will reset to 50'
         pct = 50.
      endif
      if (vel .eq. 0.0) then
         write(LERR,*)'Must enter surface velocity -- FATAL'
         stop
      endif
      if (dx .eq. 0.0) then
         write(LERR,*)'Must enter trace spacing ( -dx[] )'
         stop 666
      endif
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  fh,vel,pct,thr,ipow,dx,ntap,otap)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4     number of samples in trace
c     vel   - R*4      src/rcvr velocity
c     pct   - R*4      taper %
c     dx    - R*4      grp int
c     fh    - R*4      hi-cut freq
c     nsi   - I*4     sample interval in ms
c     ntrc  - I*4     traces per record
c     nrec  - I*4     number of records per line
c     iform - I*4     format of data
c     ntap  - C*255   input file name
c     otap  - C*255   output file name
c-----
#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec,ipow
      real        vel,dx,pct,thr
      character   ntap*(*), otap*(*)
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' notching velocity  = ',vel
            write(LERR,*) ' trace spacing      = ',dx
            write(LERR,*) ' % time/space taper = ',pct
            write(LERR,*) ' % notch width      = ',thr
            write(LERR,*) ' notch power        = ',ipow
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
