C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c     program module vred
c
c**********************************************************************c
c
c	vred takes a trace F(t) at a distance D, and forms a new
c	time series F(t -t0 -D/vred).  Zero filling is done based on
c       time -t0 on the command line.
c-----
c     declare variables
c-----
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      parameter   (nttaper=5)
      parameter   (pi=3.1415926)

      integer     itr ( 4 * SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, nsampo
      integer     luin, luout,lbytes,nbytes,lbyout,obytes
      integer     irs,ire,ns,ne,li,di
      integer     argis
      integer     ordfft

      real        dist
      real        tri ( 2*SZLNHD ), work( 2*SZLNHD )
      real        omega(2*SZLNHD )

      complex     expphi(2*SZLNHD) 

      character   ntap * 255, otap * 255, name * 4
      character   vtap * 255, stawrd * 6

      logical     verbos, query, undo, threed, tape

      external	  rshift

      data lbytes / 0 /
      data nbytes / 0 /
      data undo/.false./
      data name/'VRED'/

c-----
c
c     read program parameters from command line card image file
c
c-----
      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0 )
      if ( query )then
            call help()
            stop
      endif

c-----
c     open printout file
c-----
#include <f77/open.h>

      call gcmdln(ntap,otap,ns,ne,irs,ire,vred,it0,dmul,undo,
     1            verbos,npadend,stawrd,unit,tmul,vtap,threed,
     2            tape)
c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     if we have a velocity tape open it & read line header
c-----
      if (tape) then
         call getln(luvel, vtap, 'r', -1)
         if (luvel .lt. 0) then
            write(LERR,*)'ERROR in vred:'
            write(LERR,*)'Unable to open velocity tape ',vtap
            write(LER ,*)'ERROR in vred:'
            write(LER ,*)'Unable to open velocity tape ',vtap
            stop
         endif
         call rtape  ( luvel, itr, lbytes)
         if(lbytes .eq. 0) then
            write(LERR,*)'VRED: no header read from unit ',luvel
            write(LERR,*)'FATAL'
            stop
         endif
         call saver(itr, 'NumSmp', nsampv, LINHED)
         call saver(itr, 'SmpInt', nsiv  , LINHED)
         call saver(itr, 'NumTrc', ntrcv , LINHED)
         call saver(itr, 'NumRec', nrecv , LINHED)
         call saver(itr, 'Format', iform , LINHED)
         nvels = ntrcv * nrecv
      endif
c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LERR,*)'VRED: no header read from unit ',luin
         write(LERR,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      if (tape .AND. .not.threed) then
         if (nvels .ne. nrec) then
          write(LERR,*)' '
          write(LERR,*)'WARNING from vred:'
          write(LERR,*)'Number of velocities not equal to # input recs'
          write(LERR,*)' '
         endif
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('LinInd',ifmt_LinInd,l_LinInd,ln_LinInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

      if (stawrd(1:1) .ne. ' ') then
         call savelu(stawrd,ifmt_stawrd,l_stawrd,ln_stawrd,TRACEHEADER)
      endif

      call hlhprt (itr, lbytes, name, 4, LERR)
c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)
c-----
c     worry about undoing global padding from previous vred
c-----
      ishift = nint(float(it0)/nsi)
      iadd   = nint(float(npadend)/nsi)
      if (ishift .lt. 0) ishift = 1
      if(.not. undo) then
         nsampo = nsamp + ishift +iadd
      else
         nsampo = nsamp - ishift -iadd
      endif
      if(nsampo .lt.1)nsampo = 1
c      if(nsampo .gt. 2*SZLNHD) then
c        nsampo = 2*SZLNHD
c        write(LERR,*)'Output samples exceed twice input samples'
c        write(LERR,*)'will force output samples to be ',nsampo
c      endif
      if(undo) then
         k2 = ordfft (nsamp)
      else
         k2 = ordfft (nsampo)
      endif
      ntnew=2**k2
      domega=2.*pi/ntnew
      do 10000 iomega=1,ntnew/2
       omega(iomega)=(iomega-1)*domega
10000 continue


         write(LERR,*)' '
         write(LERR,*)' nsamp    = ',nsamp
         write(LERR,*)' nsampo   = ',nsampo
         write(LERR,*)' ntnew    = ',ntnew 
         write(LERR,*)' '

c-----
c     modify line header to reflect actual number of traces output
c-----
      obytes = SZTRHD + SZSMPD*nsampo
      call savew(itr, 'NumSmp', nsampo   , LINHED)
      call savew(itr, 'NumRec', ire-irs+1, LINHED)
      call savew(itr, 'NumTrc', ne-ns+1  , LINHED)
      call savhlh(itr,lbytes,lbyout)
      call wrtape ( luout, itr, lbyout                 )
c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
            call verbal(nsamp, nsi, ntrc, nrec, iform, vred, it0, dmul,
     1            undo,ntap,otap,npadend,stawrd,unit,tmul,
     2            vtap,threed,tape)
c-----
c     convert velocity to m/ms or ft/ms
c-----
      dt = nsi * unitsc
      dt = dt * tmul
      vred = vred * dt
c-----
c     BEGIN PROCESSING
c     read trace, do constant velocity moveout, write to output file
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)
c-----
c     process desired trace records
c-----

c++++++++
c       2D or constant velocity section
c++++++++

      IF (.not. threed) THEN

         DO   jj = irs, ire

               call trcskp(jj,1,ns-1,luin,ntrc,itr)

               if (tape) then
                  nbytes = 0
                  call rtape ( luvel, itr, nbytes)
                  if(nbytes .eq. 0) then
                    write(LERR,*)'End of file on velocity tape, rec = ',
     1              jj
                    write(LERR,*)'Will use last available velocity = ',
     1              vred
                    tape = .false.
                  endif
                  call vmov (itr(ITHWP1), 1, vred, 1, 1)
                  vred = vred * dt
               endif

               do  kk = ns, ne

                   nbytes = 0
                   call rtape ( luin, itr, nbytes)
                   if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input rec/trc ',jj,kk
                     go to 999
                   endif
                   call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

                   call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                         istatic, TRACEHEADER)

                   IF (istatic .ne. 30000) THEN
c-----
c     get absolute trace distance
c     compute shift in samples
c-----
                      call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                            idist  , TRACEHEADER)

                      dist = float( iabs(idist) ) * dmul
                      call vapply 
     1                  (itr, tri, work, omega, expphi, vred, dist,
     2                   nsamp, nsampo, ntnew, dmul, unit, ishft,
     3                   undo, stawrd, nsi, ishift,
     4                   ifmt_RecNum,l_RecNum, ln_RecNum,
     5                   ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     6                   ifmt_DphInd,l_DphInd, ln_DphInd,
     7                   ifmt_LinInd,l_LinInd, ln_LinInd,
     8                   ifmt_stawrd,l_stawrd,ln_stawrd) 

                   ELSE

                      call vclr (tri, 1, nsampo)

                   ENDIF

                   call vmov  (tri, 1, itr(ITHWP1), 1, nsampo)
                   call wrtape( luout, itr, obytes)
                   if(verbos)write(LERR,*)'rec= ',jj,' trc= ',kk,
     1                 ' dist= ',dist,' shift= ',ishft
   
               enddo

               call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)

         ENDDO

c++++++++
c       3D velocity section
c++++++++
      ELSE

         DO   jj = irs, ire
 
               do  kk = ns, ne
 
                   nbytes = 0
                   call rtape ( luin, itr, nbytes)
                   if(nbytes .eq. 0) then
                    write(LERR,*)'End of file on input rec/trc ',jj,kk
                    go to 999
                   endif
                   call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
 
                   call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                         istatic, TRACEHEADER)
 
                   IF (istatic .ne. 30000) THEN
c-----
c     for each live input trace extract velocity corresponding to trace
c     bin
c-----
                      call saver2  (itr,ifmt_DphInd,l_DphInd, ln_DphInd,
     1                              di, TRACEHEADER)
                      call saver2  (itr,ifmt_LinInd,l_LinInd, ln_LinInd,
     1                              li, TRACEHEADER)
                      call sisseek (luvel,(li-1)*ntrcv + di)
                      call rtape ( luvel, itr, nbytes)
                      if(nbytes .eq. 0) then
                        write(LERR,*)'FATAL ERROR in vred (3D option):'
                        write(LERR,*)'End of file on velocity tape at'
                        write(LERR,*)'jj,kk= ',jj,kk,' li,di= ',li,di
                        write(LER ,*)'FATAL ERROR in vred (3D option):'
                        write(LER ,*)'End of file on velocity tape at'
                        write(LER ,*)'jj,kk= ',jj,kk,' li,di= ',li,di
                        go to 999
                      endif
                      call vmov (itr(ITHWP1), 1, vred, 1, 1)
                      vred = vred * dt

c-----
c     get absolute trace distance
c     compute shift in samples
c-----
                      call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                          idist  , TRACEHEADER)
 
                      dist = float( iabs(idist) ) * dmul

                      call vapply 
     1                  (itr, tri, work, omega, expphi, vred, dist,
     2                   nsamp, nsampo, ntnew, dmul, unit, ishft,
     3                   undo, stawrd, nsi, ishift,
     4                   ifmt_RecNum,l_RecNum, ln_RecNum,
     5                   ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     6                   ifmt_DphInd,l_DphInd, ln_DphInd,
     7                   ifmt_LinInd,l_LinInd, ln_LinInd,
     8                   ifmt_stawrd,l_stawrd,ln_stawrd) 

                   ELSE
 
                      call vclr (tri, 1, nsampo)
 
                   ENDIF
 
                     call vmov  (tri, 1, itr(ITHWP1), 1, nsampo)
                     call wrtape( luout, itr, obytes)
                     if(verbos)write(LERR,*)'rec= ',jj,' trc= ',kk,
     1                   ' dist= ',dist,' shift= ',ishft
    
               enddo
 
         ENDDO

      ENDIF
c-----
c     close data files
c-----
  999 continue

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'end of vred, processed',nrec,' record(s)',
     1             ' with ',ntrc, ' traces'
      write(LER ,*)'end of vred, processed',nrec,' record(s)',
     1             ' with ',ntrc, ' traces'
      end

      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*)
     :'execute vred by typing vred and a list of program parameters.'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
        write(LER,*)' '
        write(LER,*)'Constant velocity option:'
        write(LER,*)
     :' -vred [vred] (def = infinite)  : reduction velocity (ft or m/s)'
        write(LER,*)
        write(LER,*)'Variable velocity option:'
        write(LER,*)
     :' -v [vtap]    (def = const vel) : velocity tape data file name'
        write(LER,*)
     :' -threed  if present data is unsorted 3D (see man page)'
        write(LER,*)' '
        write(LER,*)
     :' -t [it0]     (def = 0 ms)      : trc leading zero pad time (ms)'
        write(LER,*)
     :' -T [npadend] (def = 0 ms)      : trc ending  zero pad time (ms)'
        write(LER,*)
     :' -U       if present undo current command line'
        write(LER,*)' '
        write(LER,*)
     :' -sw [stawrd] (def = no store)  : opt. trc hdr word to store shif
     :t'
        write(LER,*)
     :' -u [unit] (def = 1.0)          : opt. scale factor for stored sh
     :ift'
      write(LER,*)
     :' -tmul [tmul]   (def = 1.0)     : time multiplier'
      write(LER,*)
     :' -dmul [tmul]   (def = 1.0)     : distance multiplier'
        write(LER,*)' '
        write(LER,*)' '
        write(LER,*)
     :' -ns[ns]      (default = first) : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)  : end trace number'
        write(LER,*)
     :' -rs[irs]     (default = first) : start record number'
        write(LER,*)
     :' -ne[ire]     (default = last)  : end record number'
        write(LER,*)' '
        write(LER,*)' '
         write(LER,*)
     :'usage:   vred -N[] -O[] [ [ -v[] -vred[] ] -t[ -T[] '
         write(LER,*)
     :'              -d[] -tmul[] -ns[] -ne[] -rs[] -re[]'
         write(LER,*)
     :'              [ -U -threed -V -sw[] -u[] ]'
         write(LER,*)
     :'***************************************************************'
      return
      end

      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,vred,it0,dmul,undo,
     1                  verbos,npadend,stawrd,unit,tmul,vtap,threed,
     2                  tape)
c-----
c     get command arguments
c
c     ntap  - c*255     input file name
c     otap  - c*255     output file name
c     ns    - i*4 starting trace index
c     ne    - i*4 ending trace index
c     irs   - i*4 starting record index
c     ire   - i*4 ending record index
c     vred  - r*4 reduction velocity in distance units/second
c     it0   - i*4 leading trace pad
c  npadend  - i*4 trailing trace pad
c     dmul  - r*4 multiply trace dists by this value
c     undo  - l   undo a previous vred
c     verbos- l   verbose output or not
c-----
#include <f77/iounit.h>
      character  ntap*(*), otap*(*), vtap*(*), stawrd*6
      integer    ns, ne, irs, ire, it0, npadend
      real       vred, dmul,tmul
      logical    verbos, undo, threed, tape
      integer    argis

            undo = .false.
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argstr( '-sw', stawrd, ' ', ' ' )
            call argr4( '-u', unit, 1.0, 1.0 )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argr4( '-vred', vred, 10000000.0, 10000000.0 )
            call argr4( '-dmul', dmul, 1.0, 1.0 )
            call argr4( '-tmul', tmul, 1.0, 1.0 )
            threed = ( argis( '-threed' ) .gt. 0 )
            call argi4( '-t', it0, 0, 0 )
            call argi4( '-T', npadend,0, 0 )
            call argstr( '-v', vtap, ' ', ' ' )
            tape = .false.
            if (vtap(1:1) .ne. ' ') tape = .true.
            undo   = ( argis( '-U' ) .gt. 0 )
            verbos = ( argis( '-V' ) .gt. 0 )

            if (threed .AND. .not.tape) then
               write(LERR,*)'ERROR in vred:'
               write(LERR,*)'Cannot specify -threed option without'
               write(LERR,*)'a velocity tape, -v[]'
               write(LER ,*)'ERROR in vred:'
               write(LER ,*)'Cannot specify -threed option without'
               write(LER ,*)'a velocity tape, -v[]'
               stop
            endif

            if (threed) then
               ns  = 0
               ne  = 0
               irs = 0
               ire = 0
            endif

      return
      end

      subroutine verbal(nsamp, nsi, ntrc, nrec, iform, vred, it0, dmul,
     1            undo,ntap,otap,npadend,stawrd,unit,tmul,
     2            vtap,threed,tape)
c-----
c     verbose output of processing parameters
c
c     nsamp - I*4 number of samples in trace
c     nsi   - I*4 sample interval in ms
c     ntrc  - I*4 traces per record
c     nrec  - I*4 number of records per line
c     iform - I*4 format of data
c     vred  - r*4 reduction velocity in distance units/second
c     it0   - i*4 leading trace pad
c  npadend  - i*4 trailing trace pad
c     dmul  - r*4 omultiply trace dists by this value
c     ntap  - C*255     input file name
c     otap  - C*255     output file name
c     undo  - L   if true do the inverse linear moveout
c-----
#include <f77/iounit.h>
      real*4    vred, dmul, tmul
      integer*4 nsamp, nsi, ntrc, nrec, iform
      character ntap*(*), otap*(*), vtap*(*), stawrd*6
      logical undo,threed,tape

            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace         =  ', nsamp
            write(LERR,*) ' sample interval            =  ', nsi
            write(LERR,*) ' traces per record          =  ', ntrc
            write(LERR,*) ' records per line           =  ', nrec
            write(LERR,*) ' format of data             =  ', iform
            write(LERR,*) ' reduction velocity         =  ', vred
            write(LERR,*) ' pad front of trace by (ms) = ',it0
            write(LERR,*) ' pad end   of trace by (ms) = ',npadend
            write(LERR,*) ' distance scale             =  ', dmul
            write(LERR,*) ' time scale                 =  ', tmul
            write(LERR,*) ' input data set name        = ', ntap
            write(LERR,*) ' output data set name       = ', otap
            write(LERR,*) ' static header mnemonic     = ', stawrd
            write(LERR,*) ' static scale factor        = ', unit

            if (tape) then
            write(LERR,*) ' velocity tape input:'
            write(LERR,*) ' velocity data set name     =  ',vtap
            if (threed) then
            write(LERR,*) ' 3D input data & velocity tape'
            else
            write(LERR,*) ' 2D input data & velocity tape'
            endif
            endif

            if (undo)
     1      write(LERR,*) ' undo previous vred   '
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end

