/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  This file contains the functions for
  creating and manipulating the body list 
  for the velocity builder.
*/

#include "xdisplayP.h"
#include "prototyping.h"

#define NUM_PARAMETER	9
#define V_HEIGHT	100
#define EQUAL		0L
#define PLUS		1L
#define MOD		2L

extern GC vel_gc;
extern float bar_scale;

Widget body_list;
Widget *v_parameter;
static Widget vel_r_c=NULL;
static Widget button_vel[9];
static Widget button_oper[3];

int body_init=0;		/* Indicates the velocity body list exists. */
int body_mode=0;		/* Indicate the mouse mode for body. */
Boolean body_flag=False;
Boolean show_parameters=False;
static int body_count;
static Boolean clear_flag=False;
static Pixel bg,fg;


/*
  This function creates the body
  list for the velocity builder.
*/
void body_listCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  static Widget dialog_form;
  static Widget separator;
  static Widget scroll;
  static Widget row_column;
  static Widget button;
  static Widget toggle;
  int widget_count,index_count;
  long id;	/* RJM: handle 64-bit architectures */

  id=(long)client_data;

				/* Indicate that there is a body_list. */
  body_flag=True;

				/* Store the foreground and background
				   to invert an active selection. */
  XtVaGetValues(xdisplay[id].form,
		XmNforeground,&fg,
		XmNbackground,&bg,
		NULL);

				/* Set if the form exist for the body_list. */
  if (body_list==NULL) {
    dialog_form=XmCreateFormDialogVa(xdisplay[id].window,"form",
			XmNmarginHeight,10,
			XmNmarginWidth,10,
			XmNdeleteResponse,XmDO_NOTHING,
			XmNdialogTitle,XMstr("Velocity Parameters"),
			XtNmanageChild,False,
			NULL);
    XtAddCallback(dialog_form,XmNhelpCallback,helpCB,"velocity_par.help");
    
				/* To delete body list on return. */
    body_list=dialog_form; 
    
    row_column=XmCreateRowColumnVa(dialog_form,"row_column",
				   XmNleftAttachment,XmATTACH_FORM,
				   XmNtopAttachment,XmATTACH_FORM,
				   XmNnumColumns,1,
				   XmNpacking,XmPACK_COLUMN,
				   NULL);
    
    button_vel[0]=XmCreatePushButtonVa(row_column,"Select Bodies",
		XmNforeground,white,
		XmNbackground,black,
		XmNactivateCallback,
			CBlist( reset_velCB, (XtPointer)id,
				toggle_vCB, (XtPointer)0L,
				NULL ),
		NULL);

    button_vel[4]=XmCreatePushButtonVa(row_column,"Unselect Bodies",
				       NULL);
    XtAddCallback(button_vel[4],XmNactivateCallback,reset_velCB,(XtPointer)id);
				/* Put back in to select mode. */
    XtAddCallback(button_vel[4],XmNactivateCallback,toggle_vCB,(XtPointer)0);
    
    toggle=XmCreateToggleButtonVa(dialog_form,"Show Locations",
				  XmNleftAttachment,XmATTACH_FORM,
				  XmNtopAttachment,XmATTACH_WIDGET,
				  XmNtopWidget,row_column,
				  XmNset,show_parameters,
				  NULL);
    XtAddCallback(toggle,XmNvalueChangedCallback,set_flagCB,&show_parameters);
    XtAddCallback(toggle,XmNvalueChangedCallback,refresh_imageCB,(XtPointer)id);
    
    row_column=XmCreateRowColumnVa(dialog_form,"row_column",
				   XmNleftAttachment,XmATTACH_WIDGET,
				   XmNleftWidget,row_column,
				   XmNtopAttachment,XmATTACH_FORM,
				   XmNnumColumns,2,
				   XmNpacking,XmPACK_COLUMN,
				   NULL);
    
    button_vel[3]=XmCreatePushButtonVa(row_column,"#1 V,X,Z",
				       NULL);
    XtAddCallback(button_vel[3],XmNactivateCallback,toggle_vCB,(XtPointer)3L);
    
    button_vel[1]=XmCreatePushButtonVa(row_column,"#1 Vel",
				       NULL);
    XtAddCallback(button_vel[1],XmNactivateCallback,toggle_vCB,(XtPointer)1L);
    
    button_vel[2]=XmCreatePushButtonVa(row_column,"#1 X,Z",
				       NULL);
    XtAddCallback(button_vel[2],XmNactivateCallback,toggle_vCB,(XtPointer)2L);
    
    button_vel[7]=XmCreatePushButtonVa(row_column,"#2 V,X,Z",
				       NULL);
    XtAddCallback(button_vel[7],XmNactivateCallback,toggle_vCB,(XtPointer)7L);
    
    button_vel[5]=XmCreatePushButtonVa(row_column,"#2 Vel ",
				       NULL);
    XtAddCallback(button_vel[5],XmNactivateCallback,toggle_vCB,(XtPointer)5L);
    
    button_vel[6]=XmCreatePushButtonVa(row_column,"#2 X,Z",
				       NULL);
    XtAddCallback(button_vel[6],XmNactivateCallback,toggle_vCB,(XtPointer)6L);
 
    row_column=XmCreateRowColumnVa(dialog_form,"row_column",
				   XmNleftAttachment,XmATTACH_WIDGET,
				   XmNleftWidget,row_column,
				   XmNtopAttachment,XmATTACH_FORM,
				   XmNnumColumns,3,
				   XmNpacking,XmPACK_COLUMN,
				   XmNorientation,XmHORIZONTAL,
				   NULL);
    
    button_oper[0]=XmCreatePushButtonVa(row_column,"= Replace",
					NULL);
    XtAddCallback(button_oper[0],XmNactivateCallback,set_operatorCB,(XtPointer)EQUAL);

    button_oper[1]=XmCreatePushButtonVa(row_column,"+ Add",
			  		NULL);
    XtAddCallback(button_oper[1],XmNactivateCallback,set_operatorCB,(XtPointer)PLUS);

    button_oper[2]=XmCreatePushButtonVa(row_column,"% Change",
					NULL);
    XtAddCallback(button_oper[2],XmNactivateCallback,set_operatorCB,(XtPointer)MOD);

    if (strlen(xdisplay[id].filename)==0) {
      XtVaSetValues( button_oper[1], XmNsensitive,False,
			NULL );
      XtVaSetValues( button_oper[2], XmNsensitive,False,
			NULL );
    }
    
    row_column=XmCreateRowColumnVa(dialog_form,"row_column",
				   XmNleftAttachment,XmATTACH_WIDGET,
				   XmNleftWidget,row_column,
				   XmNrightAttachment,XmATTACH_FORM,
				   XmNtopAttachment,XmATTACH_FORM,
				   XmNnumColumns,1,
				   XmNpacking,XmPACK_COLUMN,
				   NULL);

    button=XmCreatePushButtonVa(row_column," Update Velocities",
				NULL);
    XtAddCallback(button,XmNactivateCallback,update_velCB,(XtPointer)id);
    
    button=XmCreatePushButtonVa(row_column,"  Undo Velocities",
				NULL);
    XtAddCallback(button,XmNactivateCallback,undo_velCB,(XtPointer)id); 
    
    button=XmCreatePushButtonVa(row_column,"Restore Background",
				NULL);
    XtAddCallback(button,XmNactivateCallback,restore_velCB,(XtPointer)id);
    
    separator=XmCreateSeparatorVa(dialog_form,"separator",
				  XmNleftAttachment,XmATTACH_FORM,
				  XmNrightAttachment,XmATTACH_FORM,
				  XmNtopAttachment,XmATTACH_FORM,
				  XmNtopOffset,110,
				  NULL);
  }
  
  scroll=XmCreateScrolledWindowVa(dialog_form,"scroll",
				  XmNtopAttachment,XmATTACH_WIDGET,
				  XmNtopWidget,separator,
				  XmNbottomAttachment,XmATTACH_FORM,
				  XmNleftAttachment,XmATTACH_FORM,
				  XmNrightAttachment,XmATTACH_FORM,
				  /*  XmNscrollingPolicy,XmAUTOMATIC, */
				  NULL);
  
  body_count=1;
  
  row_column=XmCreateRowColumnVa(scroll,"row_column",
				 XmNtopAttachment,XmATTACH_FORM,
				 XmNbottomAttachment,XmATTACH_FORM,
				 XmNleftAttachment,XmATTACH_FORM,
				 XmNrightAttachment,XmATTACH_FORM,
				 XmNnumColumns,body_count,
				 XmNpacking,XmPACK_COLUMN,
				 XmNorientation,XmHORIZONTAL,
				 NULL);
  
  vel_r_c=row_column;

  (void)XmCreateLabelVa(row_column," V(1) ",NULL);
  (void)XmCreateLabelVa(row_column," X(1) ",NULL);
  (void)XmCreateLabelVa(row_column," Z(1) ",NULL);
  (void)XmCreateLabelVa(row_column," V(2) ",NULL);
  (void)XmCreateLabelVa(row_column," X(2) ",NULL);
  (void)XmCreateLabelVa(row_column," Z(2) ",NULL);
  (void)XmCreateLabelVa(row_column,"Vmin.",NULL);
  (void)XmCreateLabelVa(row_column,"Vmax.",NULL);
  (void)XmCreateLabelVa(row_column,"Oper.",NULL);

  if( v_parameter )
    free((char*)v_parameter); 
  v_parameter=(Widget *)malloc(v_index*NUM_PARAMETER*sizeof(Widget));

  widget_count=0;
  for (index_count=0; index_count<v_index; index_count++) {
    sprintf(buf,"%d",model_body[index_count].vel1);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;

    sprintf(buf,"%d",model_body[index_count].x1);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;

    sprintf(buf,"%d",model_body[index_count].y1);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;

    sprintf(buf,"%d",model_body[index_count].vel2);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;
    
    sprintf(buf,"%d",model_body[index_count].x2);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;

    sprintf(buf,"%d",model_body[index_count].y2);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;

    sprintf(buf,"%d",model_body[index_count].min);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;

    sprintf(buf,"%d",model_body[index_count].max);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;

    sprintf(buf,"%s",model_body[index_count].operator);
    
    v_parameter[widget_count]=XmCreateTextVa(row_column,"text",
					     XmNcolumns,5,
					     XmNvalue,buf,
					     XmNeditable,FALSE,
					     XtNmanageChild,False,
					     NULL);
    XtAddCallback(v_parameter[widget_count],XmNactivateCallback,set_parCB,(XtPointer)id);
    widget_count++;
  }

  XtManageChild(dialog_form);

  /* want set to same colormap as velocity window so doesn't flash */
  XSetWindowColormap(display,XtWindow(XtParent(dialog_form)),xdisplay[id].cmap);
#if defined(__sgi) || defined(sun)	/* added sun in case redirect disp from sun to sgi */
  /* don't need to set colormap for this window as no flashing occurs */
#endif
}



/*
  This function select a body 
  region according to index.
*/
void select_body(XEvent *event, long id)/* RJM: handle 64-bit architectures */
{
  int x,y,x_tmp,index_count,current_index;

  if (clear_flag) {
    clear_flag=False;
    XtUnmanageChild(vel_r_c);
    for (index_count=0; index_count<v_index; index_count++) 
      if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
	unmanage_text(index_count);
        generate_vel(index_count,id);
      }
    XtManageChild(vel_r_c);
  }

  x=event->xbutton.x;
  y=event->xbutton.y;
  current_index=modelread(x/fill_skip,y/fill_skip);

  current_index--;
  if (current_index<0) {
    return;
  }

  if (XtIsManaged(v_parameter[current_index*NUM_PARAMETER])) {
    unselect_body(event,id);
    return;
  }
  if (xdisplay[id].flip_flag) {
    x_tmp=(int)((etrace[id]-strace[id])/itrace[id])*itrace[id]+1;
    model_body[current_index].x=x_tmp-(x*xdisplay[id].pick_x);
  } else {
    model_body[current_index].x=x*xdisplay[id].pick_x;
  }
  model_body[current_index].y=y*xdisplay[id].pick_y;
   
  manage_text(current_index);

  generate_border(current_index);

  expose_image(id);
}



/*
  This function hightlights
  the border of a selected region.
*/
void generate_border(int current_index)
{
  int x,y;

				/* Check top and bottom for boundary. */
  for (y=0; y<model_height; y+=model_height-1)
    for (x=0; x<model_width; x++)
      if (*(model_index+x+y*model_width)==current_index+1)
	pixelwrite(x,y,white);

				/* Check left and right for boundary. */
  for (x=0; x<model_width; x+=model_width-1)
    for (y=0; y<model_height; y++)
      if (*(model_index+x+y*model_width)==current_index+1)
	pixelwrite(x,y,white);

				/* Check for borders. */
  for (x=model_body[current_index].start; x<model_body[current_index].end; x++)
    if (*(model_index+x)==current_index+1)
      if ((*(model_index+x+1)!=current_index+1)||
	  (*(model_index+x-1)!=current_index+1)||
	  (*(model_index+x-model_width)!=current_index+1)||
	  (*(model_index+x+model_width)!=current_index+1))
	pixelwrite((int)(x%model_width),(int)(x/model_width),white);
}


/*
  This function unselects a body.
*/
void unselect_body(XEvent *event, long id)/* RJM: handle 64-bit architectures */
{
  int x,y,current_index;

  if (!body_flag) {
    body_listCB(NULL,(XtPointer )id,0);
  }

  x=event->xbutton.x;
  y=event->xbutton.y;
  current_index=modelread(x/fill_skip,y/fill_skip);

  current_index--;
  if (current_index<0) return;

  generate_vel(current_index,id);

  XtUnmanageChild(vel_r_c);
  unmanage_text(current_index);
  XtManageChild(vel_r_c);

  expose_image(id);
}



/*
  This function unselects 
  all the bodies.
*/
void reset_velCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int index_count;
  long id;	/* RJM: handle 64-bit architectures */

  id=(long)client_data;

  if (body_list!=NULL) {
    XtUnmanageChild(vel_r_c);
    for (index_count=0; index_count<v_index; index_count++) 
      if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
	unmanage_text(index_count);
	generate_vel(index_count,id);
      }
    XtManageChild(vel_r_c);
    expose_image(id);
  }
}



/*
  This function set the parameter
  velocity off the vel_tape displayed.
*/
void set_vel(XEvent *event, int setting)
{
  int vel,x,y,index_count;

  x=event->xbutton.x/fill_skip;
  y=event->xbutton.y/fill_skip;

				/* Get the velocity at x,y. */
  vel=*(vel_tape+x+y*model_width);
   
  for (index_count=0; index_count<v_index; index_count++) 
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      if (setting==1) {		/* Setting vel1. */
	sprintf(buf,"%d",vel);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER],args,1);
      }
      if (setting==2) {		/* Setting vel2. */
	sprintf(buf,"%d",vel);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+3],args,1);
      }
    }
}



/*
  This function set the x,y 
  parameters from the mouse position.
*/
void set_xy(XEvent *event, long id, int setting)
{
  int x,y,x_tmp,index_count;

  if (xdisplay[id].flip_flag) {
    x_tmp=(int)((etrace[id]-strace[id])/itrace[id])*itrace[id]+1;
    x=x_tmp-(event->xbutton.x*xdisplay[id].pick_x);
  } else {
    x=event->xbutton.x*xdisplay[id].pick_x;
  }
  y=event->xbutton.y*xdisplay[id].pick_y;

  for (index_count=0; index_count<v_index; index_count++) 
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      if (setting==1) {		/* Sets x1 & y1 */
 	sprintf(buf,"%d",x+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+1],args,1);
	sprintf(buf,"%d",y+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+2],args,1);
      }
      if (setting==2) {		/* Sets x2 & y2  */
	sprintf(buf,"%d",x+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+4],args,1);
	sprintf(buf,"%d",y+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+5],args,1);
      }
    }
}



/* 
  This function set the vel,x,y parameters
  from the mouse position.
*/
void set_vxy(XEvent *event, long id, int setting)
{
  int x,x_tmp,y,index_count,vel;

  x=event->xbutton.x/fill_skip;
  y=event->xbutton.y/fill_skip;

  vel=*(vel_tape+x+y*model_width);

  if (xdisplay[id].flip_flag) {
    x_tmp=(int)((etrace[id]-strace[id])/itrace[id])*itrace[id]+1;
    x=x_tmp-(event->xbutton.x*xdisplay[id].pick_x);
  } else {
    x=event->xbutton.x*xdisplay[id].pick_x;
  }
  y=event->xbutton.y*xdisplay[id].pick_y;
   
  for (index_count=0; index_count<v_index; index_count++) 
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      if (setting==1) {		/* Set vel1,x1,y1. */
	sprintf(buf,"%d",vel);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER],args,1);
	sprintf(buf,"%d",x+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+1],args,1);
	sprintf(buf,"%d",y+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+2],args,1);
      }
      if (setting==2) {		/* Set vel2,x2,y2. */
	sprintf(buf,"%d",vel);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+3],args,1);
	sprintf(buf,"%d",x+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+4],args,1);
	sprintf(buf,"%d",y+1);
	XtSetArg(args[0],XmNvalue,buf); 
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+5],args,1);
      }
    }
}


/*
  This function hightlights (inverts)
  the selected option.
*/
void toggle_vCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  body_mode=(long)client_data & 0xffffffff;

  XtSetArg(args[0],XmNforeground,fg);
  XtSetArg(args[1],XmNbackground,bg);
  XtSetValues(button_vel[0],args,2);
  XtSetValues(button_vel[1],args,2);
  XtSetValues(button_vel[2],args,2);
  XtSetValues(button_vel[3],args,2);
  XtSetValues(button_vel[4],args,2);
  XtSetValues(button_vel[5],args,2);
  XtSetValues(button_vel[6],args,2);
  XtSetValues(button_vel[7],args,2);
  
  XtSetArg(args[0],XmNforeground,white);
  XtSetArg(args[1],XmNbackground,black);
  XtSetValues(button_vel[body_mode],args,2);
}


/*
  This function reset the min and max
  velocity for the whole model.
*/
void min_maxCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int value,index_count;
  long id;	/* RJM: handle 64-bit architectures */
  Boolean isMin;
  char *str;

  id=((MinMaxRec*)client_data)->id;
  isMin=((MinMaxRec*)client_data)->isMin;

                                /* Get velocity from text widget. */
  XtSetArg(args[0],XmNvalue,&str);
  XtGetValues(w,args,1);
  sscanf(str,"%d",&value);
  XtFree(str);

  if (isMin)
    global_min=value;
  else
    global_max=value;

  scale_range(global_min,global_max,id);

  if (xdisplay[id].mode==18) 
    for (index_count=0; index_count<v_index; index_count++)
      generate_vel(index_count,id);

  if (xdisplay[id].mode==28) 
    update(id);

  expose_image(id);
}



/* 
  This function propogates the text 
  values to all the bodies selected.
*/
void set_parCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int offset,index_count;
  long id;	/* RJM: handle 64-bit architectures */
  char *str;

  id=(long)client_data;

  clear_flag=True;

                                /* Get segment number from text widget. */
  XtSetArg(args[0],XmNvalue,&str);
  XtGetValues(w,args,1);

  offset=(-1);
  index_count=0;
  while (offset==(-1)) {
      if (w==v_parameter[index_count*NUM_PARAMETER])
	offset=0;
      if (w==v_parameter[index_count*NUM_PARAMETER+1])
	offset=1;
      if (w==v_parameter[index_count*NUM_PARAMETER+2])
	offset=2;
      if (w==v_parameter[index_count*NUM_PARAMETER+3])
	offset=3;
      if (w==v_parameter[index_count*NUM_PARAMETER+4])
	offset=4;
      if (w==v_parameter[index_count*NUM_PARAMETER+5])
	offset=5;
      if (w==v_parameter[index_count*NUM_PARAMETER+6])
	offset=6;
      if (w==v_parameter[index_count*NUM_PARAMETER+7])
	offset=7;
      if (w==v_parameter[index_count*NUM_PARAMETER+8])
	offset=8;

      index_count++;
    }

  for (index_count=0; index_count<v_index; index_count++) 
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER]))
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+offset],args,1);
  XtFree(str);
  
  update_par(id);
}



/* 
  This function assigns the velocity
  values from the color bar.
*/
void assign_vel(int vel, long id)
{
  Boolean flag;
  int index_count;

  clear_flag=True;

				/* Indicate if setting Vel #1 or #2. */
  if ((body_mode==5)||(body_mode==6)||(body_mode==7))
    flag=False;
  else
    flag=True;

  for (index_count=0; index_count<v_index; index_count++) 
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {

				/* Update text widgets. */
      sprintf(buf,"%d",vel);
      XtSetArg(args[0],XmNvalue,buf); 
      
      if (flag)
	XtSetValues(v_parameter[index_count*NUM_PARAMETER],args,1);
      else
	XtSetValues(v_parameter[index_count*NUM_PARAMETER+3],args,1);
    }

  update_par(id);
}



/*
  This function reads the values
  from the text widgets and assign
  them to the model_body structure.
*/
void update_par(long id)
{
  int index_count;
  char *str;
  
  for (index_count=0; index_count<v_index; index_count++)  {
    copy_body(index_count,index_count);
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      XtSetArg(args[0],XmNvalue,&str); 
      XtGetValues(v_parameter[index_count*NUM_PARAMETER],args,1);
      sscanf(str,"%d",&model_body[index_count].vel1);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+1],args,1);
      sscanf(str,"%d",&model_body[index_count].x1);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+2],args,1);
      sscanf(str,"%d",&model_body[index_count].y1);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+3],args,1);
      sscanf(str,"%d",&model_body[index_count].vel2);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+4],args,1);
      sscanf(str,"%d",&model_body[index_count].x2);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+5],args,1);
      sscanf(str,"%d",&model_body[index_count].y2);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+6],args,1);
      sscanf(str,"%d",&model_body[index_count].min);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+7],args,1);
      sscanf(str,"%d",&model_body[index_count].max);
      XtFree(str);	/* RJM: added */

      XtGetValues(v_parameter[index_count*NUM_PARAMETER+8],args,1);
      sscanf(str,"%1s",model_body[index_count].operator);
      XtFree(str);	/* RJM: added */
    }
  }
  
  expose_image(id);
}



/*
  This function restore the velocity
  to the backdrop values.
*/
void restore_velCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int index_count;
  

  for (index_count=0; index_count<v_index; index_count++)  {
    copy_body(index_count,index_count);
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      sprintf(buf,"%d",0);
      XtSetArg(args[0],XmNvalue,buf);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER],args,1);
      
      sprintf(buf,"%d",0);
      XtSetArg(args[0],XmNvalue,buf);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+1],args,1);
      
      sprintf(buf,"%d",0);
      XtSetArg(args[0],XmNvalue,buf);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+2],args,1);
      
      sprintf(buf,"%d",0);
      XtSetArg(args[0],XmNvalue,buf); 
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+3],args,1);

      sprintf(buf,"%d",0);
      XtSetArg(args[0],XmNvalue,buf);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+4],args,1);
      
      sprintf(buf,"%d",0);
      XtSetArg(args[0],XmNvalue,buf);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+5],args,1);
      
    }
  }
  update_velCB(w,client_data,call_data);
}



/* 
  This function update the model_body struct
  and regenerate the velocities for the selected
  bodies.
*/
void update_velCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int index_count;
  long id;	/* RJM: handle 64-bit architectures */
  char *str;
  
  id=(long)client_data;

  if (model_undo!=NULL)
    model_undo=(struct model_struct *)realloc(model_undo,v_index*sizeof(struct model_struct));
  else
    model_undo=(struct model_struct *)malloc(v_index*sizeof(struct model_struct));

  for (index_count=0; index_count<v_index; index_count++)  {
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      XtSetArg(args[0],XmNvalue,&str); 
      XtGetValues(v_parameter[index_count*NUM_PARAMETER],args,1);
      sscanf(str,"%d",&model_body[index_count].vel1);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+1],args,1);
      sscanf(str,"%d",&model_body[index_count].x1);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+2],args,1);
      sscanf(str,"%d",&model_body[index_count].y1);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+3],args,1);
      sscanf(str,"%d",&model_body[index_count].vel2);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+4],args,1);
      sscanf(str,"%d",&model_body[index_count].x2);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+5],args,1);
      sscanf(str,"%d",&model_body[index_count].y2);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+6],args,1);
      sscanf(str,"%d",&model_body[index_count].min);
      XtFree(str);	/* RJM: added */
      
      XtGetValues(v_parameter[index_count*NUM_PARAMETER+7],args,1);
      sscanf(str,"%d",&model_body[index_count].max);
      XtFree(str);	/* RJM: added */

      XtGetValues(v_parameter[index_count*NUM_PARAMETER+8],args,1);
      sscanf(str,"%1s",model_body[index_count].operator);
      XtFree(str);	/* RJM: added */

      generate_vel(index_count,id);
      generate_border(index_count);
    }
  }
  
  expose_image(id);
}



/*
  This function undoes that last
  parameter change.
*/
void undo_velCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int index_count;
  long id;	/* RJM: handle 64-bit architectures */
  
  id=(long)client_data;

  for (index_count=0; index_count<v_index; index_count++) {
    undo_body(index_count,index_count);
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      XtSetArg(args[0],XmNvalue,buf);

      sprintf(buf,"%d",model_body[index_count].vel1);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER],args,1);
      
      sprintf(buf,"%d",model_body[index_count].x1);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+1],args,1);
      
      sprintf(buf,"%d",model_body[index_count].y1);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+2],args,1);

      sprintf(buf,"%d",model_body[index_count].vel2);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+3],args,1);
      
      sprintf(buf,"%d",model_body[index_count].x2);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+4],args,1);
      
      sprintf(buf,"%d",model_body[index_count].y2);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+5],args,1);

      sprintf(buf,"%d",model_body[index_count].min);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+6],args,1);
      
      sprintf(buf,"%d",model_body[index_count].max);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+7],args,1);
      
      sprintf(buf,"%s",model_body[index_count].operator);
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+8],args,1);

      generate_vel(index_count,id);
      generate_border(index_count);
    }
  }

  expose_image(id);
}



/*
  This function computes the velocities
  for the body according to the parameters.
*/
void generate_vel(int index_count, long id)
{
  Pixel pixel;
  int x,y,vel_tmp,vel_tmp1,x_tmp1,index_tmp;
  float x1,x2,x3,y1,y2,y3,vel1,vel2,deter,a,b,c;
  float offset,scalar;

                                /* Compute offset for seismic value. */
/*offset=((end_color-start_color+1)/2+start_color); RJM: comment out */
  offset= (float)(end_color-start_color)/2+(int)start_color;	/* RJM: unbias */
  offset+=(xdisplay[id].offset*(int)(end_color-start_color+1))/100.0;
	/* RJM: note: need (int) in computation, else unsigned long types
		(end_color & start_color) convert negative offset value
		(e.g., -50) to a weird number */
  offset = MIN( offset, (float)(int)(end_color&0xffffffff) );	/* RJM: added */
  
                                /* Compute scalar for seismic value. */
  scalar=100.0*xdisplay[id].scalar;

  vel1=model_body[index_count].vel1;
  vel2=model_body[index_count].vel2;
  index_tmp=index_count+1;

				/* Check if constant veloctiy. */
  if ((model_body[index_count].x1<=0)&&
      (model_body[index_count].y1<=0)&&
      (model_body[index_count].x2<=0)&&
      (model_body[index_count].y2<=0)) {

				/* Set velocity for body. */
    vel_tmp=model_body[index_count].vel1;
    vel_tmp1=vel_tmp;
    if (model_body[index_count].vel1==0) {
      for (x=model_body[index_count].start; x<=model_body[index_count].end; x++)
	if (*(model_index+x)==index_tmp) {
				/* Compute pixel value. */
/* RJM: no need for setting here since gets set below
	  pixel=(*(vel_old+x)*scalar)+offset;
	  if (pixel<start_color) pixel=start_color;
	  if (pixel>end_color) pixel=end_color;

	*(vel_raster+x)=pixel&0xffffffff;
*/
	  vel_tmp=*(vel_old+x);
				/* Min Max checks. */
	  if (vel_tmp>model_body[index_count].max)
	    vel_tmp=model_body[index_count].max;
	  if (vel_tmp<model_body[index_count].min)
	    vel_tmp=model_body[index_count].min;
	  if (vel_tmp>global_max)
	    vel_tmp=global_max;
	  if (vel_tmp<global_min)
	    vel_tmp=global_min;
				/* Compute pixel value. */
	  pixel=(vel_tmp*scalar)+offset;
	  if (pixel<start_color) pixel=start_color;
	  if (pixel>end_color) pixel=end_color;

	  *(vel_raster+x)=pixel&0xffffffff;
	  *(vel_tape+x)=vel_tmp;
	} 
      return;
    }
				/* Overwrite background. */
    if (strcmp(model_body[index_count].operator,"=")==0) {
				/* Min Max checks. */
      if (vel_tmp>model_body[index_count].max)
	vel_tmp=model_body[index_count].max;
      if (vel_tmp<model_body[index_count].min)
	vel_tmp=model_body[index_count].min;
      if (vel_tmp>global_max)
	vel_tmp=global_max;
      if (vel_tmp<global_min)
	vel_tmp=global_min;
				/* Compute pixel value. */
      pixel=(vel_tmp*scalar)+offset;
      if (pixel<start_color) pixel=start_color;
      if (pixel>end_color) pixel=end_color;

      for (x=model_body[index_count].start; x<=model_body[index_count].end; x++)
	if (*(model_index+x)==index_tmp) {
	  *(vel_raster+x)=pixel&0xffffffff;
	  *(vel_tape+x)=vel_tmp;
	}
      return;
    }

				/* Add to background. */
    if (strcmp(model_body[index_count].operator,"+")==0) {
      for (x=model_body[index_count].start; x<=model_body[index_count].end; x++)
	if (*(model_index+x)==index_tmp) {
	  vel_tmp=vel_tmp1+*(vel_old+x);
				/* Min Max checks. */
	  if (vel_tmp>model_body[index_count].max)
	    vel_tmp=model_body[index_count].max;
	  if (vel_tmp<model_body[index_count].min)
	    vel_tmp=model_body[index_count].min;
	  if (vel_tmp>global_max)
	    vel_tmp=global_max;
	  if (vel_tmp<global_min)
	    vel_tmp=global_min;
				/* Compute pixel value. */
	  pixel=(vel_tmp*scalar)+offset;
	  if (pixel<start_color) pixel=start_color;
	  if (pixel>end_color) pixel=end_color;

	  *(vel_raster+x)=pixel&0xffffffff;
	  *(vel_tape+x)=vel_tmp;
	}
      return;
    }
				/* Percentage of background. */
    if (strcmp(model_body[index_count].operator,"%")==0) {
      for (x=model_body[index_count].start; x<=model_body[index_count].end; x++)
	if (*(model_index+x)==index_tmp) {
	  vel_tmp=*(vel_old+x)*((float)vel_tmp1/100.0)+*(vel_old+x);
				/* Min Max checks. */
	  if (vel_tmp>model_body[index_count].max)
	    vel_tmp=model_body[index_count].max;
	  if (vel_tmp<model_body[index_count].min)
	    vel_tmp=model_body[index_count].min;
	  if (vel_tmp>global_max)
	    vel_tmp=global_max;
	  if (vel_tmp<global_min)
	    vel_tmp=global_min;
				/* Compute pixel value. */
	  pixel=(vel_tmp*scalar)+offset;
	  if (pixel<start_color) pixel=start_color;
	  if (pixel>end_color) pixel=end_color;

	  *(vel_raster+x)=pixel&0xffffffff;
	  *(vel_tape+x)=vel_tmp;
	}
      return;
    }
  }

				/* Graident. */
  x1=(model_body[index_count].x1-1)/xdisplay[id].pick_x;
  y1=(model_body[index_count].y1-1)/xdisplay[id].pick_y;
  x1=x1/fill_skip;
  y1=y1/fill_skip;

  x2=(model_body[index_count].x2-1)/xdisplay[id].pick_x;
  y2=(model_body[index_count].y2-1)/xdisplay[id].pick_y;
  x2=x2/fill_skip;
  y2=y2/fill_skip;

  if (xdisplay[id].flip_flag) {
    x1=model_width-x1;
    x2=model_width-x2;
  }    

  if (x1==x2) {
    x3=x1+10;
    y3=y1;
  } else 
    if (y1==y2) {
      x3=x1;
      y3=y1+10;
    } else {
      y3=10*(x1-x2)/(y2-y1)+y1;
      x3=x1+10;
    }

  deter=x1*(y2-y3)-x2*(y1-y3)+x3*(y1-y2);
  a=(vel1*(y1-y3)-vel2*(y1-y3))/deter;
  b=(vel1*(x3-x1)+vel2*(x1-x3))/deter;
  c=(vel1*(x2*y3-x3*y2)+vel2*(x3*y1-x1*y3)+vel1*(x1*y2-x2*y1))/deter;
  
				/* Set velocity for body. */
  for (x_tmp1=model_body[index_count].start; 
       x_tmp1<=model_body[index_count].end; x_tmp1++)
    if (*(model_index+x_tmp1)==index_tmp) {
      y=(int)(x_tmp1/model_width);
      x=(int)(x_tmp1%model_width);
      
      vel_tmp=(int)(a*x+b*y+c);

				/* Add to background. */
      if (strcmp(model_body[index_count].operator,"+")==0) 
	vel_tmp+=*(vel_old+x+y*model_width);
				/* Percentage of background. */
      if (strcmp(model_body[index_count].operator,"%")==0) 
	vel_tmp=*(vel_old+x+y*model_width)*((float)vel_tmp/100.0)+*(vel_old+x+y*model_width);
				/* Min Max check. */
      if (vel_tmp>model_body[index_count].max)
	vel_tmp=model_body[index_count].max;
      if (vel_tmp<model_body[index_count].min)
	vel_tmp=model_body[index_count].min;
      if (vel_tmp>global_max)
	vel_tmp=global_max;
      if (vel_tmp<global_min)
	vel_tmp=global_min;
      
				/* Compute pixel value. */
      pixel=(vel_tmp*scalar)+offset;
      if (pixel<start_color) pixel=start_color;
      if (pixel>end_color) pixel=end_color;

      pixelwrite(x,y,pixel);
      *(vel_tape+x+y*model_width)=vel_tmp;
    }
  return;
}



/*
  Computes the correct scalar and offset
  according to the min and max value.
*/
void scale_range(int min, int max, long id)
{
/*xdisplay[id].scalar=(float)(start_color-end_color-1)/(float)(min-max); RJM comment out: stay consistent! */
  xdisplay[id].scalar=(float)(end_color-start_color+1)/(float)(max-min);
  xdisplay[id].scalar/=100.0;

  xdisplay[id].offset=(int)((int)start_color-(min*xdisplay[id].scalar*100.0));
  xdisplay[id].offset-=((int)(end_color-start_color+1)/2+(int)start_color);
/*xdisplay[id].offset=(int)(100.0*xdisplay[id].offset/(float)(end_color-start_color)); RJM: comment out: should be +1 */
  xdisplay[id].offset=(int)(100.0*xdisplay[id].offset/(float)(end_color-start_color+1));
}



/*
  This function converts a data value
  into the corresponding pixel value.
*/
Pixel data_to_pixel(int vel, long id)
{
  Pixel pixel;
  float offset,scalar;

                                /* Compute offset for seismic value. */
/*offset=((end_color-start_color+1)/2+start_color); RJM: comment out */
  offset= (float)(end_color-start_color)/2+(int)start_color;	/* RJM: unbias */
  offset+=(xdisplay[id].offset*(int)(end_color-start_color+1))/100.0;
  offset = MIN( offset, (float)(int)(end_color&0xffffffff) );	/* RJM: added */

                                /* Compute scalar for seismic value. */
  scalar=100.0*xdisplay[id].scalar;

				/* Compute pixel value. */
  pixel=(vel*scalar)+offset;

  if (pixel<start_color) pixel=start_color;
  if (pixel>end_color) pixel=end_color;

  return(pixel);
}



/*
  This function converts a pixel value
  into the corresponding data value.
*/
int pixel_to_data(Pixel pixel, long id)
{
  int vel;
  float offset,scalar;

                                /* Compute offset for seismic value. */
/*offset=((end_color-start_color+1)/2+start_color); RJM: comment out */
  offset=(float)(end_color-start_color)/2+(int)start_color; /* RJM: unbias */
  offset+=(xdisplay[id].offset*(int)(end_color-start_color+1))/100.0;
  offset = MIN( offset, (float)(int)(end_color&0xffffffff) );	/* RJM added */
  
                                /* Compute scalar for seismic value. */
  scalar=100.0*xdisplay[id].scalar;

				/* Compute velocity value. */
  vel=(int)(((int)pixel-offset)/scalar);

  return(vel);
}



/*
  This function display the velocity
  value from the velocity color bar.
*/
void vel_meter(XEvent *event, long id)
{
  int x,start,end,vel;
  float factor;

  x=event->xbutton.x-20;

  if (x<0) x=0;
  
  start=pixel_to_data(start_color,id);
  end=pixel_to_data(end_color,id);

  factor=(end-start)/(bar_scale*(int)(end_color-start_color+1));
  vel=(int)(x*factor+start);
  if (vel>end) vel=end;

  if (vel<0) 
    sprintf(buf,"                       %1.2e                ",(float)vel);
  else
    sprintf(buf,"                        %1.2e               ",(float)vel);
  
  XtVaSetValues(xdisplay[id].values,  
		XmNlabelString,XMstr(buf),
		NULL);
}


/*
  This function deletes the 
  body list.
*/
void delete_body_list( void )
{
  if (body_list!=NULL) {
    XtDestroyWidget(body_list);
    body_list=NULL;
  }

  if( v_parameter )
  {
  	free((char*)v_parameter);
	v_parameter=NULL;
  }
  body_flag=False;
}


/*
  This function set the operator.
*/
void set_operatorCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  long setting = (long)client_data;
  int index_count;

  for (index_count=0; index_count<v_index; index_count++)
    if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])) {
      switch (setting) {
      case EQUAL: strcpy(model_body[index_count].operator,"=");
	break;
      case PLUS: strcpy(model_body[index_count].operator,"+");
	break;
      case MOD: strcpy(model_body[index_count].operator,"%");
	break;
      }
      sprintf(buf,"%s",model_body[index_count].operator);
      XtSetArg(args[0],XmNvalue,buf); 
      XtSetValues(v_parameter[index_count*NUM_PARAMETER+8],args,1);
    }
}



/*
  This function quits the 
  velocity builder.
*/
void quit_model( void )
{
  if( vel_gc ) {	/* RJM: added: velocity backdrop may be null */
    XFreeGC(display,vel_gc);
    vel_gc = NULL;
  }
  body_init=0;
  model_mode=0;
}



/*
  This function manages all the 
  selected body's text widgets.
*/
void manage_text(int index_count)
{
  int n;

				/* Check if already managed. */
  if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER]))
    return;

  XtUnmanageChild(vel_r_c);

  body_count++;
  XtSetArg(args[0],XmNnumColumns,body_count);
  XtSetValues(vel_r_c,args,1);

  for (n=0; n<NUM_PARAMETER; n++) {
    XtManageChild(v_parameter[index_count*NUM_PARAMETER+n]);
  }
  XtManageChild(vel_r_c);
}



/*
  This function unmanages all the 
  selected body's text widget.
*/
void unmanage_text(int index_count)
{
  int n;

				/* Check if already unmanaged. */
  if (XtIsManaged(v_parameter[index_count*NUM_PARAMETER])==False)
    return;

  for (n=0; n<NUM_PARAMETER; n++) {
    XtUnmanageChild(v_parameter[index_count*NUM_PARAMETER+n]);
  }

  body_count--;
  if (body_count<1) 
    body_count=1;

  XtSetArg(args[0],XmNnumColumns,body_count);
  XtSetValues(vel_r_c,args,1);
}


/*
  This function resize the raster
  so that is 1 to 1.
*/
void expand_vel(long id)
{
  int x1,y1,width,skip;
  unsigned char *ptr1,*ptr2;

  width=model_width*fill_skip;
/* fprintf( stderr, "expand_vel: model_width=%d, fill_skip=%d, width=%d\n", model_width, fill_skip, width ); */
/* fprintf( stderr, "note: this function assumes raster space > vel_raster\n" ); */

  ptr1=raster[id];
  ptr2=vel_raster;

  for (y1=0; y1<model_height; y1++) {
    for (x1=0; x1<model_width; x1++) {
      for (skip=0; skip<fill_skip; skip++) {
	*(ptr1)=*(ptr2);
	ptr1++;
      }
      ptr2++;
    }
    for (skip=1; skip<fill_skip; skip++) {
      memcpy((char *)ptr1,(char *)(ptr1-width),sizeof(char)*width);
      ptr1+=width;
    }
    
  }
}
