/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  This file contains the function for
  loading, saving, and editting the colormap.
*/

#include "xdisplayP.h"
#include "rgb.h"
#include "ReadColor.h"
#include "prototyping.h"
#include "stringutils.h"

extern char colorname[80];

Boolean color_flag=False;

static CB_PROTO( cancel_color_choicesCB )

static Window top;
static GC gc;
static Colormap cmap;
static XColor color[256];
static int old_x,old_x1,old_x2,old_y;
static Widget color_editor=NULL;
static Widget color_choices=NULL;
static Widget draw_color=NULL;
static Widget color_toggle[3];
static Widget number=NULL;
static Widget draw_area=NULL;
static Pixel color_index;
static int initialize=0;
static int button_state=0;
static int points=25;



/*
  This function reset a window's
  colormap to the global colormap.
*/
void set_global_colorCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int n;
  long id;	/* RJM: handle 64-bit architectures */

  id=(long)client_data;

  raster_color_red[id]=global_red;
  raster_color_green[id]=global_green;
  raster_color_blue[id]=global_blue;

  for (n=0; n<MAX_IMAGES; n++)
    if (xdisplay[n].init==-1)
      contrast_bias(n,n);
}



/*
  This function loads a global
  colormap.
*/
void load_color_global(long id)
{
  int n,in;

  global_red=(unsigned char *)calloc(256,sizeof(char));	/* RJM: changed to calloc */
  global_green=(unsigned char *)calloc(256,sizeof(char));
  global_blue=(unsigned char *)calloc(256,sizeof(char));

  raster_color_red[id]=global_red;
  raster_color_green[id]=global_green;
  raster_color_blue[id]=global_blue;

  if (color_flag) {
    color_flag=False;
    in=open(colorname,O_RDONLY);
    if (in==-1) {
      error_message("Cannot open color file.");
      return;
    }

    load_color_file(colorname,-1L);
  } else {
    for (n=start_color; n<=end_color; n++) {
      *(global_red+n)=n;
      *(global_green+n)=n;
      *(global_blue+n)=n;
    }
  }

  *(global_red+start_pick_color-4)=255;
  *(global_green+start_pick_color-4)=0;
  *(global_blue+start_pick_color-4)=0;

  *(global_red+start_pick_color-3)=0;
  *(global_green+start_pick_color-3)=255;
  *(global_blue+start_pick_color-3)=0;

  *(global_red+start_pick_color-2)=0;
  *(global_green+start_pick_color-2)=0;
  *(global_blue+start_pick_color-2)=255;

  *(global_red+start_pick_color-1)=255;
  *(global_green+start_pick_color-1)=255;
  *(global_blue+start_pick_color-1)=0;

  *(global_red+start_pick_color)=255;
  *(global_green+start_pick_color)=0;
  *(global_blue+start_pick_color)=0;

  *(global_red+start_pick_color+1)=0;
  *(global_green+start_pick_color+1)=255;
  *(global_blue+start_pick_color+1)=0;

  *(global_red+start_pick_color+2)=0;
  *(global_green+start_pick_color+2)=0;
  *(global_blue+start_pick_color+2)=255;

  *(global_red+start_pick_color+3)=50;
  *(global_green+start_pick_color+3)=153;
  *(global_blue+start_pick_color+3)=255;

  *(global_red+start_pick_color+4)=70;
  *(global_green+start_pick_color+4)=215;
  *(global_blue+start_pick_color+4)=255;

  *(global_red+start_pick_color+5)=255;
  *(global_green+start_pick_color+5)=0;
  *(global_blue+start_pick_color+5)=255;

  *(global_red+start_pick_color+6)=160;
  *(global_green+start_pick_color+6)=32;
  *(global_blue+start_pick_color+6)=240;

  *(global_red+start_pick_color+7)=255;
  *(global_green+start_pick_color+7)=165;
  *(global_blue+start_pick_color+7)=0;

  *(global_red+start_pick_color+8)=255;
  *(global_green+start_pick_color+8)=55;
  *(global_blue+start_pick_color+8)=0;

  *(global_red+start_pick_color+9)=0;
  *(global_green+start_pick_color+9)=250;
  *(global_blue+start_pick_color+9)=154;

  *(global_red+end_color+1)=0;
  *(global_green+end_color+1)=0;
  *(global_blue+end_color+1)=0;

  *(global_red+end_color+2)=255;
  *(global_green+end_color+2)=255;
  *(global_blue+end_color+2)=255;
}


/*
  This function load the colors
  for the picking and the raster
  data.
*/
void load_color(long id)
{
  raster_color_red[id]=(unsigned char *)malloc(256*sizeof(char));
  raster_color_green[id]=(unsigned char *)malloc(256*sizeof(char));
  raster_color_blue[id]=(unsigned char *)malloc(256*sizeof(char));

  memcpy((char *)raster_color_red[id],(char *)raster_color_red[0],256*sizeof(char));
  memcpy((char *)raster_color_blue[id],(char *)raster_color_blue[0],256*sizeof(char));
  memcpy((char *)raster_color_green[id],(char *)raster_color_green[0],256*sizeof(char));
}


/*
  This function creates the edit color
  window. 
*/
void edit_colormapCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  Widget color_form;
  Widget color_window;
  Widget label;
  Widget button;
  int n;
  long id;	/* RJM: handle 64-bit architectures */

  if (initialize!=0)
  {
    XtUnmapWidget( color_editor );	/* RJM: move to foreground */
    XtMapWidget( color_editor );
    return;
  }

  id=(long)client_data;

  n=0;
  XtSetArg(args[n],XtNargc,savedArgc); n++;	/* RJM: added 1/17/97 */
  XtSetArg(args[n],XtNargv,savedArgv); n++;	/* RJM: added 1/17/97 */
  XtSetArg(args[n],XtNiconPixmap,xsd_pixmap); n++;
  XtSetArg(args[n],XtNvisual,visual); n++;	/* RJM: added */
  XtSetArg(args[n],XtNdepth,screenDepth); n++;	/* RJM: added */
/*color_editor=XtAppCreateShell("xsd","edit",
                               topLevelShellWidgetClass,display,args,n);
RJM: commented out 1/17/97 */
  color_editor=XtAppCreateShell(ApplicationName, ApplicationClass,	/* RJM: added 1/17/97 */
                               topLevelShellWidgetClass,display,args,n);
  quit_frame(color_editor);
  
  color_form=XmCreateFormVa(color_editor,"listform",
			    XmNmarginHeight,10,
			    XmNmarginWidth,10,
			    XmNdialogTitle,XMstr("Color Editor"),
			    NULL);
  XtAddCallback(color_form,XmNhelpCallback,helpCB,"editcolor.help");
  XtAddCallback(color_form,XmNdestroyCallback,free_graphCB,(XtPointer)id);

  draw_area=XmCreateDrawingAreaVa(color_form,buf,
				  XmNrightAttachment,XmATTACH_FORM,
				  XmNleftAttachment,XmATTACH_FORM,
				  XmNtopAttachment,XmATTACH_FORM,
				  XmNbackground,black,
				  XmNwidth,400,
				  XmNheight,306,
				  NULL);
  XtAddCallback(draw_area,XmNexposeCallback,expose_graphCB,(XtPointer)id);
  XtAddEventHandler(draw_area,ButtonPressMask,False,button_color_pressEH,(XtPointer)id);
  XtAddEventHandler(draw_area,ButtonReleaseMask,False,button_color_releaseEH,(XtPointer)id);
  XtAddEventHandler(draw_area,PointerMotionMask,False,button_colorEH,(XtPointer)id);

  color_window=XmCreateRowColumnVa(color_form,"colorwindow",
				   XmNpacking,XmPACK_COLUMN,
				   XmNnumColumns,1,
				   XmNleftAttachment,XmATTACH_FORM,
				   XmNbottomAttachment,XmATTACH_FORM,
				   XmNtopAttachment,XmATTACH_WIDGET,
				   XmNtopWidget,draw_area,
				   NULL);

  color_toggle[0]=XmCreateToggleButtonVa(color_window,"Red",NULL);
  color_toggle[1]=XmCreateToggleButtonVa(color_window,"Green",NULL);
  color_toggle[2]=XmCreateToggleButtonVa(color_window,"Blue",NULL);

  XmToggleButtonSetState(color_toggle[0],True,False);
  XmToggleButtonSetState(color_toggle[1],True,False);
  XmToggleButtonSetState(color_toggle[2],True,False);

  XtAddCallback(color_toggle[0],XmNvalueChangedCallback,expose_graphCB,(XtPointer)id);
  XtAddCallback(color_toggle[1],XmNvalueChangedCallback,expose_graphCB,(XtPointer)id);
  XtAddCallback(color_toggle[2],XmNvalueChangedCallback,expose_graphCB,(XtPointer)id);
  
  label=XmCreateLabelVa(color_form,"# of points",
			XmNrightAttachment,XmATTACH_FORM,
			XmNtopAttachment,XmATTACH_WIDGET,
			XmNtopWidget,draw_area,
			NULL);
  
  sprintf(buf,"%d",points);
  
  number=XmCreateTextVa(color_form,"number",
			XmNvalue,buf,
			XmNcolumns,3,
			XmNrightAttachment,XmATTACH_WIDGET,
			XmNrightWidget,label,
			XmNtopAttachment,XmATTACH_WIDGET,
			XmNtopWidget,draw_area,
			NULL);
  XtAddCallback(number,XmNactivateCallback,expose_graphCB,(XtPointer)id);
  
  color_window=XmCreateRowColumnVa(color_form,"colorwindow",
				   XmNpacking,XmPACK_COLUMN, 
				   XmNnumColumns,3,
				   XmNrightAttachment,XmATTACH_FORM,
				   XmNtopAttachment,XmATTACH_WIDGET,
			           XmNtopWidget,number,
				   XmNbottomAttachment,XmATTACH_FORM,
				   NULL);

  button=XmCreatePushButtonVa(color_window,"Load",NULL);
  XtAddCallback(button,XmNactivateCallback,set_typeCB,(XtPointer)6L);
  XtAddCallback(button,XmNactivateCallback,filer_promptCB,(XtPointer)id);

  button=XmCreatePushButtonVa(color_window,"Save",NULL);
  XtAddCallback(button,XmNactivateCallback,set_typeCB,(XtPointer)7L);
  XtAddCallback(button,XmNactivateCallback,filer_promptCB,(XtPointer)id);

  button=XmCreatePushButtonVa(color_window,"Apply",NULL);
  XtAddCallback(button,XmNactivateCallback,set_cmapCB,(XtPointer)id);

  button=XmCreatePushButtonVa(color_window,"Help",NULL);
  XtAddCallback(button,XmNactivateCallback,helpCB,"editcolor.help");

  button=XmCreatePushButtonVa(color_window,"Flip",NULL);
  XtAddCallback(button,XmNactivateCallback,flipCB,(XtPointer)id);

  button=XmCreatePushButtonVa(color_window,"Quit",NULL);
  XtAddCallback(button,XmNactivateCallback,destroyCB,color_editor);

  XtRealizeWidget(color_editor);

  XSetWindowColormap(display,XtWindow(color_editor),xdisplay[id].cmap); 
#if defined(__sgi) || defined(sun)	/* added sun in case redirect disp from sun to sgi */
  XSetWindowColormap(display,XtWindow(draw_area),xdisplay[id].cmap); 
#endif
}


/*
  This function frees up the
  colormap and GC after exiting
  the colormap editor.
*/
void free_graphCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  initialize=0;
  if( cmap ) {	/* RJM: added */
    XFreeColormap(display,cmap);
    cmap = (Colormap)NULL;
  }
  if( gc ) {	/* RJM: added */
    XFreeGC(display,gc);
    gc = NULL;
  }
}


/*
  This function applies the new
  colormap to the data.
*/
void set_cmapCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int n;
  long id = (long)client_data;	/* RJM: handle 64-bit architectures */

  for (n=start_color; n<=end_color; n++) {
    *(raster_color_red[id]+n)=color[n].red>>8;
    *(raster_color_green[id]+n)=color[n].green>>8;
    *(raster_color_blue[id]+n)=color[n].blue>>8;
  }

  for (n=0; n<MAX_IMAGES; n++)
    if (xdisplay[n].init==-1)
      contrast_bias(n,n);
}


/*
  This function handles the drawing
  area for the color editor.
*/
void expose_graphCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  XGCValues values;
  int n;
  long id = (long)client_data;	/* RJM: handle 64-bit architectures */

  if (initialize==-1) {
    XClearArea(display,top,0,0,400,306,False);
    draw_graph(306);
    return;
  }

  initialize=-1;

  top=XtWindow(w);

  XDefineCursor(display,top,
                XCreateFontCursor(display,XC_tcross));

  cmap=DefaultColormap(display,screen);

  for (n=0; n<256; n++) {
    color[n].pixel=n;
    color[n].flags=DoRed|DoGreen|DoBlue;
  }

  XQueryColors(display,cmap,color,256);

  /* NOTE: XCreateColormap window arg is only used to determine the screen */
  cmap=XCreateColormap(display,top,visual,AllocAll);

  for (n=start_color; n<=end_color; n++) {
    color[n].red=(int)*(raster_color_red[id]+n)*256;
    color[n].green=(int)*(raster_color_green[id]+n)*256;
    color[n].blue=(int)*(raster_color_blue[id]+n)*256;
    color[n].pixel=n;
    color[n].flags=DoRed|DoGreen|DoBlue;
  }

  for (n=start_pick_color-4; n<=end_pick_color; n++) {
    color[n].red=(int)*(raster_color_red[id]+n)*256;
    color[n].green=(int)*(raster_color_green[id]+n)*256;
    color[n].blue=(int)*(raster_color_blue[id]+n)*256;
    color[n].pixel=n;
    color[n].flags=DoRed|DoGreen|DoBlue;
  }

  XStoreColors(display,cmap,color,(int)end_color+1);

  values.background = black;
  values.foreground = white;
  values.font=XLoadFont(display,"9x15");

  gc = XCreateGC( display, top,GCForeground|GCBackground|GCFont,&values );

  draw_graph(306);

  XSetWindowColormap(display,XtWindow(color_editor),cmap);
#if defined(__sgi) || defined(sun)	/* added sun in case redirect disp from sun to sgi */
  XSetWindowColormap(display,XtWindow(draw_area),cmap); 
#endif
}


/*
  This funciton reads the text widget
  for the number of control points.
*/
void update_points(void)
{
  char *str;

  XtSetArg(args[0],XmNvalue,&str);
  XtGetValues(number,args,1);
  sscanf(str,"%d",&points);
  XtFree(str);
  points--;
  if (points<=1) points=2;
  if (points>199) points=199;
}


/*
  This function draws the graph
  of the RGB colors.
*/
void draw_graph(int y)
{
  int n;

  update_points();
  
  XSetForeground(display,gc,white);
  XDrawLine(display,top,gc,199,0,199,255);
  XDrawLine(display,top,gc,0,255,400,255);

  if (y>256) 
    for (n=0; n<400; n++) {
      XSetForeground(display,gc,(Pixel)((n/2)+start_color));
      XDrawLine(display,top,gc,n,266,n,306);
    }
  XSetForeground(display,gc,(Pixel)(start_pick_color-4L));
  for (n=start_color+1; n<=end_color; n++) 
    XDrawLine(display,top,gc,2*(n-(int)start_color-1),
	      256-(color[n-1].red>>8),2*(n-(int)start_color),
	      256-(color[n].red>>8));
  if (XmToggleButtonGetState(color_toggle[0])) {
    for (n=start_color; n<end_color; n+=(int)(end_color-start_color)/points) {
      XDrawLine(display,top,gc,2*(n-(int)start_color),
		251-(color[n].red>>8),2*(n-(int)start_color),
		261-(color[n].red>>8));
      XDrawLine(display,top,gc,2*(n-(int)start_color)+5,
		256-(color[n].red>>8),2*(n-(int)start_color)-5,
		256-(color[n].red>>8));
    }
    n=end_color;
    XDrawLine(display,top,gc,2*(n-(int)start_color),
	      251-(color[n].red>>8),2*(n-(int)start_color),
	      261-(color[n].red>>8));
    XDrawLine(display,top,gc,2*(n-(int)start_color)+5,
	      256-(color[n].red>>8),2*(n-(int)start_color)-5,
	      256-(color[n].red>>8));
  }

  XSetForeground(display,gc,(Pixel)(start_pick_color-3L));
  for (n=start_color+1; n<=end_color; n++) 
    XDrawLine(display,top,gc,2*(n-(int)start_color-1),
	      256-(color[n-1].green>>8),2*(n-(int)start_color),
	      256-(color[n].green>>8));
  if (XmToggleButtonGetState(color_toggle[1])) {
    for (n=start_color; n<end_color; n+=(int)(end_color-start_color)/points) {
      XDrawLine(display,top,gc,2*(n-(int)start_color),
		251-(color[n].green>>8),2*(n-(int)start_color),
		261-(color[n].green>>8));
      XDrawLine(display,top,gc,2*(n-(int)start_color)+5,
		256-(color[n].green>>8),2*(n-(int)start_color)-5,
		256-(color[n].green>>8));
    }
    n=end_color;
    XDrawLine(display,top,gc,2*(n-(int)start_color),
	      251-(color[n].green>>8),2*(n-(int)start_color),
	      261-(color[n].green>>8));
    XDrawLine(display,top,gc,2*(n-(int)start_color)+5,
	      256-(color[n].green>>8),2*(n-(int)start_color)-5,
	      256-(color[n].green>>8));
  }
  XSetForeground(display,gc,(Pixel)(start_pick_color-2L));
  for (n=start_color+1; n<=end_color; n++) 
    XDrawLine(display,top,gc,2*(n-(int)start_color-1),
	      256-(color[n-1].blue>>8),2*(n-(int)start_color),
	      256-(color[n].blue>>8));
  if (XmToggleButtonGetState(color_toggle[2])) {
    for (n=start_color; n<end_color; n+=(int)(end_color-start_color)/points) {
      XDrawLine(display,top,gc,2*(n-(int)start_color),
		251-(color[n].blue>>8),2*(n-(int)start_color),
		261-(color[n].blue>>8));
      XDrawLine(display,top,gc,2*(n-(int)start_color)+5,
		256-(color[n].blue>>8),2*(n-(int)start_color)-5,
		256-(color[n].blue>>8));
    }
    n=end_color;
    XDrawLine(display,top,gc,2*(n-(int)start_color),
	      251-(color[n].blue>>8),2*(n-(int)start_color),
	      261-(color[n].blue>>8));
    XDrawLine(display,top,gc,2*(n-(int)start_color)+5,
	      256-(color[n].blue>>8),2*(n-(int)start_color)-5,
	      256-(color[n].blue>>8));
  }
}


/*
  This function flips the current
  colormap around the middle value.
*/
void flipCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int m,n;
  static XColor color1[256];

  for (n=start_color; n<=end_color; n++) {
    m=(int)end_color-(n-(int)start_color);
    color1[m].red=color[n].red;
    color1[m].green=color[n].green;
    color1[m].blue=color[n].blue;
  }
  for (n=start_color; n<=end_color; n++) {
    color[n].red=color1[n].red;
    color[n].green=color1[n].green;
    color[n].blue=color1[n].blue;
  }
  generate_colors();
  XClearArea(display,top,0,0,400,306,False);
  draw_graph(306);
}



/* 
  This function is an event handler
  for button presses in the color editor.
*/
void button_color_pressEH(Widget w, XtPointer client_data, XEvent *event,
	Boolean *continue_to_dispatch ) /* RJM: added 4th arg */
{
  int temp,x;

  if (event->xbutton.button==Button1)
    button_state=1;
  if (event->xbutton.button==Button2)
    button_state=2;
  if (event->xbutton.button==Button3)
    button_state=3;

  update_points();
  
  temp=(int)(end_color-start_color)/points;
  if ((event->xbutton.x/2)%temp!=0) {
    x=((event->xbutton.x/2)+(temp/2))/temp;
    x=x*temp;
  } else {
    x=(event->xbutton.x/2)/temp;
    x=x*temp;
  }
  if ((event->xbutton.x>=400-(temp/4))||(x==200)) {
    old_x=199;
    old_x1=x-temp;
    old_x2=199;
  } else {
    old_x=x;
    old_x1=x-temp;
    old_x2=x+temp;
  }
  if (old_x2==200) old_x2=199;

  if (XmToggleButtonGetState(color_toggle[0]))
    old_y=256-(color[old_x+(int)start_color].red>>8);
  if (XmToggleButtonGetState(color_toggle[1]))
    old_y=256-(color[old_x+(int)start_color].green>>8);
  if (XmToggleButtonGetState(color_toggle[2]))
    old_y=256-(color[old_x+(int)start_color].blue>>8);
}


/*
  This function is an event handler
  for mouse motion in the color editor.
*/
void button_colorEH(Widget w, XtPointer client_data, XEvent *event,
	Boolean *continue_to_dispatch ) /* RJM: added 4th arg */
{
  int y;

  if (button_state==0)
    return;

  y=event->xbutton.y;

  XSetFunction(display,gc,GXequiv);

  if (XmToggleButtonGetState(color_toggle[0])) {
    XSetForeground(display,gc,(Pixel)(start_pick_color-4L));
    XDrawLine(display,top,gc,2*old_x,old_y,
	      2*old_x1,256-(color[old_x1+(int)start_color].red>>8));
    XDrawLine(display,top,gc,2*old_x,old_y,
	      2*old_x2,256-(color[old_x2+(int)start_color].red>>8));
    old_y=y;
    XDrawLine(display,top,gc,2*old_x,y,
	      2*old_x1,256-(color[old_x1+(int)start_color].red>>8));
    XDrawLine(display,top,gc,2*old_x,y,
	      2*old_x2,256-(color[old_x2+(int)start_color].red>>8));
  }

  if (XmToggleButtonGetState(color_toggle[1])) {
    XSetForeground(display,gc,(Pixel)(start_pick_color-3L));
    XDrawLine(display,top,gc,2*old_x,old_y,
	      2*old_x1,256-(color[old_x1+(int)start_color].green>>8));
    XDrawLine(display,top,gc,2*old_x,old_y,
	      2*old_x2,256-(color[old_x2+(int)start_color].green>>8));
    old_y=y;
    XDrawLine(display,top,gc,2*old_x,y,
	      2*old_x1,256-(color[old_x1+(int)start_color].green>>8));
    XDrawLine(display,top,gc,2*old_x,y,
	      2*old_x2,256-(color[old_x2+(int)start_color].green>>8));
  }
  if (XmToggleButtonGetState(color_toggle[2])) {
    XSetForeground(display,gc,(Pixel)(start_pick_color-2L));
    XDrawLine(display,top,gc,2*old_x,old_y,
	      2*old_x1,256-(color[old_x1+(int)start_color].blue>>8));
    XDrawLine(display,top,gc,2*old_x,old_y,
	      2*old_x2,256-(color[old_x2+(int)start_color].blue>>8));
    old_y=y;
    
    XDrawLine(display,top,gc,2*old_x,y,
	      2*old_x1,256-(color[old_x1+(int)start_color].blue>>8));
    XDrawLine(display,top,gc,2*old_x,y,
	      2*old_x2,256-(color[old_x2+(int)start_color].blue>>8));
  }
  
  XSetFunction(display,gc,GXcopy);
}


/*
  This function is an event handler
  for button releases in the color editor.
*/
void button_color_releaseEH(Widget w, XtPointer client_data, XEvent *event,
	Boolean *continue_to_dispatch ) /* RJM: added 4th arg */
{
  int y;

  button_state=0;

  update_points();
  
  y=256-event->xbutton.y;
  if (y<0) y=0;
  if (y>255) y=255;
  y=y<<8;

  if (XmToggleButtonGetState(color_toggle[0]))
    color[old_x+(int)start_color].red=y;
  if (XmToggleButtonGetState(color_toggle[1]))
    color[old_x+(int)start_color].green=y;
  if (XmToggleButtonGetState(color_toggle[2]))
    color[old_x+(int)start_color].blue=y;

  generate_colors();
  XClearArea(display,top,0,0,400,260,True);
}


/*
  This function computes the color 
  values based on the control points
*/
void generate_colors(void)
{
  float slope,start;
  int x,n,step;

  step=(int)(end_color-start_color)/points;

  if (XmToggleButtonGetState(color_toggle[0])) {
    for (n=start_color+step; n<=end_color; n+=step) {
      start=(float)color[n-step].red;
      slope=(float)((float)color[n].red-start)/step;
      for (x=n-step; x<=n; x++) {
	color[x].red=start;
	start+=slope;
      }
    }
    start=color[x-1].red;
    slope=(float)((float)color[end_color].red-start)/(float)((int)end_color-(n-step));
    for (x=(n-step); x<=end_color; x++) {
      color[x].red=(int)start;
      start+=slope;
    }
  }

  if (XmToggleButtonGetState(color_toggle[1])) {
    for (n=start_color+step; n<=end_color; n+=step) {
      start=(float)color[n-step].green;
      slope=(float)((float)color[n].green-start)/step;
      for (x=n-step; x<=n; x++) {
	color[x].green=start;
	start+=slope;
      }
    }
    start=color[x-1].green;
    slope=(float)((float)color[end_color].green-start)/(float)((int)end_color-(n-step));
    for (x=(n-step); x<=end_color; x++) {
      color[x].green=start;
      start+=slope;
    }
  }

  if (XmToggleButtonGetState(color_toggle[2])) {
    for (n=start_color+step; n<=end_color; n+=step) {
      start=(float)color[n-step].blue;
      slope=(float)((float)color[n].blue-start)/step;
      for (x=n-step; x<=n; x++) {
	color[x].blue=start;
	start+=slope;
      }
    }
    start=color[x-1].blue;
    slope=(float)((float)color[end_color].blue-start)/(float)((int)end_color-(n-step));
    for (x=(n-step); x<=end_color; x++) {
      color[x].blue=start;
      start+=slope;
    }
  }

  XStoreColors(display,cmap,color,(int)end_color+1);
}


/*
  This function allows the user to
  change the segment colors.
*/
void segment_colorsCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  Widget list_form;
  Widget row_column;
  Widget label;
  Widget *buttons;
  XmString item[448];
  String	tmp;
  long id = (long)client_data;	/* RJM: handle 64-bit architectures */
  Pixel m;

  if( color_choices != NULL )
    XtDestroyWidget( color_choices ); /* RJM: added because Selected Color push-
					button gets set for MOST RECENT instance
					of this widget! Hence, can only have one
					instance of this widget at a time. */

  color_choices = XmCreateFormDialogVa( xdisplay[id].window, "form",
		XmNdialogTitle,
			XMstr( tmp = concat( "Window ",
                                                itostr( xdisplay[id].sequence ),
                                                " Color Choices",
                                                NULL ) ),
		XmNvisual, visual,
		XmNdepth, screenDepth,
		XmNdeleteResponse,XmDO_NOTHING,
		XtNmanageChild,False,
		NULL );
  free( tmp );
  
  list_form=XmCreateFormVa(color_choices,"listform",
		XmNmarginHeight,10,
		XmNmarginWidth,10,
		XmNhelpCallback,
			CBl( helpCB," segmentcolor.help" ),
		NULL);

  label=XmCreateLabelVa(list_form,"Select a Color",
			XmNleftAttachment,XmATTACH_FORM,
			XmNtopAttachment,XmATTACH_FORM,
			XmNtopOffset,20,
			NULL);

  row_column=XmCreateRowColumnVa(list_form,"datewindow",
				 XmNleftAttachment,XmATTACH_WIDGET,
				 XmNleftWidget,label,
				 XmNrightAttachment,XmATTACH_FORM,
				 XmNtopAttachment,XmATTACH_FORM,
				 XmNpacking,XmPACK_COLUMN,
				 XmNnumColumns,10,
				 NULL);

#if defined(__sgi) || defined(sun)	/* added sun in case redirect disp from sun to sgi */
  buttons = (Widget*)malloc( (end_pick_color-start_pick_color+1) * sizeof( Widget ) );
#endif

  for (m=start_pick_color; m<=end_pick_color; m++)
#if defined(__sgi) || defined(sun)	/* added sun in case redirect disp from sun to sgi */
	buttons[m-start_pick_color] = XmCreatePushButtonVa(row_column,"  ",
#else
	(void)XmCreatePushButtonVa(row_column,"  ",
#endif
		XmNarmColor,black,
		XmNbackground,(Pixel)m,
		XmNtopShadowColor, black,
                XmNbottomShadowColor, black,
                XmNhighlightColor, white,
                XmNselectColor, black,
		XmNactivateCallback,
			CBl( set_colorCB, (XtPointer)m ),
		NULL);
  
  label=XmCreateLabelVa(list_form,"Selected Color ",
			XmNleftAttachment,XmATTACH_FORM,
			XmNtopAttachment,XmATTACH_WIDGET,
			XmNtopWidget,row_column,
			XmNtopOffset,10,
			NULL);
  
  color_index=start_pick_color;

  draw_color=XmCreatePushButtonVa(list_form,"  ",
		XmNleftAttachment,XmATTACH_WIDGET,
		XmNleftWidget,label,
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget,row_column,
		XmNtopOffset,2,
		XmNarmColor,black,
		XmNbackground,(Pixel)color_index,
		XmNtopShadowColor, black,
                XmNbottomShadowColor, black,
                XmNhighlightColor, white,
                XmNselectColor, black,
		NULL);

  for (m=0; m<447; m++)
    item[m]=XmStringCreateLtoR(&color_choice[m][0],XmSTRING_DEFAULT_CHARSET);

  item[m]=NULL;
  
  (void)XmCreateSelectionBoxVa(list_form,"selection",
		XmNleftAttachment,XmATTACH_FORM,
		XmNrightAttachment,XmATTACH_FORM,
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget,label,
		XmNlistItems,item,
		XmNlistItemCount,m,
		XmNlistVisibleItemCount,25,
		XmNlistLabelString,
		XMstr(" R   G   B   Color Choices"),
		XmNcancelLabelString,XMstr("Return"),
		XmNokCallback,
			CBl( assign_colorCB, (XtPointer)id ),
		XmNcancelCallback,
			CBlist( set_segmentCB, (XtPointer)id,
				cancel_color_choicesCB, (XtPointer)NULL,
				NULL ),
		NULL);

  for (m=0; m<447; m++)
    XmStringFree(item[m]);

  XtManageChild(color_choices);

  XSetWindowColormap(display,XtWindow(XtParent(color_choices)),xdisplay[id].cmap);
#if defined(__sgi) || defined(sun)	/* added sun in case redirect disp from sun to sgi */
	XSetWindowColormap(display,XtWindow(draw_color),xdisplay[id].cmap );
	for( m = 0; m < end_pick_color-start_pick_color+1; m++ )
		XSetWindowColormap(display,XtWindow(buttons[m]),xdisplay[id].cmap);
	free( buttons );
#endif
}


/*
  This function updates the images
  and exits the segment color assignment.
*/
void set_segmentCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int n;

  for (n=0; n<MAX_IMAGES; n++)
    if (xdisplay[n].init==-1)
      contrast_bias(n,n);
}


/*
  This function assigns a new
  color to a segment.
*/
void assign_colorCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  XmSelectionBoxCallbackStruct *CBst = (XmSelectionBoxCallbackStruct*)call_data;
  XColor color[256];
  char *buf;
  int red,green,blue,n;
  long id;	/* RJM: handle 64-bit architectures */

  id=(long)client_data;

  XmStringGetLtoR(CBst->value,XmSTRING_DEFAULT_CHARSET,&buf);

  if (buf==NULL)
    return;

  sscanf(buf,"%d %d %d",&red,&green,&blue);

  for (n=0; n<256; n++)
    color[n].pixel=n;
    
  XQueryColors(display,xdisplay[id].cmap,color,256);

  color[color_index].red=red<<8;
  color[color_index].green=green<<8;
  color[color_index].blue=blue<<8;
  
  XStoreColors(display,xdisplay[id].cmap,color,256);

  *(raster_color_red[id]+(int)color_index)=red;
  *(raster_color_green[id]+(int)color_index)=green;
  *(raster_color_blue[id]+(int)color_index)=blue;
}


/*
  This function set the selected
  color value.
*/
void set_colorCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  color_index = (Pixel)((long)client_data&0xff);

  XtSetArg(args[0],XmNbackground,color_index);
  XtSetValues(draw_color,args,1);
}


static CB_DECL( cancel_color_choicesCB )
{
	if( color_choices != NULL )
	{
		XtDestroyWidget( color_choices );
		color_choices = NULL;
	}
}


/*
  This function loads a colormap
  from a data file.
*/
void load_color_file(char *filename, long id)
{
  int n,temp1;
  char *rgb;

/*
************* Old way for reading in colormaps. ********
  char rgb[600];

  in=open(filename,O_RDONLY);

  if (in==-1) {
    error_message("Cannot open color file.");
    return;
  }

  length=read(in,rgb,600);

  close(in);

  if (length==384)
    length-=39;
    for (n=3; n<length; n+=3) {
      rgb[n-3]=rgb[n];
      rgb[n-2]=rgb[n+1];
      rgb[n-1]=rgb[n+2];
    }

  temp=(length/3.0)/(float)(end_color-start_color+1);
  for (n=start_color; n<=end_color; n++) {
    temp1=3*(int)((n-(int)start_color)*temp);
    if (id==-1L) {
      *(global_red+n)=(int)rgb[temp1];
      *(global_green+n)=(int)rgb[temp1+1];
      *(global_blue+n)=(int)rgb[temp1+2];
    } else {
      *(raster_color_red[id]+n)=(int)rgb[temp1];
      *(raster_color_green[id]+n)=(int)rgb[temp1+1];
      *(raster_color_blue[id]+n)=(int)rgb[temp1+2];
    }
  }
*/

  temp1=(int)(end_color-start_color+1);

				/* New colormap reader from Gary Murphy. */
  if (!(rgb=ReadColor(filename,&temp1,(int)verbose))) {
    error_message("Cannot open color file.");
    return;
  }

  for (n=start_color; n<=end_color; n++) {
    temp1=3*(n-(int)start_color);
    if (id==-1L) {
      *(global_red+n)=(unsigned char)rgb[temp1];
      *(global_green+n)=(unsigned char)rgb[temp1+1];
      *(global_blue+n)=(unsigned char)rgb[temp1+2];
    } else {
      *(raster_color_red[id]+n)=(unsigned char)rgb[temp1];
      *(raster_color_green[id]+n)=(unsigned char)rgb[temp1+1];
      *(raster_color_blue[id]+n)=(unsigned char)rgb[temp1+2];
    }
  }

  if (initialize!=0) {
    for (n=start_color; n<=end_color; n++) {
      if (id==-1L) {
	color[n].red=(int)*(global_red+n)*256;
	color[n].green=(int)*(global_green+n)*256;
	color[n].blue=(int)*(global_blue+n)*256;
      } else {
	color[n].red=(int)*(raster_color_red[id]+n)*256;
	color[n].green=(int)*(raster_color_green[id]+n)*256;
	color[n].blue=(int)*(raster_color_blue[id]+n)*256;
      }
    }
    XStoreColors(display,cmap,color,(int)end_color+1);
    XClearArea(display,top,0,0,400,306,False);
    draw_graph(306);
  }

  for (n=0; n<MAX_IMAGES; n++)
    if (xdisplay[n].init==-1)
      contrast_bias(n,n);

  free( rgb );	/* RJM: added */
}



/*
  This function save a colormap
  to a data file.
*/
void save_color_file(char *filename)
{
  int out,n;
  char rgb[600];

  out=open(filename,O_WRONLY|O_TRUNC|O_CREAT,0666);

  for (n=start_color; n<=end_color; n++) {
    rgb[3*(n-(int)start_color)]=color[n].red>>8;
    rgb[3*(n-(int)start_color)+1]=color[n].green>>8;
    rgb[3*(n-(int)start_color)+2]=color[n].blue>>8;
  }

  write(out,rgb,600);

  close(out);
}
