/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  This file contains the functions for
  writting out a velocity tape.
*/

#include "xdisplayP.h"
#include "prototyping.h"
#include "ut_defs.h"
#include "io_defs.h"

#ifdef CRAY
#define BYTES 8
#else
#define BYTES 4
#endif

extern char vel_dir[1024];

static CB_PROTO( create_vel_paint_tapeCB )
static void save_paint(int id, float dx, float dz, int unit_flag, char *filename);

static Widget vel_text[4];


/*
  Prompt use for parameters for writing velocity tape from vel. paint window.
*/
CB_DECL ( create_vel_paintCB )
{
	static Widget	dialog_form = NULL;
	Widget		rowColumn;
	char		*header;
	int		rc, n, unit_flag=1;
	float		dx=0.0, dz=0.0;
	ClientData	*message;
	long		id = (long)client_data;	/* RJM: handle 64-bit architectures */

	if( dialog_form != NULL )
	{
		XtManageChild( dialog_form );
		return;
	}

	if (strlen(xdisplay[id].filename)!=0)
	{
		rc = amoco_read_line_header( &header, xdisplay[id].filename );
		if( rc < 0 )
			reportDSerror( rc );

		C_SAVER( (int*)header, "Dx1000", &n, 0 );
		dx=(float)n/1000.0;
		C_SAVER( (int*)header, "Dz1000", &n, 0 );
		dz=(float)n/1000.0;
		C_SAVER( (int*)header, "UnitFl", &unit_flag, 0 );
		if( header )
			free( header );
	}

	dialog_form=XmCreateFormDialogVa(w,"velPaintForm",
		XmNmarginHeight,10,
		XmNmarginWidth,10,
		XmNdialogTitle,XMstr("Velocity Tape Parameters"),
		XtNmanageChild, False,
		XmNhelpCallback,
			CBl( helpCB, "vel.help" ),
		NULL);
  
	rowColumn=XmCreateRowColumnVa(dialog_form,"row_column",
		XmNpacking,XmPACK_COLUMN,
		XmNnumColumns,1,
		XmNspacing,MAGIC_SPACING,
		XmNleftAttachment,XmATTACH_FORM,
		XmNtopAttachment,XmATTACH_FORM,
		XmNtopOffset,MAGIC_SPACING,
		NULL);
 
	(void)XmCreateLabelVa(rowColumn,"Velocity File:",NULL);
	(void)XmCreateLabelVa(rowColumn,"Dx",NULL);
	(void)XmCreateLabelVa(rowColumn,"Dz",NULL);
	(void)XmCreateLabelVa(rowColumn,"UnitFl (0=ft,1=m)",NULL);
 
	rowColumn=XmCreateRowColumnVa(dialog_form,"row_column",
		XmNpacking,XmPACK_COLUMN, 
		XmNnumColumns,1,
		XmNtopAttachment,XmATTACH_FORM,
		XmNleftAttachment,XmATTACH_WIDGET,
		XmNleftWidget,rowColumn,
		XmNrightAttachment,XmATTACH_FORM,
		NULL);

	vel_text[0]=XmCreateTextVa(rowColumn,"text",
		XmNvalue,vel_dir,
		XmNcolumns,50,
		NULL);

	sprintf(buf,"%f",dx);
	vel_text[1]=XmCreateTextVa(rowColumn,"text",
		XmNvalue,buf,
		XmNcolumns,50,
		NULL);

	sprintf(buf,"%f",dz);
	vel_text[2]=XmCreateTextVa(rowColumn,"text",
		XmNvalue,buf,
		XmNcolumns,50,
		NULL);
  
	sprintf(buf,"%d",unit_flag);
	vel_text[3]=XmCreateTextVa(rowColumn,"text",
		XmNvalue,buf,
		XmNcolumns,50,
		NULL);
  
	message=(ClientData *)malloc(sizeof(ClientData));
	message->w = dialog_form;
	message->id=id;
	(void)XmCreatePushButtonVa(dialog_form,"  OK  ",
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget,rowColumn,
		XmNtopOffset,10,
		XmNbottomAttachment,XmATTACH_FORM,
		XmNleftAttachment,XmATTACH_FORM,
		XmNactivateCallback,
			CBlist( show_watchCB, (XtPointer)dialog_form,
				create_vel_paint_tapeCB, (XtPointer)message,
				hide_watchCB, (XtPointer)dialog_form,
				NULL ),
		NULL);

	(void)XmCreatePushButtonVa(dialog_form,"Cancel",
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget,rowColumn,
		XmNtopOffset,10,
		XmNbottomAttachment,XmATTACH_FORM,
		XmNrightAttachment,XmATTACH_FORM,
		NULL);
 
	XtManageChild(dialog_form);

	/* set to prevent flashing between this and painter window */
	XSetWindowColormap(display,XtWindow(XtParent(dialog_form)),xdisplay[id].cmap);
#if defined(__sgi) || defined(sun)	/* added sun in case redirect disp from sun to sgi */
	/* flashing does not occur on SGI */
#endif
}
 

/*
  Generate velocity tape via genvel system call.
*/
static CB_DECL( create_vel_paint_tapeCB )
{
	ClientData	*message=(ClientData *)client_data;
	long		id;	/* RJM: handle 64-bit architectures */
	float		dx,dz;
	int		unit_flag;
	String		sis_out, str, tmp_file;

	id=message->id;

	XtVaGetValues( vel_text[0], XmNvalue, &sis_out,
		NULL );

	if ( strcmp(xdisplay[id].hostname,"sc")==0	/* remote a Cray? */
	|| strcmp(xdisplay[id].hostname,"sk")==0
	|| strcmp(xdisplay[id].hostname,"sf")==0 )
	{
		if ((int)strlen(strrchr(sis_out,'/'))>15)
		{
			error_message("Velocity tape filename too long for the Cray (14 characters maximum).");
			XtFree(sis_out);
			return;
		}
	}

	XtVaGetValues( vel_text[1], XmNvalue, &str,
		NULL );
	(void)sscanf(str,"%f",&dx);
	XtFree(str);

	XtVaGetValues( vel_text[2], XmNvalue, &str,
		NULL );
	(void)sscanf(str,"%f",&dz);
	XtFree(str);

	XtVaGetValues( vel_text[3], XmNvalue, &str,
		NULL );
	(void)sscanf(str,"%d",&unit_flag);
	XtFree(str);

	if (strcmp(xdisplay[id].hostname,"=")!=0)   /* data on remote machine */
	{
		tmp_file=tmpnam(NULL);
		save_paint(id,dx,dz,unit_flag,tmp_file);
		sprintf(buf,"rcp %s %s:%s",tmp_file,xdisplay[id].hostname,sis_out);
		fprintf(stderr, "%s\n",buf);
		system(buf);
		unlink(tmp_file);
	} else
		save_paint(id,dx,dz,unit_flag,sis_out);

	XtFree(sis_out);
 
	if( message->w != NULL )
		XtUnmanageChild( message->w );

	free(message);
}


/*
	Write velocities to SIS tape.
*/
void save_paint(int id, float dx, float dz, int unit_flag, char *filename)
{
  int out,x,y,width,height,trace_skip,bytes;
  int header[15000];
  float *trace;

  width=etrace[id];
  height=esamp[id];

#ifdef CRAY
    bytes=2592;
#else
    bytes=1004;
#endif

  C_LBOPEN(&out,filename,"w");
  
  C_SAVEW(header,"Dx1000",(char*)(long)(dx*1000.0),0);
  C_SAVEW(header,"Dz1000",(char*)(long)(dz*1000.0),0);
  sprintf(buf,"%d",(int)(dx+0.5));
  C_SAVEW(header,"GrpInt",buf,0);	/* RJM: GrpInt is 4 char ASCII string */
  C_SAVEW(header,"DptInt",(char*)(long)(dz+0.5),0); 	/* RJM: DptInt is int */
  C_SAVEW(header,"UnitFl",(char*)(long)unit_flag,0);
  C_SAVEW(header,"MinVel",(char*)(long)global_min,0);
  C_SAVEW(header,"MaxVel",(char*)(long)global_max,0);
  C_SAVEW(header,"NumRec",(char*)1L,0);
  C_SAVEW(header,"NumTrc",(char*)(long)width,0);
  C_SAVEW(header,"NumSmp",(char*)(long)height,0);
 
  C_SAVEW(header,"HlhEnt",(char*)0L,0);
  C_SAVEW(header,"HlhByt",(char*)4L,0);
  C_SAVEW(header,"SmpInt",(char*)2L,0);
  C_SAVEW(header,"Format",(char*)3L,0);
 
  C_WRTAPE(out,(char*)header,bytes);
 
#ifdef CRAY
  trace_skip=128;
#else
  trace_skip=64;
#endif
 
  trace=(float *)malloc((height+trace_skip)*sizeof(float));
 
  for (x=0; x<width; x++) {
    C_SAVEW((int*)trace,"TrcNum",(char*)(long)(x+1),1);
    C_SAVEW((int*)trace,"RecNum",(char*)1L,1); 
    for (y=0; y<height; y++) {
      *(trace+y+trace_skip)=(float)*(vel_tape+x+y*width);
/*    fprintf(stderr, "%d %d %d\n",x,y,*(vel_tape+x+y*width)); */
    }
    bytes=BYTES*(trace_skip+height);
    C_WRTAPE(out,(char*)trace,bytes); 
  }
  
  C_LBCLOS(out);
}
