/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  This file contains the functions for
  writting out a velocity tape.
*/

#include <memory.h>
#include "xdisplayP.h"
#include "prototyping.h"

extern Widget text_min;
extern Widget text_max;

static char filename_tmp[1024];


/*
  This function save the velocity
  parameters in a file along with
  the segments.
*/
void save_vel(char *filename)
{
  FILE *out;
  int m,n,temp_count,index_count,count=0,count_max=0;
  float record,trace,sample;

  out=fopen(filename,"w");
  if (out==NULL) {
    sprintf(buf,"Can't open file %s.",filename);
    error_message(buf);
    return;
  }

  fprintf(out,"Bodies: %d\n",v_index);
  fprintf(out,"Model_width: %d\n",real_width);
  fprintf(out,"Model_height: %d\n",real_height);
  fprintf(out,"Global_Min: %d\n",global_min);
  fprintf(out,"Global_Max: %d\n",global_max);
  for (index_count=0; index_count< v_index; index_count++) {
    if (model_body[index_count].x>=0) {
      fprintf(out,"X: %d\n",model_body[index_count].x);
      fprintf(out,"Y: %d\n",model_body[index_count].y);
      fprintf(out,"Velocity1: %d\n",model_body[index_count].vel1);
      fprintf(out,"X1: %d\n",model_body[index_count].x1);
      fprintf(out,"Y1: %d\n",model_body[index_count].y1);
      fprintf(out,"Velocity2: %d\n",model_body[index_count].vel2);
      fprintf(out,"X2: %d\n",model_body[index_count].x2);
      fprintf(out,"Y2: %d\n",model_body[index_count].y2);
      fprintf(out,"Min: %d\n",model_body[index_count].min);
      fprintf(out,"Max: %d\n",model_body[index_count].max);
      fprintf(out,"Oper: %s\n",model_body[index_count].operator);
      fprintf(out,"End:\n");
    }
  }
  fclose(out);

				/* Save segments in xxx.picks. */
  strcpy(buf,filename);
  strcat(buf,".picks");

  out=fopen(buf,"w");
  if (out==NULL) {
    error_message("Cannot open file.");
    return;
  }
				/* Count the number of segments. */
  for (n=0; n<MAX_PICKS; n++) 
    if (picks[n].init==-1) {
      count++;
      if (picks[n].size>count_max)
	count_max=picks[n].size;
    }

  fprintf(out,"Units %12.6f %12.6f %12.6f ",1.0,1.0,1.0);
  fprintf(out,"%5d %5d %5d ",1,model_width,model_height);
  fprintf(out,"Offset %12.6f %12.6f %12.6f ",0.0,0.0,0.0);
  fprintf(out,"Count %5d %5d\n",count,count_max);

  temp_count=1;
  for (n=0; n<MAX_PICKS; n++) {
    if (picks[n].init==-1) {
      if (picks[n].group==0) {
	fprintf(out,"Segment = %5d ",temp_count);
	if ((int)strlen(picks[n].label)>0)
	  fprintf(out,"Name %20s ",picks[n].label);
	else
	  fprintf(out,"Name %20s ","NO_PICK_NAME_HERE");
	fprintf(out," color = %5d",picks[n].color-start_pick_color);
	fprintf(out," picks = %5d\n",picks[n].size);
	temp_count++;
	for (m=0; m<picks[n].size; m++) {
	  record=(float)*(picks[n].record+m);
	  trace=(float)*(picks[n].trace+m);
	  sample=(float)*(picks[n].sample+m);
	  fprintf(out,"%12.6f %12.6f %12.6f\n",record,trace,sample);
	} 
      }
    }
  }
  fclose(out);
  
  return;
}


/*
  This function load the velocity
  parameters and segments.
*/
void load_vel(char *filename, long id, Boolean new_picks)
{
  FILE *in;
  int index_count;
  char *error,command[1024];
  long tmp;

  strcpy(filename_tmp,filename);

  in=fopen(filename_tmp,"r");
  if (in==NULL) {
    error_message("Can't open file.");
    return;
  }

  if (model_mode!=0) {
    tmp = model_mode;	/* RJM: int to long for 64-bit machines */
    delete_windowCB(NULL,(XtPointer)tmp,NULL);
    delete_body_list();
    quit_model();
  }

  v_index=0;
  error=fgets(buf,1024,in);
  sscanf(buf,"%s",command);
  while (error!=NULL) {
    if (strcmp(command,"End:")==0)
      v_index++;
    error=fgets(buf,1024,in);
    sscanf(buf,"%s",command);
  }

  if(model_body)
    free((char*)model_body); 
  if(model_undo)
    free((char*)model_undo); 
  model_body=(struct model_struct *)malloc(v_index*sizeof(struct model_struct));
  model_undo=(struct model_struct *)malloc(v_index*sizeof(struct model_struct));
  for (index_count=0; index_count<v_index; index_count++) {
    model_body[index_count].x=-1;
    model_body[index_count].y=0;
    model_body[index_count].vel1=0;
    model_body[index_count].x1=0;
    model_body[index_count].y1=0;
    model_body[index_count].vel2=0;
    model_body[index_count].x2=0;
    model_body[index_count].y2=0;
    model_body[index_count].min=global_min;
    model_body[index_count].max=global_max;
    strcpy(model_body[index_count].operator,"=");
  }

  rewind(in);

  index_count=0;
  error=fgets(buf,1024,in);
  sscanf(buf,"%s",command);
                                /* Load all the parameters. */
  while (error!=NULL) {
    if (strcmp(command,"Global_Min:")==0)
      sscanf(buf,"Global_Min: %d",&global_min);
    if (strcmp(command,"Global_Max:")==0)
      sscanf(buf,"Global_Max: %d",&global_max);
    if (strcmp(command,"X:")==0)
      sscanf(buf,"X: %d",&model_body[index_count].x);
    if (strcmp(command,"Y:")==0)
      sscanf(buf,"Y: %d",&model_body[index_count].y);
    if (strcmp(command,"Velocity1:")==0)
      sscanf(buf,"Velocity1: %d",&model_body[index_count].vel1);
    if (strcmp(command,"X1:")==0)
      sscanf(buf,"X1: %d",&model_body[index_count].x1);
    if (strcmp(command,"Y1:")==0)
      sscanf(buf,"Y1: %d",&model_body[index_count].y1);
    if (strcmp(command,"Velocity2:")==0)
      sscanf(buf,"Velocity2: %d",&model_body[index_count].vel2);
    if (strcmp(command,"X2:")==0)
      sscanf(buf,"X2: %d",&model_body[index_count].x2);
    if (strcmp(command,"Y2:")==0)
      sscanf(buf,"Y2: %d",&model_body[index_count].y2);
    if (strcmp(command,"Min:")==0)
      sscanf(buf,"Min: %d",&model_body[index_count].min);
    if (strcmp(command,"Max:")==0)
      sscanf(buf,"Max: %d",&model_body[index_count].max);
    if (strcmp(command,"Oper:")==0)
      sscanf(buf,"Oper: %s",model_body[index_count].operator);
    if (strcmp(command,"End:")==0)
      index_count++;
    error=fgets(buf,1024,in);
    sscanf(buf,"%s",command);
  }
  fclose(in);

  if (new_picks) {
    new_picksCB(0,NULL,NULL);
    strcat(filename_tmp,".picks");
    load_picks(id,filename_tmp,0);
  }

  if (model_mode!=0) {
    scale_range(global_min,global_max,model_mode);

    sprintf(buf,"%d",global_min);
    XtSetArg(args[0],XmNvalue,buf);
    XtSetValues(text_min,args,1);
  
    sprintf(buf,"%d",global_max);
    XtSetArg(args[0],XmNvalue,buf);
    XtSetValues(text_max,args,1);

    tmp = model_mode;	/* RJM: int to long for 64-bit machines */
    regenerateCB(NULL,(XtPointer)tmp,0);
  }
}



/*
  This function regenerate the
  bodies for the model.
*/
typedef struct {                /* window: a discrete 2-D rectangle */
    int x0, y0;                 /* xmin and ymin */
    int x1, y1;                 /* xmax and ymax (inclusive) */
} Window_Seed;

typedef Pixel Pixel_Seed;


void regenerateCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  extern void fill(int, int, Window_Seed*, Pixel_Seed, int);

  Window_Seed	win;
  int x,y,color,index_count;
  int old_v_index,size;
  long id, locate;	/* RJM: 64-bit machines */
 
  id=(long)client_data;

  xdisplay[id].mode=19;
  set_mode_label(19,id);

  start_processing(xdisplay[id].form);
  show_watchCB(xdisplay[id].form,(XtPointer)xdisplay[id].form,NULL);
  reset_velCB(NULL,(XtPointer)id,NULL);

  win.x0=0; win.x1=model_width-1;
  win.y0=0; win.y1=model_height-1;
 
  for (x=0; x<model_width*model_height; x++)
    *(model_index+x)=0;
 
  memset((char *)vel_raster,(int)(white&0xffffffff),model_width*model_height*sizeof(char));
  
  expose_picks(id);

/*color=data_to_pixel(0,id); RJM */
  color=data_to_pixel((int)(white&0xffffffff),id);
  old_v_index=v_index;
  v_index=0;
  locate=0;
  while( (locate = (long)memchr( (char *)(vel_raster+locate),
			(int)(white&0xffffffff), (model_width*model_height)-locate )) != 0 ) { 
    v_index++;
    locate-=(long)vel_raster;
    fill((int)(locate%model_width),(int)(locate/model_width),&win,color,v_index);
  }

  if (v_index>old_v_index) 
    size=v_index;
  else
    size=old_v_index;

				/* Resize the parameter space. */
  if (model_body!=NULL)
    model_body=(struct model_struct *)realloc(model_body,size*sizeof(struct model_struct));
  else
    model_body=(struct model_struct *)malloc(size*sizeof(struct model_struct));

  if (model_undo!=NULL)
    model_undo=(struct model_struct *)realloc(model_undo,size*sizeof(struct model_struct));
  else
    model_undo=(struct model_struct *)malloc(size*sizeof(struct model_struct));

				/* Initialize an new space. */
  for (index_count=old_v_index; index_count<v_index; index_count++) {
    model_body[index_count].x=-1;
    model_body[index_count].y=0;
    model_body[index_count].vel1=0;
    model_body[index_count].x1=0;
    model_body[index_count].y1=0;
    model_body[index_count].vel2=0;
    model_body[index_count].x2=0;
    model_body[index_count].y2=0;
    model_body[index_count].min=global_min;
    model_body[index_count].max=global_max;
    strcpy(model_body[index_count].operator,"=");
  }

				/* Zero out all the parameter after
				   copying them in temporary storage. */
  for (index_count=0; index_count<size; index_count++) {
    copy_body(index_count,index_count);
    model_body[index_count].x=-1;
    model_body[index_count].y=0;
    model_body[index_count].vel1=0;
    model_body[index_count].x1=0;
    model_body[index_count].y1=0;
    model_body[index_count].vel2=0;
    model_body[index_count].x2=0;
    model_body[index_count].y2=0;
    model_body[index_count].min=global_min;
    model_body[index_count].max=global_max;
    model_body[index_count].start=-1;
    model_body[index_count].end=-1;
    strcpy(model_body[index_count].operator,"=");
  }

				/* Find the start of each body. */
  for (x=0; x<model_width*model_height; x++) 
    if (*(model_index+x)-1>=0) {
      if (model_body[(*(model_index+x)-1)].start==(-1)) 
	model_body[*(model_index+x)-1].start=x;
    }
				/* Find the end of each body. */
  for (x=model_width*model_height-1; x>=0; x--)
    if (*(model_index+x)-1>=0) {
      if (model_body[(*(model_index+x)-1)].end==(-1))
	model_body[*(model_index+x)-1].end=x;
    }
				/* Bodies are probably in a different
				   order so match up the parameters 
				   with the body according to x,y. */
  for (index_count=0; index_count<size; index_count++) {
    if (model_undo[index_count].x!=-1) {
      color=data_to_pixel(model_body[index_count].vel1,id);
      x=model_undo[index_count].x/xdisplay[id].pick_x;
      y=model_undo[index_count].y/xdisplay[id].pick_y;
      if (xdisplay[id].flip_flag) {
	undo_body(modelread(model_width-x/fill_skip,
			    y/fill_skip)-1,index_count);
      } else {
	undo_body(modelread(x/fill_skip,y/fill_skip)-1,index_count);
      }
    }
  }
				/* Generate the velocity for each body. */
  for (index_count=0; index_count<v_index; index_count++)
      generate_vel(index_count,id); 

  expose_image(id);

				/* Call body_listCB to update body_list. */
  body_listCB(NULL,(XtPointer)id,0);

  xdisplay[id].mode=18;
  set_mode_label(18,id);

  hide_watchCB(xdisplay[id].form,(XtPointer)xdisplay[id].form,NULL);
  stop_processing();
}



/*
  This function copies the model_body
  structure into temporary storage.
*/
void copy_body(int i1, int i2)
{
  model_undo[i1].x=model_body[i2].x;
  model_undo[i1].y=model_body[i2].y;
  model_undo[i1].vel1=model_body[i2].vel1;
  model_undo[i1].x1=model_body[i2].x1;
  model_undo[i1].y1=model_body[i2].y1;
  model_undo[i1].vel2=model_body[i2].vel2;
  model_undo[i1].x2=model_body[i2].x2;
  model_undo[i1].y2=model_body[i2].y2;
  model_undo[i1].min=model_body[i2].min;
  model_undo[i1].max=model_body[i2].max;
  strcpy(model_undo[i1].operator,model_body[i2].operator);
}



/*
  This function copy the model_body
  structure back from temporary storage.
*/
void undo_body(int i1, int i2)
{
  model_body[i1].x=model_undo[i2].x;
  model_body[i1].y=model_undo[i2].y;
  model_body[i1].vel1=model_undo[i2].vel1;
  model_body[i1].x1=model_undo[i2].x1;
  model_body[i1].y1=model_undo[i2].y1;
  model_body[i1].vel2=model_undo[i2].vel2;
  model_body[i1].x2=model_undo[i2].x2;
  model_body[i1].y2=model_undo[i2].y2;
  model_body[i1].min=model_undo[i2].min;
  model_body[i1].max=model_undo[i2].max;
  strcpy(model_body[i1].operator,model_undo[i2].operator);
}
