/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
	File	:	hlhis2c.c
	Author	:	Dean Kopesky
	Date	:	May 1988
	Changed	:	March 1990
			Changed us[ci]ctc to US[CI]CTC.
			June 1991
			Changed unpack() to unpackbytes().
	Changed	:	August 1992
			Added code for intel format conversion

	Compile with either -D IEEE or -D IBM.

	Hlhs2c translates the historical line header of an SIS file from
	SUN to CRAY-2 format.

	Hlhi2c translates the historical line header of an SIS file from
	IBM to CRAY-2 format.

	Hlhs2in translates the historical line header of an SIS file from
	SUN to INTEL format.

*/


#include	<stdio.h>

#ifdef CRAY
#ifdef	IEEE
#define	MODULE	"hlhs2c"
#endif
#ifdef	IBM
#define	MODULE	"hlhi2c"
#endif
#endif

#if ( BYTE_ORDER == LITTLE_ENDIAN )
#ifdef	IEEE
#define	MODULE	"hlhs2in"
#endif
#endif

#define	ENOTENUF	1	/* not enough data in input record */
#define	EOVRFLOW	2	/* buffer overflow on output buffer */
#define	EBYTES		3	/* byte count in record is wrong */
#define	EENTRIES	4	/* entry count in record is wrong */

#ifdef  CRAY
#define WORDSIZE 8
#define HALFWORDSIZE 8
#else
#ifdef  __i860
#define WORDSIZE 4
#define HALFWORDSIZE 2
#endif
#endif
 
void	exit();
char	*memcpy();


#ifdef  CRAY
#ifdef	IEEE
void hlhs2c( inbuf, inlen, inptr, outbuf, outlen, outptr )
#endif
#ifdef	IBM
void hlhi2c( inbuf, inlen, inptr, outbuf, outlen, outptr )
#endif

#else
#if ( BYTE_ORDER == LITTLE_ENDIAN )
#ifdef	IEEE
void hlhs2in( inbuf, inlen, inptr, outbuf, outlen, outptr )
#endif
#endif
#endif

	char *	inbuf;
	int	inlen;
	int	inptr;
	char *	outbuf;
	int	outlen;
	int *	outptr;

{
	int	nentries, nbytes, nbptr;
	int	entry, length, rounded;
	int	c1 = 1, c2 = 2;
	int	wordsize = WORDSIZE;

#ifdef DEBUG
 fprintf(stderr,"entering %s: inlen = %d, inptr = %d",MODULE,inlen,inptr);
 fprintf(stderr,", outbuf = %08x, outlen = %d, outptr = %d (%08x)\n",
	outbuf,outlen,*outptr,outptr);
#endif
	if ( inlen - inptr + 1 < 4 )
	{
		fprintf( stderr, "%s:  not enough data in input record.\n",
			MODULE );
		exit( ENOTENUF );
	}
#ifdef  CRAY
	if ( outlen - *outptr < 2 * WORDSIZE )
#endif
#if ( BYTE_ORDER == LITTLE_ENDIAN )
	if ( outlen - *outptr < 2 * HALFWORDSIZE )
#endif
	{
		fprintf( stderr, "%s:  buffer overflow on output buffer.\n",
			MODULE );
		exit( EOVRFLOW );
	}

#ifdef  CRAY
	USICTC( inbuf, &inptr, &nentries, &c1, &c2 );
	memcpy( outbuf + *outptr, ( char * ) &nentries, WORDSIZE );
#else
#if ( BYTE_ORDER == LITTLE_ENDIAN )
#ifdef DEBUG
	fprintf(stderr,"%s: decoding number of entries\n",MODULE);
	fprintf(stderr,"number in inbuf = %d ( %04x )\n",
	  *(short *)( inbuf + inptr - 1 ),*(short *)( inbuf + inptr - 1 ));
#endif
	memcpy( outbuf + *outptr, inbuf + inptr - 1, 2 );
	HTOCS( outbuf + *outptr, 1 );
#ifdef DEBUG
 fprintf(stderr,"hlhs2in: write nentries to output buffer at location %08x",
		(outbuf + *outptr));
 fprintf(stderr," value = %d\n",
		*(short *)(outbuf + *outptr));
#endif
	nentries = (long) *(short *)( outbuf + *outptr);
#ifdef DEBUG
	fprintf(stderr,"nentries = %d\n",nentries);
#endif
#endif
#endif
	inptr += 2;
	*outptr += HALFWORDSIZE;

#ifdef  CRAY
	USICTC( inbuf, &inptr, &nbytes, &c1, &c2 );
#else
#if ( BYTE_ORDER == LITTLE_ENDIAN )
#ifdef DEBUG
  fprintf(stderr,"%s: decoding number of bytes\n",MODULE);
  fprintf(stderr,"number in inbuf = %d ( %04x )\n",
	*(short *)( inbuf + inptr - 1 ),*(short *)( inbuf + inptr - 1 ));
#endif
	memcpy( (char *)(&nbytes) + 2, inbuf + inptr - 1, 2 );
	HTOCL( &nbytes, 1 );
#endif
#endif

#ifdef DEBUG
	fprintf(stderr,"%s: nbytes = %d\n",MODULE,nbytes);
#endif
	nbptr = *outptr;
	inptr += 2;
	*outptr += HALFWORDSIZE;

	if ( nentries == 0 )
	{
		if ( inlen - inptr + 1 != 0 )
		{
			fprintf( stderr,
				"%s:  byte count in record is wrong.\n",
				MODULE );
			exit( EBYTES );
		}
		nbytes = 2 * HALFWORDSIZE;
		memcpy( outbuf + nbptr, ( char * ) &nbytes, HALFWORDSIZE );
		return;
	}

	if ( nbytes - 4 != inlen - inptr + 1 )
	{
		fprintf( stderr, "%s:  byte count in record is wrong.\n",
			MODULE );
		exit( EBYTES );
	}

	nbytes = 2 * HALFWORDSIZE;

	for ( entry = 1; entry <= nentries; entry++ )
	{
		if ( inlen - inptr + 1 < 2 )
		{
			fprintf( stderr,
				"%s:  entry count in record is wrong.\n",
				MODULE );
			exit( EENTRIES );
		}
		if ( outlen - *outptr < HALFWORDSIZE )
		{
		fprintf( stderr, "%s:  buffer overflow on output buffer.\n",
				MODULE );
			exit( EOVRFLOW );
		}
#ifdef DEBUG
	printf("length of entry before rotating = %04x\n",
		* (short *)( inbuf + inptr - 1 ));
#endif

#ifdef  CRAY
		USICTC( inbuf, &inptr, &length, &c1, &c2 );
		rounded = ( ( length - 1 ) / 8 + 1 ) * 8;
		memcpy( outbuf + *outptr, ( char * ) &length, WORDSIZE );
#else
#if ( BYTE_ORDER == LITTLE_ENDIAN )
		memcpy( outbuf + *outptr, inbuf + inptr - 1, 2 );
		HTOCS( outbuf + *outptr, 1 );
		length = (int) *(short *)( outbuf + *outptr );
		rounded = length;
#endif
#endif
		inptr += 2;
		*outptr += HALFWORDSIZE;
#ifdef  DEBUG
	fprintf(stderr,"%s: length of entry #%d = %d ;",MODULE,entry,length);
#endif

		if ( inlen - inptr + 1 < length )
		{
		fprintf( stderr, "%s:  not enough data in input record.\n",
				MODULE );
			exit( ENOTENUF );
		}
		if ( outlen - *outptr < rounded )
		{
		fprintf( stderr, "%s:  buffer overflow on output buffer.\n",
				MODULE );
			exit( EOVRFLOW );
		}
#ifdef  CRAY
#ifdef	IEEE
		unpackbytes(inbuf,inptr,outbuf+*outptr,length,WORDSIZE,' ');
#endif
#ifdef	IBM
		USCCTC( inbuf, &inptr, outbuf + *outptr, &length, &wordsize );
#endif
#else
#if ( BYTE_ORDER == LITTLE_ENDIAN )
		memcpy( outbuf + *outptr, inbuf + inptr - 1, length );
#endif
#endif
		inptr += length;
		*outptr += rounded;
		nbytes += rounded + HALFWORDSIZE;
#ifdef DEBUG
		fprintf(stderr,"nbytes = %d\n",nbytes);
#endif
	}

#ifdef  CRAY
	memcpy( outbuf + nbptr, ( char * ) &nbytes, WORDSIZE );
#else
#if ( BYTE_ORDER == LITTLE_ENDIAN )
	*(short *)(outbuf + nbptr) = nbytes;
#ifdef DEBUG
  fprintf(stderr,"number in outbuf = %d ( %04x )\n",
	*(short *)( outbuf + nbptr ),*(short *)( outbuf + nbptr ));
#endif
#endif
#endif
}
