/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* ---------------------------------------------------------------------------

	File	:	revb.c
	Author	:	Dean Kopesky
	Company	:	Minnesota Supercomputer Center, Inc.
	Date	:	May 1988
	Changed	:	March 1990
			Changed usict[ci] to USICT[CI].
	Changed	:	August 1991
			Eliminated calls to USICT[CI].  Should now work on any
			machine that uses significant-byte-first ordering.

--------------------------------------------------------------------------- */


#include	<stdio.h>


static char	help[] =
"\
\n\
Revb reads a simple blocked (4) file, reblocks it, and writes to stdout\n\
an IBM RECFM=VB file optionally nested in a simple blocked (4) file.\n\
\n\
Usage:\n\
\n\
	revb [ -H ] [ -ns ] [ simple(4) ]\n\
\n\
	-H		Prints this help listing.\n\
	-ns		Specifies that the output should not be a simple\n\
			blocked file.\n\
	simple(4)	The simple blocked (4) input file (default is\n\
			stdin).\n\
\n\
";


#define	MODULE	"revb"
#define	PARAMS	"[ -H ] [ -ns ] [ simple(4) ]"

#define	ENORMAL	0	/* normal termination */
#define	EUSAGE	1	/* usage error */
#define	EBADIN	2	/* cannot open input file */
#define	ERDERR	3	/* premature EOF or read error on stdin */
#define	EWRERR	4	/* write error on stdout */
#define	EINCOMP	5	/* incomplete input file */

#define	BLKLEN	( 32760-4 )
#define	BFRLEN	( 512*1024 )


#define	GETINT(dest,source,size) \
{ \
	(dest) = 0; \
	memcpy( (char *) &(dest) + sizeof(dest) - (size), (source), (size) ); \
}

#define	PUTINT(source,dest,size) \
	memcpy( (dest), (char *) &(source) + sizeof(source) - (size), (size) );


void	exit();
void	getargs();
int	getrec();
char *	memcpy();
char *	memset();
void	putblk();
void	USICTC();
void	USICTI();


main( argc, argv )

	int	argc;
	char *	argv[];

{
	int	ns;	/* set to 1 iff -ns option is used */
	int	ifd;	/* input file descriptor */

	char	inbuf[ BLKLEN ];
	char	outbuf[ BLKLEN ];
	int	reclen, outend;

	getargs( argc, argv, &ns, &ifd );

	outend = 0;

	while ( ( reclen = getrec( ifd, inbuf ) ) > 0 )
	{
		if ( outend + reclen > BLKLEN )
		{
			putblk( ns, outbuf, outend );
			outend = 0;
		}
		reclen += 4;
		PUTINT( reclen, outbuf + outend, 2 );
		memset( outbuf + outend + 2, '\0', 2 );
		memcpy( outbuf + outend + 4, inbuf, reclen - 4 );
		outend += reclen;
	}

	if ( outend > 0 ) putblk( ns, outbuf, outend );
	putblk( ns, outbuf, 0 );  /* to flush the buffer */

	return( ENORMAL );
}


int getrec( ifd, recptr )

	int	ifd;
	char *	recptr;

{
	static char	inbuf[ BFRLEN ];
	static int	inbeg = 0, inend = 0;

	char	tmpbuf[ BFRLEN ];
	int	reclen, nread;

	if ( inend - inbeg < 4 )
	{
		memcpy( tmpbuf, inbuf + inbeg, inend - inbeg );
		memcpy( inbuf, tmpbuf, inend - inbeg );
		inend -= inbeg;
		inbeg = 0;

		nread = readpipe( ifd, inbuf + inend, BFRLEN - inend );
		inend += nread;

		if ( nread == 0 )
		{
			return( 0 );
		}
		if ( nread < 0 )
		{
		fprintf( stderr, "%s:  read error on input file.\n", MODULE );
			exit( ERDERR );
		}
	}

	GETINT( reclen, inbuf + inbeg, 4 );
	inbeg += 4;

	if ( inbeg + reclen > inend )
	{
		memcpy( tmpbuf, inbuf + inbeg, inend - inbeg );
		memcpy( inbuf, tmpbuf, inend - inbeg );
		inend -= inbeg;
		inbeg = 0;

		nread = readpipe( ifd, inbuf + inend, BFRLEN - inend );
		inend += nread;

		if ( nread == 0 )
		{
		fprintf( stderr, "%s:  incomplete input file.\n", MODULE );
			exit( EINCOMP );
		}
		if ( nread < 0 )
		{
		fprintf( stderr, "%s:  read error on input file.\n", MODULE );
			exit( ERDERR );
		}
	}

	memcpy( recptr, inbuf + inbeg, reclen );
	inbeg += reclen;

	return( reclen );
}


void putblk( ns, blkptr, blklen )

	int	ns;
	char *	blkptr;
	int	blklen;

{
	static char	outbuf[ BFRLEN ];
	static int	outend = 0;

	if ( blklen == 0 )
	{
		if ( write( 1, outbuf, ( unsigned ) outend ) != outend )
		{
		fprintf( stderr, "%s:  write error on stdout.\n", MODULE );
			exit( EWRERR );
		}
		outend = 0;
		return;
	}

	blklen += 4;
	
	if ( outend + blklen + 4 > BFRLEN )
	{
		if ( write( 1, outbuf, ( unsigned ) outend ) != outend )
		{
		fprintf( stderr, "%s:  write error on stdout.\n", MODULE );
			exit( EWRERR );
		}
		outend = 0;
	}

	if ( ! ns )
	{
		PUTINT( blklen, outbuf + outend, 4 );
		outend += 4;
	}

	PUTINT( blklen, outbuf + outend, 2 );
	memset( outbuf + outend + 2, '\0', 2 );
	memcpy( outbuf + outend + 4, blkptr, blklen - 4 );
	outend += blklen;
}


void getargs( argc, argv, ns, ifd )

	int	argc;
	char *	argv[];
	int *	ns;
	int *	ifd;

{
	*ns = 0;
	*ifd = 0;

	switch ( argc )
	{
		case 1:
			break;
		case 2:
			if ( strcmp( argv[ 1 ], "-H" ) == 0 )
			{
				fprintf( stderr, "%s", help );
				exit( ENORMAL );
			}
			if ( strcmp( argv[ 1 ], "-ns" ) == 0 )
			{
				*ns = 1;
			}
			else if ( ( *ifd = open( argv[ 1 ], 0 ) ) < 0 )
			{
fprintf( stderr, "%s:  cannot open input file (%s).\n", MODULE, argv[ 1 ] );
				exit( EBADIN );
			}
			break;
		case 3:
			if ( strcmp( argv[ 1 ], "-ns" ) == 0 )
			{
				*ns = 1;
			}
			else
			{
			fprintf( stderr, "usage: %s %s\n", MODULE, PARAMS );
				exit( EUSAGE );
			}
			if ( ( *ifd = open( argv[ 2 ], 0 ) ) < 0 )
			{
fprintf( stderr, "%s:  cannot open input file (%s).\n", MODULE, argv[ 2 ] );
				exit( EBADIN );
			}
			break;
		default:
			fprintf( stderr, "usage:  %s %s\n", MODULE, PARAMS );
			exit( EUSAGE );
			break;
	}
}
